///////////////////////////////////////////////////////////////////////////////
// Copyright     2009, 2012 Sony Corporation
// Copyright (C) 2012 Sony Computer Entertainment Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
// 
///////////////////////////////////////////////////////////////////////////////
#include "WebKitSilkPrefix.h"
#include "JSDOMWindow.h"
#include "CEJSExternal.h"
#include "ICEHtmlBrowserManager.h"

#include <runtime/JSLock.h>
#include "UString.h"
#include "PrototypeFunction.h"
#include "runtime/JSValue.h"
#include "runtime/FunctionConstructor.h"
#include "../JSUtilityFunc.h"
#include "JSDOMBinding.h"
#include <runtime/JSGlobalObject.h>
#include <runtime/ObjectPrototype.h>

#include "CEWebKit.h"
#include "ICEHtmlWebKit.h"
#include "CEJSPrivilegeExtObjectDeclaration.h"
#include "CEJSPrivilegeExtObject.h"
#include "CEJSWindowExternal.h"
#include "CEHtmlCredentialGlue.h"

// callback from JS engine
static JSC::JSValue JSC_HOST_CALL _invokeSystemMethod(
	JSC::ExecState* exec, JSC::JSObject* obj, JSC::JSValue thisValue, const JSC::ArgList& arglist);
///////////////////////////////////////////////////////////////////////////
// public function
///////////////////////////////////////////////////////////////////////////
void AddWindowExternalJavascriptInterface(WebCore::JSDOMWindow* window)
{
	JSC::JSLock lock(false);
	JSC::ExecState* exec = window->globalExec();
	//JSC::JSObject* jsext = new (exec) JSExternal(exec->lexicalGlobalObject()->emptyObjectStructure());
	JSC::JSObject* jsext = new (exec) JSExternal(window, JSExternal::createStructure(JSExternal::createPrototype(exec)));
	if( jsext )
	{
		addStaticProperty(window, JSC::Identifier(exec, "external"), jsext);

		JSC::Identifier symbol = JSC::Identifier(exec, "system");
		JSC::PrototypeFunction* propFunc = new (exec) JSC::PrototypeFunction(exec, 0, symbol, _invokeSystemMethod);
		jsext->putDirectFunction(exec, propFunc, JSC::DontEnum | JSC::ReadOnly | JSC::DontDelete);
	}
}

static JSC::JSValue JSC_HOST_CALL _invokeSystemMethod(
	JSC::ExecState* exec, JSC::JSObject* obj, JSC::JSValue thisValue, const JSC::ArgList& arglist)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	JSExternal* thisObj = static_cast<JSExternal*>(thisValue.toObject(exec));

	JSC::JSValue retValue;
	hr = JSPrivilegeExtObject::invokeSystemMethod(exec, arglist, retValue, thisObj->getPropertyId());
	if (CESucceeded(hr))
		return(retValue);

	return JSC::jsUndefined();
}
//////////////////////////////////////////////////////////////////////////////
// CEJSExternal
// this is for window.external object
//////////////////////////////////////////////////////////////////////////////
JSExternal::JSExternal(WebCore::JSDOMWindow* window, PassRefPtr<JSC::Structure> structure) 
	: JSC::JSObject(structure), _useDeclareForeParent(false), _propertyId(0)
{
	CEASSERT(window);
	_member = new _member_def;
	_member->_window = window;
	_member->_privilegeExtObjectDeclaration = 0;

	_member->_privilegeExtOblectCollection = new CEJSPrivilegeExtObjectCollection(this);
	if( _member->_privilegeExtOblectCollection )
	{
		_member->_privilegeExtOblectCollection->init();
	}

	CEHtmlCredentialGlue* credentialGlue = new CEHtmlCredentialGlue(this);
	if (credentialGlue)
	{
		_member->_credential = CEHtmlCredentialGlue::toICEHtmlCredential(credentialGlue);
	}

#	if defined(ENABLE_TEST_CODE)
	static bool initialized = false;
	if (!initialized)
	{
		initialized = true;
		CEWebKitImpl::getInstance()->setJSPrivilegeExtObjectListener(&jsPrivilegeExtObjectListener);
	}
#	endif //defined(ENABLE_TEST_CODE)
}

JSExternal::~JSExternal()
{
	if( _member )
	{
		if( _member->_privilegeExtObjectDeclaration )
		{
			delete _member->_privilegeExtObjectDeclaration;
			_member->_privilegeExtObjectDeclaration = 0;
		}
		if( _member->_privilegeExtOblectCollection )
		{
			delete _member->_privilegeExtOblectCollection;
			_member->_privilegeExtOblectCollection = 0;
		}
		delete _member;
		_member = 0;
	}
}

const JSC::ClassInfo JSExternal::s_info = {"External", 0, 0, 0};

JSC::JSObject* JSExternal::createPrototype(JSC::ExecState* exec)
{
	return new (exec) JSExternalPrototype(JSExternalPrototype::createStructure(exec->lexicalGlobalObject()->objectPrototype()));
}

void JSExternal::clearOldDeclaredObject()
{
	if (_member && _member->_privilegeExtObjectDeclaration)
	{
		delete _member->_privilegeExtObjectDeclaration;
		_member->_privilegeExtObjectDeclaration = 0;
	}
}

bool JSExternal::getOwnPropertySlot(
	JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::PropertySlot& slot)
{
	CEComDebugPrintf("JSExternal::getOwnPropertySlot: called. name=%*s\n", propertyName.size(), propertyName.data());

	bool ret = false;

	// search own slot
	JSC::JSValue val = getDirect(propertyName);
	if( val == JSC::JSValue() )
	{
		// need handling the name?
		if ( canGetItemsForName(exec, propertyName) )
		{
			// found. set custom getter
			JSC::JSValue extObjOut;
			CEHResult hr = _member->_privilegeExtOblectCollection->getPrivilegeExtObject(exec, propertyName, extObjOut);
			if( CESucceeded(hr) && extObjOut != JSC::JSValue() )
			{
				CEComDebugPrintf("  JSExternal::getOwnPropertySlot: Found it\n");
				// save it to own object
				putDirect(propertyName, extObjOut);
				slot.setValue(extObjOut);
				ret = true;
			}
			else
			{
				// not found
				ret = false;
			}
		}
		else
		{
			// not need handling the name.
			CEComDebugPrintf("  JSExternal::getOwnPropertySlot: not need handling the name.\n");
			ret = false;
		}
	}
	else if( true == val.isCell() )
	{
		// need handling the name?
		if ( canGetItemsForName(exec, propertyName) )
		{
			// found. set custom getter
			JSC::JSValue extObjOut;
			CEHResult hr = _member->_privilegeExtOblectCollection->getPrivilegeExtObject(exec, propertyName, extObjOut);
			if( CESucceeded(hr) && extObjOut != JSC::JSValue() )
			{
				CEComDebugPrintf("  JSExternal::getOwnPropertySlot: Found it\n");
				// return own object property value
				slot.setValue(val);
				ret = true;
			}
			else
			{
				// not found
				ret = false;
			}
		}
		else
		{
			// not need handling the name.
			CEComDebugPrintf("  JSExternal::getOwnPropertySlot: not need handling the name.\n");
			ret = false;
		}
	}
	else
	{
		// return own object property value
		slot.setValue(val);
		ret = true;
	}
	return(ret);
}

bool JSExternal::canGetItemsForName(
	JSC::ExecState* exec, const JSC::Identifier& propertyName)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	bool isReadOnly = false;

	if( !_member->_privilegeExtOblectCollection )
	{
		return(false);
	}

	hr = _member->_privilegeExtOblectCollection->hasPrivilegeExtObject(exec, propertyName, isReadOnly);
	if (CESucceeded(hr))
	{
		return(true);
	}

	// not handle it
	return(false);
}

void JSExternal::put(
	JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::JSValue value, JSC::PutPropertySlot& slot)
{
	JSC::JSValue val = value; // do not delete
	if ( canGetItemsForName(exec, propertyName) )
	{
		CEComDebugPrintf("JSExternal::put: set value, name=%*s, value=%*s\n", 
			propertyName.size(), propertyName.data(),  val.toString(exec).size(), val.toString(exec).data());
		return;
	}
	else
	{
		CEComDebugPrintf("JSExternal::put: set value to object. name=%*s, value=%*s\n", 
			propertyName.size(), propertyName.data(),  val.toString(exec).size(), val.toString(exec).data());
		putDirect(propertyName, value);
	}
	return;
}

CEHResult JSExternal::getCredential(CEComICEHtmlCredentialRef& credential)
{
	if (_member) {
		credential = _member->_credential;
		return CE_S_OK;
	} else {
		CEASSERT(0 && "no memory or not initialized?");
		return(CE_SILK_ERR_OPERATION_FAILED);
	}
}

void JSExternal::setPropertyId(INT32 propertyId)
{
	_propertyId = propertyId;
}

INT32 JSExternal::getPropertyId()
{
	return _propertyId;
}

CEHResult JSExternal::getJSPrivilegeExtObjectDeclaration(
	JSC::ExecState* exec, CEJSPrivilegeExtObjectDeclaration** pDeclarationOut)
{
	CEHResult hr = CE_S_OK;
	if (!_member->_privilegeExtObjectDeclaration)
	{
		hr = CEJSPrivilegeExtObjectDeclaration::create(exec, this, _member->_privilegeExtObjectDeclaration);
	}
	if( CESucceeded(hr) && _member->_privilegeExtObjectDeclaration )
	{
		//_member->_privilegeExtObjectDeclaration->AddRef();
		*pDeclarationOut = _member->_privilegeExtObjectDeclaration;
	}
	return hr;
}

///////////////////////////////////////////////////////////////////////////
// implement external prototype
///////////////////////////////////////////////////////////////////////////
const JSC::ClassInfo JSExternalPrototype::s_info = { "ExternalPrototype", 0, 0, 0 };

bool JSExternalPrototype::getOwnPropertySlot(JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::PropertySlot& slot)
{
	//return WebCore::getStaticFunctionSlot<JSC::JSObject>(exec, 0, this, propertyName, slot);
	// always false.
	return(false);
}

/////////////////////////////////
// CEJSExternalAppCallback implement
/////////////////////////////////
JSC::ExecState* CEJSExternalAppCallback::_exec = 0;

CEHResult CEJSExternalAppCallback::_getListener(CEComICEJSPrivilegeExtObjectListenerRef& listener)
{
	CEWebKitImpl* webKitImpl = CEWebKitImpl::getInstance();
	if( webKitImpl )
	{
		listener = webKitImpl->getJSPrivilegeExtObjectListener();
		CEHResult hr = (listener.object() != 0 ? CE_S_OK : CE_SILK_ERR_OPERATION_FAILED );
		return hr;
	}
	else
	{
		return(CE_SILK_ERR_OPERATION_FAILED);
	}
}

CEHResult CEJSExternalAppCallback::declarePrivilegeExtObj(
	JSC::ExecState* exec, ICEJSPrivilegeExtObjectDeclarer* iDeclarer)
{
	CEComICEJSPrivilegeExtObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSExternalAppCallback::_saveExecState saved(exec);
		hr = listener.handleDeclarePrivilegeExtObject(iDeclarer);
	}
	return hr;
}

CEHResult CEJSExternalAppCallback::declarePrivilegeExtObjWithParent(
	JSC::ExecState* exec, ICEJSPrivilegeExtObjectParent* iParent, ICEJSPrivilegeExtObjectDeclarer* iDeclarer)
{
	CEComICEJSPrivilegeExtObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSExternalAppCallback::_saveExecState saved(exec);
		hr = listener.handleDeclarePrivilegeExtObjectWithParent(iParent, iDeclarer);
	}
	return hr;
}

CEHResult CEJSExternalAppCallback::authorizeOfPrivilegeExtObj(
	JSC::ExecState* exec, INT32 classId, ICEHtmlCredential* iCredential, CEJSAuthStatus* statusOut, INT_PTR* objectIdOut)
{
	CEComICEJSPrivilegeExtObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSExternalAppCallback::_saveExecState saved(exec);
		hr = listener.handleAuthorize(classId, iCredential, statusOut, objectIdOut);
	}
	return hr;
}

CEHResult CEJSExternalAppCallback::finalizeObjectIdOfPrivilegeExtObj(
	INT_PTR objectId)
{
	CEComICEJSPrivilegeExtObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		// now finalize, don't save ExecState.
		hr = listener.handleFinalizeObjectId(objectId);
	}
	return hr;
}

CEHResult CEJSExternalAppCallback::declarePropertyOfPrivilegeExtObj(
	JSC::ExecState* exec, INT_PTR objectId, ICEJSExtPropertyDeclarer* iDeclarer)
{
	CEComICEJSPrivilegeExtObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSExternalAppCallback::_saveExecState saved(exec);
		hr = listener.handleDeclareProperty(objectId, iDeclarer);
	}
	return hr;
}

CEHResult CEJSExternalAppCallback::declarePropertyOfPrivilegeExtObjWithParent(
	JSC::ExecState* exec, INT_PTR objectId, ICEJSExtPropertyParent* iParent, ICEJSExtPropertyDeclarer* iDeclarer)
{
	CEComICEJSPrivilegeExtObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSExternalAppCallback::_saveExecState saved(exec);
		hr = listener.handleDeclarePropertyWithParent(objectId, iParent, iDeclarer);
	}
	return hr;
}

CEHResult CEJSExternalAppCallback::isDirtyPropertyOfPrivilegeExtObj(
	JSC::ExecState* exec, INT_PTR objectId, INT32 propertyId, CEJSIsDirtyStatus* statusOut)
{
	CEComICEJSPrivilegeExtObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSExternalAppCallback::_saveExecState saved(exec);
		hr = listener.handleIsDirtyProperty(objectId, propertyId, statusOut);
	}
	return hr;
}

CEHResult CEJSExternalAppCallback::getPropertyOfPrivilegeExtObj(
	JSC::ExecState* exec, INT_PTR objectId, INT32 propertyId, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	CEComICEJSPrivilegeExtObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSExternalAppCallback::_saveExecState saved(exec);
		hr = listener.handleGetProperty(objectId, propertyId, iFactory, iValueOut);
	}
	return hr;
}

CEHResult CEJSExternalAppCallback::setPropertyOfPrivilegeExtObj(
	JSC::ExecState* exec, INT_PTR objectId, INT32 propertyId, ICEJSExtVariant* iValue, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	CEComICEJSPrivilegeExtObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSExternalAppCallback::_saveExecState saved(exec);
		hr = listener.handleSetProperty(objectId, propertyId, iValue, iFactory, iValueOut);
	}
	return hr;
}

CEHResult CEJSExternalAppCallback::invokeMethodOfPrivilegeExtObj(
	JSC::ExecState* exec, INT_PTR objectId, INT32 methodId, ICEJSExtVariant** iArgValues, UINT32 argCount, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	CEComICEJSPrivilegeExtObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSExternalAppCallback::_saveExecState saved(exec);
		hr = listener.handleInvokeMethod(objectId, methodId, iArgValues, argCount, iFactory, iValueOut);
	}
	return hr;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////              test code              ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////
#if defined(ENABLE_TEST_CODE)

////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////       window.external.SampleA       ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////

const UTF16CHAR CEJSExtObjectSampleA::_name_SampleA[] = {'S','a','m','p','l','e','A'};
const UTF16CHAR CEJSExtObjectSampleA::_name_propBoolean[] = {'p','r','o','p','B','o','o','l','e','a','n'};
const UTF16CHAR CEJSExtObjectSampleA::_name_propInt32[] = {'p','r','o','p','I','n','t','3','2'};
const UTF16CHAR CEJSExtObjectSampleA::_name_propString[] = {'p','r','o','p','S','t','r','i','n','g'};
const UTF16CHAR CEJSExtObjectSampleA::_name_myAdd[] = {'m','y','A','d','d'};
const UTF16CHAR CEJSExtObjectSampleA::_name_incPropInt32[] = {'i','n','c','P','r','o','p','I','n','t','3','2'};

const CEJSExtObjectBase::ClassTemplate CEJSExtObjectSampleA::classTemplate =
{
	// declaration of window.external.SampleA
	_name_SampleA,                                         // object name
	sizeof(_name_SampleA)/sizeof(_name_SampleA[0]),        // length of object name
	CEJSPrivilegeExtObjectAttr_None,                       // attribute
	_authorizer,                                           // authorizer
	_creater                                               // creater
};

const CEJSExtObjectBase::PropertyTemplate CEJSExtObjectSampleA::_propertyTemplateTable[] =
{
	{
		// declaration of window.external.SampleA.propBoolean
		eProprtyId_propBoolean,                                   // property ID
		_name_propBoolean,                                        // property name
		sizeof(_name_propBoolean)/sizeof(_name_propBoolean[0]),   // length of property name
		CEJSExtPropertyAttr_None                                  // attribute
	},
	{
		// declaration of window.external.SampleA.propInt32
		eProprtyId_propInt32,                                     // property ID
		_name_propInt32,                                          // property name
		sizeof(_name_propInt32)/sizeof(_name_propInt32[0]),       // length of property name
		CEJSExtPropertyAttr_None                                  // attribute
	},
	{
		// declaration of window.external.SampleA.propString
		eProprtyId_propString,                                    // property ID
		_name_propString,                                         // property name
		sizeof(_name_propString)/sizeof(_name_propString[0]),     // length of property name
		CEJSExtPropertyAttr_None                                  // attribute
	}
};

const CEJSExtObjectBase::MethodTemplate CEJSExtObjectSampleA::_methodTemplateTable[] =
{
	{
		// declaration of window.external.SampleA.myAdd
		eMethodId_myAdd,                                          // method ID
		_name_myAdd,                                              // method name
		sizeof(_name_myAdd)/sizeof(_name_myAdd[0]),               // length of method name
		CEJSExtPropertyAttr_None,                                 // attribute
		2                                                         // argument count
	},
	{
		// declaration of window.external.SampleA.incPropInt32
		eMethodId_incPropInt32,                                   // method ID
		_name_incPropInt32,                                       // method name
		sizeof(_name_incPropInt32)/sizeof(_name_incPropInt32[0]), // length of method name
		CEJSExtPropertyAttr_None,                                 // attribute
		0                                                         // argument count
	}
};

CEHResult CEJSExtObjectSampleA::_creater(CEJSExtObjectBase** objOut)
{
	CEHResult hr = CE_S_OK;
	CEJSExtObjectSampleA* obj = new CEJSExtObjectSampleA;
	if (obj)
	{
		* objOut = obj;
	}
	else
	{
		hr = CE_SILK_ERR_MEMERR;
	}
	return hr;
}

CEHResult CEJSExtObjectSampleA::_authorizer(ICEHtmlCredential* iCredential, CEJSAuthStatus* statusOut)
{
	CEComICEHtmlCredentialRef rCredential = iCredential;
	*statusOut = CEJSAuthStatus_Unauthorized;
	CEHtmlSSLMode mode;
	CEHResult hr = rCredential.getSSLMode(&mode);
	if (CESucceeded(hr))
	{
		switch (mode) {
		case CEHtmlSSLMode_Loading:
			*statusOut = CEJSAuthStatus_UnauthorizedTemporarily;
			break;
		case CEHtmlSSLMode_NonSecured:
			{
				const UTF16CHAR* pCharArray16Domain;
				UINT32 numOfChars16Domain;
				hr = rCredential.getDomain(&pCharArray16Domain, &numOfChars16Domain);
				if (CESucceeded(hr))
				{
					if (_isSafeDomain(pCharArray16Domain, numOfChars16Domain))
					{
						*statusOut = CEJSAuthStatus_Authorized;
					}
					else
					{
						*statusOut = CEJSAuthStatus_Unauthorized;
					}
				}
			}
			break;
		case CEHtmlSSLMode_Secured:
		case CEHtmlSSLMode_Mixed:
			{
				UINT32 alert;
				hr = rCredential.getSSLAlert(&alert);
				if (CESucceeded(hr) && alert == CEHtmlSSLAlert_None)
				{
					const UTF16CHAR* pCharArray16Domain;
					UINT32 numOfChars16Domain;
					hr = rCredential.getDomain(&pCharArray16Domain, &numOfChars16Domain);
					if (CESucceeded(hr))
					{
						if (_isSafeDomain(pCharArray16Domain, numOfChars16Domain))
						{
							*statusOut = CEJSAuthStatus_Authorized;
						}
						else
						{
							*statusOut = CEJSAuthStatus_Unauthorized;
						}
					}
				}
			}
			break;
		}
	}
	return hr;
}

bool CEJSExtObjectSampleA::_isSafeDomain(const UTF16CHAR* pCharArray16Domain, UINT32 numOfChars16Domain)
{
	// please add domain check code!
	return true;
}

CEHResult CEJSExtObjectSampleA::declareProperty(ICEJSExtPropertyDeclarer* iDeclarer)
{
	CEHResult hr = CE_S_OK;
	CEComICEJSExtPropertyDeclarerRef iDeclarerRef = iDeclarer;
	for (UINT32 i = 0; CESucceeded(hr) && (i < (sizeof(_propertyTemplateTable) / sizeof(_propertyTemplateTable[0]))); ++i)
	{
		const PropertyTemplate& prop = _propertyTemplateTable[i];
		hr = iDeclarerRef.declareProperty(prop.propertyId, prop.pPropertyName, prop.numOfChars16, prop.attr);
	}
	for (UINT32 i = 0; CESucceeded(hr) && (i < (sizeof(_methodTemplateTable) / sizeof(_methodTemplateTable[0]))); ++i)
	{
		const MethodTemplate& method = _methodTemplateTable[i];
		hr = iDeclarerRef.declareMethod(method.methodId, method.pMethodName, method.numOfChars16, method.attr, method.argCount);
	}
	return hr;
}


CEHResult CEJSExtObjectSampleA::getProperty(INT32 propertyId, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	CEComICEJSExtVariantFactoryRef iFactoryRef = iFactory;
	switch(propertyId)
	{
	case eProprtyId_propBoolean:
		{
			hr = iFactoryRef.createFromBoolean(_propBoolean, iValueOut);
		}
		break;
	case eProprtyId_propInt32:
		{
			hr = iFactoryRef.createFromInt32(_propInt32, iValueOut);
		}
		break;
	case eProprtyId_propString:
		{
			if (_propStringLen)
			{
				hr = iFactoryRef.createFromString(_propStringValue, _propStringLen, iValueOut);
			}
			else
			{
				hr = iFactoryRef.createFromNull(iValueOut);
			}
		}
		break;
	}
	return hr;
}

CEHResult CEJSExtObjectSampleA::setProperty(INT32 propertyId, ICEJSExtVariant* iValue, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	CEComICEJSExtVariantRef iValueRef = iValue;
	CEComICEJSExtVariantFactoryRef iFactoryRef = iFactory;
	switch(propertyId)
	{
	case eProprtyId_propBoolean:
		{
			hr = iValueRef.toBoolean(&_propBoolean);
			if (CESucceeded(hr))
			{
				hr = iFactoryRef.createFromBoolean(_propBoolean, iValueOut);
			}
		}
		break;
	case eProprtyId_propInt32:
		{
			hr = iValueRef.toInt32(&_propInt32);
			if (CESucceeded(hr))
			{
				hr = iFactoryRef.createFromInt32(_propInt32, iValueOut);
				if (CESucceeded(hr))
				{
					_isDirtyPropInt32 = false;
				}
			}
		}
		break;
	case eProprtyId_propString:
		{
			const UTF16CHAR* pCharArray16;
			UINT32 numOfChars16;
			hr = iValueRef.toString(&pCharArray16, &numOfChars16);
			if (CESucceeded(hr))
			{
				if (numOfChars16 <= (sizeof(_propStringValue)/sizeof(_propStringValue[0])))
				{
					_propStringLen = numOfChars16;
					CESysCopyMemory(_propStringValue, sizeof(_propStringValue), pCharArray16, sizeof(UTF16CHAR) * _propStringLen);
					if (_propStringLen)
					{
						hr = iFactoryRef.createFromString(_propStringValue, _propStringLen, iValueOut);
					}
					else
					{
						hr = iFactoryRef.createFromNull(iValueOut);
					}
				}
				else
				{
					hr = CE_SILK_ERR_OPERATION_FAILED;
				}
			}
		}
		break;
	}
	return hr;
}

CEHResult CEJSExtObjectSampleA::isDirtyProperty(INT32 propertyId, CEJSIsDirtyStatus* statusOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	switch(propertyId)
	{
	case eProprtyId_propBoolean:
		{
			*statusOut = CEJSIsDirtyStatus_NotDirty;
			hr = CE_S_OK;
		}
		break;
	case eProprtyId_propInt32:
		{
			if (_isDirtyPropInt32)
			{
				*statusOut = CEJSIsDirtyStatus_Dirty;
				_isDirtyPropInt32 = false;
			}
			else
			{
				*statusOut = CEJSIsDirtyStatus_NotDirty;
			}
			hr = CE_S_OK;
		}
		break;
	case eProprtyId_propString:
		{
			*statusOut = CEJSIsDirtyStatus_NotDirty;
			hr = CE_S_OK;
		}
		break;
	}
	return hr;
}

CEHResult CEJSExtObjectSampleA::invokeMethod(INT32 methodId, ICEJSExtVariant** iArgValues, UINT32 argCount, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	CEComICEJSExtVariantFactoryRef iFactoryRef = iFactory;
	switch(methodId)
	{
	case eMethodId_myAdd:
		{
			if (argCount >= 2)
			{
				CEComICEJSExtVariantRef rArg0 = iArgValues[0];
				CEComICEJSExtVariantRef rArg1 = iArgValues[1];
				INT32 arg0;
				INT32 arg1;
				hr = rArg0.toInt32(&arg0);
				if (CESucceeded(hr))
				{
					hr = rArg1.toInt32(&arg1);
					if (CESucceeded(hr))
					{
						hr = iFactoryRef.createFromInt32(arg0 + arg1, iValueOut);
					}
				}
			}
			else
			{
				hr = iFactoryRef.createFromUndefined(iValueOut);
			}
		}
		break;
	case eMethodId_incPropInt32:
		{
			++_propInt32;
			_isDirtyPropInt32 = true;
			hr = CE_S_OK;
		}
		break;
	}
	return hr;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////       window.external.SampleB       ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////

const UTF16CHAR CEJSExtObjectSampleB::_name_SampleB[] = {'S','a','m','p','l','e','B'};
const UTF16CHAR CEJSExtObjectSampleB::_name_propStringConst[] = {'p','r','o','p','S','t','r','i','n','g','C','o','n','s','t'};
const UTF16CHAR CEJSExtObjectSampleB::_name_propNumberReadOnly[] = {'p','r','o','p','N','u','m','b','e','r','R','e','a','d','O','n','l','y'};
const UTF16CHAR CEJSExtObjectSampleB::_name_setPropNumberReadOnly[] = {'s','e','t','P','r','o','p','N','u','m','b','e','r','R','e','a','d','O','n','l','y'};

const CEJSExtObjectBase::ClassTemplate CEJSExtObjectSampleB::classTemplate =
{
	// declaration of window.external.SampleB
	_name_SampleB,                                         // object name
	sizeof(_name_SampleB)/sizeof(_name_SampleB[0]),        // length of object name
	CEJSPrivilegeExtObjectAttr_None,                       // attribute
	_authorizer,                                           // authorizer
	_creater                                               // creater
};

const CEJSExtObjectBase::PropertyTemplate CEJSExtObjectSampleB::_propertyTemplateTable[] =
{
	{
		// declaration of window.external.SampleB.propStringConst
		eProprtyId_propStringConst,                                                 // property ID
		_name_propStringConst,                                                      // property name
		sizeof(_name_propStringConst)/sizeof(_name_propStringConst[0]),             // length of property name
		CEJSExtPropertyAttr_Const                                                   // attribute
	},
	{
		// declaration of window.external.SampleB.propNumberReadOnly
		eProprtyId_propNumberReadOnly,                                              // property ID
		_name_propNumberReadOnly,                                                   // property name
		sizeof(_name_propNumberReadOnly)/sizeof(_name_propNumberReadOnly[0]),       // length of property name
		CEJSExtPropertyAttr_ReadOnly                                                // attribute
	},
};

const CEJSExtObjectBase::MethodTemplate CEJSExtObjectSampleB::_methodTemplateTable[] =
{
	{
		// declaration of window.external.SampleB.setPropNumberReadOnly
		eMethodId_setPropNumberReadOnly,                                            // method ID
		_name_setPropNumberReadOnly,                                                // method name
		sizeof(_name_setPropNumberReadOnly)/sizeof(_name_setPropNumberReadOnly[0]), // length of method name
		CEJSExtPropertyAttr_None,                                                   // attribute
		1                                                                           // argument count
	}
};

CEHResult CEJSExtObjectSampleB::_creater(CEJSExtObjectBase** objOut)
{
	CEHResult hr = CE_S_OK;
	CEJSExtObjectSampleB* obj = new CEJSExtObjectSampleB;
	if (obj)
	{
		* objOut = obj;
	}
	else
	{
		hr = CE_SILK_ERR_MEMERR;
	}
	return hr;
}

CEHResult CEJSExtObjectSampleB::_authorizer(ICEHtmlCredential* iCredential, CEJSAuthStatus* statusOut)
{
	CEComICEHtmlCredentialRef rCredential = iCredential;
	*statusOut = CEJSAuthStatus_Unauthorized;
	CEHtmlSSLMode mode;
	CEHResult hr = rCredential.getSSLMode(&mode);
	if (CESucceeded(hr))
	{
		switch (mode) {
		case CEHtmlSSLMode_Loading:
			*statusOut = CEJSAuthStatus_UnauthorizedTemporarily;
			break;
		case CEHtmlSSLMode_NonSecured:
			{
				const UTF16CHAR* pCharArray16Domain;
				UINT32 numOfChars16Domain;
				hr = rCredential.getDomain(&pCharArray16Domain, &numOfChars16Domain);
				if (CESucceeded(hr))
				{
					if (_isSafeDomain(pCharArray16Domain, numOfChars16Domain))
					{
						*statusOut = CEJSAuthStatus_Authorized;
					}
					else
					{
						*statusOut = CEJSAuthStatus_Unauthorized;
					}
				}
			}
			break;
		case CEHtmlSSLMode_Secured:
		case CEHtmlSSLMode_Mixed:
			{
				UINT32 alert;
				hr = rCredential.getSSLAlert(&alert);
				if (CESucceeded(hr) && alert == CEHtmlSSLAlert_None)
				{
					const UTF16CHAR* pCharArray16Domain;
					UINT32 numOfChars16Domain;
					hr = rCredential.getDomain(&pCharArray16Domain, &numOfChars16Domain);
					if (CESucceeded(hr))
					{
						if (_isSafeDomain(pCharArray16Domain, numOfChars16Domain))
						{
							*statusOut = CEJSAuthStatus_Authorized;
						}
						else
						{
							*statusOut = CEJSAuthStatus_Unauthorized;
						}
					}
				}
			}
			break;
		}
	}
	return hr;
}

bool CEJSExtObjectSampleB::_isSafeDomain(const UTF16CHAR* pCharArray16Domain, UINT32 numOfChars16Domain)
{
	// please add domain check code!
	return true;
}

CEHResult CEJSExtObjectSampleB::declareProperty(ICEJSExtPropertyDeclarer* iDeclarer)
{
	CEHResult hr = CE_S_OK;
	CEComICEJSExtPropertyDeclarerRef iDeclarerRef = iDeclarer;
	for (UINT32 i = 0; CESucceeded(hr) && (i < (sizeof(_propertyTemplateTable) / sizeof(_propertyTemplateTable[0]))); ++i)
	{
		const PropertyTemplate& prop = _propertyTemplateTable[i];
		hr = iDeclarerRef.declareProperty(prop.propertyId, prop.pPropertyName, prop.numOfChars16, prop.attr);
	}
	for (UINT32 i = 0; CESucceeded(hr) && (i < (sizeof(_methodTemplateTable) / sizeof(_methodTemplateTable[0]))); ++i)
	{
		const MethodTemplate& method = _methodTemplateTable[i];
		hr = iDeclarerRef.declareMethod(method.methodId, method.pMethodName, method.numOfChars16, method.attr, method.argCount);
	}
	return hr;
}

CEHResult CEJSExtObjectSampleB::getProperty(INT32 propertyId, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	CEComICEJSExtVariantFactoryRef iFactoryRef = iFactory;
	switch(propertyId)
	{
	case eProprtyId_propStringConst:
		{
			static const UTF16CHAR name_ABC[] = {'A','B','C'};
			hr = iFactoryRef.createFromString(name_ABC, sizeof(name_ABC)/sizeof(name_ABC[0]), iValueOut);
		}
		break;
	case eProprtyId_propNumberReadOnly:
		{
			hr = iFactoryRef.createFromNumber(_propNumberReadOnly, iValueOut);
		}
		break;
	}
	return hr;
}

CEHResult CEJSExtObjectSampleB::setProperty(INT32 propertyId, ICEJSExtVariant* iValue, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	CEComICEJSExtVariantRef iValueRef = iValue;
	CEComICEJSExtVariantFactoryRef iFactoryRef = iFactory;
	switch(propertyId)
	{
	case eProprtyId_propStringConst:
		{
			CEASSERT(0); // never called
			hr = CE_SILK_ERR_OPERATION_FAILED;
		}
		break;
	case eProprtyId_propNumberReadOnly:
		{
			CEASSERT(0); // never called
			hr = CE_SILK_ERR_OPERATION_FAILED;
		}
		break;
	}
	return hr;
}

CEHResult CEJSExtObjectSampleB::isDirtyProperty(INT32 propertyId, CEJSIsDirtyStatus* statusOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	switch(propertyId)
	{
	case eProprtyId_propStringConst:
		{
			CEASSERT(0); // never called
			hr = CE_SILK_ERR_OPERATION_FAILED;
		}
		break;
	case eProprtyId_propNumberReadOnly:
		{
			if (_isDirtyPropNumberReadOnly)
			{
				*statusOut = CEJSIsDirtyStatus_Dirty;
				_isDirtyPropNumberReadOnly = false;
			}
			else
			{
				*statusOut = CEJSIsDirtyStatus_NotDirty;
			}
			hr = CE_S_OK;
		}
		break;
	}
	return hr;
}

CEHResult CEJSExtObjectSampleB::invokeMethod(INT32 methodId, ICEJSExtVariant** iArgValues, UINT32 argCount, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	CEComICEJSExtVariantFactoryRef iFactoryRef = iFactory;
	switch(methodId)
	{
	case eMethodId_setPropNumberReadOnly:
		{
			if (argCount >= 1)
			{
				CEComICEJSExtVariantRef rArg0 = iArgValues[0];
				hr = rArg0.toNumber(&_propNumberReadOnly);
				if (CESucceeded(hr))
				{
					_isDirtyPropNumberReadOnly = true;
				}
			}
			else
			{
				_propNumberReadOnly = 0.0; 
				_isDirtyPropNumberReadOnly = true;
				hr = CE_S_OK;
			}
		}
		break;
	}
	return hr;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////// implement class of ICEJSPrivilegeExtObjectListener ////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////

const CEJSExtObjectBase::ClassTemplate* CEJSPrivilegeExtObjectListenerImpl::_classTemplateTable[] = 
{
	&(CEJSExtObjectSampleA::classTemplate),       // classId : 0, [window.external.SampleA]
	&(CEJSExtObjectSampleB::classTemplate)        // classId : 1, [window.external.SampleB]
};

CEHResult CEJSPrivilegeExtObjectListenerImpl::declarePrivilegeExtObject(ICEJSPrivilegeExtObjectDeclarer* iDeclarer)
{
	CEHResult hr = CE_S_OK;
	CEComICEJSPrivilegeExtObjectDeclarerRef iDeclarerRef = iDeclarer;
	for (UINT32 i = 0; CESucceeded(hr) && (i < (sizeof(_classTemplateTable) / sizeof(_classTemplateTable[0]))); ++i)
	{
		hr = iDeclarerRef.declareObject(i, _classTemplateTable[i]->pObjectName, _classTemplateTable[i]->numOfChars16, _classTemplateTable[i]->attribute);
	}
	return hr;
}

CEHResult CEJSPrivilegeExtObjectListenerImpl::authorize(INT32 classId, ICEHtmlCredential* iCredential, CEJSAuthStatus* statusOut, INT_PTR* objectIdOut)
{
	CEHResult hr = (*_classTemplateTable[classId]->authorizer)(iCredential, statusOut);
	if (CESucceeded(hr) && *statusOut == CEJSAuthStatus_Authorized)
	{
		CEJSExtObjectBase* obj = 0;
		hr = (*_classTemplateTable[classId]->creater)(&obj);
		if (CESucceeded(hr))
		{
			*objectIdOut = _objToObjectId(obj);
		}
	}
	return hr;
}

CEHResult CEJSPrivilegeExtObjectListenerImpl::declareProperty(INT_PTR objectId, ICEJSExtPropertyDeclarer* iDeclarer)
{
	return _objectIdToObj(objectId)->declareProperty(iDeclarer);
}

CEHResult CEJSPrivilegeExtObjectListenerImpl::getProperty(INT_PTR objectId, INT32 propertyId, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	return _objectIdToObj(objectId)->getProperty(propertyId, iFactory, iValueOut);
}

CEHResult CEJSPrivilegeExtObjectListenerImpl::setProperty(INT_PTR objectId, INT32 propertyId, ICEJSExtVariant* iValue, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	return _objectIdToObj(objectId)->setProperty(propertyId, iValue, iFactory, iValueOut);
}

CEHResult CEJSPrivilegeExtObjectListenerImpl::isDirtyProperty(INT_PTR objectId, INT32 propertyId, CEJSIsDirtyStatus* statusOut)
{
	return _objectIdToObj(objectId)->isDirtyProperty(propertyId, statusOut);
}

CEHResult CEJSPrivilegeExtObjectListenerImpl::invokeMethod(INT_PTR objectId, INT32 methodId, ICEJSExtVariant** iArgValues, UINT32 argCount, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	return _objectIdToObj(objectId)->invokeMethod(methodId, iArgValues, argCount, iFactory, iValueOut);
}

CEHResult CEJSPrivilegeExtObjectListenerImpl::finalizeObjectId(INT_PTR objectId)
{
	delete _objectIdToObj(objectId);
	return CE_S_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////// ICEJSPrivilegeExtObjectListener I/F ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////

static CEJSPrivilegeExtObjectListenerImpl impl;

static CEHResult __jsGlue_queryInterface(ICEJSPrivilegeExtObjectListener* iJSPrivilegeExtObjectListener, const UINT32 iId, void* *const iOut)
{
	// this function is dummy.
	return CE_SILK_ERR_OPERATION_FAILED;
}

static void __jsGlue_addRef(ICEJSPrivilegeExtObjectListener* iJSPrivilegeExtObjectListener)
{
	// this function is dummy.
}

static void __jsGlue_release(ICEJSPrivilegeExtObjectListener* iJSPrivilegeExtObjectListener)
{
	// this function is dummy.
}

static CEHResult __jsGlue_handleDeclarePrivilegeExtObject(ICEJSPrivilegeExtObjectListener* iJSPrivilegeExtObjectListener, ICEJSPrivilegeExtObjectDeclarer* iDeclarer)
{
	return impl.declarePrivilegeExtObject(iDeclarer);
}

static CEHResult __jsGlue_handleDeclarePrivilegeExtObjectWithParent(ICEJSPrivilegeExtObjectListener* iJSPrivilegeExtObjectListener, ICEJSPrivilegeExtObjectParent* iParent, ICEJSPrivilegeExtObjectDeclarer* iDeclarer)
{
	return impl.declarePrivilegeExtObjectWithParent(iParent, iDeclarer);
}

static CEHResult __jsGlue_handleAuthorize(ICEJSPrivilegeExtObjectListener* iJSPrivilegeExtObjectListener, INT32 classId, ICEHtmlCredential* iCredential, CEJSAuthStatus* statusOut, INT_PTR* objectIdOut)
{
	return impl.authorize(classId, iCredential, statusOut, objectIdOut);
}

static CEHResult __jsGlue_handleFinalizeObjectId(ICEJSPrivilegeExtObjectListener* iJSPrivilegeExtObjectListener, INT_PTR objectId)
{
	return impl.finalizeObjectId(objectId);
}

static CEHResult __jsGlue_handleDeclarePropertyWithParent(ICEJSPrivilegeExtObjectListener* iJSPrivilegeExtObjectListener, INT_PTR objectId, ICEJSExtPropertyParent* iParent, ICEJSExtPropertyDeclarer* iDeclarer)
{
	return impl.declarePropertyWithParent(objectId, iParent, iDeclarer);
}

static CEHResult __jsGlue_handleGetProperty(ICEJSPrivilegeExtObjectListener* iJSPrivilegeExtObjectListener, INT_PTR objectId, INT32 propertyId, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	return impl.getProperty(objectId, propertyId, iFactory, iValueOut);
}

static CEHResult __jsGlue_handleSetProperty(ICEJSPrivilegeExtObjectListener* iJSPrivilegeExtObjectListener, INT_PTR objectId, INT32 propertyId, ICEJSExtVariant* iValue, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	return impl.setProperty(objectId, propertyId, iValue, iFactory, iValueOut);
}

static CEHResult __jsGlue_handleIsDirtyProperty(ICEJSPrivilegeExtObjectListener* iJSPrivilegeExtObjectListener, INT_PTR objectId, INT32 propertyId, CEJSIsDirtyStatus* statusOut)
{
	return impl.isDirtyProperty(objectId, propertyId, statusOut);
}

static CEHResult __jsGlue_handleInvokeMethod(ICEJSPrivilegeExtObjectListener* iJSPrivilegeExtObjectListener, INT_PTR objectId, INT32 methodId, ICEJSExtVariant** iArgValues, UINT32 argCount, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	return impl.invokeMethod(objectId, methodId, iArgValues, argCount, iFactory, iValueOut);
}

static const ICEJSPrivilegeExtObjectListener_vtbl __jsGlue_jsPrivilegeExtObjectListenerVtbl =
{
	__jsGlue_queryInterface,
	__jsGlue_addRef,
	__jsGlue_release,
	__jsGlue_handleDeclarePrivilegeExtObject,
	__jsGlue_handleDeclarePrivilegeExtObjectWithParent,
	__jsGlue_handleAuthorize,
	__jsGlue_handleFinalizeObjectId,
	__jsGlue_handleDeclareProperty,
	__jsGlue_handleDeclarePropertyWithParent,
	__jsGlue_handleGetProperty,
	__jsGlue_handleSetProperty,
	__jsGlue_handleIsDirtyProperty,
	__jsGlue_handleInvokeMethod
};

ICEJSPrivilegeExtObjectListener jsPrivilegeExtObjectListener = { &__jsGlue_jsPrivilegeExtObjectListenerVtbl };

#endif // defined(ENABLE_TEST_CODE)
