/*
 * Copyright (C) 2006, 2007, 2008 Apple Inc. All rights reserved.
 * Copyright     2009, 2012 Sony Corporation
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 * 3.  Neither the name of Apple Computer, Inc. ("Apple") nor the names of
 *     its contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE AND ITS CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "WebKitSilkPrefix.h"

#include "FrameLoaderClientSilk.h"
#include "HTMLPlugInElement.h"
#include "CEWebKitFrameWindowImpl.h"
#include "CEWebKitWindow.h"
#include "RenderPart.h"
#include "FrameTree.h"
#include "PluginView.h"
#include "MIMETypeRegistry.h"
#include "PluginDatabase.h"
#include "MainResourceLoader.h"
#include "ResourceHandle.h"
#include "ResourceHandleInternal.h"
#include "CEWebKitDocumentLoader.h"
#include "Settings.h"
#include "DocLoader.h"
#include "FocusController.h"
#include "RenderView.h"
#include "GraphicsContext.h"
#include "ProgressTracker.h"

#include "NodeList.h"
#include "Element.h"
#include "ResourceHandleManagerCore.h"
#include "ICEHtmlPluginConfirmation.h"
#include "CEWebKitImpl.h"

#if CE_OS(HYDRA)
#define SHOW_INTERNAL_NETWORK_ERROR_STRING (1)
#endif

// The confirmed plugin to wait for the application's confirm.
class PluginConfirmWidget : public WebCore::Widget {

public:
	PluginConfirmWidget(WebCore::Frame* parentFrame, const IntSize& size, WebCore::Element* element, 
		const WebCore::KURL& url, const Vector<WebCore::String>& paramNames, const Vector<WebCore::String>& paramValues, 
		const WebCore::String& mimeType, bool loadManually)
		: m_parentFrame(parentFrame)
		, m_size(size)
		, m_element(element)
		, m_url(url)
		, m_paramNames(paramNames)
		, m_paramValues(paramValues)
		, m_mimeType(mimeType)
		, m_loadManually(loadManually)
	{
		resize(size);
	}

	virtual void paint(WebCore::GraphicsContext* ctx, const WebCore::IntRect& rect)
	{
		using namespace WebCore;
		ctx->save();
		ctx->clip(frameRect());
		ctx->setFillColor(Color::white);
		ctx->fillRect(frameRect());
		ctx->restore();
	}
	
	virtual void invalidateRect(const WebCore::IntRect& rect)
	{
		if (parent()) {
			WebCore::IntRect intRect(rect);
			intRect.move(x(), y());
			parent()->invalidateRect(intRect);
		}
	}

	bool swapPlugin(WebCore::RenderWidget* renderer)
	{
		bool returnValue = false;
		if(renderer)
		{
			WTF::PassRefPtr<WebCore::PluginView> pluginWidget = WebCore::PluginView::create(m_parentFrame.get(), m_size, m_element, m_url, m_paramNames, m_paramValues, m_mimeType, m_loadManually);
			if(pluginWidget)
			{
				pluginWidget->focusPluginElement();
				renderer->setWidget(pluginWidget);
				returnValue = true;
			}
		}
		return returnValue;
	}

private:
	RefPtr<WebCore::Frame>		 m_parentFrame;
	IntSize						 m_size;
	WebCore::Element*			 m_element;
	WebCore::KURL				 m_url;
	WTF::Vector<WebCore::String> m_paramNames;
	WTF::Vector<WebCore::String> m_paramValues;
	WebCore::String              m_mimeType;
	bool						 m_loadManually;


};

FrameLoaderClientSilk::FrameLoaderClientSilk(CEWebKitWindowImpl* window) :
	_window(window),
	_frame(NULL),
	_newTargetUrl(NULL),
	_isCEA2014ModeEnabled(false),
	_cert(NULL), 
	_isCommittedLoad(false)
{
}

void FrameLoaderClientSilk::frameLoaderDestroyed()
{
	if (_cert)
		WebCore::ResourceHandleManager::sharedInstance()->deleteFrameLoderClientCert(_cert);

	delete this;
}

bool FrameLoaderClientSilk::hasWebView() const
{
	return true;//?
}

void FrameLoaderClientSilk::makeRepresentation(WebCore::DocumentLoader*)
{
}

void FrameLoaderClientSilk::forceLayout()
{
}

void FrameLoaderClientSilk::forceLayoutForNonHTML()
{
}

void FrameLoaderClientSilk::setCopiesOnScroll()
{
}

void FrameLoaderClientSilk::detachedFromParent2()
{
}

void FrameLoaderClientSilk::detachedFromParent3()
{
}

static inline WebCore::ResourceHandleInternal* _getResourceHandleInternal(WebCore::DocumentLoader* documentLoader)
{
	if (documentLoader && documentLoader->mainResourceLoader() && documentLoader->mainResourceLoader()->handle())
	{
		return documentLoader->mainResourceLoader()->handle()->getInternal();
	}
	return NULL;
}
static inline CEComICEHtmlWebKitFrameWindowLoaderClientRef _getLoaderClinet(WebCore::Frame* frame)
{
	if (frame)
	{
		CEWebKitFrameWindowImpl* frameWindow = kit(frame->page());
		if (frameWindow)
		{
			return frameWindow->getLoaderClient();
		}
	}
	return NULL;
} 

void FrameLoaderClientSilk::assignIdentifierToInitialRequest(unsigned long identifier, WebCore::DocumentLoader* documentLoader, const WebCore::ResourceRequest& resourceRequest)
{
	resourceRequest.setIdentifier(identifier);

	CEWebKitDocumentLoader* wkDocumentLoader = static_cast<CEWebKitDocumentLoader*>(documentLoader);
	if (wkDocumentLoader && wkDocumentLoader->getDocumentLoaderClient())
	{
		wkDocumentLoader->getDocumentLoaderClient().assignIdentifierToInitialRequest(identifier);
	}
}

void FrameLoaderClientSilk::dispatchWillSendRequest(WebCore::DocumentLoader* documentLoader, unsigned long identifier, WebCore::ResourceRequest& resourceRequest, const WebCore::ResourceResponse& redirectResponse)
{
	WebCore::String acceptLanguageStr;
	CEComICEUStringRef acceptLanguageStrRef;
	CEWebKitImpl* webKitImpl = CEWebKitImpl::getInstance();
	if (webKitImpl)
	{
		CEComICEHtmlWebKitEnvironmentRef env = webKitImpl->getWebKitEnvironment();
		if (env)
		{
			CEHResult hr = env.getAcceptLanguage(&acceptLanguageStrRef);
			if (CESucceeded(hr) && acceptLanguageStrRef)
			{
				acceptLanguageStr.fromICEUString(acceptLanguageStrRef);
			}
		}
	}

	resourceRequest.setHTTPHeaderField("Accept-Language", acceptLanguageStr.utf8().data());
}

void FrameLoaderClientSilk::dispatchDidReceiveAuthenticationChallenge(WebCore::DocumentLoader* documentLoader, unsigned long identifier, const WebCore::AuthenticationChallenge&)
{
}

void FrameLoaderClientSilk::dispatchDidCancelAuthenticationChallenge(WebCore::DocumentLoader* documentLoader, unsigned long identifier, const WebCore::AuthenticationChallenge&)
{
}
        
void FrameLoaderClientSilk::dispatchDidReceiveResponse(WebCore::DocumentLoader* documentLoader, unsigned long identifier, const WebCore::ResourceResponse& resourceResponse)
{
	CEWebKitDocumentLoader* wkDocumentLoader = static_cast<CEWebKitDocumentLoader*>(documentLoader);
	if (wkDocumentLoader && wkDocumentLoader->getDocumentLoaderClient())
	{
        CEWebKitResouceType type = CEWebKitResouceType_None;
        WebCore::CachedResource* cachedResource = wkDocumentLoader->frame()->document()->docLoader()->cachedResource(resourceResponse.url());

	WebCore::FrameLoader* frameloader = wkDocumentLoader->frameLoader();
	if (frameloader)
	{
		WebCore::DocumentLoader* pdl = frameloader->provisionalDocumentLoader();
		if (pdl)
		{
			pdl->request().setURL(resourceResponse.url());
		}
	}

        if (cachedResource)
        {
            switch (cachedResource->type())
            {
            case WebCore::CachedResource::ImageResource:
                type = CEWebKitResouceType_Image;
                break;
            case WebCore::CachedResource::CSSStyleSheet:
                type = CEWebKitResouceType_CSS;
                break;
            case WebCore::CachedResource::Script:
                type = CEWebKitResouceType_Script;
                break;
            case WebCore::CachedResource::FontResource:
                type = CEWebKitResouceType_Font;
                break;
#if ENABLE(XSLT)
            case WebCore::CachedResource::XSLStyleSheet:
                type = CEWebKitResouceType_XSLT;
                break;
#endif
#if ENABLE(XBL)
            case WebCore::CachedResource::XBL:
                type = CEWebKitResouceType_XBL;
                break;
#endif
            }
        }
		else if (wkDocumentLoader->response() == resourceResponse)
        {
            type = CEWebKitResouceType_Body;
        }

		wkDocumentLoader->getDocumentLoaderClient().dispatchDidReceiveResponse(identifier, type, resourceResponse.isSecure(), resourceResponse.sslError());

		if (resourceResponse.responseFired() == true && resourceResponse.noContext() == false)
		{
			if(!_idMap.contains(identifier))
			{
				_idMap.set(identifier,  type == CEWebKitResouceType_Body);
			}
		}

		_isCEA2014ModeEnabled = (type == CEWebKitResouceType_Body) && resourceResponse.mimeType().contains("ce-html+xml");
	}
}

void FrameLoaderClientSilk::dispatchDidReceiveContentLength(WebCore::DocumentLoader* documentLoader, unsigned long identifier, int lengthReceived)
{
}

void FrameLoaderClientSilk::dispatchDidFinishLoading(WebCore::DocumentLoader* documentLoader, unsigned long identifier)
{
	if (_idMap.contains(identifier))
	{
		unsigned long identifierOnFrame = identifier;
		unsigned long pageId = _frame->page()->group().identifier();
		identifierOnFrame |= (pageId << PAGE_IDENTIFIER_SHIFT);
		if (_idMap.get(identifier))
		{
			RefPtr<WebCore::NodeList> list = _frame->document()->getElementsByTagName("meta");
			unsigned len = list->length();
			WebCore::String metaInfo;
			for (unsigned i = 0; i < len; i++) {
				WebCore::Element* element = static_cast<WebCore::Element*>(list->item(i));
				WebCore::AtomicString httpEquiv = element->getAttribute("http-equiv");
				WebCore::AtomicString content = element->getAttribute("content");
				if (!httpEquiv.isEmpty())
				{
					metaInfo += httpEquiv;
					metaInfo += ": ";
					metaInfo += content;
					metaInfo += "\r\n";
				}
			}
			if (!metaInfo.isEmpty())
				metaInfo += "\r\n";
			WebCore::ResourceHandleManager::sharedInstance()->didFinishLoading(identifierOnFrame, &metaInfo);
		}
		else
		{
			WebCore::ResourceHandleManager::sharedInstance()->didFinishLoading(identifierOnFrame, NULL);
		}
		_idMap.remove(identifier);
	}

	CEWebKitDocumentLoader* wkDocumentLoader = static_cast<CEWebKitDocumentLoader*>(documentLoader);
	if (wkDocumentLoader && wkDocumentLoader->getDocumentLoaderClient())
	{
		wkDocumentLoader->getDocumentLoaderClient().dispatchDidFinishLoading(identifier);
	}
}

void FrameLoaderClientSilk::dispatchDidFailLoading(WebCore::DocumentLoader* documentLoader, unsigned long identifier, const WebCore::ResourceError& err)
{
	_isCommittedLoad = false;
	CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
	if (frameWindow)
	{
		bool useAppErrPage = false;
		CEComICEUStringRef appErrPageStr = NULL;
		CEHResult hr = frameWindow->getChromeClient().checkErrorOnRequestComplete(err.resultCode(), err.lowLevelErrorCode(), err.getIsMainRresource(), NULL);
	}

	if (err.noContext() == false || err.resultCode() == CE_SILK_ERR_OPERATION_CANCELLED)
	{
		unsigned long identifierOnFrame = identifier;
		unsigned long pageId = _frame->page()->group().identifier();
		identifierOnFrame |= (pageId << PAGE_IDENTIFIER_SHIFT);

		if (_idMap.contains(identifier))
		{
			WebCore::ResourceHandleManager::sharedInstance()->didFinishLoading(identifierOnFrame, NULL);
			_idMap.remove(identifier);
		}
	}

	CEWebKitDocumentLoader* wkDocumentLoader = static_cast<CEWebKitDocumentLoader*>(documentLoader);
	if (wkDocumentLoader && wkDocumentLoader->getDocumentLoaderClient())
	{
		wkDocumentLoader->getDocumentLoaderClient().dispatchDidFailLoading(identifier);
	}
}

bool FrameLoaderClientSilk::dispatchDidLoadResourceFromMemoryCache(WebCore::DocumentLoader* documentLoader, const WebCore::ResourceRequest& resourceRequest, const WebCore::ResourceResponse& resourceResponse, int length)
{
#if defined(USE_CENETWORK) || defined(USE_XAI)
	return true;
#else
	return false;
#endif
}


void FrameLoaderClientSilk::dispatchDidHandleOnloadEvents()
{
	
}

void FrameLoaderClientSilk::dispatchDidReceiveServerRedirectForProvisionalLoad()
{
}

void FrameLoaderClientSilk::dispatchDidCancelClientRedirect()
{
}

void FrameLoaderClientSilk::dispatchWillPerformClientRedirect(const WebCore::KURL& url, double interval, double fireDate)
{
}

void FrameLoaderClientSilk::dispatchDidChangeLocationWithinPage()
{
}

void FrameLoaderClientSilk::dispatchWillClose()
{
}

void FrameLoaderClientSilk::dispatchDidReceiveIcon()
{
}

void FrameLoaderClientSilk::dispatchDidStartProvisionalLoad()
{
}

void FrameLoaderClientSilk::dispatchDidReceiveTitle(const WebCore::String& title)
{
}

void FrameLoaderClientSilk::dispatchDidCommitLoad()
{
    if (_frame == _frame->page()->focusController()->focusedFrame())
    {
        CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
        frameWindow->inactivateInputMethodState(false);
    }

	if (_frame && _frame->loader() && _frame->page())
	{
		CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
		if (frameWindow)
		{
			// History.
			bool bBack = _frame->loader()->canGoBackOrForward(-1);
			bool bFwd = _frame->loader()->canGoBackOrForward(1);
			frameWindow->getChromeClient().setCanBackAndForward(bBack, bFwd);
			frameWindow->rebuildFocusNavigation();
		}
	}
}

void FrameLoaderClientSilk::dispatchDidFailProvisionalLoad(const WebCore::ResourceError& error)
{
	_isCommittedLoad = false;
	CEComAllocatorRec* allocator = CEComGetAllocatorRec();

	if (_frame && _frame->loader() && _frame->page() && allocator && _frame->loader()->isHostedByObjectElement() == false
	    && error.resultCode() != CE_SILK_ERR_OPERATION_CANCELLED )
	{
		CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
		if (frameWindow)
		{
			CEComICEUStringRef iHtmlStringRef = NULL;

#ifdef SHOW_INTERNAL_NETWORK_ERROR_STRING
			// SCE bug 38282 work around in Browser Core.
			frameWindow->getChromeClient().checkErrorOnRequestComplete(error.resultCode(), error.lowLevelErrorCode(), error.getIsMainRresource(), &iHtmlStringRef);
			CEHResult hr = ICEUStringCreate(CEComStdClassID_CEUString, iHtmlStringRef);
			if (CESucceeded(hr))
			{
				char errStrBuf[40];
				memset(errStrBuf, 0x00, 40);
				sprintf(errStrBuf, "<br><br><br>Network Error (%s)", reinterpret_cast<const char*>(error.getUserArg()));
				printf("Network Error = [%s]\n",errStrBuf);
				hr = iHtmlStringRef.initWithCStringLiteral(reinterpret_cast<const char*>(errStrBuf));
			}
#else
			bool useAppErrPage = false;
			CEComICEUStringRef appErrPageStr = NULL;
			CEHResult hr = frameWindow->getChromeClient().checkErrorOnRequestComplete(error.resultCode(), error.lowLevelErrorCode(), error.getIsMainRresource(), &iHtmlStringRef);
			if (CESucceeded(hr) && 0 != error.lowLevelErrorCode())
			{
				_window->networkErrorEventWait(useAppErrPage, &appErrPageStr);

				if (true == useAppErrPage && appErrPageStr)
					iHtmlStringRef = appErrPageStr;
			}
#endif

			if (CESucceeded(hr) && iHtmlStringRef)
			{
				UCHAR8* utf8Char = NULL;
				UINT32 byteLength;
				hr = iHtmlStringRef.getBytesWithAlloc(eICEI18nEncoding_utf_8, allocator, &utf8Char, &byteLength);
				if (CESucceeded(hr))
				{
					const WebCore::KURL url(error.failingURL());
					WTF::RefPtr<WebCore::SharedBuffer> data = WebCore::SharedBuffer::create(utf8Char, byteLength);
					WebCore::SubstituteData substituteData(data, WebCore::String("text/html"), WebCore::String("utf-8"), url);
					_frame->loader()->load(WebCore::ResourceRequest(url), substituteData, false);
					allocator->free(allocator, utf8Char);
				}
			}
		}
	}
}

void FrameLoaderClientSilk::dispatchDidFailLoad(const WebCore::ResourceError& err)
{
	_isCommittedLoad = false;
}

void FrameLoaderClientSilk::dispatchDidFinishDocumentLoad()
{
}

void FrameLoaderClientSilk::dispatchDidFinishLoad()
{
	if (_frame && _frame->loader() && _frame->page())
	{
		CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
		if (frameWindow)
		{
			//set back&forward status.
			bool bBack = _frame->loader()->canGoBackOrForward(-1);
			bool bFwd = _frame->loader()->canGoBackOrForward(1);
			frameWindow->getChromeClient().setCanBackAndForward(bBack, bFwd);
			frameWindow->rebuildFocusNavigation();
		}

	}
}

void FrameLoaderClientSilk::dispatchDidFirstLayout()
{
	if (_frame && _frame->loader() && _frame->page())
	{
		CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
		if (frameWindow)
		{
			frameWindow->rebuildFocusNavigation();
		}
	}
}


WebCore::Frame* FrameLoaderClientSilk::dispatchCreatePage()
{
	WebCore::Page* newPage = NULL;
	if (_frame && _frame->page())
	{
		CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
		if (frameWindow)
		{

//			CEHResult hr = frameWindow->getChromeClient().windowOpen(-1, -1, -1, -1,
			CEHResult hr = frameWindow->getChromeClient().windowOpen(0, 0, 400, 300,	//TODO:Dummy-value.
				_newTargetUrl.string().charactersSilk(), _newTargetUrl.string().length(), false, reinterpret_cast<void**>(&newPage));
			if(!CESucceeded(hr))
			{
				CEComDebugPrintf("FrameLoaderClientSilk::dispatchCreatePage: Opening new window error.\n");
			}
		}
	}

	return (newPage) ? newPage->mainFrame() : NULL;
}

void FrameLoaderClientSilk::dispatchShow()
{
}


void FrameLoaderClientSilk::dispatchDecidePolicyForMIMEType(WebCore::FramePolicyFunction func, const WebCore::String& MIMEType, const WebCore::ResourceRequest& resourceRequest)
{
	if (_frame && _frame->loader() && _frame->page())
	{
		WebCore::PolicyAction action = WebCore::PolicyDownload;
		if( MIMEType.isNull() || MIMEType.isEmpty() )
		{
			action = WebCore::PolicyIgnore;
		}
		else if (canShowMIMEType(MIMEType))
		{
			action = WebCore::PolicyUse;
		}
		else
		{	
			// Thorow the error code for silk.
			CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
			if (frameWindow)
			{
				WebCore::KURL wk_url = resourceRequest.url();
				if (!wk_url.isNull() && !wk_url.isEmpty() && wk_url.isValid())
				{
					WebCore::String wk_str = wk_url.string();
					bool canDownload = false;
					CEHResult hr = frameWindow->getChromeClient().checkCanDownloadedMimeType(
						MIMEType.charactersSilk(), (UINT32)MIMEType.length(),
						wk_str.charactersSilk(), (UINT32)wk_str.length(),
						&canDownload
						);
					// Can't download.
					if(CESucceeded(hr))
					{
						if(!canDownload)
						{
							frameWindow->getChromeClient().unsupportedMimeType(
								MIMEType.charactersSilk(), (UINT32)MIMEType.length(),
								wk_str.charactersSilk(), (UINT32)wk_str.length()
								);
							action = WebCore::PolicyIgnore;
						}
					}
				}
			}
		}
		(_frame->loader()->*func)(action);
	}	
}

void FrameLoaderClientSilk::dispatchDecidePolicyForNewWindowAction(WebCore::FramePolicyFunction func, const WebCore::NavigationAction& action, const WebCore::ResourceRequest& resourceRequest, PassRefPtr<WebCore::FormState> state, const WebCore::String& frameName)
{
    ASSERT(func);
    if (!func)
        return;

	_newTargetUrl = resourceRequest.url();//keep URL for New-window.

    // FIXME: I think Qt version marshals this to another thread so when we
    // have multi-threaded download, we might need to do the same
    (_frame->loader()->*func)(WebCore::PolicyUse);
}

void FrameLoaderClientSilk::dispatchDecidePolicyForNavigationAction(WebCore::FramePolicyFunction func, const WebCore::NavigationAction& action, const WebCore::ResourceRequest& resourceRequest, PassRefPtr<WebCore::FormState> state)
{
	CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
	if (frameWindow && _frame && _frame->loader() && _frame->page())
	{
		WebCore::KURL reqUrl = resourceRequest.url();
		WebCore::KURL wk_url = action.url();
		WebCore::KURL srcUrl(_frame->page()->mainFrame()->loader()->url());

		WebCore::String wk_str = wk_url.string();
		bool supported_scheme = false;
		CEComICEURLRef iChkUrl;
		CEHResult chk_err = wk_url.createICEURL(&iChkUrl);
		if (CESucceeded(chk_err))
		{
			chk_err = iChkUrl.hasSupportedScheme(&supported_scheme);
		}
		if (CESucceeded(chk_err) && !supported_scheme)
		{
			frameWindow->getChromeClient().unsupportedScheme(wk_str.charactersSilk(), (UINT32)wk_str.length());
			(_frame->loader()->*func)(WebCore::PolicyIgnore);
			return;
		}
		if (CEFailed(chk_err))
		{
			frameWindow->getChromeClient().urlCreationFailed(wk_str.charactersSilk(), (UINT32)wk_str.length());
			(_frame->loader()->*func)(WebCore::PolicyIgnore);
			return;
		}

		// Check entering to or leaving from secure site
		bool is_next_secure = reqUrl.protocolIs("https");
		bool is_prev_secure = srcUrl.protocolIs("https");

		CEHResult hr = CE_S_OK;
		bool allowed = true;
		bool is_main_doc = _frame->page()->mainFrame() == _frame;
		if (is_main_doc)
		{
			resourceRequest.setIsMainResource();
			if (true == resourceRequest.getRedirected() && !is_next_secure && is_prev_secure)
			{
				hr = frameWindow->getChromeClient().willLeaveFromSecureSiteAtRedirection(srcUrl.string().utf8().data(), reqUrl.string().utf8().data(), &allowed);
			}
			else
			{
				if (is_next_secure && !is_prev_secure || !is_next_secure && is_prev_secure)
				{
					hr = frameWindow->getChromeClient().willEnterToOrLeaveFromSecureSite(is_next_secure, srcUrl.string().utf8().data(), reqUrl.string().utf8().data(), &allowed);
				}
			}
		}
		if (allowed && action.type() == WebCore::NavigationTypeFormResubmitted)
		{
			hr = frameWindow->getChromeClient().willResubmitForm(srcUrl.string().utf8().data(), reqUrl.string().utf8().data(), &allowed);
		}
		else if (allowed && state && !is_next_secure)
		{
			bool isPost = resourceRequest.httpMethod() == "POST";
			hr = frameWindow->getChromeClient().willSubmitInsecureForm(is_prev_secure, isPost, srcUrl.string().utf8().data(), reqUrl.string().utf8().data(), &allowed);
		}
		if (CESucceeded(hr) && !allowed)
		{
			if (action.type() == WebCore::NavigationTypeFormSubmitted || action.type() == WebCore::NavigationTypeFormResubmitted)
				_frame->loader()->resetMultipleFormSubmissionProtection();
			(_frame->loader()->*func)(WebCore::PolicyIgnore);
			return;
		}
	}

	if (resourceRequest.cachePolicy() == WebCore::ReloadIgnoringCacheData && action.type() != WebCore::NavigationTypeReload)
	{
		resourceRequest.reloadPolicyInvalid();
	}

	(_frame->loader()->*func)(WebCore::PolicyUse);

}

void FrameLoaderClientSilk::cancelPolicyCheck()
{
}

void FrameLoaderClientSilk::dispatchWillSubmitForm(WebCore::FramePolicyFunction func, PassRefPtr<WebCore::FormState>)
{
	if (_frame && _frame->loader() && _frame->page())
	{
		(_frame->loader()->*func)(WebCore::PolicyUse);
	}
}

void FrameLoaderClientSilk::dispatchDidLoadMainResource(WebCore::DocumentLoader* documentLoader)
{
}

void FrameLoaderClientSilk::revertToProvisionalState(WebCore::DocumentLoader*)
{
}

void FrameLoaderClientSilk::setMainDocumentError(WebCore::DocumentLoader* documentLoader, const WebCore::ResourceError& err)
{
	err.setIsMainResource();
}

void FrameLoaderClientSilk::setMainFrameDocumentReady(bool)
{
}

void FrameLoaderClientSilk::willChangeTitle(WebCore::DocumentLoader* documentLoader)
{
}

void FrameLoaderClientSilk::didChangeTitle(WebCore::DocumentLoader* documentLoader)
{
}


void FrameLoaderClientSilk::committedLoad(WebCore::DocumentLoader* documentLoader, const char* data, int length)
{
	if (documentLoader && _frame && _frame->loader() && _frame->loader()->documentLoader())
	{
		const WebCore::String& textEncoding = documentLoader->response().textEncodingName();

		WebCore::String encoding = _frame->loader()->documentLoader()->overrideEncoding();
		bool userChosen = !encoding.isNull();
		if (!userChosen)
		{
			encoding = textEncoding;
		}

		_frame->loader()->setEncoding(encoding, userChosen);

		if (_frame->page())
		{
			CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
			WebCore::FrameView* frameView = _frame->view();
			bool isMainFrame = (_frame == _frame->page()->mainFrame());
			if (frameView && frameWindow && isMainFrame)
			{
				WebCore::ScrollbarMode horizontalMode;
				WebCore::ScrollbarMode verticalMode;

				frameWindow->getPrivate()->getScrollbarMode(horizontalMode, verticalMode);
				frameView->setScrollbarModes(horizontalMode, verticalMode);
				frameView->updateDefaultScrollbarState();
			}
		}

		if ( (documentLoader->response().url().string() != _frame->document()->url().string()))
		{
			_frame->document()->setDocumentURI(documentLoader->response().url());
		}

		// Document on focus disable
		_isCommittedLoad = true;

		_frame->loader()->addData(data, length);
	}
}

void FrameLoaderClientSilk::finishedLoading(WebCore::DocumentLoader* documentLoader)
{
	WebCore::String szTitle = documentLoader->title();

	// Don't use urlForHistory() because here is fainal point of request.
	// Although urlForHistory() was used in FrameLoader::didChangeTitle(),
	// that point is in middle of processing request.
	// So we need to use url of last decision.
	// If you want to use another method for following code,
	// be sure to check that method create de-grade for any pattern of HTTP request.
	// (etc. redirection).
	WebCore::KURL url = documentLoader->url();

	if ( url == WebCore::blankURL() )
		committedLoad(documentLoader, 0, 0);

	setTitle(szTitle, url);
}

void FrameLoaderClientSilk::updateGlobalHistory()
{
}

bool FrameLoaderClientSilk::shouldGoToHistoryItem(WebCore::HistoryItem* historyItem) const
{
	return true;
}



void FrameLoaderClientSilk::postProgressStartedNotification()
{
	if (_frame && _frame->loader() && _frame->page() && _frame == _frame->page()->mainFrame())
	{
		CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
		if (frameWindow)
		{
			frameWindow->getChromeClient().progressStartNotification();
		}
	}
}

void FrameLoaderClientSilk::postProgressEstimateChangedNotification()
{
	if (_frame && _frame->loader() && _frame->page() && _frame == _frame->page()->mainFrame())
	{
		CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
		if (frameWindow)
		{
			WebCore::ProgressTracker* tracker = _frame->page()->progress();
			double estimated = tracker->estimatedProgress();
			frameWindow->getChromeClient().progressPercentageNotification(estimated);
		}
	}
}

void FrameLoaderClientSilk::postProgressFinishedNotification()
{
	if (_frame && _frame->loader() && _frame->page() && _frame == _frame->page()->mainFrame())
	{
		CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
		if (frameWindow)
		{
			frameWindow->getChromeClient().progressFinishedNotification();
		}
	}
}


bool FrameLoaderClientSilk::canHandleRequest(const WebCore::ResourceRequest&) const
{
	return true;
}

bool FrameLoaderClientSilk::canShowMIMEType(const WebCore::String& MIMEType) const
{
#if 1 // Bug 18240 enable image/jps temporary.
	return MIMEType == "image/jps" || WebCore::MIMETypeRegistry::isSupportedImageMIMEType(MIMEType) || WebCore::MIMETypeRegistry::isSupportedNonImageMIMEType(MIMEType) ||
		WebCore::PluginDatabase::installedPlugins()->isMIMETypeRegistered(MIMEType);
#else
	return WebCore::MIMETypeRegistry::isSupportedImageMIMEType(MIMEType) || WebCore::MIMETypeRegistry::isSupportedNonImageMIMEType(MIMEType) ||
		WebCore::PluginDatabase::installedPlugins()->isMIMETypeRegistered(MIMEType);
#endif
}

bool FrameLoaderClientSilk::representationExistsForURLScheme(const WebCore::String& URLScheme) const
{
	return false;
}

WebCore::String FrameLoaderClientSilk::generatedMIMETypeForURLScheme(const WebCore::String& URLScheme) const
{
	WebCore::String s;
	return s;
}

void FrameLoaderClientSilk::frameLoadCompleted()
{
}

void FrameLoaderClientSilk::restoreViewState()
{
}

void FrameLoaderClientSilk::provisionalLoadStarted()
{
}

bool FrameLoaderClientSilk::shouldTreatURLAsSameAsCurrent(const WebCore::KURL&) const
{
	return false; // maybe shirnk history for same url?
}

void FrameLoaderClientSilk::addHistoryItemForFragmentScroll()
{
}

void FrameLoaderClientSilk::didFinishLoad()
{
}

void FrameLoaderClientSilk::prepareForDataSourceReplacement()
{
}


PassRefPtr<WebCore::DocumentLoader> FrameLoaderClientSilk::createDocumentLoader(const WebCore::ResourceRequest& request, const WebCore::SubstituteData& substituteData)
{
	RefPtr<CEWebKitDocumentLoader> loader = CEWebKitDocumentLoader::create(request, substituteData);

	if (_frame && _frame->loader() && _frame->page())
	{
		CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
		if (frameWindow)
		{
			CEComICEHtmlWebKitDocumentLoaderClientRef iDocumentLoaderClientRef = NULL;
			CEHResult hr = frameWindow->getLoaderClient().createDocumentLoader(&iDocumentLoaderClientRef);
			if (CESucceeded(hr))
			{
				loader->setDocumentLoaderClient(iDocumentLoaderClientRef);
			}
		}
	}

	return loader.release();
	
}

void FrameLoaderClientSilk::setTitle(const WebCore::String& title, const WebCore::KURL& url)
{
	if (_frame && _frame->loader() && _frame->page())
	{
		CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
		if (frameWindow && _frame == _frame->page()->mainFrame())
		{
			// Title
			frameWindow->getChromeClient().titleChange(title.charactersSilk(), (UINT32)title.length());

			// URL
			frameWindow->getChromeClient().urlChange(url.string().charactersSilk(), url.string().length());
		}
	}
}



WebCore::String FrameLoaderClientSilk::userAgent(const WebCore::KURL& url)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;

	CEComICEHtmlWebKitDeviceInfoRef iwebkitDeviceInfoRef=0;
	CEComGetThreadContext(CEComIID_ICEHtmlWebKitDeviceInfo, reinterpret_cast<void**>(&iwebkitDeviceInfoRef));
	WebCore::String wUserAgent("");

	if(iwebkitDeviceInfoRef)
	{
		CEComICEUStringRef szUserAgent = 0;
		hr = iwebkitDeviceInfoRef.getUserAgent(&szUserAgent);
		if(CESucceeded(hr))
		{
			wUserAgent.fromICEUString(szUserAgent);
		}
	}

	return wUserAgent;
}

void FrameLoaderClientSilk::transitionToCommittedForNewPage()
{
	if(_frame && _frame->page())
	{
		bool transparent = false;
		WebCore::Color backgroundColor = transparent ? WebCore::Color::transparent : WebCore::Color::white;
		CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
		if(frameWindow)
		{
			CERectF cerect;
			CEHResult hr;
			CEComICEHtmlWebKitFrameWindowChromeClientRef iWebKitFrameWindowChromeClientRef = frameWindow->getChromeClient();
			if(iWebKitFrameWindowChromeClientRef)
			{
				hr = iWebKitFrameWindowChromeClientRef.getWindowRect(&cerect);
				WebCore::IntSize size;
				if (CESucceeded(hr))
					size = WebCore::IntSize(cerect._width, cerect._height);

				CEWebKitHtmlScrollMode scrollMode;
				hr = iWebKitFrameWindowChromeClientRef.getScrollMode(&scrollMode);
				if (!hr)
				{
					WebCore::ScrollbarMode webCoreScrollbarMode = WebCore::ScrollbarAuto;
					if (scrollMode==CEWebKitHtmlScrollMode_DocumentZoomFactor2)
					{
						webCoreScrollbarMode = WebCore::ScrollbarAlwaysOff;
					}
					_frame->createView(size, backgroundColor, transparent, WebCore::IntSize(), false, webCoreScrollbarMode, webCoreScrollbarMode);
					CEWebKitWindowImpl* ceWkWindow = kit(_frame);
					if (ceWkWindow && ceWkWindow->getView())
					{
						ceWkWindow->getView()->setFrameView(_frame->view());

						// If scrollMode is Document mode, Silk can be flick scroll mode, and Creating Platform Widget.
						if (scrollMode==CEWebKitHtmlScrollMode_Document || scrollMode==CEWebKitHtmlScrollMode_DocumentZoomFactor2)
						{
							if (_frame->view())
							{
								if (_frame == _frame->page()->mainFrame())
								{
									_frame->view()->setPlatformWidget(ceWkWindow->getView());
									hr = iWebKitFrameWindowChromeClientRef.notifyChangePage();
									if (_frame->loader())
									{
										WebCore::FrameLoadType loadType;
										loadType = _frame->loader()->loadType();
										
										switch(loadType)
										{
										case WebCore::FrameLoadTypeBack:
										case WebCore::FrameLoadTypeForward:
										case WebCore::FrameLoadTypeIndexedBackForward:
										case WebCore::FrameLoadTypeReload:
										case WebCore::FrameLoadTypeReloadFromOrigin:

											break;
										default:
											iWebKitFrameWindowChromeClientRef.setScrollPosition(0, 0);
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
}

bool FrameLoaderClientSilk::canCachePage() const
{
	return true;
}


void FrameLoaderClientSilk::saveViewStateToItem(WebCore::HistoryItem *)
{
}


WebCore::ResourceError FrameLoaderClientSilk::cancelledError(const WebCore::ResourceRequest& request)
{
	_isCommittedLoad = false;
	WebCore::ResourceError err("", 0, request.url(), "", CE_SILK_ERR_OPERATION_CANCELLED, CE_S_OK, 0);
	return err;
}

WebCore::ResourceError FrameLoaderClientSilk::blockedError(const WebCore::ResourceRequest& request)
{
	_isCommittedLoad = false;
	WebCore::ResourceError err("", 0, request.url(), "", CE_SILK_ERR_CANNOT_BE_DISPLAYED, CE_S_OK, 0);
	return err;
}

WebCore::ResourceError FrameLoaderClientSilk::cannotShowURLError(const WebCore::ResourceRequest& request)
{
	_isCommittedLoad = false;
	WebCore::ResourceError err("", 0, request.url(), "", CE_SILK_ERR_INVALID_ADDRESS, CE_S_OK, 0);
	return err;
}

WebCore::ResourceError FrameLoaderClientSilk::interruptForPolicyChangeError(const WebCore::ResourceRequest& request)
{
	_isCommittedLoad = false;
	WebCore::ResourceError err("", 0, request.url(), "", CE_SILK_ERR_OPERATION_CANCELLED, CE_S_OK, 0);
	return err;
}

WebCore::ResourceError FrameLoaderClientSilk::cannotShowMIMETypeError(const WebCore::ResourceResponse& response)
{
	_isCommittedLoad = false;
	WebCore::ResourceError err("", 0, response.url(), "", CE_SILK_ERR_CANNOT_BE_DISPLAYED, CE_S_OK, 0);
	return err;
}

WebCore::ResourceError FrameLoaderClientSilk::fileDoesNotExistError(const WebCore::ResourceResponse& response)
{
	_isCommittedLoad = false;
	WebCore::ResourceError err("", 0, response.url(), "", CE_SILK_ERR_OPERATION_FAILED, CE_S_OK, 0);
	return err;
}

WebCore::ResourceError FrameLoaderClientSilk::pluginWillHandleLoadError(const WebCore::ResourceResponse& response)
{
	_isCommittedLoad = false;
	WebCore::ResourceError err("", 0, response.url(), "", CE_SILK_ERR_OPERATION_FAILED, CE_S_OK, 0);
	return err;
}

bool FrameLoaderClientSilk::shouldFallBack(const WebCore::ResourceError&)
{
	return false;
}

void FrameLoaderClientSilk::dispatchUnableToImplementPolicy(const WebCore::ResourceError&)
{
}

void FrameLoaderClientSilk::download(WebCore::ResourceHandle*, const WebCore::ResourceRequest&, const WebCore::ResourceRequest&, const WebCore::ResourceResponse&)
{
}

PassRefPtr<WebCore::Frame> FrameLoaderClientSilk::createFrame(const WebCore::KURL& url, const WebCore::String& name, WebCore::HTMLFrameOwnerElement* ownerElement, const WebCore::String& referrer, bool allowsScrolling, int marginWidth, int marginHeight)
{
	RefPtr<WebCore::Frame> frame = NULL;
	if (_frame && _frame->page() && _frame->tree())
	{
		CEWebKitWindow* window = new CEWebKitWindow();
		if (window && window->object())
		{
			CEHResult hr = window->object()->init(_frame->page(), ownerElement);
			if (CESucceeded(hr))
			{
				frame = core(window->object());
				if (frame && frame->loader() && frame->tree())
				{
					frame->tree()->setName(name);
					_frame->tree()->appendChild(frame);
					_frame->loader()->loadURLIntoChildFrame(url, referrer, frame.get());
				}
			}
		}
	}
	return frame.release();
}

WTF::PassRefPtr<WebCore::Widget> FrameLoaderClientSilk::createPlugin(const WebCore::IntSize& pluginSize, WebCore::HTMLPlugInElement* element, const WebCore::KURL& url, const Vector<WebCore::String>& paramNames, const Vector<WebCore::String>& paramValues, const WebCore::String& mimeType, bool loadManually)
{

	using namespace WebCore;
	WTF::PassRefPtr<PluginView> prpWidget = 0;

	//-----------------------------------------------------------------------------------------------
	// IMPORTANT!!!! 
	// This is copied the code from create function on PluginView.cpp, If we update the WebKit, 
	//we should check this code.
	String mimeTypeCopy = mimeType;
	PluginPackage* plugin = PluginDatabase::installedPlugins()->findPlugin(url, mimeTypeCopy);

    // No plugin was found, try refreshing the database and searching again
    if (!plugin && PluginDatabase::installedPlugins()->refresh()) {
        mimeTypeCopy = mimeType;
        plugin = PluginDatabase::installedPlugins()->findPlugin(url, mimeTypeCopy);
    }
	// End IMPORTANT
	//-----------------------------------------------------------------------------------------------
	if(plugin && _frame && _frame->page())
	{
		//call requestConfirmation
		CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page());
		if (frameWindow)
		{
			CEComICEHtmlWebKitFrameWindowChromeClientRef chromeclient = frameWindow->getChromeClient();
			if(chromeclient)
			{
				CEComICEHtmlPluginConfirmationRef iPluginConfirmationRef;
				CEHResult hr =iPluginConfirmationRef.initByQueryInterface(chromeclient);
				if(!hr)
				{
					WTF::PassRefPtr<WebCore::Widget> pluginWidget = adoptRef(new PluginConfirmWidget(_frame, pluginSize, element, url, paramNames, paramValues, mimeType, loadManually));
					if(pluginWidget)
					{
						hr = iPluginConfirmationRef.requestConfirmPlugin(pluginWidget.get());
						if(!hr)
							return pluginWidget;
					}
				}
			}
		}
		return NULL;
		//
	}
	//
    WTF::RefPtr<PluginView> pluginView = PluginView::create(_frame, pluginSize, element, url, paramNames, paramValues, mimeType, loadManually);
	return pluginView;
}

void FrameLoaderClientSilk::redirectDataToPlugin(WebCore::Widget* pluginWidget)
{
	
}


void FrameLoaderClientSilk::startDownload(const WebCore::ResourceRequest&)
{
}


PassRefPtr<WebCore::Widget> FrameLoaderClientSilk::createJavaAppletWidget(const WebCore::IntSize&, WebCore::HTMLAppletElement*, const WebCore::KURL& baseURL, const Vector<WebCore::String>& paramNames, const Vector<WebCore::String>& paramValues)
{
	return NULL;
}


WebCore::ObjectContentType FrameLoaderClientSilk::objectContentType(const WebCore::KURL& url, const WebCore::String& mimeTypeIn)
{
	WebCore::String mimeType = mimeTypeIn;
    if (mimeType.isEmpty())
		mimeType = WebCore::MIMETypeRegistry::getMIMETypeForExtension(url.path().substring(url.path().reverseFind('.') + 1));

    if (mimeType.isEmpty())
		return WebCore::ObjectContentFrame; // Go ahead and hope that we can display the content.

	if (WebCore::MIMETypeRegistry::isSupportedImageMIMEType(mimeType))
        return WebCore::ObjectContentImage;

	if (WebCore::PluginDatabase::installedPlugins()->isMIMETypeRegistered(mimeType))
        return WebCore::ObjectContentNetscapePlugin;

	if (WebCore::MIMETypeRegistry::isSupportedNonImageMIMEType(mimeType))
        return WebCore::ObjectContentFrame;

    return WebCore::ObjectContentNone;
}

WebCore::String FrameLoaderClientSilk::overrideMediaType() const
{
	WebCore::String s;
	return s;
}

#include "CEA2014ClientSilk.h"
#include "WindowExternalImpl/CEJSWindowExternal.h"
#include "WindowSessionStorageImpl/CEJSWindowSessionStorage.h"
void FrameLoaderClientSilk::windowObjectCleared()
{

	if( _frame )
	{
		if (_isCEA2014ModeEnabled)
		{
			AddCEA2014JavascriptInterface(WebCore::toJSDOMWindow(_frame));
			_isCEA2014ModeEnabled = false;
		}
		AddWindowSessionStorageJavascriptInterface(WebCore::toJSDOMWindow(_frame));
		AddWindowExternalJavascriptInterface(WebCore::toJSDOMWindow(_frame));
	}
	else
	{
		// TODO ASSERT
	}
}

void FrameLoaderClientSilk::didPerformFirstNavigation() const
{
}


void FrameLoaderClientSilk::registerForIconNotification(bool listen)
{
}

// Check!! What is keepResult ?
bool FrameLoaderClientSilk::confirmPlugin(bool result, bool keepResult, void *const cData)
{
	using namespace WebCore;
	bool returnValue = false;
	if(!keepResult)
	{
		if(result)
		{
			// Referenced from Android's source code.
			Frame* frame = _frame;
			while(frame)
			{
				RenderView* view = frame->contentRenderer();
				if(view)
				{
					const HashSet<RenderWidget*> widgets = view->getWidgets();
					HashSet<RenderWidget*>::const_iterator it = widgets.begin();
					HashSet<RenderWidget*>::const_iterator end = widgets.end();
					for(; it!=end; ++it)
					{
						Widget* widget = (*it)->widget();
						if(widget==cData)
						{
							PluginConfirmWidget* pluginDummy = static_cast<PluginConfirmWidget*>(cData);
							ASSERT(pluginDummy);
							if(pluginDummy)
							{
								returnValue = pluginDummy->swapPlugin(*it);
							}
						}
						else
							returnValue = true;
					}
				}
				frame = frame->tree()->traverseNext();
			}
		}
		else
		{
			returnValue = true;
		}
	}
	else
	{
		returnValue = true;
	}
	return returnValue;
}

CEHResult FrameLoaderClientSilk::setCertificate(void* certIn)
{
	if (_cert) {
		WebCore::ResourceHandleManager::sharedInstance()->deleteFrameLoderClientCert(_cert);
	}
	_cert = certIn;
	return CE_S_OK;
}

CEHResult FrameLoaderClientSilk::getCertificate(const void** certOut)
{
	if (certOut)
	{
		*certOut = _cert;
		if (_cert)
		{
			_cert = NULL;
		}
	}
	return CE_S_OK;
}

void FrameLoaderClientSilk::setIdentifier(unsigned long identifier, bool isMainResource)
{
	_idMap.set(identifier,  isMainResource);
}
