/*
 * Copyright (C) 2006, 2007, 2008 Apple, Inc. All rights reserved.
 * Copyright     2009, 2012 Sony Corporation
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#include "WebKitSilkPrefix.h"

#include "HTMLNames.h"
#include "HTMLAnchorElement.h"
#include "ChromeClientSilk.h"
#include "Frame.h"
#if ENABLE(DATABASE)
#include "DatabaseTracker.h"
#endif
//for Bug 17207.
#include "FrameLoadRequest.h"
#include "WindowFeatures.h"

#include "FrameView.h"
#include "HitTestResult.h"
#include "HTMLInputElement.h"

ChromeClientSilk::ChromeClientSilk(CEWebKitFrameWindowImpl* frameWindow) :
	_frameWindow(frameWindow),
	_lastMouseOverNode(NULL)
{
}

void ChromeClientSilk::chromeDestroyed()
{
	delete this;
}


void ChromeClientSilk::setWindowRect(const WebCore::FloatRect& rect)
{
	if (_frameWindow)
	{
		CERectF cerect = { rect.x(), rect.y(), rect.width(), rect.height() };
		_frameWindow->getChromeClient().setWindowRect(&cerect);
	}
}

WebCore::FloatRect ChromeClientSilk::windowRect()
{
	WebCore::FloatRect rect;
	if (_frameWindow)
	{
		CERectF cerect;
		_frameWindow->getChromeClient().getWindowRect(&cerect);
		rect.setX(cerect._x);
		rect.setY(cerect._y);
		rect.setWidth(cerect._width);
		rect.setHeight(cerect._height);
	}	
	return rect;
}

WebCore::FloatRect ChromeClientSilk::pageRect()
{
	return windowRect();
}


float ChromeClientSilk::scaleFactor()
{
	return 1.0f;
}


void ChromeClientSilk::focus()
{
}

void ChromeClientSilk::unfocus()
{
}


bool ChromeClientSilk::canTakeFocus(WebCore::FocusDirection focusDir)
{
	return true;
}

void ChromeClientSilk::takeFocus(WebCore::FocusDirection focusDir)
{
}

void ChromeClientSilk::focusedNodeChanged(WebCore::Node* node)
{
    CEComICEURLRef url;    
    CEComICEUStringRef text;
    CEHResult hr;
    if (node && node->hasTagName(WebCore::HTMLNames::aTag))
    {
        WebCore::HTMLAnchorElement* anchor = static_cast<WebCore::HTMLAnchorElement*>(node);
        WebCore::KURL href = anchor->href();
        hr = href.createICEURL(&url);
        WebCore::String w_text(anchor->text());
        hr = w_text.createICEUString(&text);
    }

    bool ignoreFocusTextImput = false;
    _frameWindow->updateInputMethodState(&ignoreFocusTextImput);
    if (node && !ignoreFocusTextImput)
    {
	    hr = _frameWindow->getChromeClient().focusedNodeChanged(url, text, reinterpret_cast<UINT32>(node), reinterpret_cast<UINT32>(node->renderer()));
    }
}

WebCore::Page* ChromeClientSilk::createWindow(WebCore::Frame* frame, const WebCore::FrameLoadRequest& frameLoadRequest, const WebCore::WindowFeatures& windowFeature)
{
	WebCore::Page* newPage = NULL;
	if (_frameWindow)
	{
		_frameWindow->getChromeClient().windowOpen(windowFeature.x, windowFeature.y, windowFeature.width, windowFeature.height,
			frameLoadRequest.resourceRequest().url().string().charactersSilk(), frameLoadRequest.resourceRequest().url().string().length(), true, reinterpret_cast<void**>(&newPage));
	}
	return newPage;
}

void ChromeClientSilk::show()
{
}


bool ChromeClientSilk::canRunModal()
{
	return false;
}

void ChromeClientSilk::runModal()
{
}


void ChromeClientSilk::setToolbarsVisible(bool visible)
{
}

bool ChromeClientSilk::toolbarsVisible()
{
	return true;
}


void ChromeClientSilk::setStatusbarVisible(bool visible)
{
}

bool ChromeClientSilk::statusbarVisible()
{
	return true;
}


void ChromeClientSilk::setScrollbarsVisible(bool visible)
{
	_frameWindow->getChromeClient().setAllowsScrolling(visible);
}

bool ChromeClientSilk::scrollbarsVisible()
{
	return true;
}


void ChromeClientSilk::setMenubarVisible(bool visible)
{
}

bool ChromeClientSilk::menubarVisible()
{
	return true;
}


void ChromeClientSilk::setResizable(bool resizable)
{
}


void ChromeClientSilk::addMessageToConsole(WebCore::MessageSource source, WebCore::MessageLevel level, const WebCore::String& message, unsigned int lineNumber, const WebCore::String& sourceID)
{
}


bool ChromeClientSilk::canRunBeforeUnloadConfirmPanel()
{
	return true;
}

bool ChromeClientSilk::runBeforeUnloadConfirmPanel(const WebCore::String& message, WebCore::Frame* frame)
{
	return true;
}


void ChromeClientSilk::closeWindowSoon()
{
	if (_frameWindow)
	{
		_frameWindow->getChromeClient().windowClose();
	}
}


void ChromeClientSilk::runJavaScriptAlert(WebCore::Frame* frame, const WebCore::String& message)
{
	if (_frameWindow)
	{
		_frameWindow->getChromeClient().alert(message.charactersSilk(), (UINT32)message.length());
	}
}

bool ChromeClientSilk::runJavaScriptConfirm(WebCore::Frame* frame, const WebCore::String& message)
{
	bool bRet = false;

	if (_frameWindow)
	{
		_frameWindow->getChromeClient().confirm(message.charactersSilk(), (UINT32)message.length(), &bRet);
	}

	return bRet;
}

bool ChromeClientSilk::runJavaScriptPrompt(WebCore::Frame* frame, const WebCore::String& message, const WebCore::String& defaultValue, WebCore::String& result)
{
	if (_frameWindow)
	{
		CEComICEUStringRef wk_Outmsg;
		UINT32 max = NULL;

		_frameWindow->getChromeClient().prompt(
			message.charactersSilk(), (UINT32)message.length(),
			defaultValue.charactersSilk(), (UINT32)defaultValue.length(),
			&wk_Outmsg, &max);

		CEHResult hr = result.fromICEUString(wk_Outmsg);
	}

	return true;
}

void ChromeClientSilk::setStatusbarText(const WebCore::String& message)
{
}

bool ChromeClientSilk::shouldInterruptJavaScript()
{
	CEHResult hr;
	bool interrupt = true;
	if (_frameWindow)
	{
		hr = _frameWindow->getChromeClient().shouldInterruptJavaScript(&interrupt);
	}
	return interrupt;
}

bool ChromeClientSilk::tabsToLinks() const
{
	return true;//?
}


WebCore::IntRect ChromeClientSilk::windowResizerRect() const
{
	WebCore::IntRect r;
	return r;
}


void ChromeClientSilk::repaint(const WebCore::IntRect& rect, bool contentChanged, bool immediate, bool repaintContentOnly)
{
	if (_frameWindow)
	{
#if USE(ACCELERATED_COMPOSITING)
		_frameWindow->getPrivate()->invalidateExceptForLayersBackingStore();
#endif //#if USE(ACCELERATED_COMPOSITING)
		_frameWindow->repaint(rect);
	}
}

void ChromeClientSilk::scroll(const WebCore::IntSize& scrollDelta, const WebCore::IntRect& rectToScroll, const WebCore::IntRect& clipRect)
{
}

WebCore::IntPoint ChromeClientSilk::screenToWindow(const WebCore::IntPoint& pt) const
{
	return pt;
}

WebCore::IntRect ChromeClientSilk::windowToScreen(const WebCore::IntRect& pt) const
{
	return pt;
}

PlatformWidget ChromeClientSilk::platformWindow() const
{
	PlatformWidget pw = NULL;
	if (_frameWindow)
	{
		CEWebKitWidget* pWebKitWidget = NULL;
		CEHResult hr = _frameWindow->getWebKitWidget(&pWebKitWidget);
		if (!hr)
		{
			pw = pWebKitWidget;
		}
	}
	return pw;
}


void ChromeClientSilk::mouseDidMoveOverElement(const WebCore::HitTestResult& result, unsigned modifierFlags)
{
	WebCore::Node* node = result.innerNode();

	if(node && node != _lastMouseOverNode)
	{
		WebCore::KURL linkKUrl = result.absoluteLinkURL();
		CEComICEURLRef linkUrl;    
		if (!linkKUrl.isNull())
		{
			linkKUrl.createICEURL(&linkUrl);
		}
		
		WebCore::KURL imageKUrl = result.absoluteImageURL();
		CEComICEURLRef imageUrl;
		if (!imageKUrl.isNull())
		{
			imageKUrl.createICEURL(&imageUrl);
		}

		//CEHtmlFocusItemType type = CEHtmlFocusItemType_None;
		//WebCore::RenderObject* renderer = node->renderer();
		//if (renderer)
		//{
		//	if (renderer->isTextField())
		//	{
		//		type = CEHtmlFocusItemType_TextLine;
		//	}
		//	else if (renderer->isTextArea())
		//	{
		//		type = CEHtmlFocusItemType_TextArea;
		//	}		
		//	else if(renderer->isRenderButton() || renderer->isSlider() || renderer->isListBox() || renderer->isMenuList() || renderer->isWidget())
		//	{
		//		type = CEHtmlFocusItemType_Anchor;
		//	}
		//	//else if (renderer->isTextControl())
		//	//{
		//	//}
		//	//else if (renderer->isImage())
		//	//{
		//	//}
		//}		
		//if (type == CEHtmlFocusItemType_None)
		//{
		//	if (node->hasTagName(WebCore::HTMLNames::canvasTag))
		//	{
		//		type = CEHtmlFocusItemType_Canvas;
		//	}
		//	else if (node->hasTagName(WebCore::HTMLNames::inputTag))
		//	{
		//		WebCore::HTMLInputElement* input = static_cast<WebCore::HTMLInputElement*>(node);
		//		if (input->inputType() == WebCore::HTMLInputElement::CHECKBOX || input->inputType() == WebCore::HTMLInputElement::RADIO)
		//		{
		//			type = CEHtmlFocusItemType_Anchor;
		//		}
		//	}
		//	else if (node->shouldUseInputMethod())
		//	{
		//		type = CEHtmlFocusItemType_ContentEditable;
		//	}
		//	else if (!linkKUrl.isNull() && !linkKUrl.isEmpty())
		//	{
		//		type = CEHtmlFocusItemType_Anchor;
		//	}
		//}

		_frameWindow->getChromeClient().notifyMouseOverElementChanged(linkUrl, imageUrl);//, type);
		_lastMouseOverNode = node;
	}
}


void ChromeClientSilk::setToolTip(const WebCore::String& message)
{
}


void ChromeClientSilk::print(WebCore::Frame* farme)
{
}


#if ENABLE(DATABASE)
void ChromeClientSilk::exceededDatabaseQuota(WebCore::Frame* frame, const WebCore::String& databaseName)
{
    // Set to 5M for testing
    // FIXME: Make this configurable
    const unsigned long long defaultQuota = 5 * 1024 * 1024;
    WebCore::DatabaseTracker::tracker().setQuota(frame->document()->securityOrigin(), defaultQuota);
}
#endif

WebCore::String ChromeClientSilk::generateReplacementFile(const WebCore::String& path)
{
	WebCore::String truePath;
	if (_uploadFileMap.contains(path))
	{
		truePath = _uploadFileMap.get(path);
	}
	return truePath;
}

bool ChromeClientSilk::shouldReplaceWithGeneratedFileForUpload(const WebCore::String& name, WebCore::String& path)
{
#if 0
	if (_uploadFileMap.contains(name))
	{
		path = _uploadFileMap.get(name);
		return true;
	}
	return false;
#else
	path = name;
	return true;
#endif
}

void ChromeClientSilk::runOpenPanel(WebCore::Frame* frame, PassRefPtr<WebCore::FileChooser> fileChooser)
{
	RefPtr<WebCore::FileChooser> chooser = fileChooser;

	CEComICEUStringRef fileName = NULL, filePath = NULL;
	ICEUStringCreate(CEComStdClassID_CEUString, fileName);
	ICEUStringCreate(CEComStdClassID_CEUString, filePath);
	if (filePath)
	{
		_frameWindow->getChromeClient().requestSyncFileChooser(&fileName, &filePath);
		char* fname = NULL, *fpath = NULL;
		UINT32 fnameLen = 0, fpathLen = 0;
		bool isEmpty = false;
		fileName.isEmpty(&isEmpty);
		if (!isEmpty)
		{
			fileName.getBytesWithAlloc(eICEI18nEncoding_utf_8, CEComGetAllocatorRec(), reinterpret_cast<UCHAR8 ** const>(&fname), &fnameLen);
			CEComDebugPrintf("CEHtmlWebKitFrameWindowImpl::requestFileChooser() file str = [%hs]\n", fname);
		}
		filePath.isEmpty(&isEmpty);
		if (!isEmpty)
		{
			filePath.getBytesWithAlloc(eICEI18nEncoding_utf_8, CEComGetAllocatorRec(), reinterpret_cast<UCHAR8 ** const>(&fpath), &fpathLen);
		}

		if (fname && fpath)
		{
			WebCore::String filepath = WebCore::String::fromUTF8( fpath );
			WebCore::String filename = WebCore::String::fromUTF8( fname );

			_uploadFileMap.set(filename, filepath);
			chooser->chooseFile( filename );
		}
		if (fname) CEComGetAllocatorRec()->free(CEComGetAllocatorRec(), fname);
		if (fpath) CEComGetAllocatorRec()->free(CEComGetAllocatorRec(), fpath);
	}
}

#if USE(ACCELERATED_COMPOSITING)
void ChromeClientSilk::attachRootGraphicsLayer(WebCore::Frame* frame, WebCore::GraphicsLayer* graphicsLayer)
{
	if (_frameWindow) {
		CEWebKitFrameWindowPrivate* fwPrivate = _frameWindow->getPrivate();
		if (fwPrivate) {			
			fwPrivate->layerSetRootGraphicsLayer(static_cast<ICEMRLLayer*>(graphicsLayer ? graphicsLayer->nativeLayer() : 0));
		}
	}
}

void ChromeClientSilk::setNeedsOneShotDrawingSynchronization()
{
	// we want the layers to synchronize next time we update the screen anyway
	if (_frameWindow) {
		CEWebKitFrameWindowPrivate* fwPrivate = _frameWindow->getPrivate();
		if (fwPrivate)
			fwPrivate->layerMarkForSync(false);
	}
}

void ChromeClientSilk::scheduleCompositingLayerSync()
{
	// we want the layers to synchronize ASAP
	if (_frameWindow) {
		CEWebKitFrameWindowPrivate* fwPrivate = _frameWindow->getPrivate();
		if (fwPrivate)
			fwPrivate->layerMarkForSync(true);
	}
}
#endif //#if USE(ACCELERATED_COMPOSITING)

bool ChromeClientSilk::setCursor(WebCore::PlatformCursorHandle)
{
    return false;
}

void ChromeClientSilk::requestGeolocationPermissionForFrame(WebCore::Frame*, WebCore::Geolocation*)
{
}
