/*
 * Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 *                     1999 Lars Knoll <knoll@kde.org>
 *                     1999 Antti Koivisto <koivisto@kde.org>
 *                     2000 Simon Hausmann <hausmann@kde.org>
 *                     2000 Stefan Schimanski <1Stein@gmx.de>
 *                     2001 George Staikos <staikos@kde.org>
 * Copyright (C) 2004, 2005, 2006, 2007 Apple Inc. All rights reserved.
 * Copyright (C) 2005 Alexey Proskuryakov <ap@nypop.com>
 * Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies)
 * Copyright (C) 2008 Eric Seidel <eric@webkit.org>
 * Copyright     2009, 2012 Sony Corporation
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#include "WebKitSilkPrefix.h"

#include "CESysDefs.h"

#include "CEWebKitWindowImpl.h"
#include "Frame.h"
#include "Page.h"
#include "Settings.h"
#include "FrameLoader.h"
#include "WebCoreSupport/FrameLoaderClientSilk.h"
#include "CEWebKitUtility.h"
#include "GraphicsContext.h"
#include "GraphicsContextPlatformPrivateSilk.h"
#include "FrameTree.h"
#include "PlatformKeyboardEvent.h"
#include "PlatformMouseEvent.h"
#include "PlatformWheelEvent.h"
#include "EventHandler.h"
#include "KeyboardEvent.h"
#include "GCController.h"
#include "ScriptValue.h"
#include "FocusController.h"

#include "CEWebKitFrameWindowImpl.h"
#include "PlatformString.h"

CEWebKitWindowImpl::CEWebKitWindowImpl() :
	_private(NULL)
	,_networkErrorEvent(NULL)
{
}

CEHResult CEWebKitWindowImpl::init(WebCore::Page* page, WebCore::HTMLFrameOwnerElement* ownerElement)
{
	CEHResult hr = CE_S_OK;
	_private = new CEWebKitWindowPrivate(this);
	if (_private)
	{
		hr = _private->init(page, ownerElement);
	}
	else
	{
		hr = CE_SILK_ERR_MEMERR;
	}

	_networkErrorEvent = CESysCreateEvent(0, true, false, 0);

	return hr;
}

CEHResult CEWebKitWindowImpl::shutdown()
{
	CEHResult hr = CE_S_OK;
	if (_private)
	{
		_private->shutdown();
		delete _private;
		_private = NULL;
	}
	if (_networkErrorEvent)
	{
		CESysCloseHandle(_networkErrorEvent);
	}

	return hr;
}

CEHResult CEWebKitWindowImpl::load(ICEUString* iUrl, ICEUString* iReferer)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;

	if (_private)
	{
		hr = CE_S_OK;
	
		WebCore::String* url = NULL;
		WebCore::String* referer = NULL;

		if (CESucceeded(hr))
		{
			hr = CEWebKitUtility::convert(iUrl, url);
		}

		if (CESucceeded(hr) && iReferer)
		{
			hr = CEWebKitUtility::convert(iReferer, referer);
		}

		if (CESucceeded(hr))
		{
			if (url)
			{
				if (url->startsWith("javascript:"))
				{
					WebCore::String encodedURL = encodeWithURLEscapeSequences(*url);
					WebCore::KURL kurl(encodedURL);
					hr = _private->load(kurl, referer);
				}
				else
				{
					WebCore::KURL kurl = WebCore::KURL(WebCore::KURL(), *url);
					if (kurl.isValid())
					{
						hr = _private->load(kurl, referer);
					}
					else
					{
						if (kurl.protocol().isEmpty())
						{
							url->insert("http://", 0);
							kurl = WebCore::KURL(WebCore::KURL(), *url);
							hr = _private->load(kurl, referer);
						}
						else
						{
							hr = CE_SILK_ERR_BADARGS;
						}
					}
				}
			}
			else
			{
				CEASSERT(false);
				hr = CE_SILK_ERR_OPERATION_FAILED;
			}
		}

		delete url;
		delete referer;
	}

	return hr;
}

CEHResult CEWebKitWindowImpl::back()
{
	return _private->back();
}

CEHResult CEWebKitWindowImpl::forward()
{
	return _private->forward();
}

CEHResult CEWebKitWindowImpl::reload(bool skipCache)
{
	return _private->reload(skipCache);
}

CEHResult CEWebKitWindowImpl::paint(ICEVGContext* iVGC, const CERect* dirtyRect)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_private && dirtyRect)
	{
		WebCore::IntRect rect;
		rect.setX(dirtyRect->_x);
		rect.setY(dirtyRect->_y);
		rect.setWidth(dirtyRect->_width);
		rect.setHeight(dirtyRect->_height);

		WebCore::GraphicsContext gc(iVGC);
		{

			CEPointBase lwpoint = {0, 0};
			gc.save();
			{
				gc.translate(lwpoint._x, lwpoint._y);
				{
					hr = _private->paint(gc, rect);
				}
				gc.restore();
			}
		}
	}
	return hr;
}

CEHResult CEWebKitWindowImpl::mouse(eCEMouseEventType type, CEPointBase& pt, eCEMouseButton button, UINT32 modifiers, bool& consumedOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_private)
	{
		WebCore::MouseEventType mouseType = WebCore::MouseEventMoved;
		int clickCount = 0;
		switch (type)
		{
		case eCEMouseEventType_ButtonDown:
			mouseType = WebCore::MouseEventPressed;
			clickCount = 1;
			break;
		case eCEMouseEventType_ButtonUp:
			mouseType = WebCore::MouseEventReleased;
			break;
		case eCEMouseEventType_ButtonDoubleClicked:
			mouseType = WebCore::MouseEventPressed;
			clickCount = 2;
			break;
		case eCEMouseEventType_ButtonTripleClicked:
			mouseType = WebCore::MouseEventPressed;
			clickCount = 3;
			break;
		case eCEMouseEventType_Moved:
		case eCEMouseEventType_Dragged:
			mouseType = WebCore::MouseEventMoved;
			break;
		default:
			CEASSERT(false);
			hr = CE_SILK_ERR_OPERATION_FAILED;
			break;
		}
		WebCore::IntPoint pos(pt._x, pt._y);
		WebCore::IntPoint globalPos(pt._x, pt._y);
		WebCore::MouseButton mouseButton = WebCore::NoButton;
		switch (button)
		{
		case eCEMouseButton_DontCare: mouseButton = WebCore::NoButton; break;
		case eCEMouseButton_Left: mouseButton = WebCore::LeftButton; break;
		case eCEMouseButton_Middle: mouseButton = WebCore::MiddleButton; break;
		case eCEMouseButton_Right: mouseButton = WebCore::RightButton; break;
		}
		
		// if document mode. convert coordinate doc to window.
		if (_private->getFrame() && _private->getFrame()->view() && _private->getFrame()->view()->platformWidget())
		{
			WebCore::IntPoint dcPos(pos.x(), pos.y());
			pos = _private->getFrame()->view()->contentsToWindow(dcPos);
			WebCore::IntPoint dcGlobalPos(globalPos.x(), globalPos.y());
			globalPos = _private->getFrame()->view()->contentsToWindow(dcGlobalPos);
		}

		WebCore::PlatformMouseEvent mouseEvent(pos, globalPos, mouseButton, mouseType, clickCount,
											   (eCEKeyModifier_SHIFT & modifiers), 
											   (eCEKeyModifier_CTRL & modifiers), 
											   (eCEKeyModifier_ALT & modifiers), 
											   (eCEKeyModifier_ALT & modifiers),  // meta
											   0.0);
		hr = _private->mouse(mouseEvent, consumedOut);
	}
	return hr;
}

CEHResult CEWebKitWindowImpl::keyboard(CEKeyboardEventParam* ev, bool& consumedOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;

	if (_private)
	{
		WebCore::PlatformKeyboardEvent keyboardEvent(ev);

		hr = _private->keyboard(keyboardEvent, consumedOut);
	}

	return hr;
}

CEHResult CEWebKitWindowImpl::focus(bool focused)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;

	if (_private)
	{
		hr = _private->focus(focused);
	}

	return hr;
}

CEHResult CEWebKitWindowImpl::scroll(INT32 x, INT32 y, INT32 dx, INT32 dy, UINT32 modifiers, bool& consumedOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_private)
	{
		WebCore::IntPoint pos(x, y);
		WebCore::IntPoint gPos(x, y);
		// if document mode. convert coordinate doc to window.
		if (_private->getFrame() && _private->getFrame()->view() && _private->getFrame()->view()->platformWidget())
		{
			WebCore::IntPoint dcPos(pos.x(), pos.y());
			pos = _private->getFrame()->view()->contentsToWindow(dcPos);
			WebCore::IntPoint dcGlobalPos(gPos.x(), gPos.y());
			gPos = _private->getFrame()->view()->contentsToWindow(dcGlobalPos);
		}
		WebCore::PlatformWheelEvent wheelEvent(pos, gPos, dx, dy, WebCore::ScrollByPixelWheelEvent, (eCEKeyModifier_SHIFT & modifiers), (eCEKeyModifier_CTRL & modifiers), (eCEKeyModifier_ALT & modifiers), (eCEKeyModifier_ALT & modifiers));
		hr = _private->scroll(wheelEvent, consumedOut);
	}
	return hr;
}

CEHResult CEWebKitWindowImpl::shrinkHistory(UINT32 minimumSize)
{
	return _private->shrinkHistory(minimumSize);
}

CEHResult CEWebKitWindowImpl::setCustomTextEncodingName(ICEUString* iTitle)
{
	return _private->setCustomTextEncodingName(iTitle);
}

CEHResult CEWebKitWindowImpl::stop()
{
	return _private->stop();
}

CEHResult CEWebKitWindowImpl::confirmPlugin(bool result, bool keepResult, void *const cData)
{
	return _private->confirmPlugin(result, keepResult, cData);
}

CEHResult CEWebKitWindowImpl::getContentsSize(INT32* widthOut, INT32* heightOut)
{
	return _private->getContentsSize(widthOut, heightOut);
}
CEWebKitFrameViewImpl* CEWebKitWindowImpl::getView()
{ 
	return (_private != NULL) ? _private->getView() : NULL; 
}
CEWebKitWindowPrivate::CEWebKitWindowPrivate(CEWebKitWindowImpl* silkPeer) :
	_silkPeer(silkPeer),
	_frame(NULL),
	_frameView(NULL)
{
}

CEHResult CEWebKitWindowPrivate::init(WebCore::Page* page, WebCore::HTMLFrameOwnerElement* ownerElement)
{
	CEHResult hr = CE_S_OK;
	_frameView = new CEWebKitFrameViewImpl(this);
	if (!_frameView) hr = CE_SILK_ERR_MEMERR;

	if (CESucceeded(hr))
	{
		hr = _initFrame(page, ownerElement);
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::shutdown()
{
	CEHResult hr = CE_S_OK;
	if (CESucceeded(hr))
	{
		hr = _shutdownFrame();
	}
	if (_frameView)
	{
		delete _frameView;
		_frameView = NULL;
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::load(WebCore::KURL& url, WebCore::String* referer)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_frame && _frame->loader())
	{
		hr = CE_S_OK;
		if (protocolIsJavaScript(url))
		{
			WebCore::String scriptString = decodeURLEscapeSequences(url.string().substring(url.protocol().length() + sizeof(":") - 1));
			WebCore::ScriptValue result = _frame->loader()->executeScript(scriptString, true);
			if (!result.isNull())
			{
				WebCore::String resultString;
				if (result.getString(resultString))
				{
					_frame->loader()->begin();
					_frame->loader()->write(resultString);
					_frame->loader()->end();
				}
			}
		}
		else
		{
			if (referer)
			{
				_frame->loader()->load(WebCore::ResourceRequest(url, *referer), false);
			}
			else
			{
				_frame->loader()->load(WebCore::ResourceRequest(url), false);
			}
		}
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::back()
{
	CEHResult hr = CE_S_OK;
	if (_frame && _frame->page())
	{
		_frame->page()->goBack();
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::forward()
{
	CEHResult hr = CE_S_OK;
	if (_frame && _frame->page())
	{
		_frame->page()->goForward();
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::reload(bool skipCache)
{
	CEHResult hr = CE_S_OK;
	if (_frame && _frame->loader())
	{
		_frame->loader()->reload(skipCache);
	}
	return hr;
}
CEHResult CEWebKitWindowPrivate::shrinkHistory(UINT32 minimumSize)
{
	CEASSERT(1 == minimumSize);
	if (_frame && _frame->page())
	{
		_frame->page()->backForwardList()->setCapacity(0);
	}
	return CE_S_OK;
}

CEHResult CEWebKitWindowPrivate::paint(WebCore::GraphicsContext& gc, WebCore::IntRect& dirtyRect)
{
	CEHResult hr = CE_S_OK;
	if (_frame && _frame->view())
	{

#if USE(ACCELERATED_COMPOSITING)
		CEMRL_FPS_18278_SAVETIME(1, 0);
#endif //#if USE(ACCELERATED_COMPOSITING)
		_frame->view()->layoutIfNeededRecursive();
#if USE(ACCELERATED_COMPOSITING)
		CEMRL_FPS_18278_SAVETIME(1, 1);
#endif //#if USE(ACCELERATED_COMPOSITING)

		int count = _getLayoutCount(_frame);
		if (count != _layoutCount && _frame->page())
		{
			_layoutCount = count;
			if (CEWebKitFrameWindowImpl* frameWindow = kit(_frame->page()))
			{
				frameWindow->rebuildFocusNavigation();
			}
		}

		gc.save();
		gc.clip(dirtyRect);

#if USE(ACCELERATED_COMPOSITING)
		CEMRL_FPS_18278_SAVETIME(1, 2);
#endif //#if USE(ACCELERATED_COMPOSITING)
		_frame->view()->paint(&gc, dirtyRect);
#if USE(ACCELERATED_COMPOSITING)
		CEMRL_FPS_18278_SAVETIME(1, 3);
#endif //#if USE(ACCELERATED_COMPOSITING)

		gc.restore();
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::mouse(WebCore::PlatformMouseEvent& mouseEvent, bool& acceptOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_frame && _frame->eventHandler())
	{
		hr = CE_S_OK;
		switch (mouseEvent.eventType())
		{
		case WebCore::MouseEventMoved:
			acceptOut = _frame->eventHandler()->mouseMoved(mouseEvent);
			break;
		case WebCore::MouseEventPressed:
			if (_frame->loader())
			{
				FrameLoaderClientSilk* client = static_cast<FrameLoaderClientSilk*>(_frame->loader()->client());
				if (client)
				{
					client->setCommittedLoad(false);
				}
			}
			acceptOut = _frame->eventHandler()->handleMousePressEvent(mouseEvent);
			break;
		case WebCore::MouseEventReleased:
			acceptOut = _frame->eventHandler()->handleMouseReleaseEvent(mouseEvent);
			break;
		case WebCore::MouseEventScroll:
			CEASSERT(false);
			break;
		}
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::keyboard(WebCore::PlatformKeyboardEvent& keyEvent, bool& acceptOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;

	if (_frame )
	{
		WebCore::Frame* pFrame = getFocusedFrame();
		if (pFrame && pFrame->eventHandler())
		{
            acceptOut = false;
            if (keyEvent.type() == WebCore::PlatformKeyboardEvent::Char)
                acceptOut = pFrame->eventHandler()->handleAccessKey(keyEvent);
            if (!acceptOut)
                acceptOut = pFrame->eventHandler()->keyEvent(keyEvent);
			hr = CE_S_OK;
		}
	}

	return hr;
}

CEHResult CEWebKitWindowPrivate::focus(bool focused)
{
	CEHResult hr = CE_S_OK;
	if (_frame && _frame->page())
	{
		_frame->page()->focusController()->setFocused(focused);
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::scroll(WebCore::PlatformWheelEvent& wheelEvent, bool& acceptOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_frame && _frame->eventHandler())
	{
		acceptOut = _frame->eventHandler()->handleWheelEvent(wheelEvent);
		hr = CE_S_OK;
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::setCustomTextEncodingName(ICEUString* iTitle)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	WebCore::String s;

	if (_frame && _frame->loader())
	{
		if (iTitle)
		{
			hr = s.fromICEUString(iTitle);
			if (CESucceeded(hr))
			{
				//To support auto detection,
				//  set custom text encoding as default.
				//When WebKit can not judge text encoding,
				//  default encoding is used.
				_frame->page()->settings()->setDefaultTextEncodingName(s);
			}
		}
		else
		{
			hr = CE_S_OK;
		}

		if (CESucceeded(hr))
		{
			_frame->loader()->reloadWithOverrideEncoding(s);
		}
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::stop()
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_frame && _frame->loader())
	{
        _frame->loader()->stopAllLoaders();
		hr = CE_S_OK;
	}
	return hr;
}

WebCore::Frame* CEWebKitWindowPrivate::getFocusedFrame()
{
	if (_frame && _frame->page())
	{
		WebCore::FocusController* pFocusController = _frame->page()->focusController();
		if (pFocusController)
		{
			WebCore::Frame* pFrame = pFocusController->focusedOrMainFrame();
			return pFrame;
		}
	}
	return NULL;
}

CEHResult CEWebKitWindowPrivate::getContentsSize(INT32* widthOut, INT32* heightOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_frame && _frame->eventHandler())
	{
		IntSize contentSize;
		WebCore::FrameView* view = _frame->view();
		if (view)
		{
			 contentSize = view->contentsSize();
			 if (!contentSize.isEmpty())
			 {
				 *widthOut = contentSize.width();
				 *heightOut = contentSize.height();
				// view->setScrollbarModes(WebCore::ScrollbarAlwaysOff,WebCore::ScrollbarAlwaysOff);
				 hr = CE_S_OK;
			}
		}
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::confirmPlugin(bool result, bool keepResult, void *const cData)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_frame && _frame->loader())
	{
		bool reResult = static_cast<FrameLoaderClientSilk*>(_frame->loader()->client())->confirmPlugin(result, keepResult, cData);
		if (reResult)
			hr = CE_S_OK;
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::_initFrame(WebCore::Page* page, WebCore::HTMLFrameOwnerElement* ownerElement)
{
	CEHResult hr = CE_SILK_ERR_MEMERR;
	FrameLoaderClientSilk* frameLoaderClient = new FrameLoaderClientSilk(_silkPeer);
	if (frameLoaderClient)
	{
		RefPtr<WebCore::Frame> frame = WebCore::Frame::create(page, ownerElement, frameLoaderClient);
		if (frame)
		{
			frameLoaderClient->setFrame(frame.get());
			frame->init();
			_frame = frame.get();
			hr = CE_S_OK;
		}
	}
	if (CEFailed(hr))
	{
		delete frameLoaderClient;
		_frame = NULL;
	}
	return hr;
}

CEHResult CEWebKitWindowPrivate::_shutdownFrame()
{
	if (_frame && _frame->loader())
	{
		_frame->loader()->detachFromParent();
	}
	return CE_S_OK;
}

bool CEWebKitWindowPrivate::_needsLayout(WebCore::Frame* frame)	
{
	bool needsLayout = false;
	if (frame && frame->view() && frame->tree())
	{
		needsLayout |= frame->view()->needsLayout();
		frame = frame->tree()->firstChild();
		while (frame && frame->tree() && !needsLayout)
		{
			needsLayout |= _needsLayout(frame);
			frame = frame->tree()->nextSibling();
		}
	}
	return needsLayout;
}

int CEWebKitWindowPrivate::_getLayoutCount(const WebCore::Frame* frame)
{
	int count = 0;
	if (frame)
	{
		if (frame->view())
		{
			count = frame->view()->layoutCount();
		}
		if (frame->tree())
		{
			WebCore::Frame* childFrame = frame->tree()->firstChild();
			while(childFrame)
			{
				count += _getLayoutCount(childFrame);
				if (childFrame->tree())
				{
					childFrame = childFrame->tree()->nextSibling();
				}
			}
		}
	}
	return count;
}

CEWebKitWindowImpl* kit(WebCore::Frame* frame)
{
	CEWebKitWindowImpl* kit = NULL;
	if (frame && frame->loader() && frame->loader()->client())
	{
		kit = static_cast<FrameLoaderClientSilk*>(frame->loader()->client())->getWindow();
	}
	return kit;
}

WebCore::Frame* core(CEWebKitWindowImpl* window)
{
	WebCore::Frame* core = NULL;
	if (window && window->getPrivate())
	{
		core = window->getPrivate()->getFrame();
	}
	return core;
}

CEHResult CEWebKitWindowImpl::networkErrorEventWait(bool& result, ICEUString** errorPageStr)
{
	CEHResult hr = CE_S_OK;

#if 0 // SHOW_INTERNAL_NETWORK_ERROR_STRING
//#if CE_OS(HYDRA) // Wed Aug  3 16:10:42     2011 
	if (_networkErrorEvent)
	{
		UINT32 ret = CESysWaitForMultipleObjects(1, &_networkErrorEvent, false, CESYS_TIMEOUT_INFINITE);
		printf("CEWebKitWindowImpl::networkErrorEventWait() return == [%s]\n", (ret==CESYS_WAIT_OBJECT_0)?"SUCCEEDED":((ret==CESYS_WAIT_ABANDONED_0)?"ABONDONED":(ret==CESYS_WAIT_FAILED?"FAILED":"UNKNOWN_ERROR")));

		CESysResetEvent(_networkErrorEvent);
		result = _result;
		hr = _errorPageStr.copyTo(errorPageStr);
	}
#endif

	return hr;
}

CEHResult CEWebKitWindowImpl::networkErrorEventPost(bool result, ICEUString* errorPageStr)
{
#if 0 // SHOW_INTERNAL_NETWORK_ERROR_STRING
//#if CE_OS(HYDRA) // Wed Aug  3 16:10:42     2011 
	if (_networkErrorEvent)
	{
		_result = result;
		_errorPageStr = errorPageStr;

		CESysSetEvent(_networkErrorEvent);
	}
#endif

	return CE_S_OK;
}

