/*
 * Copyright     2012 Sony Corporation
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#include "WebKitSilkPrefix.h"

#include "CEWebKitFocusImpl.h"
#include "FocusController.h"
#include "CEWebKitFrameWindowImpl.h"
#include "Node.h"
#include "RenderObject.h"
#include "RenderStyle.h"
#include "RenderImage.h"
#include "RenderText.h"
#include "RenderInline.h"
#include "RenderView.h"
#include "HTMLAreaElement.h"
#include "HTMLNames.h"
#include "HTMLMapElement.h"
#include "InlineTextBox.h"
#include "RenderLineBoxList.h"

#include "HTMLFrameOwnerElement.h"
#include "CEWebKitFrameWindow.h"

//#define FOCUS_DEBUG_LOG

#if defined(_DEBUG)
#define FOCUS_DEBUG_SVG_LOG
#endif

#if defined(FOCUS_DEBUG_SVG_LOG)
static bool _debugSvg = false;

static void _dumpSvgBegin()
{
	if (_debugSvg)
	{
		CEComDebugPrintf("\n===BEGIN_LOG renderobject.svg\n");
		CEComDebugPrintf("<?xml version=\"1.0\" standalone=\"yes\" ?>"	\
						 "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 20010904//EN\" \"http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd\">" \
						 "<svg width='1600' height='3200' xmlns='http://www.w3.org/2000/svg'>");
	}
}

static void _dumpSvgEnd()
{
	if (_debugSvg)
	{
		CEComDebugPrintf("</svg>\n");
		CEComDebugPrintf("===END_LOG\n");
	}
}

static void _dumpSvgRect(const CERect& rect, UINT32 color, int no, void* item, const char* msg, int scrollDist, int forwardDist, int sideDist, bool bText = true)
{
	if (_debugSvg)
	{
		CEComDebugPrintf("<rect x='%d' y='%d' width='%d' height='%d' stroke='#%06x' fill='none' />\n",
						 rect._x, rect._y, rect._width, rect._height, color);
		if (bText)
		{
	
			CEComDebugPrintf("<text x='%d' y='%d' font-size='5pt' stroke-width='0.2px' stroke='#%06x' fill='#%06x'>%d,%08x</text>\n",
							 rect._x, rect._y + 6, color, color,
							 no, item, msg);
	
			CEComDebugPrintf("<text x='%d' y='%d' font-size='5pt' stroke-width='0.2px' stroke='#%06x' fill='#%06x'>%d,%d,%d</text>\n",
							 rect._x, rect._y + 12, color, color,
							 scrollDist, forwardDist, sideDist);

			CEComDebugPrintf("<text x='%d' y='%d' font-size='5pt' stroke-width='0.2px' stroke='#%06x' fill='#%06x'>%hs</text>\n",
							 rect._x, rect._y + 16, color, color,
							 msg);
		}
	}	
}
#endif //defined(FOCUS_DEBUG_SVG_LOG)

typedef void CEHtmlFocusItem;
static CEHtmlFocusItem* _addRenderObjectToFocusTree(WebCore::RenderObject* renderer, CEHtmlFocusItem* parent, WebCore::Node* node, WebCore::FrameView* frame, CEComICEHtmlFocusNavigatorRef iNav);
static CEHtmlFocusItem* _addFocusItem(WebCore::RenderObject* renderer, CEHtmlFocusItem* parent, WebCore::Node* node, const CERect& rect, const CEDim* dim, CEComICEHtmlFocusNavigatorRef iNav, CEHtmlFocusItemType type = CEHtmlFocusItemType_None);
static WebCore::IntRect _getAbsoluteClippedOverflowRect(WebCore::RenderObject* renderer);

static WebCore::Node* _getNode(WebCore::RenderObject* renderer)
{
	WebCore::Node* node = 0;
	if (renderer->isBoxModelObject())
	{
		node = renderer->node();
	}
	else
	{
		// from RenderLayer::enclosingElement()
		WebCore::RenderLayer* layer = renderer->enclosingLayer();
		if (layer)
		{
			for (WebCore::RenderObject* r = layer->renderer(); r; r = r->parent())
			{
				if (node = r->node())
					break;
			}
		}
	}
	return node;
}

static bool _traverseViewRecursive(WebCore::RenderObject* renderer, CEHtmlFocusItem* parent, WebCore::Node* focusElement, CEComICEHtmlFocusNavigatorRef iNav)
{
	bool hasFocusItem = false;
	bool focusableElem = false;
	
	WebCore::RenderObject* child = renderer->firstChild();
	WebCore::RenderStyle* style = renderer->style();

	WebCore::Node* node = renderer->node();


#if defined(FOCUS_DEBUG_LOG)
{
	WebCore::IntRect rect = renderer->clippedOverflowRectForRepaint(renderer->containerForRepaint());
	CEComDebugPrintf("{%p:%p (%d,%d+%d+%d)", renderer, node, rect.x(), rect.y(), rect.width(), rect.height());
	if (renderer->isApplet()) CEComDebugPrintf("-a"); 
	if (renderer->isAnonymous()) CEComDebugPrintf("-an"); 
	if (renderer->isAnonymousBlock()) CEComDebugPrintf("-anb"); 
	if (renderer->isBR()) CEComDebugPrintf("-br"); 
	if (renderer->isBlockFlow()) CEComDebugPrintf("-bf"); 
	if (renderer->isBoxModelObject()) CEComDebugPrintf("-bmo"); 
	if (renderer->isCounter()) CEComDebugPrintf("-c"); 
	if (renderer->isFieldset()) CEComDebugPrintf("-fs"); 
	if (renderer->isFrame()) CEComDebugPrintf("-fr"); 
	if (renderer->isFrameSet()) CEComDebugPrintf("-frs"); 
	if (renderer->isImage()) CEComDebugPrintf("-im"); 
	if (renderer->isInlineBlockOrInlineTable()) CEComDebugPrintf("-in"); 
	if (renderer->isInlineContinuation()) CEComDebugPrintf("-ic"); 
	if (renderer->isListBox()) CEComDebugPrintf("-lb"); 
	if (renderer->isListItem()) CEComDebugPrintf("-li"); 
	if (renderer->isListMarker()) CEComDebugPrintf("-lm"); 
	if (renderer->isMedia()) CEComDebugPrintf("-md"); 
	if (renderer->isMenuList()) CEComDebugPrintf("-ml"); 
	if (renderer->isRenderBlock()) CEComDebugPrintf("-rbl"); 
	if (renderer->isRenderButton()) CEComDebugPrintf("-rbt"); 
	if (renderer->isRenderImage()) CEComDebugPrintf("-rim"); 
	if (renderer->isRenderInline()) CEComDebugPrintf("-rin"); 
	if (renderer->isRenderPart()) CEComDebugPrintf("-rp"); 
	if (renderer->isRenderView()) CEComDebugPrintf("-rv"); 
	if (renderer->isSlider()) CEComDebugPrintf("-s"); 
	if (renderer->isTable()) CEComDebugPrintf("-t"); 
	if (renderer->isTableCell()) CEComDebugPrintf("-tcl"); 
	if (renderer->isTableCol()) CEComDebugPrintf("-tco"); 
	if (renderer->isTableRow()) CEComDebugPrintf("-tr"); 
	if (renderer->isTableSection()) CEComDebugPrintf("-ts"); 
	if (renderer->isText()) CEComDebugPrintf("-tx"); 
	if (renderer->isTextControl()) CEComDebugPrintf("-tc"); 
	if (renderer->isTextArea()) CEComDebugPrintf("-ta"); 
	if (renderer->isTextField()) CEComDebugPrintf("-tf"); 
	if (renderer->isWidget()) CEComDebugPrintf("-w");
	if (node)
	{
		if (node->onclick()) CEComDebugPrintf("-onclick");
		if (node->ondblclick()) CEComDebugPrintf("-ondblclick");
	}
	// from HitTest code
	CEComDebugPrintf(" ");
}
#endif

    if (style->visibility() != WebCore::HIDDEN && style->display() != WebCore::NONE)
	{
		bool isTextField = renderer->isTextArea() || renderer->isTextField();
		bool isControl = isTextField || renderer->isTextControl() || renderer->isRenderButton() || renderer->isSlider() || renderer->isListBox() || renderer->isMenuList() || renderer->isWidget() ;
		if (node)
		{
			bool isCanvas = node->hasTagName(WebCore::HTMLNames::canvasTag);
			bool isFocus = node->isFocusable() || node->isKeyboardFocusable(NULL) || node->isMouseFocusable()/*|| node->hasTagName(WebCore::HTMLNames::aTag)*/ || isCanvas;

			if (isFocus || isControl)
			{
#if defined(FOCUS_DEBUG_LOG)
				if (isFocus) CEComDebugPrintf("-focus ");
				if (isControl) CEComDebugPrintf("-control ");
#endif
				focusElement = node;
			}

			focusableElem = (child == 0) || ((isControl || isFocus) & !renderer->isBoxModelObject()) || isTextField || renderer->isRenderButton();

			if ((!node->isStyledElement() || node->isFrameOwnerElement()|| focusableElem) && focusElement)
			{
				CEHtmlFocusItem* item = 0;
				if (node->isFrameOwnerElement())
				{
					WebCore::HTMLFrameOwnerElement* frameOwner = reinterpret_cast<WebCore::HTMLFrameOwnerElement*>(node);
					WebCore::Document* doc = frameOwner->contentDocument();
					if (doc)
					{
						WebCore::RenderObject* ro = doc->renderer();
						if (ro)
						{
							WebCore::FrameView* frameView = doc->view();
							if (frameView)
							{
								item = _addRenderObjectToFocusTree(renderer, parent, focusElement, frameView, iNav);
								if (item != 0)
								{
									WebCore::IntRect viewRect = frameView->visibleContentRect(false);
									CEPointBase point;
									point._x = viewRect.x();
									point._y = viewRect.y();
									iNav.setScrollPos(item, &point);
									_traverseViewRecursive(ro, item, 0, iNav);
								}
								
							}

							iNav.completeConstruct(item);
						}
					}
					if (item == 0)
					{
						item = _addRenderObjectToFocusTree(renderer, parent, focusElement, 0, iNav);
					}
					hasFocusItem = (item != 0);
				}
			}
		}


		if (child && (!node || !focusableElem))
		{
			do
			{
				if (_traverseViewRecursive(child, parent, focusElement, iNav))
				{
					hasFocusItem = true;
				}
			} while ((child = child->nextSibling()));
		}

		if (renderer->isImage())
		{
			WebCore::RenderImage* image = static_cast<WebCore::RenderImage*>(renderer);
			WebCore::HTMLMapElement* map = image->imageMap();
			if (map) {
				WebCore::Node* node;
				for (node = map->firstChild(); node; 
					 node = node->traverseNextNode(map)) {
					if (!node->hasTagName(WebCore::HTMLNames::areaTag))
						continue;
					WebCore::HTMLAreaElement* area = static_cast<WebCore::HTMLAreaElement*>(node);
					if (area)
					{
						WebCore::IntRect rect = area->getRect(image);
						renderer->computeAbsoluteRepaintRect(rect);
						WebCore::FloatPoint absPos = renderer->localToAbsolute();
						rect.move(static_cast<INT32>(absPos.x()), static_cast<INT32>(absPos.y()));
						CEHtmlFocusItem* item = _addFocusItem(renderer, parent, node, rect, 0, iNav);
						hasFocusItem = item != 0;
					}
				}
			}
		}

		// if cannot register FocusItem by child-node, register FocusItem by this element.
		if (!hasFocusItem && focusElement)
		{
			CEHtmlFocusItem* item = _addRenderObjectToFocusTree(renderer, parent, focusElement, 0, iNav);
			hasFocusItem = (item != 0);
		}
	}
	
#if defined(FOCUS_DEBUG_LOG)
	CEComDebugPrintf("}");
#endif
	return hasFocusItem;
}

static CEHtmlFocusItem* _addRenderObjectToFocusTree(WebCore::RenderObject* renderer, CEHtmlFocusItem* parent, WebCore::Node* node, WebCore::FrameView* frame, CEComICEHtmlFocusNavigatorRef iNav)
{
	CEHtmlFocusItem* ret = 0;

	WebCore::RenderView* view = renderer->view();
	if (view)
	{
		if (frame)
		{
			WebCore::IntRect viewRect = frame->visibleContentRect(false);
			WebCore::IntRect rect = renderer->clippedOverflowRectForRepaint(renderer->containerForRepaint());
			CEDim size = {rect.width(), rect.height()};
			CERect outerRect = {rect.x(), rect.y(), viewRect.width(), viewRect.height()};
#if defined(FOCUS_DEBUG_LOG)
			CEComDebugPrintf("frame:");
#endif
			
			ret = _addFocusItem(reinterpret_cast<WebCore::RenderObject*>(frame), parent, node, outerRect, &size, iNav, CEHtmlFocusItemType_Frame);

#if defined(FOCUS_DEBUG_SVG_LOG)
			_dumpSvgRect(outerRect, 0xccccff, 0, ret, "frame", outerRect._width, outerRect._width, 0);
#endif			
		}
		else if (renderer->isText())
		{
#if defined(FOCUS_DEBUG_LOG)
			CEComDebugPrintf("enter text");
#endif
			
			WebCore::IntRect outerRect = _getAbsoluteClippedOverflowRect(renderer);
			WebCore::FloatPoint absPos = renderer->localToAbsolute();
			WebCore::RenderText* rt = static_cast<WebCore::RenderText*>(renderer);
			const WebCore::InlineTextBox* itb = rt->firstTextBox();
			bool success = false;
			while (itb)
			{
				CERect rect = { absPos.x() + itb->x(), absPos.y() + itb->y(), itb->width(), itb->height() };

				if (rect._x + rect._width >= 0 && rect._y + rect._height >= 0)
				{
					if (!outerRect.contains(rect))
					{
						WebCore::IntRect r = static_cast<WebCore::IntRect>(rect);
						r.intersect(outerRect);
						rect = static_cast<CERect>(r);
					}

					CEHtmlFocusItem* item = _addFocusItem(renderer, parent, node, rect, 0, iNav);	
					if (item)
					{
						success = true;
					}
#if defined(FOCUS_DEBUG_SVG_LOG)
					_dumpSvgRect(rect, 0x88aa88, 0, item, "it", rect._width, rect._width, 0);
#endif

					if (ret == 0)
					{
						ret = item;
					}
				}
			
				if (itb == rt->lastTextBox())
				{
					break;
				}
				itb = itb->nextTextBox();
			}
#if defined(FOCUS_DEBUG_LOG)
			CEComDebugPrintf("leave text");
#endif
		}
		else if (renderer->isRenderInline())
		{
			WebCore::IntRect intrect = renderer->absoluteBoundingBoxRect();
			WebCore::IntRect outerRect = _getAbsoluteClippedOverflowRect(renderer);

			if (intrect.width() > 0 && intrect.height() > 0)
			{
			
				WebCore::RenderLineBoxList* boxList = static_cast<WebCore::RenderInline*>(renderer)->lineBoxes();
				const WebCore::InlineFlowBox* ifb = boxList->firstLineBox();
				bool success = false;
				while (ifb)
				{
					CERect rect = {
						intrect.x() + ifb->x(), intrect.y() + ifb->y(),
						ifb->width(), ifb->height()
					};

					if (rect._x + rect._width >= intrect.right())
					{
						rect._width = intrect.right() - rect._x;
					}

					if (rect._y + rect._height >= intrect.bottom())
					{
						rect._height = intrect.bottom() - rect._y;
					}

					if (rect._x + rect._width >= 0 && rect._y + rect._height >= 0)
					{
						if (!outerRect.contains(rect))
						{
							WebCore::IntRect r = static_cast<WebCore::IntRect>(rect);
							r.intersect(outerRect);
							rect = static_cast<CERect>(r);
						}

						CEHtmlFocusItem* item = _addFocusItem(renderer, parent, node, rect, 0, iNav);
						if (item)
						{
							success = true;
						}
	#if defined(FOCUS_DEBUG_SVG_LOG)
						_dumpSvgRect(rect, 0x88aaaa, 0, item, "if", rect._x, rect._y, 0);
	#endif			

						if (ret == 0)
						{
							ret = item;
						}
					}
				
					if (ifb == boxList->lastLineBox())
					{
						break;
					}
					ifb = ifb->nextFlowBox();
				}

				if (!success)
				{
					ret = _addFocusItem(renderer, parent, node, static_cast<const CERect>(intrect), 0, iNav);
				}
			}

		}
		else
		{
			WebCore::FloatPoint absPos = renderer->localToAbsolute();
			WebCore::IntRect rect = _getAbsoluteClippedOverflowRect(renderer);
			rect.setWidth(rect.x() + rect.width() - absPos.x());
			rect.setHeight(rect.y() + rect.height() - absPos.y());
			rect.setX(absPos.x());
			rect.setY(absPos.y());
			ret = _addFocusItem(renderer, parent, node, static_cast<const CERect>(rect), 0, iNav);
#if defined(FOCUS_DEBUG_SVG_LOG)
			_dumpSvgRect(static_cast<const CERect>(rect), 0xffccff, 0, ret, renderer->renderName(), rect.width(), rect.height(), renderer->isReplaced());
#endif			
		}
	}
	return ret;
}

static CEHtmlFocusItem* _addFocusItem(WebCore::RenderObject* renderer, CEHtmlFocusItem* parent, WebCore::Node* node, const CERect& rect, const CEDim* dim, CEComICEHtmlFocusNavigatorRef iNav, CEHtmlFocusItemType type)
{
	CEHtmlFocusItem* ret = 0;
	
	if (rect._x + rect._width >= 0 && rect._y + rect._height >= 0)
	{
		if (type == CEHtmlFocusItemType_None)
		{
			type = CEHtmlFocusItemType_Anchor;
			if (renderer->isTextField())
			{
				type = CEHtmlFocusItemType_TextLine;
			}
			else if (renderer->isTextArea())
			{
				type = CEHtmlFocusItemType_TextArea;
			}
			else if (node && node->hasTagName(WebCore::HTMLNames::canvasTag))
			{
				type = CEHtmlFocusItemType_Canvas;
			}
			else if (node && node->shouldUseInputMethod())
			{
				type = CEHtmlFocusItemType_ContentEditable;
			}
		}
		
		iNav.setFocusItem(&ret, parent, renderer, node,
						  type,
						  //(renderer->isTextField() || renderer->isTextArea()) ? CEHtmlFocusItemType_Form : CEHtmlFocusItemType_Anchor,
						  //`(node.isLink()) ? CEHtmlFocusItemType_Anchor : CEHtmlFocusItemType_Form,
						  &rect, dim, 0);
#if defined(FOCUS_DEBUG_LOG)
		CEComDebugPrintf("addFocusItem %p RO:%p parent:%p node:%p %d,%d,%d,%d\n",
						 ret, renderer, parent, node, rect._x, rect._y, rect._width, rect._height);
#endif

	}
	else
	{
#if defined(FOCUS_DEBUG_LOG)
		CEComDebugPrintf("failed(addFocusItem %p RO:%p parent:%p node:%p %d,%d,%d,%d)\n",
						 ret, renderer, parent, node, rect._x, rect._y, rect._width, rect._height);
#endif
	}
	return ret;
}

static WebCore::IntRect _getAbsoluteClippedOverflowRect(WebCore::RenderObject* renderer)
{
	WebCore::IntRect outerRect = renderer->absoluteClippedOverflowRect();
	WebCore::RenderView* view = renderer->view();
	if (view && view->maximalOutlineSize() > 0)
	{
		outerRect.inflate(-(view->maximalOutlineSize()));
	}
	return outerRect;
}

CEWebKitFocusImpl::CEWebKitFocusImpl()
{
}

CEWebKitFocusImpl::~CEWebKitFocusImpl()
{
}

CEHResult CEWebKitFocusImpl::init(CEWebKitFrameWindowImpl* silkPeer, ICEHtmlFocusNavigator* iFocusNavigatorRef)
{
	_silkPeer = silkPeer;
	_iFocusNavigatorRef = iFocusNavigatorRef;

	return _iFocusNavigatorRef.setFocusNavigatorPeer(CEWebKitFocus::toICEHtmlFocusNavigatorPeer(this));
}

void CEWebKitFocusImpl::shutdown()
{
	_silkPeer = NULL;
	_iFocusNavigatorRef.shutdownFocusNavigatorPeer();
	_iFocusNavigatorRef = NULL;
}

void CEWebKitFocusImpl::rebuildFocusNavigation()
{
	if (_iFocusNavigatorRef)
		_iFocusNavigatorRef.notifyRebuild();
}

void CEWebKitFocusImpl::addFocusItem(WebCore::RenderObject* o)
{
	WebCore::Node* node = _getNode(o);
	while (node)
	{
		bool isFocus = node->isFocusable();
		//_addRenderObjectToFocusTree();
		if (isFocus)
		{
			rebuildFocusNavigation();
			break;
		}
		node = node->parent();
	}
}

void CEWebKitFocusImpl::removeFocusItem(WebCore::RenderObject* o)
{
	if (_iFocusNavigatorRef)
		_iFocusNavigatorRef.notifyRemoveItem(o);
}


CEHResult CEWebKitFocusImpl::isVisible(void* hView, bool* resultOut) const
{
	*resultOut = true;
	return CE_S_OK;
}

CEHResult CEWebKitFocusImpl::isFocusable(void* hView, bool* resultOut) const
{
	*resultOut = true;
	return CE_S_OK;
}

CEHResult CEWebKitFocusImpl::isEnable(void* hView, bool* resultOut) const
{
	*resultOut = true;
	return CE_S_OK;
}

CEHResult CEWebKitFocusImpl::scrollTo(void* hView, CEPointBase* pos)
{
	WebCore::FrameView* frame = _getFrame(hView);
	if (frame)
	{
		WebCore::IntPoint intScroll(pos->_x, pos->_y);
		frame->setScrollPosition(intScroll);
	}
	return CE_S_OK;
}

CEHResult CEWebKitFocusImpl::getScrollPos(void* hView, CEPointBase* posOut) const
{
	WebCore::FrameView* frame = _getFrame(hView);
	if (frame)
	{
		WebCore::IntPoint pos = frame->scrollPosition();
		posOut->_x = pos.x();
		posOut->_y = pos.y();
	}
	return CE_S_OK;
}

CEHResult CEWebKitFocusImpl::getContentSize(void* hView, CEDim* sizeOut) const
{
	WebCore::FrameView* frame = _getFrame(hView);
	if (frame)
	{
		sizeOut->_width  = frame->contentsWidth();
		sizeOut->_height = frame->contentsHeight();
	}
	return CE_S_OK;
}

CEHResult CEWebKitFocusImpl::getTabIndex(void* hView, UINT32* tabIndexOut) const
{
	return CE_S_OK;
}

CEHResult CEWebKitFocusImpl::notifyFocused(void* hView)
{
	if (hView)
	{
		if (WebCore::Node* node = reinterpret_cast<WebCore::Node*>(hView))
		{
			if (node->isElementNode())
			{
				static_cast<WebCore::Element*>(node)->focus();
			}
		}
	}
	else
	{
		// remove focus if hView==0
		if (_silkPeer && _silkPeer->getPrivate())
		{
			WebCore::Page* page = _silkPeer->getPrivate()->getPage();
			if (page && page->focusController())
			{
#if 0
				const WebCore::Frame* frame = page->focusController()->focusedFrame();
				if (frame)
				{
					WebCore::Document* doc = frame->document();
					if (doc)
					{
						doc->focusedNodeRemoved();
					}
				}
#else
				page->focusController()->setFocusedNode(0, page->focusController()->focusedOrMainFrame());
#endif
			}
		}
	}
	return CE_S_OK;
}

CEHResult CEWebKitFocusImpl::notifyDestroy(void* hView)
{
	return CE_S_OK;
}

CEHResult CEWebKitFocusImpl::constructFocusItemTree()
{
#if defined(FOCUS_DEBUG_SVG_LOG)
	_dumpSvgBegin();
#endif
#if defined(FOCUS_DEBUG_LOG)
	CEComDebugPrintf("constructFocusItemTree()\n");
#endif
	if (_silkPeer && _iFocusNavigatorRef)
	{
		CEWebKitFrameWindowPrivate* silkPeerPrivate = _silkPeer->getPrivate();
		if (silkPeerPrivate && silkPeerPrivate->getPage())
		{
			WebCore::Frame* frame = silkPeerPrivate->getPage()->mainFrame();
			if (frame)
			{
				const WebCore::Document* doc = frame->document();
				if (doc)
				{
					WebCore::RenderView* ro = doc->renderView();
					WebCore::FrameView* frameView = doc->view();
					if (ro && frameView)
					{
						WebCore::IntRect viewRect = frameView->visibleContentRect(false);
						
						void* root;
						CERect outer = {ro->x(), ro->y(), ro->width(), ro->height()};
						CEDim dim = {viewRect.width(), viewRect.height()};
						_iFocusNavigatorRef.setFocusItem(
							&root, 0, frameView, 0, CEHtmlFocusItemType_Frame,
							&outer, &dim, 0);

						if (root)
						{
							_traverseViewRecursive(ro, root, 0, _iFocusNavigatorRef);
							_iFocusNavigatorRef.completeConstruct(root);
						}
					}
				}
			}
		}
	}
#if defined(FOCUS_DEBUG_SVG_LOG)
	_dumpSvgEnd();
#endif
	return CE_S_OK;
}

CEHResult CEWebKitFocusImpl::click(void* hView)
{
	WebCore::RenderObject* ro = reinterpret_cast<WebCore::RenderObject*>(hView);
	if (ro)
	{
		WebCore::Node* node = ro->node();
		if (node)
		{
			node->dispatchSimulatedClick(0, true, true);
		}
	}
	return CE_S_OK;
}


CEHResult CEWebKitFocusImpl::setStatus(CEHtmlFocusNavigatorPeerStatus status, INT32 value)
{
	CEHResult ret = CE_S_OK;
	switch (status)
	{
#if defined(FOCUS_DEBUG_SVG_LOG)
		case CEHtmlFocusNavigatorPeerStatus_Debug_DumpSvg:
			_debugSvg = value;
			break;
#endif
		case CEHtmlFocusNavigatorPeerStatus_None:
		default:
			ret = CE_SILK_ERR_OPERATION_FAILED;
			break;
	}
	return ret;
}

CEHResult CEWebKitFocusImpl::getStatus(CEHtmlFocusNavigatorPeerStatus status, INT32* value) const
{
	CEHResult ret = CE_S_OK;
	switch (status)
	{
#if defined(FOCUS_DEBUG_SVG_LOG)
		case CEHtmlFocusNavigatorPeerStatus_Debug_DumpSvg:
			*value = _debugSvg;
			break;
#endif
		case CEHtmlFocusNavigatorPeerStatus_None:
		default:
			ret = CE_SILK_ERR_OPERATION_FAILED;
			break;
	}
	return ret;
}

WebCore::FrameView* CEWebKitFocusImpl::_getFrame(void* hView) const
{
	return reinterpret_cast<WebCore::FrameView*>(hView);
}
