/*
 * Copyright (C) 2006, 2007 Apple Inc.  All rights reserved.
 * Copyright (C) 2008 Collabora Ltd. All rights reserved.
 * Copyright     2009, 2012 Sony Corporation
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
#include "config.h"
#include "HostWindow.h"
#include "PluginView.h"
#include <runtime/JSLock.h>

#include "EventNames.h"
#include "Frame.h"
#include "FrameView.h"
#include "GraphicsContext.h"
#include "HTMLNames.h"
#include "HTMLPlugInElement.h"
#include "NotImplemented.h"
#include "npapi.h"
#include "npruntime_impl.h"
#include "PluginMainThreadScheduler.h"
#include "PluginPackage.h"
#include "ScriptController.h"
#include "Timer.h"
#include "MouseEvent.h"
#include "KeyboardEvent.h"
#include "PlatformKeyboardEvent.h"

#include "CEUITypes.h"
#include "ICEHtmlWebKit.h"
#include "npapi_silk.h"

using std::max;
using std::min;

namespace WebCore {
using namespace HTMLNames;

class PluginTimer : public TimerBase
{
public:
	PluginTimer(PluginView* view) : TimerBase(), m_view(view), m_markForDelete(false) {}
	virtual ~PluginTimer();

    virtual void fired();
	PluginView* getPluginView() { return m_view; }
	void markForDelete() { m_markForDelete = true; }

private:
	PluginView* m_view;
	bool m_markForDelete;
};

static WTF::Vector<PluginTimer*> pluginTimerVector;

PluginTimer::~PluginTimer() {
	size_t position = pluginTimerVector.find(this);
	pluginTimerVector.remove(position);
}

void PluginTimer::fired() {
	if (!m_markForDelete) {

		// To do:
		// I have to decide timer event paramters.
		CETimerEventParam timerParam;
		::memset(&timerParam, 0x00, sizeof(CETimerEventParam));

		NPEvent npEvent;
		npEvent.event = CENP_WM_TIMER;
		npEvent.lParam = reinterpret_cast<uint32>(&timerParam);

		if (m_view->plugin()->pluginFuncs()->event) {
			JSC::JSLock::DropAllLocks dropAllLocks(false);
			m_view->ref();
			m_view->plugin()->pluginFuncs()->event(m_view->instance(), &npEvent);
			m_view->deref();
		}
		//
	}

	delete this; // don't reuse me!
}

void PluginView::init()
{
    if (m_haveInitialized)
        return;
	// Silk's plugin only supports windowless plugin.
	m_isWindowed = false;
    m_haveInitialized = true;

    if (!m_plugin) {
        ASSERT(m_status == PluginStatusCanNotFindPlugin);
        return;
    }

    if (!m_plugin->load()) {
        m_plugin = 0;
        m_status = PluginStatusCanNotLoadPlugin;
        return;
    }

    if (!start()) {
        m_status = PluginStatusCanNotLoadPlugin;
        return;
    }

	m_matDocToPluginCoordinateScale.makeIdentity();
	m_matPlugToDocCoordinate.makeIdentity();

    m_npWindow.type = NPWindowTypeDrawable; // The Silk plugin always has fake window less plugin. 
	m_npWindow.window = 0;
    m_npWindow.x = 0;
    m_npWindow.y = 0;
    m_npWindow.width = 0;
    m_npWindow.height = 0;
    m_npWindow.clipRect.left = 0;
    m_npWindow.clipRect.top = 0;
    m_npWindow.clipRect.right = 0;
    m_npWindow.clipRect.bottom = 0;

	if (!m_plugin->quirks().contains(PluginQuirkDeferFirstSetWindowCall))
		setNPWindowRect(frameRect());

	show();
    m_status = PluginStatusLoadedSuccessfully;
}

void PluginView::updatePluginWidget()
{

	/******************************************/
	// The definition of coordinate
	/******************************************/
	// m_windowRect --> The TopFrame coordinate
	// m_clipRect --> The plugin coordinate

	if (!parent())
        return;

    ASSERT(parent()->isFrameView());
    FrameView* frameView = static_cast<FrameView*>(parent());

    IntRect oldWindowRect = m_windowRect;
    IntRect oldClipRect = m_clipRect;
	
	m_windowRect = IntRect(frameView->contentsToWindow(frameRect().location()), frameRect().size());

	//m_clipRect is indicated by PluginCoordinate
    m_clipRect = windowClipRect();
    m_clipRect.move(-m_windowRect.x(), -m_windowRect.y()); // convert for plugin coordinate

    if (platformPluginWidget() && (m_windowRect != oldWindowRect || m_clipRect != oldClipRect)) {

        setCallingPlugin(true);

        // To prevent flashes while scrolling, we disable drawing during the window
        // update process by clipping the window to the zero rect.

        bool clipToZeroRect = !m_plugin->quirks().contains(PluginQuirkDontClipToZeroRectWhenScrolling);
        setCallingPlugin(false);
    }

}

void PluginView::setFocus()
{
	Widget::setFocus();
}

void PluginView::show()
{
	Widget::show();
    setSelfVisible(true);
}

void PluginView::hide()
{
	Widget::hide();
    setSelfVisible(false);
}

void PluginView::handleKeyboardEvent(KeyboardEvent* event)
{
	//CEComDebugPrintf("PluginView::handleKeyboardEvent: keycode:%d charcode:%d  \n", event->keyCode(), event->charCode());
	NPEvent npEvent;
	CENPKeyboardEventParam cekeyboardEventParam;
	const PlatformKeyboardEvent* platformKeyboardEvent = event->keyEvent();
	if (platformKeyboardEvent)
	{
		if (event->type()==eventNames().keydownEvent)
		{
			npEvent.event = CENP_WM_KEYDOWN;
			cekeyboardEventParam.virtualKey = static_cast<eCENPVirtualKey>(platformKeyboardEvent->windowsVirtualKeyCode());
		}
		else if (event->type()==eventNames().keyupEvent)
		{
			npEvent.event = CENP_WM_KEYUP;
			cekeyboardEventParam.virtualKey = static_cast<eCENPVirtualKey>(platformKeyboardEvent->windowsVirtualKeyCode());
		}
		else if (event->type()==eventNames().keypressEvent)
		{
			npEvent.event = CENP_WM_KEYTYPED;
			cekeyboardEventParam.virtualKey = eCENPVK_UNDEFINED;
		}
		else
		{
			ASSERT(0);
		}
		cekeyboardEventParam.keyChar = event->charCode();
		cekeyboardEventParam.modifiers = eCENPKeyModifier_NONE;

		if (platformKeyboardEvent->ctrlKey())
		{
			cekeyboardEventParam.modifiers |= eCENPKeyModifier_CTRL;
		}

		if (platformKeyboardEvent->shiftKey())
		{
			cekeyboardEventParam.modifiers |= eCENPKeyModifier_SHIFT;
		}

		if (platformKeyboardEvent->altKey())
		{
			cekeyboardEventParam.modifiers |= eCENPKeyModifier_ALT;
		}

		npEvent.lParam = reinterpret_cast<uint32>(&cekeyboardEventParam);
		npEvent.wParam = 0;

		JSC::JSLock::DropAllLocks dropAllLocks(false);
		if (!dispatchNPEvent(npEvent))
			event->setDefaultHandled();
	}
}

extern bool ignoreNextSetCursor;

void PluginView::handleMouseEvent(MouseEvent* event)
{
	NPEvent npEvent;
	IntPoint p = static_cast<FrameView*>(parent())->contentsToWindow(IntPoint(event->pageX(), event->pageY()));

	p.setX(p.x()-m_windowRect.x());
	p.setY(p.y()-m_windowRect.y());

	double plgX;
	double plgY;
	m_matDocToPluginCoordinateScale.map(p.x(), p.y(), plgX, plgY);
	
	CENPPoint point;
	point._x = ceil(plgX);
	point._y = ceil(plgY);

	npEvent.lParam = reinterpret_cast<uint32>(&point);
	npEvent.wParam = 0;

	if (event->type() == eventNames().mousemoveEvent) 
	{
		npEvent.event = CENP_WM_MOUSEMOVE;
	}
	else if (event->type() == eventNames().mousedownEvent)
	{
		focusPluginElement();
		switch (event->button()) 
		{
		case 0:
			npEvent.event = CENP_WM_LBUTTONDOWN;
			break;
		default:
			break;
		}
	} 
	else if (event->type() == eventNames().mouseupEvent) 
	{
		switch (event->button()) 
		{
		case 0:
			npEvent.event = CENP_WM_LBUTTONUP;
			break;
		default:
			break;
		}
	}
	else
		return;

	JSC::JSLock::DropAllLocks dropAllLocks(false);
	if (!dispatchNPEvent(npEvent))
		event->setDefaultHandled();

	ignoreNextSetCursor = true;
}

void PluginView::setParent(ScrollView* parent)
{
    Widget::setParent(parent);

    if (parent)
	{
        init();
    }
}

void PluginView::setNPWindowRect(const IntRect& rect)
{	
    if (!m_isStarted || !parent() || !m_plugin->pluginFuncs()->setwindow)
        return;

    m_npWindow.window = NULL;
	m_npWindow.x = 0;
    m_npWindow.y = 0;

    m_npWindow.width = rect.width();
    m_npWindow.height = rect.height();

    m_npWindow.clipRect.left = m_clipRect.x();
    m_npWindow.clipRect.top = m_clipRect.y();
    m_npWindow.clipRect.right = m_clipRect.width()+m_clipRect.x();
    m_npWindow.clipRect.bottom = m_clipRect.height()+m_clipRect.y();

	PluginView::setCurrentPluginView(this);
    JSC::JSLock::DropAllLocks dropAllLocks(false);
    setCallingPlugin(true);
    m_plugin->pluginFuncs()->setwindow(m_instance, &m_npWindow);
    setCallingPlugin(false);
    PluginView::setCurrentPluginView(0);
}

void PluginView::setParentVisible(bool visible)
{
	if (isParentVisible() == visible)
        return;
    
	Widget::setParentVisible(visible);
}

void PluginView::stop()
{
    if (!m_isStarted)
        return;

    HashSet<RefPtr<PluginStream> > streams = m_streams;
    HashSet<RefPtr<PluginStream> >::iterator end = streams.end();
    for (HashSet<RefPtr<PluginStream> >::iterator it = streams.begin(); it != end; ++it) {
        (*it)->stop();
        disconnectStream((*it).get());
    }

    ASSERT(m_streams.isEmpty());

    m_isStarted = false;

    JSC::JSLock::DropAllLocks dropAllLocks(false);

    m_npWindow.window = 0;
    if (m_plugin->pluginFuncs()->setwindow && !m_plugin->quirks().contains(PluginQuirkDontSetNullWindowHandleOnDestroy)) {
        PluginView::setCurrentPluginView(this);
        setCallingPlugin(true);
        m_plugin->pluginFuncs()->setwindow(m_instance, &m_npWindow);
        setCallingPlugin(false);
        PluginView::setCurrentPluginView(0);
    }

    PluginMainThreadScheduler::scheduler().unregisterPlugin(m_instance);

    // Destroy the plugin
    PluginView::setCurrentPluginView(this);
    setCallingPlugin(true);
    m_plugin->pluginFuncs()->destroy(m_instance, 0);
    setCallingPlugin(false);
    PluginView::setCurrentPluginView(0);

	for (Vector<PluginTimer*>::iterator it = pluginTimerVector.begin(); it != pluginTimerVector.end(); ++it) {
		PluginTimer* timer = *it;
		if (timer && timer->getPluginView() == this) {
			timer->markForDelete();
		}
	}

    m_instance->pdata = 0;
}

static const char* MozillaUserAgent = "Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.8.1) Gecko/20061010 Firefox/2.0";
const char* PluginView::userAgent()
{
	if (m_plugin->quirks().contains(PluginQuirkWantsMozillaUserAgent))
        return MozillaUserAgent;

    if (m_userAgent.isNull())
        m_userAgent = m_parentFrame->loader()->userAgent(m_url).utf8();
    return m_userAgent.data();
}

const char* PluginView::userAgentStatic()
{
	return MozillaUserAgent;
}

NPError PluginView::handlePostReadFile(Vector<char>& buffer, uint32 len, const char* buf)
{
	return NPERR_NO_ERROR;
}

NPError PluginView::getValueStatic(NPNVariable variable, void* value)
{
	if (value)
	{
		switch (variable) {
		case NPNVToolkit:
			*((uint32 *)value) = 0;
			return NPERR_NO_ERROR;
		case NPNVSupportsXEmbedBool:
			*((uint32 *)value) = false;
			return NPERR_NO_ERROR;
		case NPNVjavascriptEnabledBool:
			*((uint32 *)value) = true;
			return NPERR_NO_ERROR;
		default:
			return NPERR_GENERIC_ERROR;
		}
	}
	return NPERR_GENERIC_ERROR;
}

NPError PluginView::getValue(NPNVariable variable, void* value)
{
	if (value)
	{
		switch (variable) {
			case NPNVWindowNPObject: {
				if (m_isJavaScriptPaused)
					return NPERR_GENERIC_ERROR;

				NPObject* windowScriptObject = m_parentFrame->script()->windowScriptNPObject();

				// Return value is expected to be retained, as described here: <http://www.mozilla.org/projects/plugin/npruntime.html>
				if (windowScriptObject)
					_NPN_RetainObject(windowScriptObject);

				void** v = (void**)value;
				*v = windowScriptObject;
            
				return NPERR_NO_ERROR;
			}

			case NPNVPluginElementNPObject: {
				if (m_isJavaScriptPaused)
					return NPERR_GENERIC_ERROR;

				NPObject* pluginScriptObject = 0;

				if (m_element->hasTagName(appletTag) || m_element->hasTagName(embedTag) || m_element->hasTagName(objectTag))
					pluginScriptObject = static_cast<HTMLPlugInElement*>(m_element)->getNPObject();

				// Return value is expected to be retained, as described here: <http://www.mozilla.org/projects/plugin/npruntime.html>
				if (pluginScriptObject)
					_NPN_RetainObject(pluginScriptObject);

				void** v = (void**)value;
				*v = pluginScriptObject;

				return NPERR_NO_ERROR;
			}
			default:
        	    return getValueStatic(variable, value);
		}
	}
	return NPERR_GENERIC_ERROR;
}

NPError PluginView::setValue(NPPVariable variable, void* value)
{
	NPError npErr = NPERR_NO_ERROR;
    switch (variable) {
    case NPPVpluginWindowBool:
		// ignore
        // m_isWindowed = value;
        return npErr;
    case NPPVpluginTransparentBool:
        m_isTransparent = value;
        return npErr;
    case NPPVpluginTimerInterval: {
		PluginTimer* timer = new PluginTimer(this);
		if (timer && value)
		{
			pluginTimerVector.append(timer);

			int usec = *static_cast<int*>(value);
			double sec = static_cast<double>(usec) / 1000.0 / 1000.0;
			timer->startOneShot(sec);
		}
        return npErr;
	}
	case NPPVpluginInitializeIME: {
		if (root() && root()->hostWindow())
		{
			PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
			if(platformWindow)
			{
				CENPInitializeIME* paramIME = static_cast<CENPInitializeIME*>(value);
				if(paramIME)
				{
					CEWebKitWidgetInputType inputType = CEWebKitWidgetInputType_Undefined;
					if((paramIME->imeMode & CENPImeModeFlag_Type_Text) == CENPImeModeFlag_Type_Text)
					{
						inputType = CEWebKitWidgetInputType_TEXT;
					}
					else if((paramIME->imeMode & CENPImeModeFlag_Type_Search) == CENPImeModeFlag_Type_Search)
					{
						inputType = CEWebKitWidgetInputType_SEARCH;
					}
					else if((paramIME->imeMode & CENPImeModeFlag_Type_Tel) == CENPImeModeFlag_Type_Tel)
					{
						inputType = CEWebKitWidgetInputType_TELEPHONE;
					}
					else if((paramIME->imeMode & CENPImeModeFlag_Type_Url) == CENPImeModeFlag_Type_Url)
					{
						inputType = CEWebKitWidgetInputType_URL;
					}
					else if((paramIME->imeMode & CENPImeModeFlag_Type_Email) == CENPImeModeFlag_Type_Email)
					{
						inputType = CEWebKitWidgetInputType_EMAIL;
					}		
					else if((paramIME->imeMode & CENPImeModeFlag_Type_Password) == CENPImeModeFlag_Type_Password)
					{
						inputType = CEWebKitWidgetInputType_PASSWORD;
					}
					else if((paramIME->imeMode & CENPImeModeFlag_Type_Number) == CENPImeModeFlag_Type_Number)
					{
						inputType = CEWebKitWidgetInputType_NUMBER;
					}
					else
					{
						npErr = NPERR_INVALID_PARAM;
					}

					bool isPassword = false;
					if((paramIME->imeMode & CENPImeModeFlag_Password) == CENPImeModeFlag_Password)
					{
						isPassword = true;
					}

					bool isSingleLine = false;
					if((paramIME->imeMode & CENPImeModeFlag_SingleLine) == CENPImeModeFlag_SingleLine)
					{
						isSingleLine = true;
					}

					if(npErr == NPERR_NO_ERROR)
					{

#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
						printf("initializeIME len:%d \n",  paramIME->iDefaultStringLen);
						for(int i=0; i< paramIME->iDefaultStringLen; i++)
						{
							printf("%x ", paramIME->iDefaultStringStr[i]);
						}
						printf("\n");
#endif
						CEHResult hr = platformWindow->initializeIME(isPassword, isSingleLine, inputType, paramIME->iDefaultStringStr,  paramIME->iDefaultStringLen, paramIME->maxLength, 0, 0, 0, 0, true, this);
						if(hr)
						{
							npErr = NPERR_GENERIC_ERROR;
						}
					}
				}
			}
		}
	}
		return npErr;
	case NPPVpluginShutdownIME:{
		if (root() && root()->hostWindow())
		{
			PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
			if(platformWindow)
			{
				CEHResult hr = platformWindow->shutdownIME(false); 
				if(hr)
				{
					npErr = NPERR_GENERIC_ERROR;
				}
			}
		}
	}
		return npErr;

	case NPPVsetcursor:
	{
		if (root() && root()->hostWindow())
		{
			PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
			if (platformWindow)
			{
				CENPSetCursor* paramCursor = static_cast<CENPSetCursor*>(value);
				if(paramCursor)
				{	
					CEHResult hr = CE_S_OK;
					eCECursorType setType = eCECursorType_Arrow;

					switch (paramCursor->type)
					{
					case CENPCursorType_Arrow:
						setType = eCECursorType_Arrow;
						break;
					case CENPCursorType_Hand:
						setType = eCECursorType_Hand;
						break;
					case CENPCursorType_IBeam:
						setType = eCECursorType_IBeam;
						break;
					case CENPCursorType_None:
						setType = eCECursorType_None;
						break;
					default:
						hr = CE_SILK_ERR_BADARGS;
						break;
					}
					if (!hr)
					{
						 hr = platformWindow->setCursor(setType);
					}
					if (hr)
					{
						npErr = NPERR_GENERIC_ERROR;
					}
				}
			}
		}
		return npErr;
	}
#if CE_OS(XAI) // todo delete this.
    case 9999:
    {
	    fprintf(stdout,"[SILK_Webkit] Memory usedSize = %dbyte \n", CESysHeapGetStat(eCESysHeapStat_UsedBytes));
	    return npErr;
    }
#endif
    default:
        notImplemented();
        return NPERR_GENERIC_ERROR;
    }
}

void PluginView::invalidateRect(const IntRect& rect)
{
	//The rect is indicated by pluginCoordinate
	if (parent()&&root()&&root()->hostWindow()) {
		PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
		if(platformWindow)
		{
			CEWebKitScrollMode scrollMode;
			platformWindow->getScrollMode(&scrollMode);
			if (scrollMode != CEWebKitScrollMode_Document)
			{
				_invalidateRect(rect);
			}
			else
			{
				ASSERT(0); //The test haven't done yet.
				_invalidateRectDocumentMode(rect);
			}
		}
	}
}

void PluginView::invalidateRect(NPRect* rect)
{
	IntRect diryRect;
	diryRect.setX(0);
	diryRect.setY(0);
	diryRect.setWidth(0);
	diryRect.setHeight(0);
	if (rect) {
		diryRect.setX(rect->left);
		diryRect.setY(rect->top);
		diryRect.setWidth(rect->right - rect->left);
		diryRect.setHeight(rect->bottom - rect->top);
	}
	invalidateRect(diryRect);
}

void PluginView::forceRedraw()
{
}

bool PluginView::dispatchNPEvent(NPEvent& npEvent)
{
    if (!m_plugin->pluginFuncs()->event)
        return true;

    JSC::JSLock::DropAllLocks dropAllLocks(false);
    setCallingPlugin(true);
    this->ref();
    bool result = m_plugin->pluginFuncs()->event(m_instance, &npEvent);
    this->deref();
    setCallingPlugin(false);

    return result;
}

PluginView::~PluginView()
{
    stop();
	m_matDocToPluginCoordinateScale.makeIdentity();
	m_matPlugToDocCoordinate.makeIdentity();

    deleteAllValues(m_requests);

    freeStringArray(m_paramNames, m_paramCount);
    freeStringArray(m_paramValues, m_paramCount);

    m_parentFrame->script()->cleanupScriptObjectsForPlugin(this);

    if (m_plugin && !(m_plugin->quirks().contains(PluginQuirkDontUnloadPlugin)))
        m_plugin->unload();
}

bool PluginView::setCompositionText(unsigned char* iTextStr, unsigned int iTextLen)
{
	CENPSetCompositionText compositionParam;
	::memset(&compositionParam, 0x00, sizeof(CENPSetCompositionText));
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
	printf("insertText len:%d \n", iTextLen);
	for(int i=0; i<iTextLen; i++)
	{
		printf("%x ", iTextStr[i]);
	}
	printf("\n");
#endif
	NPEvent npEvent;
	npEvent.event = CENP_WM_IME_COMPOSITIONTEXT;
	compositionParam.iTextStr = iTextStr;
	compositionParam.iTextLen = iTextLen;
	npEvent.lParam = reinterpret_cast<uint32>(&compositionParam);
	npEvent.wParam = 0;

	if (m_plugin->pluginFuncs()->event) {
		JSC::JSLock::DropAllLocks dropAllLocks(false);
		m_plugin->pluginFuncs()->event(m_instance, &npEvent);
	}

	return true;
}
bool PluginView::endComposition()
{
	NPEvent npEvent;
	npEvent.event = CENP_WM_IME_ENDCOMPOSITION;
	npEvent.lParam = 0;
	npEvent.wParam = 0;
	if (m_plugin->pluginFuncs()->event) {
		JSC::JSLock::DropAllLocks dropAllLocks(false);
		m_plugin->pluginFuncs()->event(m_instance, &npEvent);
	}
	return true;
}
bool PluginView::clearComponentText()
{
	NPEvent npEvent;
	npEvent.event = CENP_WM_IME_CLEARCOMPONENTTEXT;
	npEvent.lParam = 0;
	npEvent.wParam = 0;
	if (m_plugin->pluginFuncs()->event) {
		JSC::JSLock::DropAllLocks dropAllLocks(false);
		m_plugin->pluginFuncs()->event(m_instance, &npEvent);
	}
	return true;
}
bool PluginView::exitComponent()
{
	NPEvent npEvent;
	npEvent.event = CENP_WM_IME_EXITCOMPONENT;
	npEvent.lParam = 0;
	npEvent.wParam = 0;
	if (m_plugin->pluginFuncs()->event) {
		JSC::JSLock::DropAllLocks dropAllLocks(false);
		m_plugin->pluginFuncs()->event(m_instance, &npEvent);
	}
	return true;
}
void PluginView::suspend(bool suspend)
{
	NPBool value = static_cast<NPBool>(suspend);
	if (m_isStarted && m_plugin && m_plugin->pluginFuncs())
	{
		if (m_plugin->pluginFuncs()->setvalue)
		{
			m_plugin->pluginFuncs()->setvalue(m_instance, NPNVsuspend, static_cast<void*>(&value));
		}
	}
}

void PluginView::paint(GraphicsContext* context, const IntRect& rect)
{
	if (!m_isStarted)
	{
		paintMissingPluginIcon(context, rect);
        return; 
	}

#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
	CEComDebugPrintf("this:%08x plugin view rect: x:%d, y:%d, width:%d, height:%d \n", this, rect.x(), rect.y(), rect.width(), rect.height() );
	CEComDebugPrintf("this:%08x plugin view frameRect x:%d, y:%d, w:%d, h:%d\n", this, frameRect().location().x(), frameRect().location().y(), frameRect().width(), frameRect().height());
	CEComDebugPrintf("this:%08x plugin view x(), y(): x:%d y:%d\n", this, x(), y());
#endif
    if (context->paintingDisabled())
        return;

	if (parent())
	{
		ASSERT(parent()->isFrameView());

		PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
		if (platformWindow)
		{
			CEWebKitScrollMode scrollMode;
			platformWindow->getScrollMode(&scrollMode);
			if (scrollMode == CEWebKitScrollMode_Document)
			{
				_paintDocumentMode(context, rect);
			}
			else
			{
				_paint(context, rect);
			}
		}
	}
}

void PluginView::_paintDocumentMode(GraphicsContext* context, const IntRect& rect)
{
	FrameView* frameView = static_cast<FrameView*>(parent());
	if (frameView)
	{
		IntRect trect = rect;
		IntRect tframeRect = frameRect();
		IntRect tframeRectNoClip = tframeRect;
	
		// For IFrame
		// The document coordinate is used by PluginViewSilk::paint, when point is calculating.
		// Converting from FrameView coordinate to DocumentCoordinate. This is inverse calculation of ScrollView::paint()
		PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
		if(platformWindow)
		{
			ScrollView* parentScroll = frameView->parent();
			while(parentScroll)
			{
				trect.move(-frameView->scrollX(), -frameView->scrollY());
				trect.move(frameView->x(), frameView->y());
				tframeRect.move(-frameView->scrollX(), -frameView->scrollY());
				tframeRect.move(frameView->frameRect().location().x(), frameView->frameRect().location().y());
				tframeRect.intersect(frameView->frameRect());

				frameView = static_cast<FrameView*>(parentScroll);
				parentScroll = parentScroll->parent();
			}
		}
		//
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
		CEComDebugPrintf("this:%08x plugin view trect: x:%d, y:%d, width:%d, height:%d \n", this, trect.x(), trect.y(), trect.width(), trect.height() );
		CEComDebugPrintf("this:%08x plugin view tframeRect x:%d, y:%d, w:%d, h:%d\n", this, tframeRect.location().x(), tframeRect.location().y(), tframeRect.width(), tframeRect.height());
		CEComDebugPrintf("this:%08x plugin view tframeRectNoClip x:%d, y:%d, w:%d, h:%d\n", this, tframeRectNoClip.location().x(), tframeRectNoClip.location().y(), tframeRectNoClip.width(), tframeRectNoClip.height());
#endif

		m_npWindow.type = NPWindowTypeDrawable;
		m_npWindow.window = 0;
		if (context && context->platformContext())
		{
			context->save();
			CEComICEUnknownRef iUnkGCRef = NULL;
			CEHResult hr = CE_S_OK;

			CEComICEVGContextRef vgc(context->platformContext());
			hr = iUnkGCRef.initByQueryInterface(vgc);

			if (CESucceeded(hr)) 
			{		
				CEComICEVGSurfaceRef iSurface;
				CEHResult err = vgc.getTargetSurface(&iSurface);
				if (!err)
				{
					CEDim dim;
					err = iSurface.getDimension(&dim);
					if (!err && dim._width>0 && dim._height>0 )
					{
						PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
						if (platformWindow)
						{
							WebCore::FloatRect updateDoc; 
							err = platformWindow->getUpdateDocRect(&updateDoc);
							if (!err)
							{
								if (updateDoc.width() > 0.0f && updateDoc.height() > 0.0f)
								{
									TransformationMatrix matDocToPluginCoordinate; // The convert matrixt from document coordinate to Plugin coordinate
									matDocToPluginCoordinate.setMatrix(static_cast<double>(dim._width)/static_cast<double>(updateDoc.width()), 0.0, 0.0, static_cast<double>(dim._height)/static_cast<double>(updateDoc.height()), 0.0, 0.0);
									TransformationMatrix matMove(1.0, 0.0, 0.0, 1.0f, -tframeRect.x(), -tframeRect.y());
									matDocToPluginCoordinate.multLeft(matMove);

									//m_matPlugToDocCoordinate = matDocToPluginCoordinate.inverse();

									CEDXMatrix2DAffineD mat;
									vgc.getMatrix2DAffine(&mat);
									TransformationMatrix matDocToPluginCoordinateVirtual; // The convert matrixt from document coordinate to Plugin coordinate
									matDocToPluginCoordinateVirtual.setMatrix(mat._xx, 0.0, 0.0,mat._yy, 0.0, 0.0);
									matDocToPluginCoordinateVirtual.multLeft(matMove);

									m_matPlugToDocCoordinate = matDocToPluginCoordinateVirtual.inverse();
									m_matDocToPluginCoordinateScale.setMatrix(mat._xx, 0.0, 0.0,mat._yy, 0.0, 0.0);
								
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
							CEComDebugPrintf("this:%08x plugin view matDocToPluginCoordinate:\n m11:%f m12:%f m13:%f m14:%f\n m21:%f m22:%f m23:%f m24:%f\n m31:%f m32:%f m33:%f m34:%f\n m41:%f m42:%f m43:%f m44:%f\n\n", this, 
								matDocToPluginCoordinate.m11(), matDocToPluginCoordinate.m12(), matDocToPluginCoordinate.m13(), matDocToPluginCoordinate.m14(),
								matDocToPluginCoordinate.m21(), matDocToPluginCoordinate.m22(), matDocToPluginCoordinate.m23(), matDocToPluginCoordinate.m24(),
								matDocToPluginCoordinate.m31(), matDocToPluginCoordinate.m32(), matDocToPluginCoordinate.m33(), matDocToPluginCoordinate.m34(),
								matDocToPluginCoordinate.m41(), matDocToPluginCoordinate.m42(), matDocToPluginCoordinate.m43(), matDocToPluginCoordinate.m44()
								);
#endif
									IntRect docClipRect = tframeRect; // clipRect on DocumentCoordinate.
									docClipRect.intersect(trect);

									double tempResultL; // left
									double tempResultT; // top
									double tempResultR; // right
									double tempResultB; // botom

									matDocToPluginCoordinate.map(docClipRect.x(), docClipRect.y(), tempResultL, tempResultT);
									matDocToPluginCoordinate.map(docClipRect.x()+docClipRect.width(), docClipRect.y()+docClipRect.height(), tempResultR, tempResultB);
									FloatRect plgVirtualClipRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);

									INT32 l = static_cast<INT32>(floorf(plgVirtualClipRect.x()));
									INT32 t = static_cast<INT32>(floorf(plgVirtualClipRect.y()));
									INT32 r = static_cast<INT32>(floorf(plgVirtualClipRect.right()));
									INT32 b = static_cast<INT32>(floorf(plgVirtualClipRect.bottom()));

									IntRect plgClipRect = IntRect(l, t, r - l, b - t);

									m_clipRect = plgClipRect;

#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
									CEComDebugPrintf("this:%08x plugin view x(), y(): x:%d y:%d\n", this, x(), y());
									CEComDebugPrintf("this:%08x plugin view plgVirtualClipRect: x:%f y:%f, width:%f, height:%f \n", this, plgVirtualClipRect.x(), plgVirtualClipRect.y(), plgVirtualClipRect.width(), plgVirtualClipRect.height());
									CEComDebugPrintf("this:%08x plugin view m_clipRect: x:%d y:%d, width:%d, height:%d \n", this, m_clipRect.x(), m_clipRect.y(), m_clipRect.width(), m_clipRect.height());
									CEComDebugPrintf("this:%08x plugin view dim: width:%d, height:%d \n", this, dim._width, dim._height);
#endif
									// <<FrameRect On PluginCoord>>
									// Calc PluginRect size.
									matDocToPluginCoordinateVirtual.map(tframeRect.x(), tframeRect.y(), tempResultL, tempResultT);
									matDocToPluginCoordinateVirtual.map(tframeRect.x()+tframeRectNoClip.width(), tframeRect.y()+tframeRectNoClip.height(), tempResultR, tempResultB);
									FloatRect plgVirtualFrameRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);

									l = static_cast<INT32>(floorf(plgVirtualFrameRect.x()));
									t = static_cast<INT32>(floorf(plgVirtualFrameRect.y()));
									r = static_cast<INT32>(floorf(plgVirtualFrameRect.right()));
									b = static_cast<INT32>(floorf(plgVirtualFrameRect.bottom()));

									IntRect plgFrameRect = IntRect(l, t, r - l, b - t);

#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
									CEComDebugPrintf("this:%08x plugin view plgFrameRect: x:%d y:%d, width:%d, height:%d \n", this, plgFrameRect.x(), plgFrameRect.y(), plgFrameRect.width(), plgFrameRect.height());
#endif
									setNPWindowRect(plgFrameRect);
									//

									//<<NPRect>>
									IntRect docrect = IntRect(static_cast<INT32>(updateDoc.x()), static_cast<INT32>(updateDoc.y()), static_cast<INT32>(updateDoc.width()), static_cast<INT32>(updateDoc.height()));
									matDocToPluginCoordinate.map(docrect.x(), docrect.y(), tempResultL, tempResultT);
									matDocToPluginCoordinate.map(docrect.x()+docrect.width(), docrect.y()+docrect.height(), tempResultR, tempResultB);

									FloatRect plgVirtualDocRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);
									l = static_cast<INT32>(floorf(plgVirtualDocRect.x()));
									t = static_cast<INT32>(floorf(plgVirtualDocRect.y()));
									r = static_cast<INT32>(floorf(plgVirtualDocRect.right()));
									b = static_cast<INT32>(floorf(plgVirtualDocRect.bottom()));

									IntRect plgRect = IntRect(l, t, r - l, b - t);
									
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
									CEComDebugPrintf("this:%08x plugin view plgRect: x:%d y:%d, width:%d, height:%d \n", this, plgRect.x(), plgRect.y(), plgRect.width(), plgRect.height());
#endif
									unsigned char* pAddr =NULL;
									INT32 stride=0;
									err = iSurface.getImageDataRef( &stride, (UCHAR8* *const)&pAddr );
									if (!err)
									{
										NPEvent npEvent;
										npEvent.event = CENP_WM_PAINT;
										CENPBuffer npBuffer;
										npBuffer.pitch = stride;
										npBuffer.bytePerPixel = static_cast<int32>(stride/dim._width);

										npBuffer.pbuffer = pAddr;
										npEvent.wParam = reinterpret_cast<uint32>(&npBuffer);

										CENPRect npClipRect;
										npClipRect._left = plgRect.x();
										npClipRect._right = plgRect.x()+plgRect.width();
										npClipRect._top = plgRect.y();
										npClipRect._bottom =plgRect.y()+plgRect.height();

										npEvent.lParam = reinterpret_cast<uint32>(&npClipRect);
						
										dispatchNPEvent(npEvent);
									}
								}
							}
						}
					}
				}
			}
		}
		context->restore();
	}
}

void PluginView::_paint(GraphicsContext* context, const IntRect& rect)
{
	IntRect frameRectInWindow = static_cast<FrameView*>(parent())->contentsToWindow(frameRect());
	IntRect rectInWindow = static_cast<FrameView*>(parent())->contentsToWindow(rect); 

	PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
	if (platformWindow)
	{
		CEWebKitScrollMode scrollMode;
		platformWindow->getScrollMode(&scrollMode);
		if (scrollMode == CEWebKitScrollMode_DocumentZoomFactor2)
		{		
			frameRectInWindow = frameRect();
			rectInWindow = rect;
			ASSERT(parent()->isFrameView());
			FrameView* frameView = static_cast<FrameView*>(parent());
			if (frameView)
			{
				FrameView* nowFrameView = frameView;
				FrameView* firstParentView = frameView;
				bool isParent = true;
				while(isParent)
				{
					ScrollView* parentScroll = nowFrameView->parent();
					if (parentScroll)
					{
						frameRectInWindow.move(-nowFrameView->scrollX(), -nowFrameView->scrollY());
						frameRectInWindow.move(nowFrameView->x(), nowFrameView->y());
						rectInWindow.move(-nowFrameView->scrollX(), -nowFrameView->scrollY());
						rectInWindow.move(nowFrameView->frameRect().location().x(), nowFrameView->frameRect().location().y());

						nowFrameView = static_cast<FrameView*>(parentScroll);
					}
					else
					{
						isParent = false;
					}
				}	
				if (!isParent)
				{

				}
			}
		}
	}
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
	CEComDebugPrintf("this:%08x plugin rectInWindow: x:%d, y:%d, width:%d, height:%d \n", this, rectInWindow.x(), rectInWindow.y(), rectInWindow.width(), rectInWindow.height() );
	CEComDebugPrintf("this:%08x plugin frameRectInWindow: x:%d, y:%d, width:%d, height:%d \n", this, frameRectInWindow.x(), frameRectInWindow.y(), frameRectInWindow.width(), frameRectInWindow.height() );
#endif
	IntRect clipRect = frameRectInWindow;
	clipRect.intersect(rectInWindow);

	TransformationMatrix matDocToPluginCoordinate; // The convert matrixt from document coordinate to Plugin coordinate
	TransformationMatrix matMove(1.0, 0.0, 0.0, 1.0f, -frameRectInWindow.x(), -frameRectInWindow.y());
	matDocToPluginCoordinate.multLeft(matMove);
	m_matPlugToDocCoordinate = matDocToPluginCoordinate.inverse();

	double tempResultL; // left
	double tempResultT; // top
	double tempResultR; // right
	double tempResultB; // botom

	matDocToPluginCoordinate.map(clipRect.x(), clipRect.y(), tempResultL, tempResultT);
	matDocToPluginCoordinate.map(clipRect.x()+clipRect.width(), clipRect.y()+clipRect.height(), tempResultR, tempResultB);

	FloatRect plgVirtualClipRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);
	INT32 l = static_cast<INT32>(floorf(plgVirtualClipRect.x()));
	INT32 t = static_cast<INT32>(floorf(plgVirtualClipRect.y()));
	INT32 r = static_cast<INT32>(floorf(plgVirtualClipRect.right()));
	INT32 b = static_cast<INT32>(floorf(plgVirtualClipRect.bottom()));

	IntRect plgRect = IntRect(l, t, r - l, b - t);
	m_clipRect = plgRect;
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
	CEComDebugPrintf("this:%08x plugin After Intersection clipRect: x:%d, y:%d, width:%d, height:%d \n", this, m_clipRect.x(), m_clipRect.y(), m_clipRect.width(), m_clipRect.height() );
#endif
	setNPWindowRect(frameRectInWindow);

	if (context && context->platformContext())
	{
		CEComICEUnknownRef iUnkGCRef = NULL;
		CEHResult hr = CE_S_OK;

		CEComICEVGContextRef vgc(context->platformContext());
		hr = iUnkGCRef.initByQueryInterface(vgc);

		if (CESucceeded(hr)) 
		{		
			CEComICEVGSurfaceRef iSurface;
			hr = vgc.getTargetSurface(&iSurface);
			if (!hr)
			{
				unsigned char* pAddr =NULL;
				INT32 stride=0;
				hr = iSurface.getImageDataRef( &stride, (UCHAR8* *const)&pAddr );
				if (!hr)
				{
					WebCore::FloatRect updateDoc; 
					hr = platformWindow->getUpdateDocRect(&updateDoc);
					if (!hr)
					{	
						if (updateDoc.width() > 0.0f && updateDoc.height() > 0.0f)
						{
							if (!hr)
							{
								CEDim dim;
								hr = iSurface.getDimension(&dim);
								if (!hr)
								{
									//<<NPRect>>
									IntRect docrect = IntRect(static_cast<INT32>(updateDoc.x()), static_cast<INT32>(updateDoc.y()), static_cast<INT32>(updateDoc.width()), static_cast<INT32>(updateDoc.height()));
									matDocToPluginCoordinate.map(docrect.x(), docrect.y(), tempResultL, tempResultT);
									matDocToPluginCoordinate.map(docrect.x()+docrect.width(), docrect.y()+docrect.height(), tempResultR, tempResultB);

									FloatRect plgVirtualDocRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);
									INT32 l = static_cast<INT32>(floorf(plgVirtualDocRect.x()));
									INT32 t = static_cast<INT32>(floorf(plgVirtualDocRect.y()));
									INT32 r = static_cast<INT32>(floorf(plgVirtualDocRect.right()));
									INT32 b = static_cast<INT32>(floorf(plgVirtualDocRect.bottom()));

									IntRect plgRect = IntRect(l, t, r - l, b - t);
								
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
								CEComDebugPrintf("this:%08x plugin view plgRect: x:%d y:%d, width:%d, height:%d \n\n", this, plgRect.x(), plgRect.y(), plgRect.width(), plgRect.height());
#endif


									NPEvent npEvent;
									npEvent.event = CENP_WM_PAINT;
									CENPBuffer npBuffer;
									npBuffer.pitch = stride;
									npBuffer.bytePerPixel = static_cast<int32>(stride/dim._width);

									npBuffer.pbuffer = pAddr;
									npEvent.wParam = reinterpret_cast<uint32>(&npBuffer);

									CENPRect npClipRect;
									npClipRect._left = plgRect.x();
									npClipRect._right = plgRect.x()+plgRect.width();
									npClipRect._top = plgRect.y();
									npClipRect._bottom =plgRect.y()+plgRect.height();

									npEvent.lParam = reinterpret_cast<uint32>(&npClipRect);
						
									dispatchNPEvent(npEvent);
								}
							}
						}
					}	
				}
			}
		}
	}
}

void PluginView::_invalidateRect(const IntRect& plgRect)
{
	IntRect docRect = plgRect;

	if (docRect.width()==0 || docRect.height()==0)
	{
		docRect = frameRect(); // doc coordinate
		docRect.setX(0);
		docRect.setY(0);
	}
	invalidateWindowlessPluginRect(docRect);
}

void PluginView::_invalidateRectDocumentMode(const IntRect& plgRect)
{
	IntRect docRect = plgRect;
	if (plgRect.width()>0 && plgRect.height()>0)
	{
		double tempResultL; // left
		double tempResultT; // top
		double tempResultR; // right
		double tempResultB; // botom
		m_matPlugToDocCoordinate.map(plgRect.x(), plgRect.y(), tempResultL, tempResultT);
		m_matPlugToDocCoordinate.map(plgRect.right(), plgRect.bottom(), tempResultR, tempResultB);

		FloatRect fdocInvalidateRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);
		INT32 l = static_cast<INT32>(floorf(fdocInvalidateRect.x()));
		INT32 t = static_cast<INT32>(floorf(fdocInvalidateRect.y()));
		INT32 r = static_cast<INT32>(ceilf(fdocInvalidateRect.right()));
		INT32 b = static_cast<INT32>(ceilf(fdocInvalidateRect.bottom()));

		docRect = IntRect(l, t, r - l, b - t);
	}
	else
	{
		docRect = frameRect(); // doc coordinate
		docRect.setX(0);
		docRect.setY(0);
	}
	ASSERT(parent()->isFrameView());
	
	FrameView* frameView = static_cast<FrameView*>(parent());
	if (frameView)
	{
		ScrollView* parentScroll = frameView->parent();
		while(parentScroll)
		{
			frameView = static_cast<FrameView*>(parentScroll);
			parentScroll = parentScroll->parent();
		}
	}
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
	//	CEComDebugPrintf("PluginView::invalidateRect plg coord x:%d, y:%d, right:%d, bottom:%d, width:%d, height:%d\n", rect.x(), rect.y(), rect.right(), rect.bottom(), rect.width(), rect.height());
	//	CEComDebugPrintf("PluginView::invalidateRect doc coord x:%d, y:%d, right:%d, bottom:%d, width:%d, height:%d\n", dirtyRect.x(), dirtyRect.y(), dirtyRect.right(), dirtyRect.bottom(), dirtyRect.width(), dirtyRect.height());
#endif
	parent()->invalidateRect(docRect);
}

} // namespace WebCore
 
