/*
 * Copyright (C) 2006, 2008 Apple Inc. All rights reserved.
 * Copyright     2009 Sony Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#include "config.h"
#include "CString.h"
#include "PopupMenu.h"
#include "FrameView.h"
#include "ICEHtmlWebKit.h"
#include "ceulinearlists.h"

namespace WebCore {

PopupMenu::PopupMenu(PopupMenuClient* client)
    : m_popupClient(client), m_private(0)
{
}

PopupMenu::~PopupMenu()
{
	ICEHtmlWebKitFrameWindowWidget *frameWindow = (ICEHtmlWebKitFrameWindowWidget*)m_private;
	if(frameWindow)
	{
		CEComICEHtmlWebKitFrameWindowWidgetRef frameWindowRef = frameWindow;
		frameWindowRef.hidePopupMenu();
	}
}

void PopupMenu::updateFromElement()
{
    client()->setTextFromItem(client()->selectedIndex());
}

bool PopupMenu::itemWritingDirectionIsNatural()
{
    return true;
}

//format is specified in ICEApxPopupMenu.idl
#define POPUPMENU_ITEM		0x0
#define POPUPMENU_LABEL		0x1
#define POPUPMENU_SEPARATOR	0x2

#define POPUPMENU_ITEM_DISABLE	(0x0<<3)
#define POPUPMENU_ITEM_ENABLE	(0x1<<3)

void PopupMenu::show(const IntRect& rect, FrameView* view, int index)
{
	ASSERT(client());

	CEHResult err = CE_S_OK;

	UINT32 listSize = client()->listSize();
	UINT32 i;
	CEHResult hr;

	CEUUINT32List *attrList = new CEUUINT32List();
	CEUStringList *itemList = new CEUStringList();

	if(attrList && itemList)
	{
		for(i = 0; i < listSize; i++) {

			//set attribute
			{
				UINT32 attr = 0;
				if(client()->itemIsSeparator(i))
				{ attr |= POPUPMENU_SEPARATOR; }//separator
				else if(client()->itemIsLabel(i))
				{ attr |= POPUPMENU_LABEL; }//label
				else
				{ attr |= POPUPMENU_ITEM; }//item
				
				// set enable dor disable
				attr |= (client()->itemIsEnabled(i)) ? POPUPMENU_ITEM_ENABLE : POPUPMENU_ITEM_DISABLE;
				
				CEUUINT32ListElement *newAttr = new CEUUINT32ListElement(attr);
				if(newAttr) {
					attrList->addToTail(newAttr);
				} else {
					err = CE_SILK_ERR_MEMERR;
					break;
				}
			}

			//set item
			{
				CEComICEUStringRef str;
				WebCore::String wStr = client()->itemText(i);
				hr = wStr.createICEUString(&str);
				CEUStringListElement *newItem = new CEUStringListElement(str);
				if(newItem) {
					itemList->addToTail(newItem);
				} else {
					err = CE_SILK_ERR_MEMERR;
					break;
				}
			}
		}

		if(!err) {
		
			if(!m_private)
			{
				CEComGetThreadContext(CEComIID_ICEHtmlWebKitFrameWindowWidget, reinterpret_cast<void**>(&m_private));
			}

			ICEHtmlWebKitFrameWindowWidget *frameWindow = (ICEHtmlWebKitFrameWindowWidget*)m_private;
			if(frameWindow)
			{
				CEComICEHtmlWebKitFrameWindowWidgetRef frameWindowRef = frameWindow;
				WebCore::IntRect rectOnWindow = view->contentsToWindow(rect);
				err = frameWindowRef.showPopupMenu(attrList, itemList, client()->selectedIndex(),
								   rectOnWindow.x(), rectOnWindow.y(),
								   rectOnWindow.width(), rectOnWindow.height(),
								   reinterpret_cast<void*>(this));
			}
		}
	}

	if(err)
	{
		client()->popupDidHide();
	}

	//free all list elements
	if(attrList) {
		attrList->disposeAllElements();
		delete attrList;
	}
	if(itemList) {
		itemList->disposeAllElements();
		delete itemList;
	}
}


void PopupMenu::hide()
{
	ICEHtmlWebKitFrameWindowWidget *frameWindow = (ICEHtmlWebKitFrameWindowWidget*)m_private;
	if(frameWindow)
	{
		CEComICEHtmlWebKitFrameWindowWidgetRef frameWindowRef = frameWindow;
		frameWindowRef.hidePopupMenu();
	}
	client()->popupDidHide();

	//Remove pointer to frame 'cause Popup is closed.
	m_private = 0;
}

void PopupMenu::selected()
{
	//Remove pointer to frame 'cause Item is selected.
	m_private = 0;
}

} // namespace WebCore
