///////////////////////////////////////////////////////////////////////////////
// Copyright     2008,2009,2012 Sony Corporation
// Copyright (C) 2012 Sony Computer Entertainment Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
// 
///////////////////////////////////////////////////////////////////////////////
#include "config.h"
#include "FileSystem.h"
#include "CString.h"

#include "NotImplemented.h"

#include "CEApiFSFileSystem.h"
#include "ICETextSupport.h"
#include "CEFSPathnameUtil.h"
#include "ICEHtmlPlugin.h"

#include <wtf/CurrentTime.h>


namespace WebCore {

static ICEFSFileSystem* getFileSystem()
{
	static ICEFSFileSystem* iFileSystem = NULL;
	if (!iFileSystem)
	{
		CEComGetThreadContext(CEComIID_ICEFSFileSystem, reinterpret_cast<void**>(&iFileSystem));
		iFileSystem->_vtbl->_release(iFileSystem); // will keep alive after webcore die.
	}
	return iFileSystem;
}

static ICEHtmlPluginModuleLoader* getModuleLoader()
{
	// file system will keep alive after webcore die.
	static ICEHtmlPluginModuleLoader* iModuleLoader = NULL;
	if (!iModuleLoader)
	{
		CEComGetThreadContext(CEComIID_ICEHtmlPluginModuleLoader, reinterpret_cast<void**>(&iModuleLoader));
		iModuleLoader->_vtbl->_release(iModuleLoader); // will keep alive after webcore die.
	}
	return iModuleLoader;
}

static CEHResult convertStringToPathname(const String& path, CEComICEFSPathnameRef& pathnameOut)
{
	CEComICEUStringRef iStrRef = NULL;
	CEHResult hr = path.createICEUString(&iStrRef);
	if (CESucceeded(hr))
	{
		hr = ICEFSPathnameCreate(iStrRef, &pathnameOut);
	}
	return hr;
}

static CEHResult convertICEUStringToString(CEComICEUStringRef iStrRef, String& strOut)
{
	return strOut.fromICEUString(iStrRef);
}

static CEHResult convertPathnameToString(CEComICEFSPathnameRef iPathRef, String& strOut)
{
	CEComICEUStringRef iStrRef;
	CEHResult hr = iPathRef.toString(&iStrRef);
	if (CESucceeded(hr))
	{
		hr = convertICEUStringToString(iStrRef, strOut);
	}
	return hr;
}


bool fileExists(const String& path)
{
	bool result = false;
	CEComICEFSFileSystemRef iFileSystemRef = getFileSystem();
	CEComICEFSPathnameRef iPathRef = NULL;
	CEHResult hr = convertStringToPathname(path, iPathRef);
	if (CESucceeded(hr) && iFileSystemRef && iPathRef)
	{
		int status = 0;
		UINT32 size = 0;
		hr = iFileSystemRef.getStatus(iPathRef, &status, &size);
		if (CESucceeded(hr))
		{
			result = (status & eCEFileStatusFlag_Exist);
		}
	}
	return result;
}

bool deleteFile(const String& path)
{
    CEComICEFSFileSystemRef fs = getFileSystem();
    CEComICEFSPathnameRef iPath;
    CEHResult hr = convertStringToPathname(path, iPath);
    if (CESucceeded(hr) && fs && iPath)
    {
        hr = fs.unlink(iPath);
    }
    return CESucceeded(hr);
}

bool deleteEmptyDirectory(const String& path)
{
    CEComICEFSFileSystemRef fs = getFileSystem();
    CEComICEFSPathnameRef iPath;
    CEHResult hr = convertStringToPathname(path, iPath);
    if (CESucceeded(hr) && fs && iPath)
    {
        hr = fs.rmdir(iPath);
    }
    return CESucceeded(hr);
}

bool getFileSize(const String& path, long long& resultSize)
{
    CEComICEFSFileSystemRef fs = getFileSystem();
    CEComICEFSPathnameRef iPath;
    CEHResult hr = convertStringToPathname(path, iPath);
    if (CESucceeded(hr) && fs && iPath)
    {
        int status = 0;
        UINT32 size = 0;
        hr = fs.getStatus(iPath, &status, &size);
        resultSize = size;
    }
    return CESucceeded(hr);
}

bool getFileModificationTime(const String& path, time_t& modifiedTime)
{
	bool result = true;
	modifiedTime = currentTime() * 1000;
    return result;
}

String pathByAppendingComponent(const String& path, const String& component)
{
    String result;
    CEComICEFSPathnameRef iPath, iComponent;
    CEHResult hr1 = convertStringToPathname(path, iPath);
    CEHResult hr2 = convertStringToPathname(component, iComponent);
    if (CESucceeded(hr1) && CESucceeded(hr1))
    {
        CEComICEFSPathnameRef t;
        CEHResult hr = iPath.append(iComponent, &t);
        if (CESucceeded(hr))
            hr = convertPathnameToString(t, result);
        ASSERT(CESucceeded(hr));
    }
    return result;
}

static CEHResult makeAllDirectories(CEComICEFSPathnameRef& path)
{
    CEHResult hr;
    CEComICEFSFileSystemRef fs = getFileSystem();
    int status = 0;
    UINT32 size = 0;
    hr = fs.getStatus(path, &status, &size);
    if (!(CESucceeded(hr) && status & eCEFileStatusFlag_Exist && status & eCEFileStatusFlag_Directory))
    {
        CEComICEFSPathnameRef parent;
        hr = path.getSubPath(&parent);
        if (CESucceeded(hr))
        {
            hr = makeAllDirectories(parent);
            if (CESucceeded(hr))
            {
                hr = fs.mkdir(path, 0);
            }
        }
    }
    return hr;
}

bool makeAllDirectories(const String& path)
{
    bool ok = false;
    CEComICEFSFileSystemRef fs = getFileSystem();
    CEComICEFSPathnameRef iPath;
    CEHResult hr = convertStringToPathname(path, iPath);
    if (CESucceeded(hr) && fs && iPath)
    {
        hr = makeAllDirectories(iPath);
    }
    return CESucceeded(hr);
}

String homeDirectoryPath()
{
    notImplemented();
    return String();
}

String pathGetFileName(const String& pathName)
{
	String result;
	CEComICEFSPathnameRef iPathRef;
	CEHResult hr = convertStringToPathname(pathName, iPathRef);
	if (CESucceeded(hr))
	{
		CEComICEUStringRef iLastEntryRef;
		hr = iPathRef.getLastDirEntryName(&iLastEntryRef);
		if (CESucceeded(hr))
		{
			hr = convertICEUStringToString(iLastEntryRef, result);
		}
	}
    return result;
}

String directoryName(const String& path)
{
    notImplemented();
    return String();
}

Vector<String> listDirectory(const String& path, const String& filter)
{
	Vector<String> result;
 	CEComICEFSFileSystemRef iFileSystemRef = getFileSystem();
	CEComICEFSPathnameRef iDirPathRef = NULL;
	CEHResult hr = convertStringToPathname(path, iDirPathRef);
	if (CESucceeded(hr) && iFileSystemRef && iDirPathRef)
	{
		CEComICEFSDirectoryEnumeratorRef iEnumeratorRef;
		UINT32 optFlag = eCEDirectoryEnumerationFlag_skipDot |
			eCEDirectoryEnumerationFlag_skipDotDot |
			eCEDirectoryEnumerationFlag_skipDirectory |
			eCEDirectoryEnumerationFlag_skipHidden;
		hr = iFileSystemRef.createDirectoryEnumerator(iDirPathRef, optFlag, &iEnumeratorRef);
		bool hasMoreFile = false;
		while (CESucceeded(hr) && !(hr = iEnumeratorRef.hasMoreFile(&hasMoreFile)) && hasMoreFile)
		{
			CEComICEFSPathnameRef iFilePathRef;
			hr = iEnumeratorRef.getNext(&iFilePathRef);
			if (CESucceeded(hr))
			{
				String filePath;
				hr = convertPathnameToString(iFilePathRef, filePath);
				if (CESucceeded(hr))
				{
					result.append(filePath);
				}
			}
		}
	}
    return result;
}

CString openTemporaryFile(const char* prefix, PlatformFileHandle& handle)
{
    notImplemented();
    return CString();
}

void closeFile(PlatformFileHandle& handle)
{
    notImplemented();
}

int writeToFile(PlatformFileHandle handle, const char* data, int length)
{
    notImplemented();
    return 0;
}

bool unloadModule(PlatformModule module)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	CEComICEHtmlPluginModuleLoaderRef iModuleLoaderRef = getModuleLoader();
	if (iModuleLoaderRef)
	{
		hr = iModuleLoaderRef.unloadModule(module);
	}
    return CESucceeded(hr);
}

// Called by SharedBuffer::createWithContentsOfFile
CEHResult readContentsOfFile(const String& filePath, std::auto_ptr<char>& bufOut, int& sizeOut)
{
    CEComICEFSFileSystemRef fs = getFileSystem();
    CEComICEFSPathnameRef path;
    CEHResult hr = convertStringToPathname(filePath, path);
    if (CESucceeded(hr) && fs && path)
    {
        int status = 0;
        UINT32 size = 0;
        hr = fs.getStatus(path, &status, &size);
        if (CESucceeded(hr))
        {
            std::auto_ptr<char> tmp(new char[size]);
            if (!tmp.get())
            {
                return CE_SILK_ERR_MEMERR;
            }
            CEComICEFSFileRef file;
            hr = fs.open(&file, path, eCEFileOpenFlag_Read, 0);
            if (CESucceeded(hr))
            {
                int read_bytes;
                hr = file.read(tmp.get(), size, &read_bytes);
                if (CESucceeded(hr))
                {
                    bufOut = tmp;
                    sizeOut = size;
                }
            }
        }
    }
    return hr;
}
}
