/*
 * Copyright 2008,2009 Sony Corporation
 * Copyright (C) 2004, 2006 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2006 Michael Emmel mike.emmel@gmail.com
 * Copyright (C) 2007 Alp Toker <alp@atoker.com>
 * Copyright (C) 2007 Holger Hans Peter Freyther
 * Copyright (C) 2008 Collabora Ltd.
 * Copyright (C) 2008 Nuanti Ltd.
 * All rights reserved.
 * Copyright     2011, 2012 Sony Corporation
 * Copyright (C) 2011, 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "config.h"

#include "ResourceHandleInternal.h"
#include "Base64.h"
#include "HTTPParsers.h"
#include "MIMETypeRegistry.h"
#include "NotImplemented.h"
#include "ResourceError.h"
#include "ResourceHandle.h"
#include "MainResourceLoader.h"
#include "TextEncoding.h"

#include "ResourceHandleManagerCore.h"
#include "WebCoreSupport/FrameLoaderClientSilk.h"

#include "ICENetwork.h"
#include "ICENetworkSupport.h"
#include "CEComClassID.h"

#undef malloc
#undef realloc
#undef free

#define LPCSTR_TO_LPUINT8(pointer) reinterpret_cast<UINT8*>(const_cast<char*>(pointer))
#define LPSTR_TO_LPUINT8(pointer) reinterpret_cast<UINT8*>(pointer)

#define LPSTR_FREE(pointer) \
	do { \
		if (pointer) CEFREEEX(CENetwork, pointer) ; pointer=NULL; \
	} while (0)

static char* LPSTR_MALLOC(size_t allocsize) 
{
	return reinterpret_cast<char*>(CEMALLOCEX(CENetwork, allocsize));
}

static void LPVOID_FREE(void* pointer) 
{
	LPSTR_FREE(pointer);
}

static char* LPSTR_STRDUP(const char* src)
{
	return CESTRDUPEX(CENetwork, src);
}



namespace WebCore {

class ResourceHandleManager;

ResourceHandleManager* m_silkhandle= 0;
CEComICENetworkRef m_cenetwork = NULL;

void prefetchDNS(const String& hostname)
{
    notImplemented();
}

ResourceHandleManager* ResourceHandleManager::sharedInstance()
{
	if (!m_silkhandle)
		m_silkhandle = new ResourceHandleManager();
	return m_silkhandle;
}

ResourceHandleManager::ResourceHandleManager()
	: m_cookieJarFileName(NULL)
{
}

ResourceHandleManager::~ResourceHandleManager()
{
	LPVOID_FREE(m_cookieJarFileName);
}

void ResourceHandleManager::setCookies(const KURL& url, const String& value)
{
	String hostStr = url.string().substring(url.hostStart(), url.hostEnd() - url.hostStart());
	String pathStr = url.string().substring(url.pathStart(), url.pathEnd() - url.pathStart());
	CERawDataBuffer hostData = { 0 };
	CERawDataBuffer pathData = { 0 };
	CERawDataBuffer valueData = { 0 };

// work around for Bug 19597.
#if CE_OS(XAI) || CE_OS(HYDRA)
	char* lpstrHost = LPSTR_STRDUP(url.string().isEmpty()?"":url.string().utf8().data());
#else
	char* lpstrHost = LPSTR_STRDUP(hostStr.isEmpty()?"":hostStr.utf8().data());
#endif
	char* lpstrPath = LPSTR_STRDUP(pathStr.isEmpty()?"":pathStr.utf8().data());
	char* lpstrValue = LPSTR_STRDUP(value.isEmpty()?"":value.utf8().data());
	hostData._data = LPSTR_TO_LPUINT8(lpstrHost);
	pathData._data = LPSTR_TO_LPUINT8(lpstrPath);
	valueData._data = LPSTR_TO_LPUINT8(lpstrValue);
	m_cenetwork.setCookies(
		&hostData,
		&pathData,
		&valueData
		);
	LPSTR_FREE(lpstrHost);
	LPSTR_FREE(lpstrPath);
	LPSTR_FREE(lpstrValue);
}

String ResourceHandleManager::getCookies(const KURL& url)
{
	const size_t bufSize = 4096;
	char* buf = LPSTR_MALLOC(bufSize);
	if (buf)
	{
		String hostStr = url.string().substring(url.hostStart(), url.hostEnd() - url.hostStart());
		String pathStr = url.string().substring(url.pathStart(), url.pathEnd() - url.pathStart());
		CERawDataBuffer hostData = {0};
		CERawDataBuffer pathData = {0};
// work around for Bug 19597.
#if CE_OS(XAI) || CE_OS(HYDRA)
		char* lpstrHost = LPSTR_STRDUP(url.string().isEmpty()?"":url.string().utf8().data());
#else
		char* lpstrHost = LPSTR_STRDUP(hostStr.isEmpty()?"":hostStr.utf8().data());
#endif
		char* lpstrPath = LPSTR_STRDUP(pathStr.isEmpty()?"":pathStr.utf8().data());
		hostData._data = LPSTR_TO_LPUINT8(lpstrHost);
		pathData._data = LPSTR_TO_LPUINT8(lpstrPath);
		unsigned resSize = 0;
		m_cenetwork.getCookies(
			&hostData,
			&pathData,
			false, buf, bufSize, &resSize
			);
		if (resSize >= bufSize)
		{
			buf = static_cast<char*>(CEComGetAllocatorRec()->realloc(CEComGetAllocatorRec(), buf, resSize+1));
			if (buf)
				m_cenetwork.getCookies(
					&hostData,
					&pathData,
					false, buf, resSize+1, &resSize
					);
		}
		LPSTR_FREE(lpstrHost);
		LPSTR_FREE(lpstrPath);
	}
	String cookie(buf);
	LPSTR_FREE(buf);
	return cookie;
}

bool ResourceHandleManager::cookiesEnabled()
{
	bool cookiesEnabled = false;
	m_cenetwork.cookiesEnabled(&cookiesEnabled);
	return cookiesEnabled;
}

void ResourceHandleManager::setCookieJarFileName(const char* cookieJarFileName)
{
	m_cookieJarFileName = LPSTR_STRDUP(cookieJarFileName);
}

void ResourceHandleManager::downloadTimerCallback(Timer<ResourceHandleManager>*)
{
}

bool ResourceHandleManager::removeScheduledJob(ResourceHandle* job)
{
	int size = m_resourceHandleList.size();
	for (int i = 0; i < size; i++) {
		if (job == m_resourceHandleList[i]) {
			m_resourceHandleList.remove(i);
#ifdef RESOURCEHANDLEMANAGER_DEBUG
			CEComDebugPrintf("####### ResourceHandleManager::removeScheduledJob() job->defef() job = [%p]#######\n", job);
#endif
			job->deref();
			return true;
		}
	}
	return false;
}

void ResourceHandleManager::startJob(ResourceHandle*)
{
}

CEHResult ResourceHandleManager::startJob(ResourceHandle* job, bool syncRequest, WebCore::Frame* frame)
{
	CEHResult hr = CE_S_OK;

	unsigned long identifier = 0;
	KURL kurl = job->request().url();

	ResourceHandleInternal* d = job->getInternal();
	// Remove any fragment part, otherwise curl will send it as part of the request.
	String url = kurl.string();
	d->m_url = LPSTR_STRDUP(url.utf8().data());

	char* customHeadersCStr = NULL;
	UINT32 headerCount = 0;
	char* postData = NULL;
	UINT32 postSize = 0;
	bool chunkedTransfer = false;
	UINT32 chunkedTransferFileSize = 0;
	UINT32 totallen = 0;
	bool requireRead = false;
	if (job->request().httpHeaderFields().size() > 0) {
		HTTPHeaderMap customHeaders = job->request().httpHeaderFields();
		HTTPHeaderMap::const_iterator end = customHeaders.end();
		for (HTTPHeaderMap::const_iterator it = customHeaders.begin(); it != end; ++it) {
			String key = it->first;
			String value = it->second;
			String headerString(key);
			headerString.append(": ");
			headerString.append(value);
			CString headerLatin1 = headerString.latin1();
			UINT32 length = headerLatin1.length();
			customHeadersCStr = reinterpret_cast<char*>(CEREALLOCEX(CENetwork, 
										customHeadersCStr,
									    totallen + length + 2
									    ));
			if (customHeadersCStr)
			{
				memcpy(customHeadersCStr + totallen, headerLatin1.data(), length);
				totallen += length;
				customHeadersCStr[totallen] = '\r';
				customHeadersCStr[totallen + 1] = '\n';
				totallen += 2;
				headerCount++;
			}
			else
			{
				CEASSERT(false);
			}							
		}
		if (customHeadersCStr)
		{
			totallen += 1;
			customHeadersCStr = reinterpret_cast<char*>(CEREALLOCEX(CENetwork,
									    customHeadersCStr,
									    totallen
									    ));
			if (customHeadersCStr)
				customHeadersCStr[totallen - 1] = '\0';
		}
	}

	String fullPathStr;
	if (job->request().httpBody())
	{
		Vector<FormDataElement> elements = job->request().httpBody()->elements();
		size_t numElements = elements.size();
		if (!numElements)
		{
			CEASSERT(false);
			hr = CE_SILK_ERR_OPERATION_FAILED;
		}

		if (CESucceeded(hr))
		{
			// Do not stream for simple POST data
			if (numElements == 1) {
				job->request().httpBody()->flatten(d->m_postBytes);
				if (d->m_postBytes.size() != 0) {
					postSize = d->m_postBytes.size();
					postData = LPSTR_MALLOC(postSize + 1);
					if (postData)
					{
						memcpy(postData, d->m_postBytes.data(), postSize);
						postData[postSize] = '\0';
					}
				}
			}
			else
			{
				for (size_t i = 0; i < numElements; i++)
				{
					FormDataElement elm = elements[i];
					if (elm.m_shouldGenerateFile == true)
					{
						fullPathStr.append(elm.m_generatedFilename);
						fullPathStr.append("\r\n");
						requireRead = true;
						job->request().setIsUpLoad(true);
					}
				}

				job->request().httpBody()->flatten(d->m_postBytes);
				if (d->m_postBytes.size() != 0) {
					postSize = d->m_postBytes.size();
					postData = LPSTR_MALLOC(postSize + 1);
					if (postData)
					{
						memcpy(postData, d->m_postBytes.data(), postSize);
						postData[postSize] = '\0';
					}
				}
			}
		}
	}

	if (CESucceeded(hr))
	{
		CEComICENetworkEventListenerRef networkListener = NULL;
		networkListener = reinterpret_cast<ICENetworkEventListener*>(ResourceHandleManager::getNetworkListenerFromJob(job, syncRequest, frame));
		if (syncRequest == true)
		{
			identifier = job->request().getIdentifier();
		}
		else
		{
			ResourceLoader* client = reinterpret_cast<ResourceLoader*>(job->getInternal()->client());
			if (client)
				identifier = client->identifier();
		}

		CENetworkHttpMethod method = CENETWORK_HTTP_GET;

		if (job->request().httpMethod().contains("HEAD", false) == true)
		{
			method = CENETWORK_HTTP_HEAD;
		}
		else if (job->request().httpMethod().contains("POST", false) == true)
		{
			method = CENETWORK_HTTP_POST;
		}
		else if (job->request().httpMethod().contains("PUT", false) == true)
		{
			method = CENETWORK_HTTP_PUT;
		}

		if (frame)
		{
			String hostStr = kurl.string().substring(kurl.hostStart(), kurl.hostEnd() - kurl.hostStart());
			String pathStr = kurl.string().substring(kurl.pathStart(), kurl.pathEnd() - kurl.pathStart());
			CERawDataBuffer hostData = {0};
			CERawDataBuffer pathData = {0};
			CERawDataBuffer protocolData = {0};
			CERawDataBuffer queryData = {0};
			CERawDataBuffer hashData = {0};
			CERawDataBuffer uploadPathArrayData = {0};

			char* lpstrHost = LPSTR_STRDUP(hostStr.isEmpty()?"":hostStr.utf8().data());
			char* lpstrPath = LPSTR_STRDUP(pathStr.isEmpty()?"":pathStr.utf8().data());
			char* lpstrProtocol = LPSTR_STRDUP(kurl.protocol().isEmpty()?"":kurl.protocol().utf8().data());
			char* lpstrQuery = LPSTR_STRDUP(kurl.query().isEmpty()?"":kurl.query().utf8().data());
			char* lpstrHash = LPSTR_STRDUP(kurl.ref().isEmpty()?"":kurl.ref().utf8().data());
			char* lpstrPathArray = LPSTR_STRDUP(requireRead==true?fullPathStr.utf8().data():NULL);

			hostData._data = LPSTR_TO_LPUINT8(lpstrHost);
			pathData._data = LPSTR_TO_LPUINT8(lpstrPath);
			protocolData._data = LPSTR_TO_LPUINT8(lpstrProtocol);
			queryData._data = LPSTR_TO_LPUINT8(lpstrQuery);
			hashData._data = LPSTR_TO_LPUINT8(lpstrHash);
			uploadPathArrayData._data = LPSTR_TO_LPUINT8(lpstrPathArray);

			unsigned long idOnFrame = identifier;
			unsigned long pageId = frame->page()->group().identifier();
			pageId = pageId << PAGE_IDENTIFIER_SHIFT;
			idOnFrame |= pageId;

			bool toReload = (ReloadIgnoringCacheData == job->request().cachePolicy());
			bool contextCreated = false;
			if (syncRequest)
			{
				hr = m_cenetwork.syncSubmit(
					job->request().getIsMainRresource(),
					idOnFrame,
					reinterpret_cast<void*>(job),
					networkListener,
					&protocolData,
					method,
					&hostData,
					kurl.port(),
					&pathData,
					&queryData,
					&hashData,
					customHeadersCStr,
					postData,
					postSize,
					requireRead,
					&uploadPathArrayData,  // array of full-path of upload file
					&contextCreated,
					NULL); // connected form data
			}
			else
			{
				hr = m_cenetwork.submit(
					toReload,
					job->request().getIsMainRresource(),
					idOnFrame,
					reinterpret_cast<void*>(job),
					networkListener,
					&protocolData,
					method,
					&hostData,
					kurl.port(),
					&pathData,
					&queryData,
					&hashData,
					customHeadersCStr,
					postData,
					postSize,
					requireRead,
					&uploadPathArrayData, // array of full-path of upload file
					&contextCreated,
					NULL); // connected form data
			}
			LPSTR_FREE(lpstrHost);
			LPSTR_FREE(lpstrPath);
			LPSTR_FREE(lpstrProtocol);
			LPSTR_FREE(lpstrQuery);
			LPSTR_FREE(lpstrPathArray);
			if (frame && frame->loader() && true == contextCreated)
			{
				FrameLoaderClientSilk* clientsilk = reinterpret_cast<FrameLoaderClientSilk*>(frame->loader()->client());
				if (clientsilk)
				{
					clientsilk->setIdentifier(identifier, job->request().getIsMainRresource());
				}
			}
		}
	}

	LPSTR_FREE(customHeadersCStr);
	LPSTR_FREE(postData);
	return hr;
}

CEHResult ResourceHandleManager::init(ICENetwork* iCENetwork)
{
	if (iCENetwork)
	{
		CEASSERT(!m_silkhandle);
		m_silkhandle = new ResourceHandleManager();
		if (m_silkhandle)
		{
			if (!m_cenetwork)
			{
				m_cenetwork = iCENetwork;
				return CE_S_OK;
			}
			return CE_SILK_ERR_BADSTATE;
		}
		else
			return CE_SILK_ERR_MEMERR;
	}
	return CE_SILK_ERR_BADARGS;
}

void ResourceHandleManager::shutdown()
{
	if (m_silkhandle)
	{
		delete m_silkhandle;
		m_silkhandle = 0;
	}
	if (m_cenetwork)
	{
		m_cenetwork = NULL;
	}
}

CEHResult ResourceHandleManager::setCookieEnable(bool cookiesEnabled)
{
	CEHResult hr = CE_S_OK;
	if (m_cenetwork)
	{
		if (cookiesEnabled == true)
		{
			hr = m_cenetwork.enableCookies();
		}
		else
		{
			hr = m_cenetwork.disableCookies();
		}
	}
	return hr;
}

CEHResult ResourceHandleManager::deleteAllCookies()
{
	CEHResult hr = CE_S_OK;
	if (m_cenetwork)
	{
		hr = m_cenetwork.deleteAllCookies();
	}
	return hr;
}

CEHResult ResourceHandleManager::_handleDataAvailableCallback(ResourceHandle* webCoreJob, UINT32 httpCode, const char* url, void* data, UINT32 size)
{
	CEHResult hr = CE_S_OK;
	if (webCoreJob)
	{
		ResourceHandleInternal* d = NULL;
		d = webCoreJob->getInternal();

		if (d)
		{
			if (d->m_cancelled)
				return CE_S_OK;

			if (httpCode >= 300 && httpCode < 400)
			{
				String location = d->m_response.httpHeaderField("location");
				if (!location.isEmpty()) {
					return CE_S_OK;
				}
			}

			// since the code in headerCallback will not have run for local files
			// the code to set the URL and fire didReceiveResponse is never run,
			// which means the ResourceLoader's response does not contain the URL.
			// Run the code here for local files to resolve the issue.
			// TODO: See if there is a better approach for handling this.
			if (!d->m_response.responseFired()) 
			{
				String urlString = String(url);
				if (!urlString.isEmpty())
				{
					String encodedStr = encodeWithURLEscapeSequences(urlString);
					d->m_response.setURL(KURL(encodedStr));
				}
				if (d->client())
				{
					d->m_response.setResponseFired(true);
					d->client()->didReceiveResponse(webCoreJob, d->m_response);
				}
			}

			if (d->client())
			{
				d->client()->didReceiveData(webCoreJob, reinterpret_cast<char*>(data), size, 0);
			}
		}
	}
	else
	{
		CEASSERT(false);
	}
	return hr;
}

CEHResult ResourceHandleManager::DataAvailableCallback(ResourceHandle* webCoreJob, UINT32 httpCode, const char* url, void* data, UINT32 size)
{
#ifdef RESOURCEHANDLEMANAGER_DEBUG
	CEComDebugPrintf("####### ResourceHandleManager::DataAvailableCallback() webCoreJob = [%p]#######\n", webCoreJob);
//	CEComDebugPrintf("####### ResourceHandleManager::DataAvailableCallback() data = [%hs]#######\n", reinterpret_cast<char*>(data));
	CEComDebugPrintf("####### ResourceHandleManager::DataAvailableCallback() dataSize = [%d]#######\n", size);
#endif
	CEHResult hr = CE_S_OK;
	hr = _handleDataAvailableCallback(webCoreJob, httpCode, url, data, size);
    return hr;
}

CEHResult ResourceHandleManager::_handleHeaderCallbackInternal(ResourceHandle* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderLen, const char* url, UINT32 httpCode, UINT32 expectedContentLength, ResourceResponse& response)
{
	CEHResult hr = CE_S_OK;

	if (webCoreJob)
	{
		KURL kurl = webCoreJob->request().url();

		String responseHeaders(responseHeaderBuffer, responseHeaderLen);
		int splitHeaderPos = 0;
		int splitLinePos = responseHeaders.find("\r\n");
		if (splitLinePos != -1)
		{
			do
			{
				String responseHeader = responseHeaders.substring(0, splitLinePos);
				String responseHeadersNext = responseHeaders.substring(splitLinePos + 2);
				if (responseHeader.isEmpty() == false)
				{
					splitHeaderPos = responseHeader.find(": ");
					if (splitHeaderPos != -1)
					{
						String headerNameStr = responseHeader.left(splitHeaderPos);
						String headerValueStr = responseHeader.substring(splitHeaderPos + 1).stripWhiteSpace();
						response.setHTTPHeaderField(AtomicString(
														String(headerNameStr)), 
													String(headerValueStr)
							);
#ifdef RESOURCEHANDLEMANAGER_DEBUG
						CEComDebugPrintf("####### ResourceHandleManager::headerCallback() headerNameStr = [%hs]#######\n", headerNameStr.utf8().data());
						CEComDebugPrintf("####### ResourceHandleManager::headerCallback() headerValueStr = [%hs]#######\n", headerValueStr.utf8().data());
#endif
					}
				}
				else
				{
					break;
				}
				responseHeaders = responseHeadersNext;
				splitLinePos = responseHeaders.find("\r\n");
			}while(splitLinePos != 1);
		}

		String urlString = String(url);
		if (!urlString.isEmpty())
		{
			response.setURL(KURL(urlString));
		}
		response.setHTTPStatusCode(httpCode);
		if (expectedContentLength != -1)
		{
			response.setExpectedContentLength(static_cast<long long int>(expectedContentLength));
		}

		// set following header fields explicitly.
		String ContentType = extractMIMETypeFromMediaType(response.httpHeaderField(AtomicString("Content-Type")));
		String ContentEncoding = extractCharsetFromMediaType(response.httpHeaderField(AtomicString("Content-Type")));
		if (ContentEncoding.isEmpty())
		{
			ContentEncoding = extractCharsetFromMediaType(response.httpHeaderField(AtomicString("Content-Encoding")));
		}
		String ContentDisposition = filenameFromHTTPContentDisposition(response.httpHeaderField(AtomicString("Content-Disposition")));


		if (ContentType.isEmpty() == true)
		{
			ContentType = String("text/html");
		}
#if 1 // Bug 18240 enable image/jps temporary.
		if (urlString.endsWith("jps", true))
		{
			ContentType = String("image/jps");
		}
#endif

		response.setMimeType(ContentType);
		if (ContentEncoding.isEmpty() == false)
			response.setTextEncodingName(ContentEncoding);
		if (ContentDisposition.isEmpty() == false)
			response.setSuggestedFilename(ContentDisposition);
	}
	else
	{
		hr = CE_SILK_ERR_OPERATION_FAILED;
	}

	return hr;
}

CEHResult ResourceHandleManager::_handleHeaderCallback(ResourceHandle* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderLen, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext)
{
	CEHResult hr = CE_S_OK;
	if (webCoreJob)
	{
		ResourceHandleInternal* d = webCoreJob->getInternal();
		ResourceHandleClient* client = NULL;
		if (d && d->m_cancelled == false)
		{
			_handleHeaderCallbackInternal(webCoreJob, responseHeaderBuffer, responseHeaderLen, url, httpCode, expectedContentLength, d->m_response);
			client = d->client();
			
			if (client)
			{
				d->m_response.setNoContext(noContext);
				d->m_response.setResponseFired(true);
				d->m_response.setURL(KURL(url));
				client->didReceiveResponse(webCoreJob, d->m_response);
#ifdef RESOURCEHANDLEMANAGER_DEBUG
				CEComDebugPrintf("####### ResourceHandleManager::headerCallback() pMessage = [%p]#######\n", pMessage);
#endif
			}
		}
	}
	else
	{
		// TODO: error callback
		CEASSERT(false);
	}
	return hr;
}

CEHResult ResourceHandleManager::headerCallback(ResourceHandle* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderLen, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext)
{
	CEHResult hr = CE_S_OK;

	if (webCoreJob)
	{
		_handleHeaderCallback(webCoreJob, responseHeaderBuffer, responseHeaderLen, url, httpCode, expectedContentLength, noContext);
	}
	else
	{
		// TODO: error callback
		CEASSERT(false);
	}

	return hr;
}

CEHResult ResourceHandleManager::redirectCallback(ResourceHandle* webCoreJob, const char* url)
{
#ifdef RESOURCEHANDLEMANAGER_DEBUG
	CEComDebugPrintf("####### ResourceHandleManager::redirectCallback() webCoreJob = [%p]#######\n", webCoreJob);
#endif

	CEASSERT(webCoreJob);

	CEHResult hr = CE_S_OK;
	if (webCoreJob)
	{
		ResourceHandleInternal* d = NULL;
		ResourceHandleClient* client = NULL;
		ResourceRequest redirectedRequest;
		d = webCoreJob->getInternal();
		if (d && d->m_cancelled == false) client = d->client();

		if (client && url) 
		{
			String location = String(url);
			KURL newURL = KURL(webCoreJob->request().url(), location);

			ResourceRequest redirectedRequest = webCoreJob->request();
			redirectedRequest.setURL(newURL);

			redirectedRequest.setRedirected(true);
			d->m_response.setRedirected(true);

			if (client)
			{
				client->willSendRequest(webCoreJob, redirectedRequest, d->m_response);
			}

			redirectedRequest.setRedirected(false);
			d->m_response.setRedirected(false);

			d->m_request.setURL(newURL);

		}
	}
	else
	{
		hr = CE_SILK_ERR_BADARGS;
	}
	return hr;
}

CEHResult ResourceHandleManager::allInOneCallback(ResourceHandle* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderLen, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext, void* data, UINT32 size)
{
	CEHResult hr = CE_S_OK;

	CEASSERT(webCoreJob);

	if (webCoreJob)
	{
		hr = _handleHeaderCallback(webCoreJob, responseHeaderBuffer, responseHeaderLen, url, httpCode, expectedContentLength, noContext);
		if (CESucceeded(hr))
		{
			hr = _handleDataAvailableCallback(webCoreJob, httpCode, url, data, size);
		}
		if (CESucceeded(hr))
		{
			hr = _handleCompleteCallback(webCoreJob);
		}
	}
	else
	{
		hr = CE_SILK_ERR_BADARGS;
	}
	return hr;
}

CEHResult ResourceHandleManager::_handleCompleteCallback(ResourceHandle* webCoreJob)
{
	CEHResult hr = CE_S_OK;

	CEASSERT(webCoreJob);

	if (webCoreJob)
	{
		ResourceHandleInternal* d = NULL;
		ResourceHandleClient* client = NULL;
		d = webCoreJob->getInternal();

		if (d && d->m_cancelled == false)
			client = d->client();

		if (client)
		{
			client->didFinishLoading(webCoreJob);
		}
#ifdef RESOURCEHANDLEMANAGER_DEBUG
		CEComDebugPrintf("####### ResourceHandleManager::CompleteCallback() job->defef() job = [%p]#######\n", job);
#endif
	}
	else
	{
		hr = CE_SILK_ERR_BADARGS;
	}
	return hr;
}

CEHResult ResourceHandleManager::CompleteCallback(ResourceHandle* webCoreJob)
{
#ifdef RESOURCEHANDLEMANAGER_DEBUG
	CEComDebugPrintf("####### ResourceHandleManager::CompleteCallback() webCoreJob = [%p]#######\n", webCoreJob);
#endif
	CEHResult hr = CE_S_OK;

	CEASSERT(webCoreJob);

	if (webCoreJob)
	{
		_handleCompleteCallback(webCoreJob);
	}
	else
	{
		hr = CE_SILK_ERR_BADARGS;
	}

	return hr;
}

CEHResult ResourceHandleManager::readCallback(ResourceHandle* webCoreJob, void* postData, UINT32 blockSize, UINT32 blockNum)
{
	CEHResult hr = CE_S_OK;
#if !CE_OS(XAI)
	if (webCoreJob && postData)
	{
		ResourceHandleInternal* d = webCoreJob->getInternal();
		if (d->m_cancelled)
			return CE_SILK_ERR_OPERATION_CANCELLED;

		if (!d->m_formDataStream.hasMoreElements())
			return CE_SILK_ERR_BADSTATE;

		size_t sent = d->m_formDataStream.read(postData, blockSize, blockNum);
		// send blocking message to curl's read callback.

		// Something went wrong so cancel the job.
		if (!sent)
			webCoreJob->cancel();
	}
#endif
	return hr;
}

CEHResult ResourceHandleManager::errorCallback(ResourceHandle* webCoreJob, INT32 lowLevelError, CEHResult resultCode, bool noContext, void* userArg)
{
	CEHResult hr = CE_S_OK;
	CEASSERT(webCoreJob);
	if (webCoreJob)
	{
		ResourceHandleInternal* d = NULL;
		ResourceHandleClient* client = NULL;
		d = webCoreJob->getInternal();
		if (d && d->m_cancelled == false) client = d->client();

		if (client)
		{
			CEHResult internalErr = resultCode;

			ResourceError resourceErr
				(webCoreJob->request().url().host(),
				 -1,
				 webCoreJob->request().url().string(),
				 WebCore::String(),
				 resultCode,
				 internalErr,
				 lowLevelError);
			if (true == webCoreJob->request().getIsMainRresource())
				resourceErr.setIsMainResource();
			if (userArg)
				resourceErr.setUserArg(userArg);

			resourceErr.setNoContext(noContext);
			client->didFail(webCoreJob, resourceErr);
		}
	}
	else
	{
		hr = CE_SILK_ERR_BADARGS;
	}
	return hr;
}

void ResourceHandleManager::add(ResourceHandle* job, WebCore::Frame* frame)
{
	job->ref();
	startJob(job, false, frame);
}

void ResourceHandleManager::cancel(ResourceHandle* job)
{
#ifdef RESOURCEHANDLEMANAGER_DEBUG
	CEComDebugPrintf("####### ResourceHandleManager::cancel() job = [%p]#######\n", job);
#endif

	ResourceHandleInternal* d = job->getInternal();
	if (d && d->m_cancelled == false)
	{
		d->m_cancelled = true;
		WebCore::ResourceLoader* loader = reinterpret_cast<WebCore::ResourceLoader*>(d->client());
		if (loader)
		{
			unsigned long idOnFrame = loader->identifier();
			unsigned long pageId = 0;
			if (loader->frameLoader() && loader->frameLoader()->frame() && loader->frameLoader()->frame()->page() && loader->frameLoader()->frame()->page())
			{	
				pageId = loader->frameLoader()->frame()->page()->group().identifier();
				pageId = pageId << PAGE_IDENTIFIER_SHIFT;
				idOnFrame |= pageId;
				m_cenetwork.cancel(idOnFrame, NULL);
			}
		}		
	}
}

void ResourceHandleManager::cancel(unsigned long identifier)
{
	m_cenetwork.cancel(identifier, NULL);
}

void ResourceHandleManager::dispatchSynchronousJob(ResourceHandle* job, WebCore::Frame* frame)
{
	KURL kurl = job->request().url();

	if (kurl.protocolIs("data")) {
		return;
	}

	ResourceHandleInternal* handle = job->getInternal();

	CEHResult hr = CE_S_OK;

	hr = startJob(job, true, frame);
}

void ResourceHandleManager::didFinishLoading(unsigned long identifier, const String* metaTagInfo)
{
	CERawDataBuffer metaData = {0};
	if (metaTagInfo)
		metaData._data = reinterpret_cast<UINT8*>(LPSTR_STRDUP(metaTagInfo->utf8().data()));
	m_cenetwork.didFinishLoading(identifier,
								 metaTagInfo? &metaData : NULL);
	LPSTR_FREE(metaData._data);
}

void ResourceHandleManager::deleteFrameLoderClientCert(void* cert)
{
	m_cenetwork.deleteCertForCredential(cert);
}

ICENetworkEventListener* ResourceHandleManager::getNetworkListenerFromJob(ResourceHandle* job, bool syncRequest, WebCore::Frame* frame)
{
	CEComICENetworkEventListenerRef networkListener = NULL;

	if (job)
	{
		WebCore::ResourceHandleInternal* d = job->getInternal();
		if (syncRequest)
		{
			WebCore::WebCoreSynchronousLoader* client = NULL;
			client = (WebCoreSynchronousLoader*)(job->getInternal()->client());
			if (client)
				networkListener = client->getNetworkListener();
		}
		else
		{
			CEHResult hr = CE_S_OK;

			if (d && frame && frame->view() && frame->view()->hostWindow() )
			{
				PlatformWidget widget = frame->view()->hostWindow()->platformWindow();
				if (widget)
				{
					hr = widget->getNetworkEventListener(&networkListener);
				}
			}
			if (CEFailed(hr))
				networkListener = NULL;
		}
	}
	return networkListener;
}

CEHResult ResourceHandleManager::_ICEUStringToCString(ICEUString* in, char** out)
{
	CEHResult err = CE_S_OK;
	if (in && out)
	{
		UINT32 len = 0;
		CEComICEUStringRef str = in;
		err = str.getNumOfChars32(&len);
		if (!err && len)
			err = str.getBytesWithAlloc(eICEI18nEncoding_us_ascii, CEComGetAllocatorRec(), reinterpret_cast<UCHAR8**>(out), &len);
	}
	else
	{
		err = CE_SILK_ERR_BADARGS;
	}

	return err;
}
void ResourceHandleManager::_freeCStringFromICEUString(char* cstr)
{
	if (cstr)
	{
		CEComGetAllocatorRec()->free(CEComGetAllocatorRec(), cstr);
	}
}

CEHResult ResourceHandleManager::replyBadCertCallback(unsigned long identifier, bool sslContinue)
{
	CEHResult hr = CE_S_OK;
	hr = m_cenetwork.replyBadCertMessage(identifier, sslContinue);
	return hr;
}

CEHResult ResourceHandleManager::replyAuthCallback(unsigned long identifier, ICEUString* site, ICEUString* realm, ICEUString* username, ICEUString* password, bool authContinue, bool updateSavedFlag)
{
	char* usernameCStr = NULL;
	char* passwordCStr = NULL;
	char* siteCStr = NULL;
	char* realmCStr = NULL;
	ResourceHandleManager::_ICEUStringToCString(username, &usernameCStr);
	ResourceHandleManager::_ICEUStringToCString(password, &passwordCStr);
	ResourceHandleManager::_ICEUStringToCString(site, &siteCStr);
	ResourceHandleManager::_ICEUStringToCString(realm, &realmCStr);
	CERawDataBuffer usernameData = {0};
	CERawDataBuffer passwordData = {0};
	CERawDataBuffer siteData = {0};
	CERawDataBuffer realmData = {0};
	usernameData._data = LPCSTR_TO_LPUINT8(usernameCStr);
	passwordData._data = LPCSTR_TO_LPUINT8(passwordCStr);
	siteData._data = LPCSTR_TO_LPUINT8(siteCStr);
	realmData._data = LPCSTR_TO_LPUINT8(realmCStr);

	m_cenetwork.replyAuthMessage(identifier, &siteData, &realmData, &usernameData, &passwordData, authContinue, updateSavedFlag);
	ResourceHandleManager::_freeCStringFromICEUString(usernameCStr);
	ResourceHandleManager::_freeCStringFromICEUString(passwordCStr);
	ResourceHandleManager::_freeCStringFromICEUString(siteCStr);
	ResourceHandleManager::_freeCStringFromICEUString(realmCStr);
	return CE_S_OK;
}

}

CEHResult CENetworkSupport_init(ICENetwork* iCENetwork)
{
	return WebCore::ResourceHandleManager::init(iCENetwork);
}

void CENetworkSupport_shutdown()
{
	WebCore::ResourceHandleManager::shutdown();
}

