/*
 * Copyright (C) 2004, 2006 Apple Computer, Inc.  All rights resreved.
 * Copyright (C) 2005, 2006 Michael Emmel mike.emmel@gmail.com
 * All rights reserved.
 * Copyright     2011 Sony Corporation
 * Copyright (C) 2011 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "ResourceHandle.h"

#include "DocLoader.h"
#include "NotImplemented.h"
#include "ResourceHandleInternal.h"
#include "ResourceHandleManagerCore.h"


namespace WebCore {

WebCoreSynchronousLoader::WebCoreSynchronousLoader(ICENetworkEventListener* networkListener) :
	m_networklistener(networkListener)
{
}

void WebCoreSynchronousLoader::didReceiveResponse(ResourceHandle* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderLen, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext)
{
	if (webCoreJob)
	{
		ResourceHandleManager::_handleHeaderCallbackInternal(webCoreJob, responseHeaderBuffer, responseHeaderLen, url, httpCode, expectedContentLength, m_response);
		m_response.setNoContext(noContext);
		m_response.setResponseFired(true);
	}
}

void WebCoreSynchronousLoader::didReceiveData(ResourceHandle* webCoreJob, void* data, UINT32 size)
{
	m_data.append(reinterpret_cast<char*>(data), size);
}

void WebCoreSynchronousLoader::didFinishLoading(ResourceHandle* webCoreJob)
{
	CEComDebugPrintf("WebCoreSynchronousLoader::didFinishLoading() [%hs]:[%d]\n", __FILE__, __LINE__);
}

void WebCoreSynchronousLoader::didFail(ResourceHandle* webCoreJob)
{
	CEComDebugPrintf("WebCoreSynchronousLoader::didFail() [%hs]:[%d]\n", __FILE__, __LINE__);
}


static HashSet<String>& allowsAnyHTTPSCertificateHosts()
{
    static HashSet<String> hosts;

    return hosts;
}

ResourceHandleInternal::~ResourceHandleInternal()
{
	if (m_client)
	{
		m_client = NULL;
	}

	if (m_url && m_request.getIsSync() == false)
	{
		free(m_url);
		m_url = NULL;
	}
}

ResourceHandle::~ResourceHandle()
{
    cancel();
}

static bool hasSupportedScheme(const KURL& url)
{
	const String& scheme = url.protocol().lower();
	return (scheme == "http") || (scheme == "https") || (scheme == "file") ||
		   (scheme == "data") || (scheme == "about") || (scheme == "javascript");
}

bool ResourceHandle::start(Frame* frame)
{
    ASSERT(frame);
    if (hasSupportedScheme(request().url()))
	    ResourceHandleManager::sharedInstance()->add(this, frame);
    else
	    scheduleFailure(InvalidURLFailure);
    return true;
}

void ResourceHandle::cancel()
{
    ResourceHandleManager::sharedInstance()->cancel(this);
}

PassRefPtr<SharedBuffer> ResourceHandle::bufferedData()
{
    return 0;
}

bool ResourceHandle::supportsBufferedData()
{
    return false;
}

void ResourceHandle::setDefersLoading(bool defers)
{
    if (d->m_defersLoading == defers)
        return;

    d->m_defersLoading = defers;
}

bool ResourceHandle::willLoadFromCache(ResourceRequest&)
{
    notImplemented();
    return false;
}

bool ResourceHandle::loadsBlocked()
{
    notImplemented();
    return false;
}

typedef HashMap<unsigned, WTF::Vector<char> > xhr_store;

static int _max_sync_xhr_cache = 40;
static HashMap<unsigned, WTF::Vector<char> > _sync_xhr_cache;

void ResourceHandle::loadResourceSynchronously(const ResourceRequest& request, StoredCredentials storedCredentials, ResourceError& error, ResourceResponse& response, Vector<char>& data, Frame* frame)
{
	KURL kurl = request.url();
	unsigned url_hash = KURLHash::hash(kurl);
	xhr_store::iterator itr = _sync_xhr_cache.find(url_hash);
	if ( _sync_xhr_cache.end() != itr)
	{
		response.setURL(kurl);
		response.setHTTPStatusCode(200);
		response.setNoContext(true);
		error.setNoContext(true);
		response.setResponseFired(true);
		data = itr->second;
	}
	else
	{
		CEComICENetworkEventListenerRef networkListener = NULL;
		CEHResult hr = CE_S_OK;
		if (frame && frame->view() && frame->view()->hostWindow() )
		{
			PlatformWidget widget = frame->view()->hostWindow()->platformWindow();
			if (widget)
			{
				hr = widget->getNetworkEventListener(&networkListener);
			}
			request.setFrame(reinterpret_cast<void*>(frame));
		}
		if (CESucceeded(hr) && networkListener)
		{
			WebCoreSynchronousLoader syncLoader(networkListener);

			RefPtr<ResourceHandle> newHandle(adoptRef(new ResourceHandle(request, &syncLoader, true, false, true)));
			if (newHandle->getInternal())
			{
				newHandle->getInternal()->m_request.setIsSync(true);

				ResourceHandleManager* manager = ResourceHandleManager::sharedInstance();

				if (manager)
				{
					manager->dispatchSynchronousJob(newHandle.get(), frame);

					error = syncLoader.resourceError();
					data = syncLoader.data();
					response = syncLoader.resourceResponse();

					if ( _sync_xhr_cache.size() >= _max_sync_xhr_cache )
					{
						xhr_store::iterator itr = _sync_xhr_cache.begin();
						_sync_xhr_cache.remove(itr);
					}
					if ( 200 == response.httpStatusCode() )
						_sync_xhr_cache.add( url_hash, data );
					newHandle->setClient(0);
				}
			}
			newHandle.release();
		}
	}
}

//stubs needed for windows version
void ResourceHandle::didReceiveAuthenticationChallenge(const AuthenticationChallenge&) 
{
    notImplemented();
}

void ResourceHandle::receivedCredential(const AuthenticationChallenge&, const Credential&) 
{
    notImplemented();
}

void ResourceHandle::receivedRequestToContinueWithoutCredential(const AuthenticationChallenge&) 
{
    notImplemented();
}

void ResourceHandle::receivedCancellation(const AuthenticationChallenge&)
{
    notImplemented();
}

} // namespace WebCore


