/*
 * Copyright (C) 2003, 2006 Apple Computer, Inc.  All rights reserved.
 *                     2006, 2008 Rob Buis <buis@kde.org>
 * Copyright     2009 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"
#include "Path.h"

#if PLATFORM(SILK)

#include "StrokeStyleApplier.h"
#include "TransformationMatrix.h"
#include "FloatRect.h"
#include "GraphicsContext.h"
#include "IntRect.h"
#include "PlatformString.h"
#include "SilkPath.h"

namespace WebCore {
Path::Path()
{
	m_path = new SilkPath();
}

Path::~Path()
{
	if(m_path)
		delete m_path;
	m_path = 0;
}

Path::Path(const Path& other)
{
	m_path = new SilkPath();
	if (m_path)
	{
		m_path->m_vgPath.appendPath(other.m_path->m_vgPath);
	}
}

Path& Path::operator=(const Path& other)
{
	if(&other == this )
	{
		return *this;
	}
	clear();
	if (m_path)
	{
		m_path->m_vgPath.appendPath(other.m_path->m_vgPath);
	}
	return *this;
}

void Path::clear()
{
	platformPath()->m_vgPath.newPath();
}

bool Path::isEmpty() const
{
	return (platformPath()->m_vgPath.getCurrentPointF(0, 0) != CE_S_OK);
}

#if defined(VGC_LATEST_COLORSPACE_17634)
bool Path::hasCurrentPoint() const
{
	return !isEmpty();
}
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)

void Path::translate(const FloatSize& p)
{
	platformPath()->m_vgPath.translateF(p.width(), p.height());
}

void Path::moveTo(const FloatPoint& p)
{
	platformPath()->m_vgPath.moveToF(p.x(), p.y());
}

void Path::addLineTo(const FloatPoint& p)
{
	platformPath()->m_vgPath.lineToF(p.x(), p.y());
}

void Path::addRect(const FloatRect& rect)
{
	platformPath()->m_vgPath.addRectangleF(rect.x(), rect.y(), rect.width(), rect.height());
}

/*
* inspired by libsvg-cairo and PathCairo
*/
void Path::addQuadCurveTo(const FloatPoint& controlPoint, const FloatPoint& point)
{
	CEComICEVGContextRef vgPath = platformPath()->m_vgPath;
	float x = 0;
	float y = 0;
	float x1 = controlPoint.x();
	float y1 = controlPoint.y();
	float x2 = point.x();
	float y2 = point.y();

	if (CEFailed(vgPath.getCurrentPointF(&x, &y)))
		x = y = 0;

	vgPath.curveToF(x  + 2.0 / 3.0 * (x1 - x),  y  + 2.0 / 3.0 * (y1 - y),
		x2 + 2.0 / 3.0 * (x1 - x2), y2 + 2.0 / 3.0 * (y1 - y2),
		x2, y2);
}

void Path::addBezierCurveTo(const FloatPoint& controlPoint1, const FloatPoint& controlPoint2, const FloatPoint& controlPoint3)
{
	platformPath()->m_vgPath.curveToF(controlPoint1.x(), controlPoint1.y(),
		controlPoint2.x(), controlPoint2.y(),
		controlPoint3.x(), controlPoint3.y());
}

void Path::addArc(const FloatPoint& p, float r, float sa, float ea, bool anticlockwise)
{
	// http://bugs.webkit.org/show_bug.cgi?id=16449
	// cairo_arc() functions hang or crash when passed inf as radius or start/end angle
	if (!isfinite(r) || !isfinite(sa) || !isfinite(ea))
		return;

	CEComICEVGContextRef vgPath = platformPath()->m_vgPath;
	if (anticlockwise)
		vgPath.addArcNegativeF(p.x(), p.y(), r, sa, ea);
	else
		vgPath.addArcF(p.x(), p.y(), r, sa, ea);
}

void Path::addArcTo(const FloatPoint& p1, const FloatPoint& p2, float radius)
{
	if (isEmpty())
		return;

	CEComICEVGContextRef vgPath = platformPath()->m_vgPath;

	float x0, y0;
	vgPath.getCurrentPointF(&x0, &y0);
	FloatPoint p0(x0, y0);
	if ((p1.x() == p0.x() && p1.y() == p0.y()) || (p1.x() == p2.x() && p1.y() == p2.y()) || radius == 0.f) {
		vgPath.lineToF(p1.x(), p1.y());
		return;
	}

	FloatPoint p1p0((p0.x() - p1.x()),(p0.y() - p1.y()));
	FloatPoint p1p2((p2.x() - p1.x()),(p2.y() - p1.y()));
	float p1p0_length = sqrtf(p1p0.x() * p1p0.x() + p1p0.y() * p1p0.y());
	float p1p2_length = sqrtf(p1p2.x() * p1p2.x() + p1p2.y() * p1p2.y());

	double cos_phi = (p1p0.x() * p1p2.x() + p1p0.y() * p1p2.y()) / (p1p0_length * p1p2_length);
	// all points on a line logic
	if (cos_phi == -1) {
		vgPath.lineToF(p1.x(), p1.y());
		return;
	}
	if (cos_phi == 1) {
		// add infinite far away point
		unsigned int max_length = 65535;
		double factor_max = max_length / p1p0_length;
		FloatPoint ep((p0.x() + factor_max * p1p0.x()), (p0.y() + factor_max * p1p0.y()));
		vgPath.lineToF(ep.x(), ep.y());
		return;
	}

	float tangent = radius / tan(acos(cos_phi) / 2);
	float factor_p1p0 = tangent / p1p0_length;
	FloatPoint t_p1p0((p1.x() + factor_p1p0 * p1p0.x()), (p1.y() + factor_p1p0 * p1p0.y()));

	FloatPoint orth_p1p0(p1p0.y(), -p1p0.x());
	float orth_p1p0_length = sqrt(orth_p1p0.x() * orth_p1p0.x() + orth_p1p0.y() * orth_p1p0.y());
	float factor_ra = radius / orth_p1p0_length;

	// angle between orth_p1p0 and p1p2 to get the right vector orthographic to p1p0
	double cos_alpha = (orth_p1p0.x() * p1p2.x() + orth_p1p0.y() * p1p2.y()) / (orth_p1p0_length * p1p2_length);
	if (cos_alpha < 0.f)
		orth_p1p0 = FloatPoint(-orth_p1p0.x(), -orth_p1p0.y());

	FloatPoint p((t_p1p0.x() + factor_ra * orth_p1p0.x()), (t_p1p0.y() + factor_ra * orth_p1p0.y()));

	// calculate angles for addArc
	orth_p1p0 = FloatPoint(-orth_p1p0.x(), -orth_p1p0.y());
	float sa = acos(orth_p1p0.x() / orth_p1p0_length);
	if (orth_p1p0.y() < 0.f)
		sa = 2 * piDouble - sa;

	// anticlockwise logic
	bool anticlockwise = false;

	float factor_p1p2 = tangent / p1p2_length;
	FloatPoint t_p1p2((p1.x() + factor_p1p2 * p1p2.x()), (p1.y() + factor_p1p2 * p1p2.y()));
	FloatPoint orth_p1p2((t_p1p2.x() - p.x()),(t_p1p2.y() - p.y()));
	float orth_p1p2_length = sqrtf(orth_p1p2.x() * orth_p1p2.x() + orth_p1p2.y() * orth_p1p2.y());
	float ea = acos(orth_p1p2.x() / orth_p1p2_length);
	if (orth_p1p2.y() < 0)
		ea = 2 * piDouble - ea;
	if ((sa > ea) && ((sa - ea) < piDouble))
		anticlockwise = true;
	if ((sa < ea) && ((ea - sa) > piDouble))
		anticlockwise = true;

	vgPath.lineToF(t_p1p0.x(), t_p1p0.y());

	addArc(p, radius, sa, ea, anticlockwise);
}

void Path::addEllipse(const FloatRect& rect)
{
	CEComICEVGContextRef vgPath = platformPath()->m_vgPath;
	vgPath.save();
	float yRadius = .5 * rect.height();
	float xRadius = .5 * rect.width();
	vgPath.translateF(rect.x() + xRadius, rect.y() + yRadius);
	vgPath.scaleF(xRadius, yRadius);
	vgPath.addArcF(0., 0., 1., 0., 2 * piDouble);
	vgPath.restore();
}

void Path::closeSubpath()
{
	platformPath()->m_vgPath.closePath();
}

FloatRect Path::boundingRect() const
{
	float x, y, w, h;
	platformPath()->m_vgPath.getPathExtentsF(&x, &y, &w, &h);
	return FloatRect(x, y, w, h);
}

FloatRect Path::strokeBoundingRect(StrokeStyleApplier* applier)
{
	float x = 0;
	float y = 0;
	float w = 0;
	float h = 0;
	CEComICEVGContextRef vgPath = platformPath()->m_vgPath;
	if (applier) {
		GraphicsContext gc(vgPath);
		applier->strokeStyle(&gc);
		platformPath()->m_vgPath.getStrokeExtentsF(&x, &y, &w, &h);
	}
	return FloatRect(x, y, w, h);
}

bool Path::contains(const FloatPoint& point, WindRule rule) const
{
	if (!boundingRect().contains(point))
		return false;
	CEComICEVGContextRef vgPath = platformPath()->m_vgPath;
	eCEVGFillRule cur = eCEVGFillRule_WINDING;
	vgPath.getFillRule(&cur);
	vgPath.setFillRule(rule == RULE_EVENODD ? eCEVGFillRule_EVENODD : eCEVGFillRule_WINDING);
	bool contains = false;
	vgPath.isInFillF(point.x(), point.y(), &contains);
	vgPath.setFillRule(cur);
	return contains;
}

bool Path::strokeContains(StrokeStyleApplier* applier, const FloatPoint& point) const
{
	ASSERT(applier);
	CEComICEVGContextRef vgPath = platformPath()->m_vgPath;
	GraphicsContext gc(vgPath);
	applier->strokeStyle(&gc);
	bool isInStroke = false;
	vgPath.isInStroke(point.x(), point.y(), &isInStroke);
	return isInStroke;
}

void Path::apply(void* info, PathApplierFunction function) const
{
	CEComICEVGContextRef vgPath = platformPath()->m_vgPath;
	CEComICEVGPathEnumeratorRef en;

	if (CESucceeded(vgPath.copyPathData(&en))) {
		PathElement pelement;
		FloatPoint points[3];
		pelement.points = points;

		INT_PTR nPoints = 3;
		CEPointF pointFs[3];
		eCEVGPathDataType type = eCEVGPathDataType_MOVE_TO;
		while (CESucceeded(en.nextF(&type, pointFs, &nPoints))) {
			switch (type) {
					case eCEVGPathDataType_MOVE_TO:
						pelement.type = PathElementMoveToPoint;
						pelement.points[0] = FloatPoint(pointFs[0]._x, pointFs[0]._y);
						function(info, &pelement);
						break;
					case eCEVGPathDataType_LINE_TO:
						pelement.type = PathElementAddLineToPoint;
						pelement.points[0] = FloatPoint(pointFs[0]._x, pointFs[0]._y);
						function(info, &pelement);
						break;
					case eCEVGPathDataType_CURVE_TO:
						pelement.type = PathElementAddCurveToPoint;
						pelement.points[0] = FloatPoint(pointFs[0]._x, pointFs[0]._y);
						pelement.points[1] = FloatPoint(pointFs[1]._x, pointFs[1]._y);
						pelement.points[2] = FloatPoint(pointFs[2]._x, pointFs[2]._y);
						function(info, &pelement);
						break;
					case eCEVGPathDataType_CLOSE_PATH:
						pelement.type = PathElementCloseSubpath;
						function(info, &pelement);
						break;
			}
			nPoints = 3;
		}
	}
}

void Path::transform(const TransformationMatrix& trans)
{
	CEComICEVGContextRef vgPath = platformPath()->m_vgPath;
	CEDXMatrix2DAffineD dxmatrix = GraphicsContextPlatformPrivate::initCEDXMatrix2DAffineD(trans);
	CEComICEVGRef vg = GraphicsContextPlatformPrivate::Env::getVG();
	vg.invertMatrix2DAffine(&dxmatrix);
	vgPath.transformMatrix2DAffine(&dxmatrix);
}

String Path::debugString() const
{
	if (isEmpty())
		return String();

	String pathString;
	CEComICEVGContextRef vgPath = platformPath()->m_vgPath;
	CEComICEVGPathEnumeratorRef en;

	if (CESucceeded(vgPath.copyPathData(&en))) {
		en.reset();
		CEPointF pointFs[3];
		eCEVGPathDataType type = eCEVGPathDataType_MOVE_TO;
		INT_PTR nPoints = 3;
		while (CESucceeded(en.nextF(&type, pointFs, &nPoints))) {
			switch (type) {
					case eCEVGPathDataType_MOVE_TO:
						pathString += String::format("M%.2f,%.2f ",
							pointFs[0]._x, pointFs[0]._y);
						break;
					case eCEVGPathDataType_LINE_TO:
						pathString += String::format("L%.2f,%.2f ",
							pointFs[0]._x, pointFs[0]._y);
						break;
					case eCEVGPathDataType_CURVE_TO:

						pathString += String::format("C%.2f,%.2f,%.2f,%.2f,%.2f,%.2f ",
							pointFs[0]._x, pointFs[0]._y,
							pointFs[1]._x, pointFs[1]._y,
							pointFs[2]._x, pointFs[2]._y);
						break;
					case eCEVGPathDataType_CLOSE_PATH:
						pathString += "Z ";
						break;
			}
			nPoints = 3;
		}
	}
	return pathString.simplifyWhiteSpace();
}
}; // namespace namespace WebCore

#endif // PLATFORM(SILK)
