/*
 * Copyright (C) 2006 Apple Computer, Inc.  All rights reserved.
 * Copyright     2008 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "BitmapImage.h"

#if PLATFORM(SILK)
#include "ICEVG.h"
#include "TransformationMatrix.h"
#include "FloatConversion.h"
#include "FloatRect.h"
#include "GraphicsContext.h"
#include "GraphicsContextPlatformPrivateSilk.h"
#include "ImageObserver.h"
#include "CEImageBufferGraphicsContextImpl.h"

#include "PlatformString.h"
#include "ICEHtmlWebKit.h"
#include <wtf/OwnPtr.h>

#define NOT_DRAW_ERROR_IMAGE_AFTER_DRAWCOMPRESSEIMAGE
//#define CALC_TIME
#if defined(CALC_TIME)
#include "CESysGetTimeOfDay.h"
#endif
namespace WebCore {

bool FrameData::clear(bool clearMetadata)
{
    if (clearMetadata)
        m_haveMetadata = false;

    if (m_frame) {
        m_frame = 0;
        return true;
    }
    return false;
}

static void drawDownloadingImage(ICEVGContext* iContext, const CERect* pDestRect )
{
	CEDim imageDim = {0,0};
	CEHResult hr = CE_SILK_ERR_BADARGS;
	CEComICEVGSurfaceRef missingImageRef = 0;
	// initialize
	CEComICEHtmlWebKitThemeRef iwebkitThemeRef=0;
	CEComGetThreadContext(CEComIID_ICEHtmlWebKitTheme, reinterpret_cast<void**>(&iwebkitThemeRef));
	if(iwebkitThemeRef)
	{
		hr = iwebkitThemeRef.getMissingImage(&missingImageRef);
		if(CESucceeded(hr))
		{
			hr = missingImageRef.getDimension(&imageDim);
		}
	}

	if(missingImageRef && imageDim._width>0 && imageDim._height>0 && iContext && pDestRect)
	{
		/////////////////////////////////////
		//Calculate center rect
		CERect targetRect;
		targetRect._width = imageDim._width;
		targetRect._height= imageDim._height;
		
		//Decide x position
		if(pDestRect->_width - imageDim._width>=2)
			targetRect._x = pDestRect->_x+( (pDestRect->_width - imageDim._width)>>1 );
		else
			targetRect._x = pDestRect->_x;

		//Decide y position
		if(pDestRect->_height - imageDim._height>=2)
			targetRect._y = pDestRect->_y+( (pDestRect->_height - imageDim._height)>>1 );
		else
			targetRect._y = pDestRect->_y;
		/////////////////////////////////////
		
		CEComICEVGContextRef contextRef = iContext;
		contextRef.save();
		CEComICEVGPatternRef pattern;
		CEComICEVGFactoryRef vgFactory = GraphicsContextPlatformPrivate::Env::getVGFactory();
		vgFactory.createPattern(missingImageRef, &pattern);
		contextRef.translateF(targetRect._x, targetRect._y);
		contextRef.setSourcePattern(pattern);
		contextRef.addRectangleF(0, 0, targetRect._width, targetRect._height);
		contextRef.paint();
		contextRef.setClip();
		contextRef.restore();
	}

}
// ================================================
// Image Class
// ================================================

BitmapImage::BitmapImage(NativeImagePtr nativeImage, ImageObserver* observer)
    : Image(observer)
    , m_currentFrame(0)
    , m_frames(0)
    , m_frameTimer(0)
    , m_repetitionCount(cAnimationNone)
    , m_repetitionCountStatus(Unknown)
    , m_repetitionsComplete(0)
    , m_desiredFrameStartTime(0)
    , m_isSolidColor(false)
    , m_checkedForSolidColor(false)
    , m_animationFinished(false)
    , m_allDataReceived(false)
    , m_haveSize(false)
    , m_sizeAvailable(false)
    , m_hasUniformFrameSize(true)
    , m_decodedSize(0)
    , m_haveFrameCount(false)
    , m_frameCount(0)
{
	initPlatformData();

	// duty hack. if iImage is NULL then jps image.
	if (nativeImage == NULL)
	{
		m_source.initAsStereoImage();
	}
	else
	{
		m_animationFinished = true;
		m_allDataReceived = true;
		m_haveSize = true;
		m_sizeAvailable = true;
		m_haveFrameCount = true;
		m_frameCount = 1;

		CEDim dim;
		nativeImage.getDimension(&dim);

		unsigned int height = static_cast<unsigned int>(dim._height);
		unsigned int width  = static_cast<unsigned int>(dim._width);

		// To do. BytePerPixel always four??
		m_decodedSize = height*width*4;
		m_size = IntSize(width, height);
		
		m_frames.grow(1);
		m_frames[0].m_frame = nativeImage;
		m_frames[0].m_hasAlpha = true;
		m_frames[0].m_haveMetadata = true;
		checkForSolidColor();

		// create Listener
		// To do:
		// The specification of setData might change.
		// If it is changed, you should check ImageSourceSilk::setData.
		// I suppose that setData specification, if the first variable is NULL and second one is true, 
		// only create the Listener for ImageDecorder thread and setting the isDraw = true.  
		m_source.setData(NULL,true);
	}
}

// Drawing Routines

void BitmapImage::checkForSolidColor()
{
	m_isSolidColor = false;
    m_checkedForSolidColor = true;
}

#define _MY_MULTPLY_255(x, a) ((x*a)/255)
#if defined(VGC_LATEST_COLORSPACE_17634)
void BitmapImage::draw(GraphicsContext* context, const FloatRect& dst, const FloatRect& src, ColorSpace styleColorSpace, CompositeOperator op)
#else //#if defined(VGC_LATEST_COLORSPACE_17634)
void BitmapImage::draw(GraphicsContext* context, const FloatRect& dst, const FloatRect& src, CompositeOperator op)
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)
{
	// Is execute the cacheFrame()?
	bool isExecuteCacheFrame = (m_currentFrame >= m_frames.size() || !m_frames[m_currentFrame].m_frame) ? true : false;
	bool isFirstDecoded = m_decodedSize ? false : true;
#if defined(CALC_TIME)
		INT64 tmsA = CESysGetMonotonicTick();
#endif
	// Add WebCore's cacheSize.
	CEComICEVGSurfaceRef iImage = frameAtIndex(m_currentFrame);

	if(!iImage.object())
		return;

	// If (executeCacheFrame && !firstFrame) Remove WebCore's cacheSize.
	if(isExecuteCacheFrame && !isFirstDecoded)
	{
		this->destroyMetadataAndNotify(1);
	}

	FloatRect srcRect(src);
	FloatRect dstRect(dst);

	if (dstRect.width() == 0.0f || dstRect.height() == 0.0f ||
		srcRect.width() == 0.0f || srcRect.height() == 0.0f)
		return;

	startAnimation(false);

	INT32 orgWidth = srcRect.width();//init value.
	INT32 orgHeight = srcRect.height();
	INT32 dstWidth = dstRect.width();
	INT32 dstHeight = dstRect.height();

	//get image-size.
	CEDim cdm;
	CEHResult err = iImage.getDimension(&cdm);
	if(!err)
	{
		orgWidth = cdm._width;
		orgHeight = cdm._height;
		dstWidth = orgWidth;
		dstHeight = orgHeight;
	}

	CEComICEVGFactoryRef vgFactory = GraphicsContextPlatformPrivate::Env::getVGFactory();

	if (mayFillWithSolidColor()) {
#if defined(VGC_LATEST_COLORSPACE_17634)
		fillWithSolidColor(context, dstRect, solidColor(), styleColorSpace, op);
#else //#if defined(VGC_LATEST_COLORSPACE_17634)
		fillWithSolidColor(context, dstRect, solidColor(), op);
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)
		return;
	}

	IntSize selfSize = size();

	CEComICEVGContextRef vgc = context->platformContext();
	context->save();

	// Set the compositing operation.
	if (op == CompositeSourceOver && !frameHasAlphaAtIndex(m_currentFrame))
		context->setCompositeOperation(CompositeCopy);
	else
		context->setCompositeOperation(op);

	// set frame index.
	CEComICEVGImageSurfaceSourceRef iImageSurfaceRef;
	err = iImageSurfaceRef.initByQueryInterface(iImage);
	if(!err)
	{
		iImageSurfaceRef.setFrameIndex(m_currentFrame);
	}

#if defined(CALC_TIME)
	INT64 tsmC = ::CESysGetMonotonicTick();
#endif
	// If we're drawing a sub portion of the image or scaling then create
	// a pattern transformation on the image and draw the transformed pattern.
	// Test using example site at http://www.meyerweb.com/eric/css/edge/complexspiral/demo.html
	CEComICEVGPatternRef pattern;
	err = vgFactory.createPattern(iImage, &pattern);
	if(err)
	{
		CERect destRectBeforeDownloadImage = {dstRect.x(), dstRect.y(), dstRect.width(), dstRect.height()};
		drawDownloadingImage(context->platformContext(), &destRectBeforeDownloadImage);
		context->restore();
		return;
	}
#if defined(CALC_TIME)
	printf("BitmapImage ::drawImage src:x=%f y=%f width=%f height=%f ImageSize: width=%d, height=%d decorde time:=%dus\n", srcRect.x(), srcRect.y(), srcRect.width(), srcRect.height(), cdm._width, cdm._height, (INT32)((::CESysGetMonotonicTick() - tsmC)));
#endif

	// For Bug 18887
	// Extend_Pad is very slow.
#if 0
	pattern.setExtend(eCEVGExtend_PAD);
#else
	// To avoid the unwanted gradient effect (#14017) we use
    // CAIRO_FILTER_NEAREST now, but the real fix will be to have
    // CAIRO_EXTEND_PAD implemented for surfaces in Cairo allowing us to still
    // use bilinear filtering
	pattern.setExtend(eCEVGExtend_NONE);
	pattern.setFilter(eCEVGFilter_GOOD);
#endif

	// scaling.
	float scaleX = srcRect.width() / dstRect.width();
	float scaleY = srcRect.height() / dstRect.height();
	CEDXMatrix2DAffineD matrix = { scaleX, 0, 0, scaleY, srcRect.x(), srcRect.y() };
	pattern.setMatrix2DAffine(&matrix);

	// Draw the shadow
#if ENABLE(FILTERS)
	IntSize shadowSize;
	int shadowBlur;
	Color shadowColor;
	if (context->getShadow(shadowSize, shadowBlur, shadowColor)) {
		IntSize shadowBufferSize;
		FloatRect shadowRect;
		float kernelSize (0.0);
		GraphicsContext::calculateShadowBufferDimensions(shadowBufferSize, shadowRect, kernelSize, dstRect, shadowSize, shadowBlur);
		shadowColor = colorWithOverrideAlpha(shadowColor.rgb(), (shadowColor.alpha() *  context->getAlpha()) / 255.f);

		//draw shadow into a new ImageBuffer
#if defined(VGC_FILTERS_LATEST_17935)
		OwnPtr<ImageBuffer> shadowBuffer = ImageBuffer::create(shadowBufferSize);
#else //#if defined(VGC_FILTERS_LATEST_17935)
		OwnPtr<ImageBuffer> shadowBuffer = ImageBuffer::create(shadowBufferSize, false);
#endif //#if defined(VGC_FILTERS_LATEST_17935)
		CEComICEVGContextRef shadowContext = shadowBuffer->context()->platformContext();
		shadowContext.setSourcePattern(pattern);
		shadowContext.translateF(-dstRect.x(), -dstRect.y());
		shadowContext.addRectangleF(0, 0, dstRect.width(), dstRect.height());
		shadowContext.fill();

		context->createPlatformShadow(shadowBuffer.release(), shadowColor, shadowRect, kernelSize);
	}
#endif

	// Draw the image.
	vgc.translateF(dstRect.x(), dstRect.y());
	vgc.setSourcePattern(pattern);
	vgc.addRectangleF(0, 0, dstRect.width(), dstRect.height());
	vgc.setClip();
	vgc.paintWidthAlphaF(context->getAlpha());
	context->restore();

	if (imageObserver())
		imageObserver()->didDraw(this);

#if defined(CALC_TIME)
	INT64 tmsB = CESysGetMonotonicTick();
	INT32 time = (INT32)((tmsB-tmsA));
	static INT32 imageNum = 0;
	static INT32 allTime = 0;

	printf("Time between BitmapImage::draw count:%d, time=%d, all time:%d \n", imageNum, time, allTime);
	imageNum++;
	allTime += time;
#endif

}

#if defined(VGC_LATEST_COLORSPACE_17634)
void Image::drawPattern(GraphicsContext* context, const FloatRect& tileRect, const TransformationMatrix& patternTransform,
						const FloatPoint& phase, ColorSpace, CompositeOperator op, const FloatRect& destRect)
#else //#if defined(VGC_LATEST_COLORSPACE_17634)
void Image::drawPattern(GraphicsContext* context, const FloatRect& tileRect, const TransformationMatrix& patternTransform,
                        const FloatPoint& phase, CompositeOperator op, const FloatRect& destRect)
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)
{

#if defined(CALC_TIME)
		INT64 tmsA = CESysGetMonotonicTick();
#endif

	CEComICEVGSurfaceRef iImage = nativeImageForCurrentFrame();
	if (!iImage.object()) // If it's too early we won't have an image yet.
		return;

	context->save();
	CEComICEVGContextRef vgc = context->platformContext();
	IntRect imageSize = enclosingIntRect(tileRect);

	INT32 orgWidth = 0;
	INT32 orgHeight = 0;
	INT32 dstWidth = 0;
	INT32 dstHeight = 0;
	CEDim cdm;
	CEHResult err = iImage.getDimension(&cdm);
	if(!err)
	{
		orgWidth = cdm._width;
		orgHeight = cdm._height;
		dstWidth = orgWidth;
		dstHeight = orgHeight;
	}

#if defined(VGC_LATEST_COLORSPACE_17634)
	OwnPtr<ImageBuffer> imageSurface = ImageBuffer::create(imageSize.size());

	if (!imageSurface)
		return;

	if (tileRect.size() != size()) {
		CEComICEVGContextRef clippedImageContext = imageSurface->context()->platformContext();
		clippedImageContext.setSourceSurfaceF(iImage, -tileRect.x(), -tileRect.y());
		clippedImageContext.paint();
		iImage = imageSurface->image()->nativeImageForCurrentFrame();
	}
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)

	CEComICEVGFactoryRef vgFactory = GraphicsContextPlatformPrivate::Env::getVGFactory();
	CEComICEVGPatternRef pattern;
	err =vgFactory.createPattern(iImage, &pattern);
	if(err)
	{
		CERect destRectBeforeDownloadImage = {destRect.x(), destRect.y(), destRect.width(), destRect.height()};
		drawDownloadingImage(context->platformContext(), &destRectBeforeDownloadImage);
		context->restore();
		return;
	}

	pattern.setExtend(eCEVGExtend_REPEAT);
	CEDXMatrix2DAffineD pattern_matrix = GraphicsContextPlatformPrivate::initCEDXMatrix2DAffineD(patternTransform);
	CEDXMatrix2DAffineD phase_matrix = {1, 0, 0, 1, phase.x(), phase.y()};
	CEDXMatrix2DAffineD combined;

	vgFactory.multiplyMatrix2DAffine(&combined, &pattern_matrix, &phase_matrix);
	vgFactory.invertMatrix2DAffine(&combined);
	pattern.setMatrix2DAffine(&combined);

	context->setCompositeOperation(op);
	vgc.setSourcePattern(pattern);
	vgc.addRectangleF(destRect.x(), destRect.y(), destRect.width(), destRect.height());
	vgc.fill();

	context->restore();

	if (imageObserver())
		imageObserver()->didDraw(this);
#if defined(CALC_TIME)
	INT64 tmsB = CESysGetMonotonicTick();
	INT32 time = (INT32)((tmsB-tmsA));
	static INT32 imageNum = 0;
	static INT32 allTime = 0;
	printf("Time between BitmapImage::drawPattern count:%d, time=%d, all time:%d \n", imageNum, time, allTime);
	imageNum++;
	allTime += time;
#endif

}

void BitmapImage::drawPattern(GraphicsContext* context, const FloatRect& tileRect, const TransformationMatrix& patternTransform,
                        const FloatPoint& phase, CompositeOperator op, const FloatRect& destRect)
{
	Image::drawPattern(context, tileRect, patternTransform, phase, op, destRect);
}

PassRefPtr<Image> Image::loadPlatformResource(const char *name)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	
	CEComICEHtmlWebKitThemeRef iwebkitThemeRef=0;
	hr = CEComGetThreadContext(CEComIID_ICEHtmlWebKitTheme, reinterpret_cast<void**>(&iwebkitThemeRef));

	if (!hr)
	{
		CEComICEVGSurfaceRef vgSurface = 0;

		if (!hr)
		{
			WebCore::String resourceName;
			resourceName.append(name);

			if(resourceName=="missingImage")
			{
				//Get missing image
				hr = iwebkitThemeRef.getMissingImage(&vgSurface);
			}
			else if(resourceName=="nullPlugin")
			{
				//Get null plugin image
				hr = iwebkitThemeRef.getNoPluginImage(&vgSurface);
			}
			else
			{
				// To do:
				// You should add other resources. 
				CEASSERT(0);
			}

			if (!hr)
			{
				RefPtr<Image> image = BitmapImage::create(vgSurface);
				return image.release();
			}	
		}
		else
		{
			CEASSERT(0);
		}
	}
	
	return Image::nullImage();
}

}

#endif // PLATFORM(SILK)
