///////////////////////////////////////////////////////////////////////////////
// Copyright 2010 Sony Corporation
///////////////////////////////////////////////////////////////////////////////
#include "config.h"
#if USE(ACCELERATED_COMPOSITING)

#include "GraphicsLayerSilk.h"
#include "RenderLayerBacking.h"
#include "CEGLayerCallback.h"
#include "ICEVG.h"

namespace WebCore {

CEComICEMRLFactoryRef GraphicsLayerSilk::Env::_mrlFactory;

static eCEMRLCompositingCoordinatesOrientation _toECompositingCoordinatesOrientation
	(
	GraphicsLayer::CompositingCoordinatesOrientation orientation
	)
{
	eCEMRLCompositingCoordinatesOrientation ret = eCEMRLCompositingCoordinatesOrientation_TopDown;

	switch (orientation)
	{
	case GraphicsLayer::CompositingCoordinatesTopDown:
		ret = eCEMRLCompositingCoordinatesOrientation_TopDown;
		break;
	case GraphicsLayer::CompositingCoordinatesBottomUp:
		ret = eCEMRLCompositingCoordinatesOrientation_BottomUp;
		break;
	default:
		ASSERT(0 && "fix this error");
		break;
	}
	return ret;
}


////////////////////////////////////////////////
// GraphicsLayerSilk impls.
////////////////////////////////////////////////

GraphicsLayer* GraphicsLayer::createGraphicsLayer(GraphicsLayerClient* client)
{
	return new GraphicsLayerSilk(client);
}

GraphicsLayer::CompositingCoordinatesOrientation GraphicsLayer::compositingCoordinatesOrientation()
{
	return CompositingCoordinatesBottomUp;
}

#ifndef NDEBUG
bool GraphicsLayer::showDebugBorders()
{
	return false;
}
#endif //#ifndef NDEBUG


////////////////////////////////////////////////
// GraphicsLayerSilk virtual impls.
////////////////////////////////////////////////

GraphicsLayerSilk::GraphicsLayerSilk(GraphicsLayerClient* client)
: GraphicsLayer(client)
, _cemrlLayer(0)
, _needsSyncChildren(false)
{
	CEComICEMRLFactoryRef mrl = Env::getMRLFactory();
	ASSERT(mrl);

	mrl.createLayer(client ? eCEMRLLayerType_Layer : eCEMRLLayerType_RootLayer, &_cemrlLayer);
	if (_cemrlLayer)
	{
		_cemrlLayer.setUserData(this);

		//if (client && !static_cast<RenderLayerBacking*>(client)->paintingGoesToWindow())
		{
			CEGLayerCallback* pCb = new CEGLayerCallback();
			if (pCb)
			{
				CEComICEMRLLayerCallbackRef cbRef(CEGLayerCallback::toICEMRLLayerCallback(pCb));
				pCb->object()->init(this);
				_cemrlLayer.setCallback(cbRef);
			}
		}
		_cemrlLayer.setDrawsContent(false);
	}
}

GraphicsLayerSilk::~GraphicsLayerSilk()
{
	if (_cemrlLayer)
	{
		CEComICEMRLLayerRef parent;
		_cemrlLayer.parent(&parent);
		CEU_UNUSED(parent);
		ASSERT(!parent);

		// clear 
		_cemrlLayer.setCallback(0);
	}
	_cemrlLayer = 0;
}

void GraphicsLayerSilk::setName(const String& name)
{
	GraphicsLayer::setName(name);
	CEComICEUStringRef istr;
	name.createICEUString(&istr);
	_cemrlLayer.setName(istr);
}

NativeLayer GraphicsLayerSilk::nativeLayer() const
{
	return _cemrlLayer;
}

#define NOTIFY_SYNC_REQUIRED  if (client()) {client()->notifySyncRequired(this);}

///////////////////////////////////////////////////////////////
// layer tree structure functions
#define NOTIFY_SYNC_CHILDREN_REQUIRED  _needsSyncChildren = true;

void GraphicsLayerSilk::_syncChildren()
{
	if (_needsSyncChildren)
	{
		if (_cemrlLayer)
		{
			_cemrlLayer.removeAllChildren();
			const Vector<GraphicsLayer*> newChildren = children();
			size_t sz = newChildren.size();

			size_t i = 0;
			for (i = 0; i < sz; i++)
			{
				if (newChildren[i])
				{
					GraphicsLayerSilk* gLayerSilk = GLayerSILK(newChildren[i]);
					_cemrlLayer.addChild(gLayerSilk->_cemrlLayer);
				}
			}
		}
		_needsSyncChildren = false;
	}
}


void GraphicsLayerSilk::addChild(GraphicsLayer* layer)
{
	GraphicsLayer::addChild(layer);
	NOTIFY_SYNC_CHILDREN_REQUIRED;
}

void GraphicsLayerSilk::addChildAtIndex(GraphicsLayer* layer, int index)
{
	GraphicsLayer::addChildAtIndex(layer, index);
	NOTIFY_SYNC_CHILDREN_REQUIRED;
}

void GraphicsLayerSilk::addChildAbove(GraphicsLayer* layer, GraphicsLayer* sibling)
{
	GraphicsLayer::addChildAbove(layer, sibling);
	NOTIFY_SYNC_CHILDREN_REQUIRED;
}

void GraphicsLayerSilk::addChildBelow(GraphicsLayer* layer, GraphicsLayer* sibling)
{
	GraphicsLayer::addChildBelow(layer, sibling);
	NOTIFY_SYNC_CHILDREN_REQUIRED;
}

bool GraphicsLayerSilk::replaceChild(GraphicsLayer* oldChild, GraphicsLayer* newChild)
{
	bool ret = GraphicsLayer::replaceChild(oldChild, newChild);
	if (ret)
	{
		NOTIFY_SYNC_CHILDREN_REQUIRED;
	}
	return ret;
}

void GraphicsLayerSilk::removeFromParent()
{
	GraphicsLayer::removeFromParent();
	_cemrlLayer.removeFromParent();
}

// layer tree structure functions
///////////////////////////////////////////////////////////////


void GraphicsLayerSilk::setPosition(const FloatPoint& p)
{
	_cemrlLayer.setPosition(p.x(), p.y());
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setPosition(p);
}

void GraphicsLayerSilk::setAnchorPoint(const FloatPoint3D& p)
{
	_cemrlLayer.setAnchorPoint(p.x(), p.y(), p.z());
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setAnchorPoint(p);
}

void GraphicsLayerSilk::setSize(const FloatSize& size)
{
	_cemrlLayer.setSize(size.width(), size.height());
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setSize(size);
}

typedef float CEMRLGLMatrixF[16];
void _toCEMRLGLMatrixF(const TransformationMatrix& m, float* v)
{
	v[0] = m.m11();
	v[1] = m.m12();
	v[2] = m.m13();
	v[3] = m.m14();

	v[4] = m.m21();
	v[5] = m.m22();
	v[6] = m.m23();
	v[7] = m.m24();

	v[8] = m.m31();
	v[9] = m.m32();
	v[10] = m.m33();
	v[11] = m.m34();

	v[12] = m.m41();
	v[13] = m.m42();
	v[14] = m.m43();
	v[15] = m.m44();
}


void GraphicsLayerSilk::setTransform(const TransformationMatrix& t)
{
#if 0
	{
		CEComDebugPrintf("%p: tr [%g, %g, %g, %g], [%g, %g, %g, %g], [%g, %g, %g, %g], [%g, %g, %g, %g]\n",
			_cemrlLayer,
			t.m11(), t.m12(), t.m13(), t.m14(),
			t.m21(), t.m22(), t.m23(), t.m24(),
			t.m31(), t.m32(), t.m33(), t.m34(),
			t.m41(), t.m42(), t.m43(), t.m44());
	}
#endif //#if 0

	CEMRLGLMatrixF mf;
	_toCEMRLGLMatrixF(t, mf);
	_cemrlLayer.setTransform(mf);
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setTransform(t);
}

void GraphicsLayerSilk::setChildrenTransform(const TransformationMatrix& t)
{
	CEMRLGLMatrixF mf;
	_toCEMRLGLMatrixF(t, mf);
	_cemrlLayer.setChildrenTransform(mf);
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setChildrenTransform(t);
}

void GraphicsLayerSilk::setPreserves3D(bool b)
{
	_cemrlLayer.setPreserves3D(b);
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setPreserves3D(b);
}

void GraphicsLayerSilk::setMasksToBounds(bool b)
{
	_cemrlLayer.setMasksToBounds(b);
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setMasksToBounds(b);
}

void GraphicsLayerSilk::setDrawsContent(bool b)
{
	_cemrlLayer.setDrawsContent(b);
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setDrawsContent(b);
}

void GraphicsLayerSilk::setBackgroundColor(const Color& c)
{
	float r, g, b, a;
	c.getRGBA(r, g, b, a);
	_cemrlLayer.setBackgroundColor(r, g, b, a);
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setBackgroundColor(c);
}

void GraphicsLayerSilk::clearBackgroundColor()
{
	_cemrlLayer.clearBackgroundColor();
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::clearBackgroundColor();
}

void GraphicsLayerSilk::setContentsOpaque(bool b)
{
	_cemrlLayer.setContentsOpaque(b);
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setContentsOpaque(b);
}

void GraphicsLayerSilk::setBackfaceVisibility(bool b)
{
	_cemrlLayer.setBackfaceVisibility(b);
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setBackfaceVisibility(b);	
}

void GraphicsLayerSilk::setOpacity(float o)
{
	_cemrlLayer.setOpacity(o);
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setOpacity(o);
}

void GraphicsLayerSilk::suspendAnimations()
{
	GraphicsLayer::suspendAnimations();
	//_ceLayer->suspendAnimations(..);
}

void GraphicsLayerSilk::resumeAnimations()
{
	GraphicsLayer::resumeAnimations();
	//_ceLayer->resumeAnimations();
}

void GraphicsLayerSilk::setContentsToImage(Image* image)
{
	_cemrlLayer.setContentsToImage(image ? image->nativeImageForCurrentFrame() : 0);
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setContentsToImage(image);
}

void GraphicsLayerSilk::setContentsToVideo(PlatformLayer* videoLayer)
{
	_cemrlLayer.setContentsToMedia(reinterpret_cast<ICEUnknown*>(videoLayer));
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setContentsToVideo(videoLayer);
}

void GraphicsLayerSilk::setContentsBackgroundColor(const Color& c)
{
	float r, g, b, a;
	c.getRGBA(r, g, b, a);
	_cemrlLayer.setContentsBackgroundColor(r, g, b, a);
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setContentsBackgroundColor(c);
}

PlatformLayer* GraphicsLayerSilk::platformLayer() const
{
	return reinterpret_cast<PlatformLayer*>(_cemrlLayer.object());
}

#ifndef NDEBUG
void GraphicsLayerSilk::setDebugBackgroundColor(const Color& c)
{
	GraphicsLayer::setDebugBackgroundColor(c);
	// FIXME: what is this?
}

void GraphicsLayerSilk::setDebugBorder(const Color& c, float borderWidth)
{
	GraphicsLayer::setDebugBorder(c, borderWidth);
	// FIXME: what is this?
}
float GraphicsLayerSilk::zPosition() const
{
	// FIXME: actual platform z-position is required?
	return GraphicsLayer::zPosition();
}

void GraphicsLayerSilk::setZPosition(float z)
{
	ASSERT("fix me");
	//_cemrlLayer.setZPosition(z);
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setZPosition(z);	
}
#endif //#ifndef NDEBUG

// Set the geometry orientation (top-down, or bottom-up) for this layer, which also controls sublayer geometry.
void GraphicsLayerSilk::setGeometryOrientation(CompositingCoordinatesOrientation orientation)
{
	_cemrlLayer.setGeometryOrientation(_toECompositingCoordinatesOrientation(orientation));
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setGeometryOrientation(orientation);
}

// Flippedness of the contents of this layer. Does not affect sublayer geometry.
void GraphicsLayerSilk::setContentsOrientation(CompositingCoordinatesOrientation orientation)
{
	_cemrlLayer.setContentsOrientation(_toECompositingCoordinatesOrientation(orientation));
	NOTIFY_SYNC_REQUIRED;
	GraphicsLayer::setContentsOrientation(orientation);
}

// flush pending changes of this layer and all descendants.
void GraphicsLayerSilk::syncCompositingState()
{
	_cemrlLayer.syncCompositingState();
	GraphicsLayer::syncCompositingState();
	//CEComDebugPrintf("%p: GraphicsLayerSilk::syncCompositingState()\n", this);
}

void GraphicsLayerSilk::setOpacityInternal(float accumulatedOpacity)
{
	//[(preserves3D() ? m_layer.get() : primaryLayer()) setOpacity:accumulatedOpacity];
	ASSERT(0 && "fix this error");
}


////////////////////////////////////////////////
// GraphicsLayerSilk pure virtual impls.
////////////////////////////////////////////////
void GraphicsLayerSilk::setNeedsDisplay()
{
	_cemrlLayer.setNeedsDisplay();
	NOTIFY_SYNC_REQUIRED;
	//CEComDebugPrintf("%p: GraphicsLayerSilk::setNeedsDisplay()\n", this);
}

void GraphicsLayerSilk::setNeedsDisplayInRect(const FloatRect& rect)
{
	_cemrlLayer.setNeedsDisplayInRect(rect.x(), rect.y(), rect.width(), rect.height());
	NOTIFY_SYNC_REQUIRED;
#if 0
	CEComDebugPrintf("%p: GraphicsLayerSilk::setNeedsDisplayInRect(%f, %f, %f, %f)\n", this,
		rect.x(), rect.y(), rect.width(), rect.height()
		);
#endif //0
}

}; // namespace WebCore
#endif //#if USE(ACCELERATED_COMPOSITING)

