/*
 * Copyright (C) 2003, 2004, 2005, 2006, 2007 Apple Inc. All rights reserved.
 * Copyright (C) 2008 Eric Seidel <eric@webkit.org>
 * Copyright     2009 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"
#include "GraphicsContext.h"

#include "TransformationMatrix.h"
#include "FloatConversion.h"
#include "GraphicsContextPrivate.h"
#include "GraphicsContextPlatformPrivateSilk.h"
#include "ImageBuffer.h"
#include "CEImageBufferGraphicsContext.h"
#include "NotImplemented.h"
#include "Path.h"
#include "Pattern.h"
#include "SilkPath.h"

#if ENABLE(FILTERS)
#if defined(VGC_FILTERS_LATEST_17935)
#include "FEGaussianBlur.h"
#include "ImageBufferFilter.h"
#else //#if defined(VGC_FILTERS_LATEST_17935)
#if defined(VGC_FILTERS_17935)
// see cl52825.
#include "FEGaussianBlurSilk.h"
#include "ImageBufferFilterSilk.h"
#endif //#if defined(VGC_FILTERS_17935)
#endif //#if defined(VGC_FILTERS_LATEST_17935)

#include "SourceGraphic.h"
#include "Filter.h"
#endif //#if ENABLE(FILTERS)

#include "TransformationMatrix.h"

//#define CALC_TIME
#if defined(CALC_TIME)
#include "CESysGetTimeOfDay.h"
#endif

CEHResult VGSupport_init(ICEVGFactory* ivgFactory)
{
	CEU_VALIDATE_PTR(ivgFactory);
	WebCore::GraphicsContextPlatformPrivate::Env::init(ivgFactory);
	return CE_S_OK;
}

void VGSupport_shutdown()
{
	WebCore::GraphicsContextPlatformPrivate::Env::shutdown();
}

namespace WebCore {

CEComICEVGFactoryRef GraphicsContextPlatformPrivate::Env::_vgFactory;	

static void _converColorToRGBAColor(const Color& inColor, CERGBAColor& outColor)
{
	outColor._color._components.alpha = static_cast<UINT8>(inColor.alpha());
	outColor._color._components.red = static_cast<UINT8>(inColor.red());
	outColor._color._components.green = static_cast<UINT8>(inColor.green());
	outColor._color._components.blue = static_cast<UINT8>(inColor.blue());
}

static void _converColorToRGBColor(const Color& inColor, CERGBColor& outColor)
{
	CERGBAColor rgba;
	_converColorToRGBAColor(inColor, rgba);
	outColor = reinterpret_cast<CERGBColor&>(rgba);
}

static void _convertLineStyle(StrokeStyle base, CELineStyle& style)
{
	switch (base) {
       case SolidStroke:
		style = LineStyle_Solid;
           break;
       case DottedStroke:
		style = LineStyle_Dotted;
           break;
       case DashedStroke:
		style = LineStyle_Dashed;
           break;
	default://
		ASSERT(0);
		break;
	}
}

static StrokeStyle _toStrokeStyle(CELineStyle v)
{
	StrokeStyle ret = SolidStroke;
	switch (v) {
	   case LineStyle_Solid:
		   ret = SolidStroke;
		   break;
	   case LineStyle_Dotted:
		   ret = DottedStroke;
		   break;
	   case LineStyle_Dashed:
		   ret = DashedStroke;
		   break;
	   default://
		   ASSERT(0);
		   break;
	}
	return ret;
}

#if !defined(DISABLE_PIXBOUNDARY_FIX_17849)
void adjustLineToPixelBoundaries(FloatPoint& p1, FloatPoint& p2, float strokeWidth, const StrokeStyle& penStyle)
{
	// For odd widths, we add in 0.5 to the appropriate x/y so that the float arithmetic
	// works out.  For example, with a border width of 3, WebKit will pass us (y1+y2)/2, e.g.,
	// (50+53)/2 = 103/2 = 51 when we want 51.5.  It is always true that an even width gave
	// us a perfect position, but an odd width gave us a position that is off by exactly 0.5.
	if (penStyle == DottedStroke || penStyle == DashedStroke) {
		if (p1.x() == p2.x()) {
			p1.setY(p1.y() + strokeWidth);
			p2.setY(p2.y() - strokeWidth);
		} else {
			p1.setX(p1.x() + strokeWidth);
			p2.setX(p2.x() - strokeWidth);
		}
	}

	if (static_cast<int>(strokeWidth) % 2) { //odd
		if (p1.x() == p2.x()) {
			// We're a vertical line.  Adjust our x.
			p1.setX(p1.x() + 0.5f);
			p2.setX(p2.x() + 0.5f);
		} else {
			// We're a horizontal line. Adjust our y.
			p1.setY(p1.y() + 0.5f);
			p2.setY(p2.y() + 0.5f);
		}
	}
}
#endif //#if !defined(DISABLE_PIXBOUNDARY_FIX_17849)

static void _drawLineI(CEComICEVGContextRef& vgc, const IntPoint& point1, const IntPoint& point2, CELineStyle lineStyle)
{
	vgc.save();
	double width = 0;
	vgc.getLineWidthD(&width);
	
	if (width < 1)
		width = 1;

	FloatPoint p1 = point1;
	FloatPoint p2 = point2;
	bool isVerticalLine = (p1.x() == p2.x());

	adjustLineToPixelBoundaries(p1, p2, width, _toStrokeStyle(lineStyle));
	vgc.setLineWidthD(width);

	int patWidth = 0;
	switch (lineStyle) {
	case LineStyle_Solid:
		break;
	case LineStyle_Dotted:
		patWidth = static_cast<int>(width);
		break;
	case LineStyle_Dashed:
		patWidth = 3*static_cast<int>(width);
		break;
	}
	vgc.setAntialias(eCEVGAntialias_NONE);

	if (patWidth) {
		// Do a rect fill of our endpoints.  This ensures we always have the
		// appearance of being a border.  We then draw the actual dotted/dashed line.
		if (isVerticalLine) {
			CERect rc1 = {static_cast<INT32>(p1.x() - width/2), static_cast<INT32>(p1.y() - width), static_cast<INT32>(width), static_cast<INT32>(width)};
			vgc.drawRectangleI(&rc1, eCEVGDrawMode_FILL, eCEVGOperator_OVER);
			CERect rc2 = {static_cast<INT32>(p2.x() - width/2), static_cast<INT32>(p2.y()), static_cast<INT32>(width), static_cast<INT32>(width)};
			vgc.drawRectangleI(&rc2, eCEVGDrawMode_FILL, eCEVGOperator_OVER);
		} else {
			CERect rc3 = {static_cast<INT32>(p1.x() - width), static_cast<INT32>(p1.y() - width/2), static_cast<INT32>(width), static_cast<INT32>(width)};
			vgc.drawRectangleI(&rc3, eCEVGDrawMode_FILL, eCEVGOperator_OVER);
			CERect rc4 = {static_cast<INT32>(p2.x()), static_cast<INT32>(p2.y() - width/2), static_cast<INT32>(width), static_cast<INT32>(width)};
			vgc.drawRectangleI(&rc4, eCEVGDrawMode_FILL, eCEVGOperator_OVER);
		}

		// Example: 80 pixels with a width of 30 pixels.
		// Remainder is 20.  The maximum pixels of line we could paint
		// will be 50 pixels.
		int distance = (isVerticalLine ? (point2.y() - point1.y()) : (point2.x() - point1.x())) - 2*static_cast<int>(width);
		int remainder = distance%patWidth;
		int coverage = distance-remainder;
		int numSegments = coverage/patWidth;

		float patternOffset = 0;
		// Special case 1px dotted borders for speed.
		if (patWidth == 1)
			patternOffset = 1.0;
		else {
			bool evenNumberOfSegments = numSegments%2 == 0;
			if (remainder)
				evenNumberOfSegments = !evenNumberOfSegments;
			if (evenNumberOfSegments) {
				if (remainder) {
					patternOffset += patWidth - remainder;
					patternOffset += remainder/2;
				}
				else
					patternOffset = static_cast<float>(patWidth/2);
			}
			else if (!evenNumberOfSegments) {
				if (remainder)
					patternOffset = static_cast<float>((patWidth - remainder)/2);
			}
		}

		double dash = patWidth;
		vgc.setDashD(&dash, 1, patternOffset);
	}
	vgc.moveToF(p1.x(), p1.y());
	vgc.lineToF(p2.x(), p2.y());

	vgc.stroke();
	vgc.restore();
}

static inline void setColor(CEComICEVGContextRef& vgc, const Color& col)
{
	CERGBAColor rgbc;
	_converColorToRGBAColor(col, rgbc);
	vgc.setSourceRGBA(&rgbc);
}

static inline void setPlatformFill(GraphicsContext* context, CEComICEVGContextRef& cr, GraphicsContextPrivate* gcp)
{
	cr.save();
	switch (gcp->state.fillColorSpace){
	case PatternColorSpace:
		{
			TransformationMatrix affine;
			cr.setSourcePattern(gcp->state.fillPattern->createPlatformPattern(affine));
			break;
		}
	case GradientColorSpace:
		cr.setSourcePattern(gcp->state.fillGradient->platformGradient());
		break;
	case SolidColorSpace:
		setColor(cr, context->fillColor());
		break;
	default:
		ASSERT_NOT_REACHED();
		break;
	}
	cr.setClipPreserve();
	cr.paintWidthAlphaF(gcp->state.globalAlpha);
	cr.restore();
}

static inline void setPlatformStroke(GraphicsContext* context, CEComICEVGContextRef& cr, GraphicsContextPrivate* gcp)
{
	cr.save();
    
	switch (gcp->state.strokeColorSpace) {
	case PatternColorSpace:
		{
			TransformationMatrix affine;
			cr.setSourcePattern(gcp->state.strokePattern->createPlatformPattern(affine));
		}
		break;
	case GradientColorSpace:
		cr.setSourcePattern(gcp->state.strokeGradient->platformGradient());
		break;
	case SolidColorSpace:
		{
			Color strokeColor = colorWithOverrideAlpha(context->strokeColor().rgb(), context->strokeColor().alpha() / 255.f * gcp->state.globalAlpha);
			setColor(cr, strokeColor);
		}
		break;
	default:
		ASSERT_NOT_REACHED();
		break;
	}

    if (gcp->state.globalAlpha < 1.0f && (gcp->state.strokePattern || gcp->state.strokeGradient)) {
		cr.pushGroup();
		cr.paintWidthAlphaF(gcp->state.globalAlpha);
		cr.popGroupToSource();
    }
	cr.strokePreserve();
	cr.restore();
}


static inline void copyContextProperties(CEComICEVGContextRef& srcVgc, CEComICEVGContextRef& dstVgc)
{
	srcVgc.copyPropertiesTo(dstVgc);
}

#if defined(VGC_FILTERS_17935)
void GraphicsContext::calculateShadowBufferDimensions(IntSize& shadowBufferSize, FloatRect& shadowRect, float& kernelSize, const FloatRect& sourceRect, const IntSize& shadowSize, int shadowBlur)
{
#if ENABLE(FILTERS)
	// calculate the kernel size according to the HTML5 canvas shadow specification
	kernelSize = (shadowBlur < 8 ? shadowBlur / 2.f : sqrt(shadowBlur * 2.f));
	int blurRadius = ceil(kernelSize);

	shadowBufferSize = IntSize(sourceRect.width() + blurRadius * 2, sourceRect.height() + blurRadius * 2);

	// determine dimensions of shadow rect
	shadowRect = FloatRect(sourceRect.location(), shadowBufferSize);
	shadowRect.move(shadowSize.width() - kernelSize, shadowSize.height() - kernelSize);
#endif
}
#endif //#if defined(VGC_FILTERS_17935)

static inline void drawPathShadow(GraphicsContext* context, GraphicsContextPrivate* gcp, bool fillShadow, bool strokeShadow)
{
#if ENABLE(FILTERS)
	IntSize shadowSize;
	int shadowBlur;
	Color shadowColor;
	if (!context->getShadow(shadowSize, shadowBlur, shadowColor))
		return;

	// Calculate filter values to create appropriate shadow.
	CEComICEVGContextRef vgc = context->platformContext();
	CEComICEVGPathEnumeratorRef pathEn;
	vgc.copyPathData(&pathEn);
		
	float x, y, w, h;
	if (strokeShadow)
		vgc.getStrokeExtentsF(&x, &y, &w, &h);
	else
		vgc.getFillExtentsF(&x, &y, &w, &h);
	FloatRect rect(x, y, w, h);

	IntSize shadowBufferSize;
	FloatRect shadowRect;
	float kernelSize = 0;
	GraphicsContext::calculateShadowBufferDimensions(shadowBufferSize, shadowRect, kernelSize, rect, shadowSize, shadowBlur);

	// Create suitably-sized ImageBuffer to hold the shadow.
#if defined(VGC_LATEST_COLORSPACE_17634)
	OwnPtr<ImageBuffer> shadowBuffer = ImageBuffer::create(shadowBufferSize);
#else //#if defined(VGC_LATEST_COLORSPACE_17634)
	OwnPtr<ImageBuffer> shadowBuffer = ImageBuffer::create(shadowBufferSize, false);
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)

	// Draw shadow into a new ImageBuffer.
	CEComICEVGContextRef shadowContext = shadowBuffer->context()->platformContext();

	copyContextProperties(vgc, shadowContext);
	shadowContext.translateF(-rect.x() + kernelSize, -rect.y() + kernelSize);

	shadowContext.newPath();
	shadowContext.appendPathEn(pathEn);

	if (fillShadow)
		setPlatformFill(context, shadowContext, gcp);
	if (strokeShadow)
		setPlatformStroke(context, shadowContext, gcp);

	context->createPlatformShadow(shadowBuffer.release(), shadowColor, shadowRect, kernelSize);
#endif
}

#define SILK_VGC (m_data->m_vgc)

GraphicsContext::GraphicsContext(ICEVGContext* pContext)
    :m_common(createGraphicsContextPrivate())
    ,m_data(new GraphicsContextPlatformPrivate(pContext))
	,m_fillColor(255,255,255,255)
	,m_strokeColor(255,255,255,255)
{
	if(pContext)
	{
		setPaintingDisabled(false);
		setPlatformFillColor(fillColor());
		setPlatformStrokeColor(strokeColor());
	}
	else
		setPaintingDisabled(true);
}

GraphicsContext::~GraphicsContext()
{
	destroyGraphicsContextPrivate(m_common);
	delete m_data;	
}

ICEVGContext* GraphicsContext::platformContext() const
{
	ASSERT(!paintingDisabled());
	ASSERT(m_data->m_vgc);
	// The return value no reference counter type.
	// therefore, this method doesn't operate reference counter.
	return m_data->m_vgc;
}

TransformationMatrix GraphicsContext::getCTM() const
{
	CEComICEVGContextRef vgc = SILK_VGC;
	CEDXMatrix2DAffineD m;
	vgc.getMatrix2DAffine(&m);
	return TransformationMatrix(m._xx, m._yx, m._xy, m._yy, m._x0, m._y0);
}

void GraphicsContext::savePlatformState()
{
	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.save();
}

void GraphicsContext::restorePlatformState()
{
	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.restore();
}

void GraphicsContext::setPlatformStrokeStyle(const StrokeStyle& strokeStyle)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	
	static double dashPattern[] = {5.0, 5.0};
	static double dotPattern[] = {1.0, 1.0};

	switch (strokeStyle) {
	case NoStroke:
		// FIXME: is it the right way to emulate NoStroke?
		vgc.setLineWidthD(0);
		break;
	case SolidStroke:
		vgc.setDashD(0, 0, 0);
		break;
	case DottedStroke:
		vgc.setDashD(dotPattern, 2, 0);
		break;
	case DashedStroke:
		vgc.setDashD(dashPattern, 2, 0);
		break;
	}

	// TODO: remove this variable.
	m_strokeStyle = strokeStyle;
}

// Draws a filled rectangle with a stroked border.
void GraphicsContext::drawRect(const IntRect& rect)
{
#if defined(CALC_TIME)
		INT64 tmsA = CESysGetMonotonicTick();
#endif
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;

	CEHResult hr = vgc.save();
	if(CESucceeded(hr))
	{
		if(fillColor().alpha())
		{
			setColor(vgc, m_fillColor);
			CERect srect(rect);
			hr = vgc.drawRectangleI(&srect, eCEVGDrawMode_FILL, eCEVGOperator_OVER);
		}

		if (CESucceeded(hr) && strokeStyle() != NoStroke && strokeColor().alpha())
		{
			// We do a fill of four rects to simulate the stroke of a border.
			setColor(vgc, m_strokeColor);
			{
				IntRect rect2 = rect;
				rect2.inflate(1);
				CERect inrect(rect2);
				CELineStyle style;
				_convertLineStyle(strokeStyle(), style);
				hr = vgc.drawRectangleI(&inrect, eCEVGDrawMode_STROKE, eCEVGOperator_OVER);
			}	
		}
		vgc.restore();
	}

#if defined(CALC_TIME)
	INT64 tmsB = CESysGetMonotonicTick();
	INT32 time = (INT32)((tmsB-tmsA));
	static INT32 rectNum = 0;
	static INT32 allTime = 0;
	printf("Time betweenGraphicsContext::drawRect count:%d, time=%d, all time:%d \n", rectNum, time, allTime );
	rectNum++;
	allTime+=time;

#endif
}

// This is only used to draw borders.
void GraphicsContext::drawLine(const IntPoint& point1, const IntPoint& point2)
{
	if (paintingDisabled())
		return;

	if(strokeStyle() == NoStroke || !strokeColor().alpha())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;

	CEHResult hr = vgc.save();
	if(CESucceeded(hr))
	{
		vgc.setLineWidthD(strokeThickness());
		setColor(vgc, strokeColor());
		{
			//Draw the Line
			CELineStyle style;
			_convertLineStyle(strokeStyle(), style);
			_drawLineI(vgc, point1, point2, style);
		}
		vgc.restore();
	}
}
#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

// This method is only used to draw the little circles used in lists.
void GraphicsContext::drawEllipse(const IntRect& rect)
{
	if(paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;

	float yRadius = 0.5f*rect.height();
	float xRadius = 0.5f*rect.width();
	
	CEHResult hr = vgc.save();
	if(CESucceeded(hr))
	{
		vgc.translateF(rect.x() + xRadius, rect.y() + yRadius);
		vgc.scaleF(xRadius, yRadius);
		vgc.addArcF(0., 0., 1., 0., 2 * M_PI);
		vgc.restore();   // really?
	}		
	if (fillColor().alpha()) {
		setColor(vgc, fillColor());
		vgc.fillPreserve();
	}

	if (strokeStyle() != NoStroke) {
		setColor(vgc, strokeColor());
		vgc.setLineWidthD(strokeThickness());
		vgc.stroke();
	}
	vgc.newPath();
}

void GraphicsContext::strokeArc(const IntRect& rect, int startAngle, int angleSpan)
{ 
	if (paintingDisabled() || strokeStyle() == NoStroke)
		return;

	int x = rect.x();
	int y = rect.y();
	float w = rect.width();
	float h = rect.height();
	float scaleFactor = h / w;
	float reverseScaleFactor = w / h;

	float hRadius = w / 2;
	float vRadius = h / 2;
	float fa = startAngle;
	float falen =  fa + angleSpan;

	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.save();

	if (w != h)
		vgc.scaleF(1., scaleFactor);

	vgc.addArcNegativeF(x + hRadius, (y + vRadius) * reverseScaleFactor, hRadius, -fa * M_PI/180, -falen * M_PI/180);

	if (w != h)
		vgc.scaleF(1., reverseScaleFactor);

	float width = strokeThickness();
	int patWidth = 0;

	switch (strokeStyle()) {
		case DottedStroke:
			patWidth = static_cast<int>(width / 2);
			break;
		case DashedStroke:
			patWidth = 3 * static_cast<int>(width / 2);
			break;
		default:
			break;
	}

	setColor(vgc, strokeColor());

	if (patWidth) {
		// Example: 80 pixels with a width of 30 pixels.
		// Remainder is 20.  The maximum pixels of line we could paint
		// will be 50 pixels.
		int distance;
		if (hRadius == vRadius)
			distance = static_cast<int>((M_PI * hRadius) / 2.0);
		else // We are elliptical and will have to estimate the distance
			distance = static_cast<int>((M_PI * sqrtf((hRadius * hRadius + vRadius * vRadius) / 2.0)) / 2.0);

		int remainder = distance % patWidth;
		int coverage = distance - remainder;
		int numSegments = coverage / patWidth;

		float patternOffset = 0.0;
		// Special case 1px dotted borders for speed.
		if (patWidth == 1)
			patternOffset = 1.0;
		else {
			bool evenNumberOfSegments = !(numSegments % 2);
			if (remainder)
				evenNumberOfSegments = !evenNumberOfSegments;
			if (evenNumberOfSegments) {
				if (remainder) {
					patternOffset += patWidth - remainder;
					patternOffset += remainder / 2.0;
				} else
					patternOffset = patWidth / 2.0;
			} else {
				if (remainder)
					patternOffset = (patWidth - remainder) / 2.0;
			}
		}

		double dash = patWidth;
		vgc.setDashD(&dash, 1, patternOffset);
	}
	vgc.stroke();
	vgc.restore();
}

void GraphicsContext::drawConvexPolygon(size_t npoints, const FloatPoint* points, bool shouldAntialias)
{
	if (paintingDisabled() || !fillColor().alpha() && (strokeThickness() <= 0 || strokeStyle() == NoStroke))
		return;
	
	if (npoints <= 1)
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.save();
	vgc.setAntialias(shouldAntialias ? eCEVGAntialias_DEFAULT : eCEVGAntialias_NONE);
	vgc.moveToF(points[0].x(), points[0].y());
	for (size_t i = 1; i < npoints; i++)
		vgc.lineToF(points[i].x(), points[i].y());
	vgc.closePath();

	if (fillColor().alpha()) {
		setColor(vgc, fillColor());
		vgc.setFillRule(eCEVGFillRule_EVENODD);
		vgc.fillPreserve();
	}

	if (strokeStyle() != NoStroke) {
		setColor(vgc, strokeColor());
		vgc.setLineWidthD(strokeThickness());
		vgc.stroke();
	}
	vgc.newPath();
	vgc.restore();
}

void GraphicsContext::drawPath()
{	
	if (paintingDisabled())
		return;

	// TODO: test this method.
	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.setFillRule(fillRule() == RULE_EVENODD ? eCEVGFillRule_EVENODD : eCEVGFillRule_WINDING);
	drawPathShadow(this, m_common, true, true);

	setPlatformFill(this, vgc, m_common);
	setPlatformStroke(this, vgc, m_common);
	vgc.newPath();
}

static void drawBorderlessRectShadow(GraphicsContext* context, const FloatRect& rect, const Color& rectColor)
{
#if ENABLE(FILTERS)
	IntSize shadowSize;
	int shadowBlur;
	Color shadowColor;

	if (!context->getShadow(shadowSize, shadowBlur, shadowColor))
		return;

	IntSize shadowBufferSize;
	FloatRect shadowRect;
	float kernelSize = 0;
	GraphicsContext::calculateShadowBufferDimensions(shadowBufferSize, shadowRect, kernelSize, rect, shadowSize, shadowBlur);

	// Draw shadow into a new ImageBuffer
#if defined(VGC_LATEST_COLORSPACE_17634)
	OwnPtr<ImageBuffer> shadowBuffer = ImageBuffer::create(shadowBufferSize);
#else //#if defined(VGC_LATEST_COLORSPACE_17634)
	OwnPtr<ImageBuffer> shadowBuffer = ImageBuffer::create(shadowBufferSize, false);
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)
	
	GraphicsContext* shadowContext = shadowBuffer->context();

#if defined(VGC_LATEST_COLORSPACE_17634)
	shadowContext->fillRect(FloatRect(FloatPoint(kernelSize, kernelSize), rect.size()), rectColor, DeviceColorSpace);
#else // #if defined(VGC_LATEST_COLORSPACE_17634)
	shadowContext->fillRect(FloatRect(FloatPoint(kernelSize, kernelSize), rect.size()), rectColor);
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)

	context->createPlatformShadow(shadowBuffer.release(), shadowColor, shadowRect, kernelSize);
#endif
}

void GraphicsContext::fillPath()
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.setFillRule(fillRule() == RULE_EVENODD ? eCEVGFillRule_EVENODD : eCEVGFillRule_WINDING);

	drawPathShadow(this, m_common, true, false);
	setPlatformFill(this,vgc , m_common);
	vgc.newPath();
}

void GraphicsContext::strokePath()
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	drawPathShadow(this, m_common, false, true);

	setPlatformStroke(this, vgc, m_common);
	vgc.newPath();
}

void GraphicsContext::fillRect(const FloatRect& r)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.addRectangleF(r.x(), r.y(), r.width(), r.height());
	fillPath();
}


#if defined(VGC_LATEST_COLORSPACE_17634)
void GraphicsContext::fillRect(const FloatRect& rect, const Color& color, ColorSpace colorSpace)
#else //#if defined(VGC_LATEST_COLORSPACE_17634)
void GraphicsContext::fillRect(const FloatRect& rect, const Color& color)
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)
{
#if defined(CALC_TIME)
		INT64 tmsA = CESysGetMonotonicTick();
#endif
	if (paintingDisabled())
		return;
	drawBorderlessRectShadow(this, rect, color);
	if(color.alpha())
	{
		CEComICEVGContextRef vgc = SILK_VGC;
		CEHResult hr = vgc.save();
		if(CESucceeded(hr)) {
			setColor(vgc, color);
			const CERect rectI = {static_cast<INT32>(rect.x()), static_cast<INT32>(rect.y()),
				static_cast<INT32>(rect.width()), static_cast<INT32>(rect.height())  };
			hr = vgc.drawRectangleI(&rectI, eCEVGDrawMode_FILL, eCEVGOperator_OVER);
			vgc.restore();
		}
	}

#if defined(CALC_TIME)
	INT64 tmsB = CESysGetMonotonicTick();
	INT32 time = (INT32)((tmsB-tmsA));
	static INT32 rectNum = 0;
	static INT32 allTime = 0;
	printf("Time betweenGraphicsContext::fillRect count:%d, time=%d, all time:%d \n",  rectNum, time, allTime );
	rectNum++;
	allTime+=time;
#endif
}

#if defined(VGC_LATEST_COLORSPACE_17634)
void GraphicsContext::fillRoundedRect(const IntRect& r, const IntSize& topLeft, const IntSize& topRight, const IntSize& bottomLeft, const IntSize& bottomRight, const Color& color, ColorSpace colorSpace)
#else //#if defined(VGC_LATEST_COLORSPACE_17634)
void GraphicsContext::fillRoundedRect(const IntRect& rect, const IntSize& topLeft, const IntSize& topRight, const IntSize& bottomLeft, const IntSize& bottomRight, const Color& color)
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)
{
	if (paintingDisabled())
		return;
	if(color.alpha())
	{
		CEComICEVGContextRef vgc = SILK_VGC;
		vgc.save();
		beginPath();
		addPath(Path::createRoundedRectangle(rect, topLeft, topRight, bottomLeft, bottomRight));
		setColor(vgc, color);
		drawPathShadow(this, m_common, true, false);
		vgc.fill();
		vgc.restore();
	}
}

#if defined(VGC_LATEST_COLORSPACE_17634)
void GraphicsContext::drawFocusRing(const Vector<Path>& paths, int width, int offset, const Color& color)
{
	// FIXME: implement
}

void GraphicsContext::drawFocusRing(const Vector<IntRect>& rects, int width, int /* offset */, const Color& color)
{
}
#else //#if defined(VGC_LATEST_COLORSPACE_17634)
void GraphicsContext::drawFocusRing(const Color& color)
{
	if (paintingDisabled())
		return;

#if 0
	// focus navigator will draw focus ring.

	const Vector<IntRect>& rects = focusRingRects();
	unsigned rectCount = rects.size();

	CEComICEVGContextRef vgc = SILK_VGC;
	
	vgc.save();
	vgc.pushGroup();
	vgc.newPath();

	int radius = (focusRingWidth() - 1) / 2;
	for (unsigned i = 0; i < rectCount; i++)
		addPath(Path::createRoundedRectangle(rects[i], FloatSize(radius, radius)));

	// Force the alpha to 50%.  This matches what the Mac does with outline rings.
	Color ringColor(color.red(), color.green(), color.blue(), 127);
	setColor(vgc, ringColor);
	vgc.setLineWidthD(focusRingWidth());
	setPlatformStrokeStyle(SolidStroke);

	vgc.setOperator(eCEVGOperator_OVER);
	vgc.strokePreserve();

	vgc.setOperator(eCEVGOperator_CLEAR);
	vgc.setFillRule(eCEVGFillRule_WINDING);
	vgc.fill();

	vgc.popGroupToSource();
	vgc.setOperator(eCEVGOperator_OVER);

	vgc.paint();
	vgc.restore();
#endif
}
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)

void GraphicsContext::drawLineForMisspellingOrBadGrammar(const IntPoint& point, int width, bool grammar)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;

	CEHResult hr = vgc.save();
	if(CESucceeded(hr))
	{
		CERGBAColor dxcolor;

		if (grammar)
		{	
			dxcolor._color._components.alpha = 255;
			dxcolor._color._components.red = 0;
			dxcolor._color._components.green = 255;
			dxcolor._color._components.blue = 0;
		}
		else
		{
			dxcolor._color._components.alpha = 255;
			dxcolor._color._components.red = 255;
			dxcolor._color._components.green = 0;
			dxcolor._color._components.blue = 0;
		}

		hr = vgc.setSourceRGBA(&dxcolor);
		if(CESucceeded(hr))
		{
			IntPoint p2;
			p2.setX(point.x() + width);
			p2.setY(point.y());
			_drawLineI(vgc, point, p2, LineStyle_Solid);
		}
		vgc.restore();
	}
}

void GraphicsContext::clip(const FloatRect& rect)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	CERectF rectF = { rect.x(), rect.y(),
		rect.width(), rect.height() };

	CEHResult hr = vgc.setClipRectF(&rectF, eCEVGFillRule_WINDING);
	CEU_UNUSED(hr);
}

void GraphicsContext::clipOut(const IntRect& r)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;

	CERectF frect;
	vgc.getClipExtents(&frect._x, &frect._y, &frect._width, &frect._height);
	vgc.addRectangleF(frect._x, frect._y, frect._width, frect._height);
	vgc.addRectangleF(r.x(), r.y(), r.width(), r.height());
	
	eCEVGFillRule savedFillRule = eCEVGFillRule_WINDING;
	vgc.getFillRule(&savedFillRule);
	vgc.setFillRule(eCEVGFillRule_EVENODD);
	vgc.setClip();
	vgc.setFillRule(savedFillRule);
}

void GraphicsContext::clipPath(WindRule clipRule)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.setFillRule(clipRule == RULE_EVENODD ? eCEVGFillRule_EVENODD : eCEVGFillRule_WINDING);
	vgc.setClip();
}

void GraphicsContext::clipOutEllipseInRect(const IntRect& r)
{
	if (paintingDisabled())
		return;

	Path p;
	p.addEllipse(r);
	clipOut(p);

}

void GraphicsContext::addInnerRoundedRectClip(const IntRect& rect, int thickness)
{
	if (paintingDisabled())
		return;

	clip(rect);

	Path p;
	FloatRect r(rect);
	// Add outer ellipse
	p.addEllipse(r);
	// Add inner ellipse
	r.inflate(-thickness);
	p.addEllipse(r);
	addPath(p);

	CEComICEVGContextRef vgc = SILK_VGC;
	eCEVGFillRule savedFillRule = eCEVGFillRule_WINDING;
	vgc.getFillRule(&savedFillRule);
	vgc.setFillRule(eCEVGFillRule_EVENODD);
	vgc.setClip();
	vgc.setFillRule(savedFillRule);
}

void GraphicsContext::beginTransparencyLayer(float opacity)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	CEHResult hr = vgc.pushGroup();
	CEU_UNUSED(hr);
	m_data->layers.append(opacity);
}

void GraphicsContext::endTransparencyLayer()
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	CEHResult hr = vgc.popGroupToSource();
	CEU_UNUSED(hr);
	vgc.paintWidthAlphaF(m_data->layers.last());
	m_data->layers.removeLast();
}

void GraphicsContext::clipToImageBuffer(const FloatRect& rect, const ImageBuffer* imageBuffer)
{
	notImplemented();
}

#if defined(VGC_LATEST_COLORSPACE_17634)
void GraphicsContext::setPlatformShadow(IntSize const& size, int, Color const&, ColorSpace)
#else //#if defined(VGC_LATEST_COLORSPACE_17634)
void GraphicsContext::setPlatformShadow(const IntSize& size, int blur, const Color& color)
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)
{
	// inspired by GraphicsContextCairo.
	if (m_common->state.shadowsIgnoreTransforms) {
		m_common->state.shadowSize = IntSize(size.width(), -size.height());
	}
}

#if defined(VGC_FILTERS_FAKE_17935)
void GraphicsContext::createPlatformShadow(PassOwnPtr<ImageBuffer> buffer, const Color& shadowColor, const FloatRect& shadowRect, float kernelSize)
{
#if ENABLE(FILTERS)
	CEComICEVGContextRef vgc = SILK_VGC;

	// draw the shadow without blurring, if kernelSize is zero
	if (!kernelSize) {
		setColor(vgc, shadowColor);
		NativeImageSilk* pImageSilk = buffer->image()->nativeImageForCurrentFrame();
		if (pImageSilk)
		{
			vgc.maskSurfaceF(pImageSilk->getVGSurface(), shadowRect.x(), shadowRect.y());
		}
		return;
	}

	// limit kernel size to 1000, this is what CG is doing.
	kernelSize = std::min(1000.f, kernelSize);

	// create filter
#if defined(VGC_FILTERS_LATEST_17935)
	RefPtr<Filter> filter = ImageBufferFilter::create();
#else //#if defined(VGC_FILTERS_LATEST_17935)
	RefPtr<Filter> filter = ImageBufferFilterSilk::create();
#endif //#if defined(VGC_FILTERS_LATEST_17935)
	
	filter->setSourceImage(buffer.release());
	RefPtr<FilterEffect> source = SourceGraphic::create();

#if defined(VGC_FILTERS_LATEST_17935)
	source->setScaledSubRegion(FloatRect(FloatPoint(), shadowRect.size()));
	source->setIsAlphaImage(true);
	RefPtr<FilterEffect> blur = FEGaussianBlur::create(source.get(), kernelSize, kernelSize);
	blur->setScaledSubRegion(FloatRect(FloatPoint(), shadowRect.size()));
#else //#if defined(VGC_FILTERS_LATEST_17935)
	source->setSubRegion(FloatRect(FloatPoint(), shadowRect.size())); 
	source->setIsAlphaImage(true);
	RefPtr<FilterEffect> blur = FEGaussianBlurSilk::create(source.get(), kernelSize, kernelSize);
	blur->setSubRegion(FloatRect(FloatPoint(), shadowRect.size()));
#endif //#if defined(VGC_FILTERS_LATEST_17935)

	blur->apply(filter.get());

	// Mask the filter with the shadow color and draw it to the context.
	// Masking makes it possible to just blur the alpha channel.
	setColor(vgc, shadowColor);
	
	{
		ImageBuffer* pIbuff = blur->resultImage();
		if (pIbuff)
		{
			Image* pImg = pIbuff->image();
			if (pImg)
			{
				NativeImageSilk* pImageSilk = pImg->nativeImageForCurrentFrame();
				if (pImageSilk)
				{
					vgc.maskSurfaceF(pImageSilk->getVGSurface(), shadowRect.x(), shadowRect.y());
				}
			}
		}
	}
#endif //#if ENABLE(FILTERS)
}

#endif //#if defined(VGC_FILTERS_FAKE_17935)

void GraphicsContext::clearPlatformShadow()
{
	notImplemented();
}

void GraphicsContext::setMiterLimit(float limit)
{
	if (paintingDisabled())
		return;
	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.setMiterLimitF(limit);
}

void GraphicsContext::setAlpha(float alpha)
{
	m_common->state.globalAlpha = alpha;
}

float GraphicsContext::getAlpha()
{
	return m_common->state.globalAlpha;
}

void GraphicsContext::clearRect(const FloatRect& r)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	CEHResult hr = vgc.save();
	if(CESucceeded(hr))
	{		
		CERect rect = {static_cast<INT32>(r.x()), static_cast<INT32>(r.y()), static_cast<INT32>(r.width()), static_cast<INT32>(r.height())};
		hr = vgc.drawRectangleI(&rect, eCEVGDrawMode_FILL, eCEVGOperator_CLEAR);
		vgc.restore();
	}
}

void GraphicsContext::strokeRect(const FloatRect& rect, float lineWidth)
{
	// Ignore the lineWidth.
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;

	vgc.save();
	vgc.addRectangleF(rect.x(), rect.y(), rect.width(), rect.height());
	vgc.setLineWidthD(lineWidth);
	strokePath();
	vgc.restore();
}

void GraphicsContext::setLineCap(LineCap cap)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	eCEVGLineCap v = eCEVGLineCap_BUTT;
	switch (cap)
	{
	case RoundCap:
		v = eCEVGLineCap_ROUND;
		break;
	case SquareCap:
		v = eCEVGLineCap_SQUARE;
		break;
	case ButtCap:
	default:
		break;
	}
	vgc.setLineCap(v);
}

void GraphicsContext::setLineDash(const DashArray& dashes, float dashOffset)
{
	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.setDashD(dashes.data(), dashes.size(), (double)dashOffset);
}

void GraphicsContext::setLineJoin(LineJoin join)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	eCEVGLineJoin v = eCEVGLineJoin_MITER;
	switch (join)
	{
		case RoundJoin:
			v = eCEVGLineJoin_ROUND;
			break;
		case BevelJoin:
			v = eCEVGLineJoin_BEVEL;
			break;
		case MiterJoin:
		default:
			break;
	}
	vgc.setLineJoin(v);
}

void GraphicsContext::beginPath()
{
	if (paintingDisabled())
		return;
	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.newPath();
}

void GraphicsContext::addPath(const Path& path)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.appendPath(path.platformPath()->m_vgPath);
}

void GraphicsContext::clip(const Path& path)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.appendPath(path.platformPath()->m_vgPath);

	eCEVGFillRule savedRule = eCEVGFillRule_WINDING;
	vgc.getFillRule(&savedRule);
	vgc.setFillRule(eCEVGFillRule_WINDING);
	vgc.setClip();
	vgc.setFillRule(savedRule);
}

#if defined(VGC_LATEST_COLORSPACE_17634)
void GraphicsContext::canvasClip(const Path& path)
{
	clip(path);
}
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)

void GraphicsContext::clipOut(const Path& path)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	float x = 0;
	float y = 0;
	float w = 0;
	float h = 0;
	vgc.getClipExtents(&x, &y, &w, &h);
	vgc.addRectangleF(x, y, w, h);
	addPath(path);

	eCEVGFillRule savedRule = eCEVGFillRule_WINDING;
	vgc.getFillRule(&savedRule);
	vgc.setFillRule(eCEVGFillRule_EVENODD);
	vgc.setClip();
	vgc.setFillRule(savedRule);
}

void GraphicsContext::scale(const FloatSize& size)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.scaleF(size.width(), size.height());
}

void GraphicsContext::rotate(float angle)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.rotateF(angle);
}

void GraphicsContext::translate(float x, float y)
{	
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	CEHResult hr = vgc.translateF(x, y);
	CEU_UNUSED(hr);
}

IntPoint GraphicsContext::origin()
{
	CEDXMatrix2DAffineD matrix;
	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.getMatrix2DAffine(&matrix);
	return IntPoint(static_cast<int>(matrix._x0), static_cast<int>(matrix._y0));
}

void GraphicsContext::concatCTM(const TransformationMatrix& transform)
{
	if (paintingDisabled())
		return;

	const CEDXMatrix2DAffineD matrix = GraphicsContextPlatformPrivate::initCEDXMatrix2DAffineD(transform);
	{
		CEComICEVGRef vg = GraphicsContextPlatformPrivate::Env::getVG();
		CEDXMatrix2DAffineD tm = matrix;
		if (CEFailed(vg.invertMatrix2DAffine(&tm)))
			return;
	}
	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.transformMatrix2DAffine(&matrix);
}

FloatRect GraphicsContext::roundToDevicePixels(const FloatRect& frect)
{
    FloatRect result;
    float x = frect.x();
    float y = frect.y();
    CEComICEVGContextRef vgc = SILK_VGC;
	vgc.userToDeviceF(&x, &y);
    x = round(x);
    y = round(y);
	vgc.deviceToUserF(&x, &y);
    result.setX(static_cast<float>(x));
    result.setY(static_cast<float>(y));
    x = frect.width();
    y = frect.height();
	vgc.userToDeviceDistanceF(&x, &y);
    x = round(x);
    y = round(y);
	vgc.deviceToUserDistanceF(&x, &y);
    result.setWidth(static_cast<float>(x));
    result.setHeight(static_cast<float>(y));
    return result;
}

void GraphicsContext::drawLineForText(const IntPoint& point, int width, bool printing)
{
	if (paintingDisabled())
		return;
	
	if (width <= 0)
		return;
	
	// This is a workaround for http://bugs.webkit.org/show_bug.cgi?id=15659
	StrokeStyle savedStrokeStyle = strokeStyle();
	setStrokeStyle(SolidStroke);

	IntPoint endPoint = point + IntSize(width, 0);
	drawLine(point, endPoint);

	setStrokeStyle(savedStrokeStyle);
}

void GraphicsContext::setURLForRect(const KURL& link, const IntRect& destRect)
{
	//Silk ICEGUIGraphicsContext cannot read the PDF.
	notImplemented();
}

void GraphicsContext::setImageInterpolationQuality(InterpolationQuality mode)
{
	//Silk ICEGUIGraphicsContext execute only Nearest neighbor method.
    notImplemented();
}

InterpolationQuality GraphicsContext::imageInterpolationQuality() const
{
	//Silk ICEGUIGraphicsContext execute only Nearest neighbor method.
    //notImplemented();
	return InterpolationDefault;
}

void GraphicsContext::setPlatformStrokeColor(const Color& color)
{
	m_strokeColor = color;
}

void GraphicsContext::setPlatformFillColor(const Color& color)
{
	m_fillColor = color;
}

void GraphicsContext::setPlatformStrokeThickness(float thickness)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.setLineWidthD((double)thickness);
}

void GraphicsContext::setPlatformShouldAntialias(bool enable)
{
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.setAntialias(enable ? eCEVGAntialias_DEFAULT : eCEVGAntialias_NONE);
}

static inline eCEVGOperator toVGOperator(CompositeOperator op)
{
	switch (op) {
		case CompositeClear:
			return eCEVGOperator_CLEAR;
		case CompositeCopy:
			return eCEVGOperator_SOURCE;
		case CompositeSourceOver:
			return eCEVGOperator_OVER;
		case CompositeSourceIn:
			return eCEVGOperator_IN;
		case CompositeSourceOut:
			return eCEVGOperator_OUT;
		case CompositeSourceAtop:
			return eCEVGOperator_ATOP;
		case CompositeDestinationOver:
			return eCEVGOperator_DEST_OVER;
		case CompositeDestinationIn:
			return eCEVGOperator_DEST_IN;
		case CompositeDestinationOut:
			return eCEVGOperator_DEST_OUT;
		case CompositeDestinationAtop:
			return eCEVGOperator_DEST_ATOP;
		case CompositeXOR:
			return eCEVGOperator_XOR;
		case CompositePlusDarker:
			return eCEVGOperator_SATURATE;
		case CompositeHighlight:
			// There is no Cairo equivalent for CompositeHighlight.
			return eCEVGOperator_OVER;
		case CompositePlusLighter:
			return eCEVGOperator_ADD;
		default:
			return eCEVGOperator_SOURCE;
	}
}

void GraphicsContext::setCompositeOperation(CompositeOperator mode)
{   
	if (paintingDisabled())
		return;

	CEComICEVGContextRef vgc = SILK_VGC;
	vgc.setOperator(toVGOperator(mode));
}
};
