/*
 * This file is part of the internal font implementation.  It should not be included by anyone other than
 * FontMac.cpp, FontWin.cpp and Font.cpp.
 *
 * Copyright (C) 2006, 2007, 2008 Apple Inc.
 * Copyright 2008, 2009 Sony Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#ifndef FontPlatformData_Silk_H
#define FontPlatformData_Silk_H

#include "StringImpl.h"
#include <wtf/PassRefPtr.h>
#include <wtf/RefCounted.h>

#if USE(SILK_CEUIFONT)
#include "CECom.h"
#include "ICEUIFont.h"
#include "ICEI18n.h"
#include "ICEPFRasterizer.h"
//////////////////////////////////////////////////////////
typedef unsigned short Glyph;

//////////////////////////////////////////////////////////

#if !defined(SILK_DISABLE_CEUI_CUSTOM_FONT)
#define WTF_USE_SILK_CEUI_CUSTOM_FONT 1
#endif //if !defined(SILK_DISABLE_CEUI_CUSTOM_FONT)

class CEComICEVGContextRef;

namespace WebCore {

typedef ICEUIPlatformFont* ICEUIPlatformFontRef;

class FontDescription;
class Font;
class GraphicsContext;
class FontPlatformData;
class FloatPoint;

namespace CEUIFontSupport {
	CEHResult toCEUIFontFamily(const WebCore::FontDescription& fontDescription, CEComICEUIFontFamilyRef& ifamily);
	CEHResult toCEUIFontFamily(CEComICEUStringRef& str, CEComICEUIFontFamilyRef& ffOut);
	CEHResult toCEUIFontFamily(const Font& font, CEComICEUIFontFamilyRef& ffOut);
	CEHResult toCEUIFontFamily(const WebCore::AtomicString& astr, CEComICEUIFontFamilyRef& ffOut);
	CEComICEUICompositeFontFactoryRef&	getCFFactory();
	void setupPlatformGC(GraphicsContext& graphicsContext, CEComICEVGContextRef& vgc, bool bSetStrokeColor);
	CEPFFontStyle toCEUIFontStyle(const FontDescription& fontDescription);
	void getSynthesizeFlags(const CEPFFontStyle& requiredStyle, CEComICEUIPlatformFontRef& pf, bool& synthesizeBoldOut, bool& synthesizeItalicOut);
	void getCFLang(CEComICEI18nLocaleRef& langOut);
	void initFontPlatformData(int szPix, CEComICEUIPlatformFontRef& pf, FontPlatformData& fpdOut);
	void drawGlyphs(GraphicsContext* graphicsContext, ICEUIFontGlyphBuffer* iGlyphBuffer, const FloatPoint& point);
};

class FontPlatformData {
public:
	FontPlatformData() :
		  m_pfFont(0)
		, m_size(0)
        , m_syntheticBold(false)
        , m_syntheticOblique(false)
    {
    }

	FontPlatformData(float size, bool bold, bool oblique);
    FontPlatformData(ICEUIPlatformFontRef pfFont, float size, bool bold, bool oblique);
    ~FontPlatformData();

    FontPlatformData(WTF::HashTableDeletedValueType) : m_pfFont(WTF::HashTableDeletedValue) { }
    bool isHashTableDeletedValue() const { return m_pfFont.isHashTableDeletedValue(); }

	ICEUIPlatformFontRef pfFont() const { return m_pfFont ? m_pfFont->pfFont() : 0; }

	float size() const { return m_size; }
    void setSize(float size) { m_size = size; }
    bool syntheticBold() const { return m_syntheticBold; }
    bool syntheticOblique() const { return m_syntheticOblique; }

    unsigned hash() const
    {
        return m_pfFont->hash();
    }

    bool operator==(const FontPlatformData& other) const
    { 
        return (m_pfFont == other.m_pfFont ||
			   (!isHashTableDeletedValue() && !other.isHashTableDeletedValue() &&
			   pfFont() == other.pfFont())) &&
               m_size == other.m_size &&
               m_syntheticBold == other.m_syntheticBold && m_syntheticOblique == other.m_syntheticOblique
			   ;
    }

private:

    class RefCountedICEUIPlatformFont : public RefCounted<RefCountedICEUIPlatformFont> {
    public:
        static PassRefPtr<RefCountedICEUIPlatformFont> create(ICEUIPlatformFontRef pfFont) 
		{
			CEComICEUIPlatformFontRef ref(pfFont);
			if (ref)
				ref.detach();
			return adoptRef(new RefCountedICEUIPlatformFont(pfFont)); 
		}

        static PassRefPtr<RefCountedICEUIPlatformFont> createDeleted() 
		{
			return adoptRef(new RefCountedICEUIPlatformFont(reinterpret_cast<ICEUIPlatformFontRef>(-1)));
		}

        ~RefCountedICEUIPlatformFont();

        ICEUIPlatformFontRef pfFont() const { return m_pfFont; }
        unsigned hash() const
        {
            return StringImpl::computeHash(reinterpret_cast<const UChar*>(&m_pfFont), sizeof(ICEUIPlatformFontRef) / sizeof(UChar));
        }

    private:
        RefCountedICEUIPlatformFont(ICEUIPlatformFontRef pfFont)
            : m_pfFont(pfFont)
        {
		}

        mutable ICEUIPlatformFontRef m_pfFont;
    };

    void platformDataInit(ICEUIPlatformFontRef pfFont, float size, UChar* faceName);

    RefPtr<RefCountedICEUIPlatformFont> m_pfFont;

    float m_size;
    bool m_syntheticBold;
    bool m_syntheticOblique;
};

} //namespace
#else //#if USE(SILK_CEUIFONT)
bad config.
#endif //#if USE(SILK_CEUIFONT)
#endif //FontPlatformData_Silk_H

