/*
 * Copyright (C) 2006, 2007, 2008 Apple Inc.  All rights reserved.
 * Copyright     2008, 2009 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"

#if USE(SILK_CEUIFONT)
#include "Font.h"

#include "TransformationMatrix.h"
#include "FloatConversion.h"
#include "GlyphBuffer.h"
#include "GraphicsContext.h"
#include "ImageBuffer.h"
#include "IntRect.h"
#include "SimpleFontData.h"
#include <wtf/MathExtras.h>

#include "GraphicsContextPlatformPrivateSilk.h"
#include "ICEVG.h"

#include "Gradient.h"
#include "Pattern.h"

#if ENABLE(FILTERS)
#include "ImageBuffer.h"
#endif //#if ENABLE(FILTERS)
namespace WebCore {

namespace CEUIFontSupport {
void drawGlyphs(GraphicsContext* graphicsContext, ICEUIFontGlyphBuffer* iGlyphBuffer, const FloatPoint& point)
{
	if (!graphicsContext) return;

	CEPointBase posb;
	posb._x = point.x();
	posb._y = point.y();
	CEComICEUIFontGlyphBufferRef gb(iGlyphBuffer);
	CEComICEVGContextRef vgc = graphicsContext->platformContext();
	if (vgc) {
		Color fillColor = graphicsContext->fillColor();

		// Text shadow, inspired by FontMac
		IntSize shadowSize;
		int shadowBlur = 0;
		Color shadowColor;
		bool hasShadow = graphicsContext->textDrawingMode() == cTextFill &&
			graphicsContext->getShadow(shadowSize, shadowBlur, shadowColor);

		// TODO: Blur support
		if (hasShadow) {
			// Disable graphics context shadows (not yet implemented) and paint them manually
			graphicsContext->clearShadow();
			Color shadowFillColor(shadowColor.red(), shadowColor.green(), shadowColor.blue(), shadowColor.alpha() * fillColor.alpha() / 255);
			vgc.save();

			CERGBAColor ac;
			GraphicsContextPlatformPrivate::Color2CERGBAColor(shadowColor, ac);
			vgc.setSourceRGBA(&ac);

#if ENABLE(FILTERS)
			CEUITextExtents extents;
			gb.getGlyphExtents(&extents);

			FloatRect rect(FloatPoint(), FloatSize(extents.width, extents.height));
			IntSize shadowBufferSize;
			FloatRect shadowRect;
			float kernelSize = 0.f;
			GraphicsContext::calculateShadowBufferDimensions(shadowBufferSize, shadowRect, kernelSize, rect, shadowSize, shadowBlur);

			// Draw shadow into a new ImageBuffer
#if defined(VGC_FILTERS_LATEST_17935)
			OwnPtr<ImageBuffer> shadowBuffer = ImageBuffer::create(shadowBufferSize);
#else //#if defined(VGC_FILTERS_LATEST_17935)
			OwnPtr<ImageBuffer> shadowBuffer = ImageBuffer::create(shadowBufferSize, false);
#endif //#if defined(VGC_FILTERS_LATEST_17935)
			GraphicsContext* shadowContext = shadowBuffer->context();
			CEComICEVGContextRef shadowCr = shadowContext->platformContext();
			CEPointBase sgpos = {kernelSize, extents.height + kernelSize};
			shadowCr.drawGlyphsI(&sgpos, iGlyphBuffer);

			vgc.translateF(0.0 + posb._x, posb._y - extents.height);
			graphicsContext->createPlatformShadow(shadowBuffer.release(), shadowColor, shadowRect, kernelSize);
#else
			vgc.translateF(shadowSize.width(), shadowSize.height());
			vgc.drawGlyphsI(&posb, iGlyphBuffer);
#endif
			vgc.restore();
		}
		vgc.save();
		{
			int tdmode = graphicsContext->textDrawingMode();

			if (tdmode & cTextFill) {
				switch (graphicsContext->fillColorSpace()) {
				case GradientColorSpace:
					{
						vgc.setSourcePattern(graphicsContext->fillGradient()->platformGradient());
						if (graphicsContext->getAlpha() < 1.0f) {
							vgc.pushGroup();
							vgc.paintWidthAlphaF(graphicsContext->getAlpha());
							vgc.popGroupToSource();
						}
					}
					break;
				case PatternColorSpace:
					{
						TransformationMatrix affine;
						CEComICEVGPatternRef vgPattern;
						vgPattern.attach(graphicsContext->fillPattern()->createPlatformPattern(affine));
						vgc.setSourcePattern(vgPattern);
						if (graphicsContext->getAlpha() < 1.0f) {
							vgc.pushGroup();
							vgc.paintWidthAlphaF(graphicsContext->getAlpha());
							vgc.popGroupToSource();
						}
					}
					break;
					case SolidColorSpace: 
					{
						float red, green, blue, alpha;
						fillColor.getRGBA(red, green, blue, alpha);
						vgc.setSourceRGBAF(red, green, blue, alpha * graphicsContext->getAlpha());
					}
					break;
				
				default:
					ASSERT_NOT_REACHED();
					break;
				}

				CEPointBase posb;
				posb._x = point.x();
				posb._y = point.y();
				if (tdmode & cTextStroke)
				{
					vgc.addGlyphPathsI(&posb, iGlyphBuffer);
					vgc.fill();
				}
				else
				{
					vgc.drawGlyphsI(&posb, iGlyphBuffer);
				}
			}

			if (tdmode & cTextStroke) {
				switch (graphicsContext->strokeColorSpace()) {
				case GradientColorSpace:
					{
						vgc.setSourcePattern(graphicsContext->strokeGradient()->platformGradient());
						if (graphicsContext->getAlpha() < 1.0f) {
							vgc.pushGroup();
							vgc.paintWidthAlphaF(graphicsContext->getAlpha());
							vgc.popGroupToSource();
						}
					}
					break;
				case PatternColorSpace:
					{
						TransformationMatrix affine;
						CEComICEVGPatternRef vgPattern;
						vgPattern.attach(graphicsContext->strokePattern()->createPlatformPattern(affine));
						vgc.setSourcePattern(vgPattern);
						if (graphicsContext->getAlpha() < 1.0f) {
							vgc.pushGroup();
							vgc.paintWidthAlphaF(graphicsContext->getAlpha());
							vgc.popGroupToSource();
						}
					}
					break;
				case SolidColorSpace: 
					{
						Color strokeColor = graphicsContext->strokeColor();
						float red, green, blue, alpha;
						strokeColor.getRGBA(red, green, blue, alpha);
						vgc.setSourceRGBAF(red, green, blue, alpha * graphicsContext->getAlpha());
					} 
					break;
				default:
					ASSERT_NOT_REACHED();
					break;
				}

				vgc.setLineWidthD(graphicsContext->strokeThickness());
				CEPointBase posb;
				posb._x = point.x();
				posb._y = point.y();
				vgc.addGlyphPathsI(&posb, iGlyphBuffer);
				vgc.stroke();
			}
		}
		// Re-enable the platform shadow we disabled earlier
		if (hasShadow)
#if defined(VGC_LATEST_COLORSPACE_17634)
			graphicsContext->setShadow(shadowSize, shadowBlur, shadowColor, DeviceColorSpace);
#else //#if defined(VGC_LATEST_COLORSPACE_17634)
			graphicsContext->setShadow(shadowSize, shadowBlur, shadowColor);
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)
		vgc.restore();
	}
	else {
		CEASSERT(0 && "fix this error");
	}
}
} // namespace CEUIFontSupport

void Font::drawGlyphs(GraphicsContext* graphicsContext, const SimpleFontData* font, const GlyphBuffer& glyphBuffer, 
					  int from, int numGlyphs, const FloatPoint& point) const
{
	if (!graphicsContext) return;

	CEComICEUIPlatformFontRef pf = font->platformData().pfFont();

	if (pf)	{
		CEComICEUIFontGlyphBufferRef gb;
		if (CESucceeded(pf.createGlyphBuffer(&gb))) {
			CEASSERT(sizeof(WebCore::GlyphBufferAdvance) == sizeof(CEDimF));
			CEASSERT(sizeof(WebCore::Glyph) == sizeof(UINT16));
			if (CESucceeded(gb.initG16F(pf, glyphBuffer.glyphs(from), reinterpret_cast<const CEDimF*>(glyphBuffer.advances(from)), numGlyphs))) {
				CEUIFontSupport::drawGlyphs(graphicsContext, gb, point);
			}
		}
	}
}

}
#else //#if !USE(SILK_CEUIFONT)
bad config.
#endif //#if !USE(SILK_CEUIFONT)
