/*
 * Copyright (C) 2007 Apple Inc. All rights reserved.
 * Copyright (C) 2007 Justin Haygood (jhaygood@reaktix.com)
 * Copyright (C) 2009 Google Inc. All rights reserved.
 * Copyright     2009 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer. 
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 * 3.  Neither the name of Apple Computer, Inc. ("Apple") nor the names of
 *     its contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE AND ITS CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "config.h"
#include "Threading.h"

#if PLATFORM(SILK)
#include <wtf/HashMap.h>
#include <wtf/CurrentTime.h>
#include "CESysCriticalSection.h"
#include "CESysThread.h"
#include "ICEApartment.h"

namespace WTF {

void setThreadNameInternal(const char* szThreadName)
{
}

static ThreadIdentifier mainThreadIdentifier = 0;
static Mutex* atomicallyInitializedStaticMutex = NULL;

void lockAtomicallyInitializedStaticMutex()
{
    ASSERT(atomicallyInitializedStaticMutex);
    atomicallyInitializedStaticMutex->lock();
}

void unlockAtomicallyInitializedStaticMutex()
{
    atomicallyInitializedStaticMutex->unlock();
}

static Mutex& threadMapMutex()
{
    static Mutex mutex;
    return mutex;
}

void initializeThreading()
{
    if (!atomicallyInitializedStaticMutex) {
        atomicallyInitializedStaticMutex = new Mutex;
        threadMapMutex();
        mainThreadIdentifier = currentThread();
    }
}

static HashMap<CESysThreadId, CESysThreadHandle>& threadMap()
{
    static HashMap<CESysThreadId, CESysThreadHandle> map;
    return map;
}

static void storeThreadHandleByIdentifier(CESysThreadId threadID, CESysThreadHandle threadHandle)
{
    MutexLocker locker(threadMapMutex());
    ASSERT(!threadMap().contains(threadID));
    threadMap().add(threadID, threadHandle);
}

static CESysThreadHandle threadHandleForIdentifier(ThreadIdentifier id)
{
    MutexLocker locker(threadMapMutex());
    return threadMap().get(id);
}

static void clearThreadHandleForIdentifier(ThreadIdentifier id)
{
    MutexLocker locker(threadMapMutex());
    ASSERT(threadMap().contains(id));
    threadMap().remove(id);
}

struct ThreadFunctionInvocation {
    ThreadFunctionInvocation(ThreadFunction function, void* data) : function(function), data(data) {}

    ThreadFunction function;
    void* data;
};

typedef union
{
    CEHResult hr;
    void* retval;
} CESysThreadExitCode;

static CEHResult wtfThreadEntryPoint(void* param)
{
    ThreadFunctionInvocation invocation = *static_cast<ThreadFunctionInvocation*>(param);
    delete static_cast<ThreadFunctionInvocation*>(param);

    CEComAttachThread(CEComIID_ICESingleThreadApartment);
    void* result = invocation.function(invocation.data);
    CEComDetachThread();

#if 0 // TODO: Bug 17412
    ThreadSpecificThreadExit();
#endif

    CESysThreadExitCode exitCode;
    exitCode.retval = result;
    return exitCode.hr;
}

ThreadIdentifier createThreadInternal(ThreadFunction entryPoint, void* data, const char* threadName)
{
    CESysThreadId threadIdentifier = 0;
    ThreadFunctionInvocation* invocation = new ThreadFunctionInvocation(entryPoint, data);
    CESysThreadHandle threadHandle = CESysCreateThread(eCESysThreadType_WTF, wtfThreadEntryPoint, invocation, &threadIdentifier);
    if (!threadHandle) {
        delete invocation;
        LOG_ERROR("Failed to create thread at entry point %p with data %p", entryPoint, data);
        return 0;
    }

    storeThreadHandleByIdentifier(threadIdentifier, threadHandle);
    return static_cast<ThreadIdentifier>(threadIdentifier);
}

int waitForThreadCompletion(ThreadIdentifier threadID, void** result)
{
    ASSERT(threadID);
    
    CESysThreadHandle threadHandle = threadHandleForIdentifier(threadID);
    if (!threadHandle)
        LOG_ERROR("ThreadIdentifier %u did not correspond to an active thread when trying to quit", threadID);

    CEHResult exitValue;
    CEHResult joinResult = CESysWaitForThreadExit(threadHandle, &exitValue);
    if (CEFailed(joinResult))
        LOG_ERROR("ThreadIdentifier %u was found to be deadlocked trying to quit", threadID);

    clearThreadHandleForIdentifier(threadID);
    *result = reinterpret_cast<void*>(exitValue);
    return CEFailed(joinResult);
}

void detachThread(ThreadIdentifier threadID)
{
    ASSERT(threadID);

    CESysThreadHandle threadHandle = threadHandleForIdentifier(threadID);
    if (threadHandle)
        CESysThreadCloseHandle(threadHandle);
    clearThreadHandleForIdentifier(threadID);
}

ThreadIdentifier currentThread()
{
    return static_cast<ThreadIdentifier>(CESysGetCurrentThreadId());
}

bool isMainThread()
{
    return currentThread() == mainThreadIdentifier;
}

Mutex::Mutex()
{
    m_mutex.m_recursionCount = 0;
    CESysInitializeCriticalSection(&m_mutex.m_internalMutex, "WebKitWTF");
}

Mutex::~Mutex()
{
    CESysDeleteCriticalSection(&m_mutex.m_internalMutex);
}

void Mutex::lock()
{
    CESysEnterCriticalSection(&m_mutex.m_internalMutex);
    ++m_mutex.m_recursionCount;
}

bool Mutex::tryLock()
{
    bool ok = CESysTryEnterCriticalSection(&m_mutex.m_internalMutex);
    if (ok) { // We got the lock
        if (m_mutex.m_recursionCount > 0) {
            CESysLeaveCriticalSection(&m_mutex.m_internalMutex);
            return false;
        }
        ++m_mutex.m_recursionCount;
        return true;
    }
    return false;
}

void Mutex::unlock()
{
    --m_mutex.m_recursionCount;
    CESysLeaveCriticalSection(&m_mutex.m_internalMutex);
}

ThreadCondition::ThreadCondition(Mutex& mutex)
{
    CESysInitializeConditionVariableCS(&mutex.impl().m_internalMutex, &m_condition);
}

ThreadCondition::ThreadCondition()
{
}

void ThreadCondition::setMutex(Mutex& mutex)
{
    CESysInitializeConditionVariableCS(&mutex.impl().m_internalMutex, &m_condition);
}

ThreadCondition::~ThreadCondition()
{
    CESysDeleteConditionVariable(&m_condition);
}

void ThreadCondition::wait(Mutex& mutex)
{
    CESysSleepConditionVariableCS(&m_condition, CESYS_TIMEOUT_INFINITE);
}

bool ThreadCondition::timedWait(Mutex& mutex, double absoluteTime)
{
    int32_t t;
    if (std::numeric_limits<double>::max() == absoluteTime)
        t = CESYS_TIMEOUT_INFINITE;
    else
        t = std::max(absoluteTime - currentTime(), 0.0);
    return CESysSleepConditionVariableCS(&m_condition, t);
}

void ThreadCondition::signal()
{
    CESysWakeConditionVariable(&m_condition);
}

void ThreadCondition::broadcast()
{
    CESysWakeAllConditionVariable(&m_condition);
}

};
#endif // PLATFORM(SILK)
