/* cairo - a vector graphics library with display and print output
 *
 * Copyright © 2004 Red Hat, Inc.
 * Copyright © 2005 Red Hat, Inc.
 *
 * "The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY
 * OF ANY KIND, either express or implied. See the LGPL or the MPL for
 * the specific language governing rights and limitations.
 *
 * The Original Code is the cairo graphics library.
 *
 * The Initial Developer of the Original Code is Red Hat, Inc.
 *
 * Contributor(s):
 *      Keith Packard <keithp@keithp.com>
 *	Graydon Hoare <graydon@redhat.com>
 *	Carl Worth <cworth@cworth.org>
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the _____ license (the  "[___] License"), in which case the
 * provisions of [______] License are applicable instead of those
 * above. If you wish to allow use of your version of this file only
 * under the terms of the [____] License and not to allow others to use
 * your version of this file under the MPL, indicate your decision by
 * deleting the provisions above and replace them with the notice and
 * other provisions required by the [___] License. If you do not delete
 * the provisions above, a recipient may use your version of this file
 * under either the MPL or the [___] License."
 *
 * NOTE: The text of this Exhibit A may differ slightly from the text
 * of the notices in the Source Code files of the Original Code. You
 * should use the text of this Exhibit A rather than the text found in
 * the Original Code Source Code for Your Modifications. 
 */

#ifndef CAIRO_CACHE_PRIVATE_H
#define CAIRO_CACHE_PRIVATE_H

#include "cairo-compiler-private.h"
#include "cairo-types-private.h"

/**
 * cairo_cache_entry_t:
 *
 * A #cairo_cache_entry_t contains both a key and a value for
 * #cairo_cache_t. User-derived types for #cairo_cache_entry_t must
 * have a #cairo_cache_entry_t as their first field. For example:
 *
 * 	typedef _my_entry {
 *	    cairo_cache_entry_t base;
 *	    ... Remainder of key and value fields here ..
 *	} my_entry_t;
 *
 * which then allows a pointer to my_entry_t to be passed to any of
 * the #cairo_cache_t functions as follows without requiring a cast:
 *
 *	_cairo_cache_insert (cache, &my_entry->base, size);
 *
 * IMPORTANT: The caller is responsible for initializing
 * my_entry->base.hash with a hash code derived from the key.  The
 * essential property of the hash code is that keys_equal must never
 * return %TRUE for two keys that have different hashes. The best hash
 * code will reduce the frequency of two keys with the same code for
 * which keys_equal returns %FALSE.
 *
 * The user must also initialize my_entry->base.size to indicate
 * the size of the current entry. What units to use for size is
 * entirely up to the caller, (though the same units must be used for
 * the max_size parameter passed to _cairo_cache_create()). If all
 * entries are close to the same size, the simplest thing to do is to
 * just use units of "entries", (eg. set size==1 in all entries and
 * set max_size to the number of entries which you want to be saved
 * in the cache).
 *
 * Which parts of the entry make up the "key" and which part make up
 * the value are entirely up to the caller, (as determined by the
 * computation going into base.hash as well as the keys_equal
 * function). A few of the #cairo_cache_t functions accept an entry which
 * will be used exclusively as a "key", (indicated by a parameter name
 * of key). In these cases, the value-related fields of the entry need
 * not be initialized if so desired.
 **/
typedef struct _cairo_cache_entry {
    unsigned long hash;
    unsigned long size;
} cairo_cache_entry_t;

typedef cairo_bool_t
(*cairo_cache_keys_equal_func_t) (const void *key_a, const void *key_b);

typedef void
(*cairo_cache_callback_func_t) (void *entry,
				void *closure);

cairo_private cairo_cache_t *
_cairo_cache_create (cairo_cache_keys_equal_func_t keys_equal,
		     cairo_destroy_func_t	   entry_destroy,
		     unsigned long		   max_size);

cairo_private void
_cairo_cache_destroy (cairo_cache_t *cache);

cairo_private void
_cairo_cache_freeze (cairo_cache_t *cache);

cairo_private void
_cairo_cache_thaw (cairo_cache_t *cache);

cairo_private cairo_bool_t
_cairo_cache_lookup (cairo_cache_t	  *cache,
		     cairo_cache_entry_t  *key,
		     cairo_cache_entry_t **entry_return);

cairo_private cairo_status_t
_cairo_cache_insert (cairo_cache_t	 *cache,
		     cairo_cache_entry_t *entry);

cairo_private void
_cairo_cache_foreach (cairo_cache_t 	      	 *cache,
		      cairo_cache_callback_func_t cache_callback,
		      void			 *closure);

#endif
