#!/usr/bin/python2.6

# Copyright (c) 2009 Google Inc. All rights reserved.
# Copyright (C) 2012 Sony Computer Entertainment Inc. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Make the format of a vcproj really pretty.

   It outputs the resulting xml to stdout.
"""

import os
import sys

from xml.dom.minidom import parse
from xml.dom.minidom import Node

class CmpTuple:
  """Compare function between 2 tuple."""
  def __call__(self, x, y):
    (key1, value1) = x
    (key2, value2) = y
    return cmp(key1, key2)

class CmpNode:
  """Compare function between 2 xml nodes."""

  def get_string(self, node):
    node_string = "node"
    node_string += node.nodeName
    if node.nodeValue:
      node_string += node.nodeValue

    if node.attributes:
      # We first sort by name, if present.
      node_string += node.getAttribute("Name")

      all_nodes = []
      for (name, value) in node.attributes.items():
        all_nodes.append((name, value))

      all_nodes.sort(CmpTuple())
      for (name, value) in all_nodes:
        node_string += name
        node_string += value

    return node_string

  def __call__(self, x, y):
    return cmp(self.get_string(x), self.get_string(y))


def _write_data(writer, data):
    "Writes datachars to writer."
    data = data.replace("&", "&amp;").replace("<", "&lt;")
    data = data.replace("\"", "&quot;").replace(">", "&gt;")
    writer.write(data)

def writexml(node, writer, indent="", addindent="", newl=""):
    # indent = current indentation
    # addindent = indentation to add to higher levels
    # newl = newline string
    writer.write(indent+"<" + node.tagName)

    attrs = node._get_attributes()
    a_names = attrs.keys()

    for a_name in a_names:
        writer.write("\n%s%s" % (indent,addindent))
        writer.write("%s=\"" % a_name)
        _write_data(writer, attrs[a_name].value)
        writer.write("\"")
    if node.childNodes:
        writer.write("\n%s%s" % (indent,addindent))
        writer.write(">%s"%(newl))
        for c in node.childNodes:
            writexml(c,writer,indent+addindent,addindent,newl)
        writer.write("%s</%s>%s" % (indent,node.tagName,newl))
    else:
        writer.write("\n%s" % (indent))
        writer.write("/>%s"%(newl))


def CleanupVcproj(node):
  # For each sub node, we call recursively this function.
  for sub_node in node.childNodes:
    CleanupVcproj(sub_node)

  # Normalize the node, and remove all extranous whitespaces.
  for sub_node in node.childNodes:
    if sub_node.nodeType == Node.TEXT_NODE:
        sub_node.data = sub_node.data.replace("\r", "")
        sub_node.data = sub_node.data.replace("\n", "")
        sub_node.data = sub_node.data.rstrip()

  if node.childNodes:
    node.normalize()

  # For each node, take a copy, and remove it from the list.
  node_array = []
  filters = []
  files = []
  while node.childNodes and node.childNodes[0]:
    # Take a copy of the node and remove it from the list.
    current = node.childNodes[0]
    if current.nodeName == 'Filter':
      filters.append(current)
    elif current.nodeName == 'File':
      files.append(current)
    else:
      node_array.append(current)
    node.removeChild(current)

  # Sort the list.
  filters.sort(CmpNode())
  files.sort(CmpNode())

  for new_node in node_array + filters + files:
    node.appendChild(new_node)

def main(argv):
  """Main function of this vcproj prettifier."""

  # check if we have exactly 1 parameter.
  if len(argv) < 1:
    print ('Usage: %s <vcproj.vcproj>' % argv[0])
    return
  # Open the vcproj and parse the xml.
  dom = parse(argv[1])
  CleanupVcproj(dom.documentElement)

  sys.stdout.write('<?xml version="1.0" encoding="Windows-1252"?>\n')
  writexml(dom.documentElement, sys.stdout, "", "\t", "\n")

if __name__ == '__main__':
  main(sys.argv)
