/*
 * Copyright (C) 2008 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebViewPopupMenu.h"

#include "PopupMenuClient.h"
#include "WebViewPrivate.h"

WebViewPopupMenu::WebViewPopupMenu(WebKit::WebViewPrivate* webView)
    : m_webView(webView)
    , m_client(0)
    , m_itemCount(0)
    , m_value(0)
    , m_selected(0)
{
}

WebViewPopupMenu::~WebViewPopupMenu()
{
    clear();
}

void WebViewPopupMenu::clear()
{
    m_itemCount = 0;
    m_selected = -1;
    m_value = -1;
}

void WebViewPopupMenu::setValue(int index)
{
    setSelected(index);

    m_value = m_selected;

    if (m_client)
        valueChanged(index);
}

void WebViewPopupMenu::show(WebCore::PopupMenuClient* client, int index, const WebCore::IntRect& rect)
{
    if (isActive())
        hide(true);

    // update client
    m_client = client;

    m_selected = index;
    m_itemCount = client->listSize();
    if (m_itemCount > WEBVIEWPOPUPMENU_MAX_ITEMS)
        m_itemCount = WEBVIEWPOPUPMENU_MAX_ITEMS;

    // show callback.
    m_webView->client().showPopupMenu(m_itemCount, m_selected, m_client->multiple());

    for (int i = 0; i < m_itemCount; i++)
        m_webView->client().popupMenuItem(m_client->itemText(i), m_client->itemIsSelected(i), m_client->itemIsEnabled(i));
}

void WebViewPopupMenu::hide(bool cancel)
{
    if (!isActive())
        return;

    m_client->popupDidHide();
    m_client = 0;

    // clear items
    clear();
    m_webView->client().hidePopupMenu();
}

bool WebViewPopupMenu::isActive()
{
    return m_client ? true : false;
}

void WebViewPopupMenu::getItem(int index)
{
}

void WebViewPopupMenu::setSelected(int index)
{
    if (!m_itemCount)
        return;

    if (index < 0)
        index = 0;
    else if (index >= m_itemCount)
        index = m_itemCount - 1;

    if (m_selected == index)
        return;

    m_selected = index;
}

void WebViewPopupMenu::valueChanged(int index)
{
    if (!isActive())
        return;

    if (m_client->multiple())
        m_client->listBoxSelectItem(index, true, false);
    else {
        if (index >= 0)
            m_client->valueChanged(index);
        hide(false);
    }
}
