/*
    Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies)
    Copyright (C) 2014 Sony Computer Entertainment Inc.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "config.h"
#include "GraphicsLayerAcagi.h"

#include "GraphicsContext.h"
#include "GraphicsLayerAnimation.h"
#include "GraphicsLayerFactory.h"
#include "ImageBuffer.h"
#include <wtf/CurrentTime.h>

#if USE(ACCELERATED_COMPOSITING) && USE(ACAGI)

namespace WebCore {

AcagiLayer* toAcagiLayer(GraphicsLayer* layer)
{
    return layer ? toGraphicsLayerAcagi(layer)->layer() : 0;
}

PassOwnPtr<GraphicsLayer> GraphicsLayer::create(GraphicsLayerFactory* factory, GraphicsLayerClient* client)
{
    if (!factory)
        return adoptPtr(new GraphicsLayerAcagi(client));

    return factory->createGraphicsLayer(client);
}

PassOwnPtr<GraphicsLayer> GraphicsLayer::create(GraphicsLayerClient* client)
{
    return adoptPtr(new GraphicsLayerAcagi(client));
}

GraphicsLayerAcagi::GraphicsLayerAcagi(GraphicsLayerClient* client)
    : GraphicsLayer(client)
    , m_layer(adoptPtr(new AcagiLayer()))
    , m_compositedNativeImagePtr(0)
    , m_changeMask(NoChanges)
    , m_needsDisplay(false)
    , m_fixedToViewport(false)
    , m_debugBorderWidth(0)
    , m_contentsLayer(0)
    , m_animationStartTime(0)
    , m_isScrollable(false)
{
}

void GraphicsLayerAcagi::notifyChange(ChangeMask changeMask)
{
    m_changeMask |= changeMask;
    if (!client())
        return;
    client()->notifyFlushRequired(this);
}

void GraphicsLayerAcagi::setName(const String& name)
{
    GraphicsLayer::setName(name);
}

GraphicsLayerAcagi::~GraphicsLayerAcagi()
{
    willBeDestroyed();
}

void GraphicsLayerAcagi::willBeDestroyed()
{
    GraphicsLayer::willBeDestroyed();
}

/* \reimp (GraphicsLayer.h): The current size might change, thus we need to update the whole display.
*/
void GraphicsLayerAcagi::setNeedsDisplay()
{
    if (!drawsContent())
        return;

    m_needsDisplay = true;
    notifyChange(DisplayChange);
    addRepaintRect(FloatRect(FloatPoint(), m_size));
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setContentsNeedsDisplay()
{
    notifyChange(DisplayChange);
    addRepaintRect(contentsRect());
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setNeedsDisplayInRect(const FloatRect& rect)
{
    if (!drawsContent())
        return;

    if (m_needsDisplay)
        return;
    m_needsDisplayRect.unite(rect);
    notifyChange(DisplayChange);
    addRepaintRect(rect);
}

/* \reimp (GraphicsLayer.h)
*/
bool GraphicsLayerAcagi::setChildren(const Vector<GraphicsLayer*>& children)
{
    if (GraphicsLayer::setChildren(children)) {
        notifyChange(ChildrenChange);
        return true;
    }
    return false;
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::addChild(GraphicsLayer* layer)
{
    notifyChange(ChildrenChange);
    GraphicsLayer::addChild(layer);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::addChildAtIndex(GraphicsLayer* layer, int index)
{
    GraphicsLayer::addChildAtIndex(layer, index);
    notifyChange(ChildrenChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::addChildAbove(GraphicsLayer* layer, GraphicsLayer* sibling)
{
    GraphicsLayer::addChildAbove(layer, sibling);
    notifyChange(ChildrenChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::addChildBelow(GraphicsLayer* layer, GraphicsLayer* sibling)
{
    GraphicsLayer::addChildBelow(layer, sibling);
    notifyChange(ChildrenChange);
}

/* \reimp (GraphicsLayer.h)
*/
bool GraphicsLayerAcagi::replaceChild(GraphicsLayer* oldChild, GraphicsLayer* newChild)
{
    if (GraphicsLayer::replaceChild(oldChild, newChild)) {
        notifyChange(ChildrenChange);
        return true;
    }
    return false;
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setMaskLayer(GraphicsLayer* value)
{
    if (value == maskLayer())
        return;
    GraphicsLayer::setMaskLayer(value);
    notifyChange(MaskLayerChange);

    if (!value)
        return;
    value->setSize(size());
    value->setContentsVisible(contentsAreVisible());
}


/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setReplicatedByLayer(GraphicsLayer* value)
{
    if (value == replicaLayer())
        return;
    GraphicsLayer::setReplicatedByLayer(value);
    notifyChange(ReplicaLayerChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setPosition(const FloatPoint& value)
{
    if (value == position())
        return;
    GraphicsLayer::setPosition(value);
    notifyChange(PositionChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setAnchorPoint(const FloatPoint3D& value)
{
    if (value == anchorPoint())
        return;
    GraphicsLayer::setAnchorPoint(value);
    notifyChange(AnchorPointChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setSize(const FloatSize& value)
{
    if (value == size())
        return;

    GraphicsLayer::setSize(value);
    if (maskLayer())
        maskLayer()->setSize(value);
    notifyChange(SizeChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setTransform(const TransformationMatrix& value)
{
    if (value == transform())
        return;

    GraphicsLayer::setTransform(value);
    notifyChange(TransformChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setChildrenTransform(const TransformationMatrix& value)
{
    if (value == childrenTransform())
        return;
    GraphicsLayer::setChildrenTransform(value);
    notifyChange(ChildrenTransformChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setPreserves3D(bool value)
{
    if (value == preserves3D())
        return;
    GraphicsLayer::setPreserves3D(value);
    notifyChange(Preserves3DChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setMasksToBounds(bool value)
{
    if (value == masksToBounds())
        return;
    GraphicsLayer::setMasksToBounds(value);
    notifyChange(MasksToBoundsChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setDrawsContent(bool value)
{
    if (value == drawsContent())
        return;
    GraphicsLayer::setDrawsContent(value);
    notifyChange(DrawsContentChange);

    if (value)
        setNeedsDisplay();
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setContentsVisible(bool value)
{
    if (value == contentsAreVisible())
        return;
    notifyChange(ContentsVisibleChange);
    GraphicsLayer::setContentsVisible(value);
    if (maskLayer())
        maskLayer()->setContentsVisible(value);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setContentsOpaque(bool value)
{
    if (value == contentsOpaque())
        return;
    notifyChange(ContentsOpaqueChange);
    GraphicsLayer::setContentsOpaque(value);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setBackfaceVisibility(bool value)
{
    if (value == backfaceVisibility())
        return;
    GraphicsLayer::setBackfaceVisibility(value);
    notifyChange(BackfaceVisibilityChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setOpacity(float value)
{
    if (value == opacity())
        return;
    GraphicsLayer::setOpacity(value);
    notifyChange(OpacityChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setContentsRect(const IntRect& value)
{
    if (value == contentsRect())
        return;
    GraphicsLayer::setContentsRect(value);
    notifyChange(ContentsRectChange);
}

void GraphicsLayerAcagi::setContentsToSolidColor(const Color& color)
{
    if (color == m_solidColor)
        return;

    m_solidColor = color;
    notifyChange(BackgroundColorChange);
}


/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::setContentsToImage(Image* image)
{
    if (image) {
        // Make the decision about whether the image has changed.
        // This code makes the assumption that pointer equality on a NativeImagePtr is a valid way to tell if the image is changed.
        // This assumption is true in Qt, GTK and EFL.
        NativeImagePtr newNativeImagePtr = image->nativeImageForCurrentFrame();
        if (!newNativeImagePtr)
            return;

        if (newNativeImagePtr == m_compositedNativeImagePtr)
            return;

        m_compositedNativeImagePtr = newNativeImagePtr;
        if (!m_compositedImage)
            m_compositedImage = AcagiTiledBackingStore::create();
        m_compositedImage->setContentsToImage(image);
    } else {
        m_compositedNativeImagePtr = 0;
        m_compositedImage = 0;
    }

    setContentsToMedia(m_compositedImage.get());
    notifyChange(ContentChange);
    GraphicsLayer::setContentsToImage(image);
}

void GraphicsLayerAcagi::setContentsToMedia(PlatformLayerAcagi* media)
{
    if (media == m_contentsLayer)
        return;

    GraphicsLayer::setContentsToMedia(media);
    notifyChange(ContentChange);
    m_contentsLayer = media;

    m_contentsLayer->setClient(this);
}

void GraphicsLayerAcagi::setShowDebugBorder(bool show)
{
    if (isShowingDebugBorder() == show)
        return;

    GraphicsLayer::setShowDebugBorder(show);
    notifyChange(DebugVisualsChange);
}

void GraphicsLayerAcagi::setShowRepaintCounter(bool show)
{
    if (isShowingRepaintCounter() == show)
        return;

    GraphicsLayer::setShowRepaintCounter(show);
    notifyChange(DebugVisualsChange);
}

void GraphicsLayerAcagi::didCommitScrollOffset(const IntSize& offset)
{
    if (offset.isZero())
        return;

    m_committedScrollOffset = offset;
    notifyChange(CommittedScrollOffsetChange);
}

void GraphicsLayerAcagi::setIsScrollable(bool isScrollable)
{
    if (m_isScrollable == isScrollable)
        return;

    m_isScrollable = isScrollable;
    notifyChange(IsScrollableChange);
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::flushCompositingStateForThisLayerOnly()
{
    prepareBackingStoreIfNeeded();
    commitLayerChanges();
    m_layer->syncAnimations();
    updateBackingStoreIfNeeded();
}

void GraphicsLayerAcagi::prepareBackingStoreIfNeeded()
{
    if (!shouldHaveBackingStore()) {
        m_backingStore.clear();
        m_changeMask |= BackingStoreChange;
    } else {
        if (!m_backingStore) {
            m_backingStore = AcagiTiledBackingStore::create();
            m_changeMask |= BackingStoreChange;
        }
    }

    updateDebugBorderAndRepaintCount();
}

void GraphicsLayerAcagi::updateDebugBorderAndRepaintCount()
{
    if (isShowingDebugBorder())
        updateDebugIndicators();

    // When this has its own backing store (e.g. Qt WK1), update the repaint count before calling AcagiLayer::flushCompositingStateForThisLayerOnly().
    bool needsToRepaint = shouldHaveBackingStore() && (m_needsDisplay || !m_needsDisplayRect.isEmpty());
    if (isShowingRepaintCounter() && needsToRepaint) {
        incrementRepaintCount();
        m_changeMask |= RepaintCountChange;
    }
}

void GraphicsLayerAcagi::setDebugBorder(const Color& color, float width)
{
    m_debugBorderColor = color;
    m_debugBorderWidth = width;
    m_changeMask |= DebugVisualsChange;
}

static void toAcagiLayerVector(const Vector<GraphicsLayer*>& layers, Vector<AcagiLayer*>& texmapLayers)
{
    texmapLayers.reserveCapacity(layers.size());
    for (size_t i = 0; i < layers.size(); ++i)
        texmapLayers.append(toAcagiLayer(layers[i]));
}

void GraphicsLayerAcagi::commitLayerChanges()
{
    if (m_changeMask == NoChanges)
        return;

    if (m_changeMask & ChildrenChange) {
        Vector<AcagiLayer*> acagiLayerChildren;
        toAcagiLayerVector(children(), acagiLayerChildren);
        m_layer->setChildren(acagiLayerChildren);
    }

    if (m_changeMask & MaskLayerChange)
        m_layer->setMaskLayer(toAcagiLayer(maskLayer()));

    if (m_changeMask & ReplicaLayerChange)
        m_layer->setReplicaLayer(toAcagiLayer(replicaLayer()));

    if (m_changeMask & PositionChange)
        m_layer->setPosition(position());

    if (m_changeMask & AnchorPointChange)
        m_layer->setAnchorPoint(anchorPoint());

    if (m_changeMask & SizeChange)
        m_layer->setSize(size());

    if (m_changeMask & TransformChange)
        m_layer->setTransform(transform());

    if (m_changeMask & ChildrenTransformChange)
        m_layer->setChildrenTransform(childrenTransform());

    if (m_changeMask & Preserves3DChange)
        m_layer->setPreserves3D(preserves3D());

    if (m_changeMask & ContentsRectChange)
        m_layer->setContentsRect(contentsRect());

    if (m_changeMask & MasksToBoundsChange)
        m_layer->setMasksToBounds(masksToBounds());

    if (m_changeMask & DrawsContentChange)
        m_layer->setDrawsContent(drawsContent());

    if (m_changeMask & ContentsVisibleChange)
        m_layer->setContentsVisible(contentsAreVisible());

    if (m_changeMask & ContentsOpaqueChange)
        m_layer->setContentsOpaque(contentsOpaque());

    if (m_changeMask & BackfaceVisibilityChange)
        m_layer->setBackfaceVisibility(backfaceVisibility());

    if (m_changeMask & OpacityChange)
        m_layer->setOpacity(opacity());

    if (m_changeMask & BackgroundColorChange)
        m_layer->setSolidColor(solidColor());

#if ENABLE(CSS_FILTERS)
    if (m_changeMask & FilterChange)
        m_layer->setFilters(filters());
#endif

    if (m_changeMask & BackingStoreChange)
        m_layer->setBackingStore(m_backingStore);

    if (m_changeMask & DebugVisualsChange)
        m_layer->setDebugVisuals(isShowingDebugBorder(), debugBorderColor(), debugBorderWidth(), isShowingRepaintCounter());

    if (m_changeMask & RepaintCountChange)
        m_layer->setRepaintCount(repaintCount());

    if (m_changeMask & ContentChange)
        m_layer->setContentsLayer(platformLayer());

    if (m_changeMask & AnimationChange)
        m_layer->setAnimations(m_animations);

    if (m_changeMask & AnimationStarted)
        client()->notifyAnimationStarted(this, m_animationStartTime);

    if (m_changeMask & FixedToViewporChange)
        m_layer->setFixedToViewport(fixedToViewport());

    if (m_changeMask & IsScrollableChange)
        m_layer->setIsScrollable(isScrollable());

    if (m_changeMask & CommittedScrollOffsetChange)
        m_layer->didCommitScrollOffset(m_committedScrollOffset);

    m_changeMask = NoChanges;
}

/* \reimp (GraphicsLayer.h)
*/
void GraphicsLayerAcagi::flushCompositingState(const FloatRect& visibleRectInMainFrameView)
{
    if (!m_layer->compositor())
        return;

    // Mapping visible rect in main frame view coordinate onto layer's coordinate.
    FloatRect visibleRect = m_layer->combinedTransform().inverse().mapQuad(FloatQuad(visibleRectInMainFrameView)).boundingBox();
    setActualVisibleRect(enclosingIntRect(visibleRect));
    flushCompositingStateForThisLayerOnly();

    if (maskLayer())
        maskLayer()->flushCompositingState(visibleRectInMainFrameView);
    if (replicaLayer())
        replicaLayer()->flushCompositingState(visibleRectInMainFrameView);
    for (size_t i = 0; i < children().size(); ++i)
        children()[i]->flushCompositingState(visibleRectInMainFrameView);
}

void GraphicsLayerAcagi::updateBackingStoreIfNeeded()
{
    AcagiCompositor* compositor = m_layer->compositor();
    if (!compositor)
        return;

    if (!shouldHaveBackingStore()) {
        ASSERT(!m_backingStore);
        return;
    }
    ASSERT(m_backingStore);

    IntRect dirtyRect = enclosingIntRect(FloatRect(FloatPoint::zero(), m_size));
    if (!m_needsDisplay)
        dirtyRect.intersect(enclosingIntRect(m_needsDisplayRect));

    AcagiTiledBackingStore* backingStore = static_cast<AcagiTiledBackingStore*>(m_backingStore.get());

    FloatRect rect = m_actualVisibleRect;
    rect.inflateX(compositor->maxTextureSize().width() / compositor->contentsScale() * 0.0); // FIXME: 
    rect.inflateY(compositor->maxTextureSize().height() / compositor->contentsScale() * 0.0); // FIXME: 
    backingStore->updateContents(compositor, this, m_size, rect, dirtyRect, AcagiBitmapTexture::UpdateCanModifyOriginalImageData);

    m_needsDisplay = false;
    m_needsDisplayRect = IntRect();
}

bool GraphicsLayerAcagi::shouldHaveBackingStore() const
{
    return drawsContent() && contentsAreVisible() && !m_size.isEmpty();
}

bool GraphicsLayerAcagi::addAnimation(const KeyframeValueList& valueList, const IntSize& boxSize, const Animation* anim, const String& keyframesName, double timeOffset)
{
    ASSERT(!keyframesName.isEmpty());

    if (!anim || anim->isEmptyOrZeroDuration() || valueList.size() < 2 || (valueList.property() != AnimatedPropertyWebkitTransform && valueList.property() != AnimatedPropertyOpacity))
        return false;

    bool listsMatch = false;
    bool hasBigRotation;

    if (valueList.property() == AnimatedPropertyWebkitTransform)
        listsMatch = validateTransformOperations(valueList, hasBigRotation) >= 0;

    const double currentTime = WTF::currentTime();
    m_animations.add(GraphicsLayerAnimation(keyframesName, valueList, boxSize, anim, currentTime - timeOffset, listsMatch));
    // m_animationStartTime is the time of the first real frame of animation, now or delayed by a negative offset.
    if (timeOffset > 0)
        m_animationStartTime = currentTime;
    else
        m_animationStartTime = currentTime - timeOffset;
    notifyChange(AnimationChange);
    notifyChange(AnimationStarted);
    return true;
}

void GraphicsLayerAcagi::setAnimations(const GraphicsLayerAnimations& animations)
{
    m_animations = animations;
    notifyChange(AnimationChange);
}


void GraphicsLayerAcagi::pauseAnimation(const String& animationName, double timeOffset)
{
    m_animations.pause(animationName, timeOffset);
}

void GraphicsLayerAcagi::removeAnimation(const String& animationName)
{
    m_animations.remove(animationName);
}

#if ENABLE(CSS_FILTERS)
bool GraphicsLayerAcagi::setFilters(const FilterOperations& filters)
{
    notifyChange(FilterChange);
    return GraphicsLayer::setFilters(filters);
}
#endif

void GraphicsLayerAcagi::setFixedToViewport(bool fixed)
{
    if (m_fixedToViewport == fixed)
        return;

    m_fixedToViewport = fixed;
    notifyChange(FixedToViewporChange);
}

void GraphicsLayerAcagi::setRepaintCount(int repaintCount)
{
    m_repaintCount = repaintCount;
    notifyChange(RepaintCountChange);
}

}
#endif // USE(ACCELERATED_COMPOSITING) && USE(ACAGI)
