/*
 Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies)
 Copyright (C) 2014 Sony Computer Entertainment Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public License
 along with this library; see the file COPYING.LIB.  If not, write to
 the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 Boston, MA 02110-1301, USA.
 */

#ifndef AcagiCompositor_h
#define AcagiCompositor_h

#if USE(ACCELERATED_COMPOSITING) && USE(ACAGI)

#include "AcagiBitmapTexture.h"
#include "AcagiPlatformGraphics.h"
#include "GraphicsContext.h"
#include "IntRect.h"
#include "IntSize.h"
#include "TransformationMatrix.h"

namespace WebCore {

class BitmapTexturePool;
class GraphicsLayer;

class AcagiCompositor {
    WTF_MAKE_FAST_ALLOCATED;
    WTF_MAKE_NONCOPYABLE(AcagiCompositor);
    friend class AcagiBitmapTexture;
    
    typedef AcagiPlatformGraphics PlatformGraphics;
public:
    struct ClipState {
        IntRect scissorBox;
        ClipState(const IntRect& scissors = IntRect())
            : scissorBox(scissors)
        { }
    };

    class ClipStack {
    public:
        ClipStack()
            : clipStateDirty(false)
        { }

        void push();
        void pop();
        void apply(PlatformGraphics*);
        void applyIfNeeded(PlatformGraphics*);
        inline ClipState& current() { return clipState; }
        void reset(const IntRect&);
        void intersect(const IntRect&);
        inline bool isCurrentScissorBoxEmpty() const
        {
            return !clipStack.isEmpty() && clipState.scissorBox.isEmpty();
        }

    private:
        ClipState clipState;
        Vector<ClipState> clipStack;
        bool clipStateDirty;
        IntSize size;
    };
    
    enum WrapMode {
        StretchWrap,
        RepeatWrap
    };

    static PassOwnPtr<AcagiCompositor> create(PlatformGraphics*);
    ~AcagiCompositor();

    enum ExposedEdges {
        NoEdges = 0,
        LeftEdge = 1 << 0,
        RightEdge = 1 << 1,
        TopEdge = 1 << 2,
        BottomEdge = 1 << 3,
        AllEdges = LeftEdge | RightEdge | TopEdge | BottomEdge,
    };

    void drawBorder(const Color&, float borderWidth, const FloatRect&, const TransformationMatrix&);
    void drawNumber(int number, const Color&, const FloatPoint&, const TransformationMatrix&);

    void drawTexture(const AcagiBitmapTexture&, const FloatRect& target, const TransformationMatrix& modelViewMatrix = TransformationMatrix(), float opacity = 1.0f, unsigned exposedEdges = AllEdges);
    void drawSolidColor(const FloatRect&, const TransformationMatrix&, const Color&);

    // makes a surface the target for the following drawTexture calls.
    void bindSurface(AcagiBitmapTexture* surface);
    void setGraphicsContext(GraphicsContext* context) { m_context = context; }
    GraphicsContext* graphicsContext() { return m_context; }
    void beginClip(const TransformationMatrix&, const FloatRect&);
    void endClip();
    IntRect clipBounds();
    PassRefPtr<AcagiBitmapTexture> createTexture();
    PassRefPtr<AcagiBitmapTexture> duplicateTexture(PassRefPtr<AcagiBitmapTexture>);

    void beginPainting();
    void endPainting();

    void setMaskMode(bool m) { m_isMaskMode = m; }

    IntSize maxTextureSize() const;

    PassRefPtr<AcagiBitmapTexture> acquireTextureFromPool(const IntSize&);

    void setPatternTransform(const TransformationMatrix& p) { m_patternTransform = p; }
    void setWrapMode(WrapMode m) { m_wrapMode = m; }
    void setContentsScale(float scale) { m_contentsScale = scale; }
    float contentsScale() const { return m_contentsScale; }

    ClipStack& clipStack();

private:
    explicit AcagiCompositor(PlatformGraphics*);

    bool isInMaskMode() const { return m_isMaskMode; }
    WrapMode wrapMode() const { return m_wrapMode; }
    const TransformationMatrix& patternTransform() const { return m_patternTransform; }

private:
    PlatformGraphics* m_graphics;
    GraphicsContext* m_context;
    OwnPtr<BitmapTexturePool> m_texturePool;
    bool m_isMaskMode;
    TransformationMatrix m_patternTransform;
    WrapMode m_wrapMode;
    ClipStack m_clipStack;
    float m_contentsScale;
};

}

#endif

#endif
