/*
 * Copyright (C) 2014 Sony Computer Entertainment Inc.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "URLRequestJobFactory.h"

#if USE(NTF)

#include "KURL.h"
#include "ResourceHandle.h"
#include "ResourceRequest.h"
#include "URLRequestData.h"
#include "URLRequestHttp.h"

namespace WebCore {

bool URLRequestJobFactory::create(ResourceHandle* resourceHandle, NTF::URLRequestJob** job, NTF::URLRequestListener** listener)
{
    NTF::URLRequestMessage* message = 0;

    *job = 0;
    *listener = 0;

    KURL requestURL = resourceHandle->firstRequest().url();
    if (requestURL.protocolIsInHTTPFamily()) {
        // HTTP
        message = URLRequestHttpFactory::createRequestMessage(resourceHandle);
        *listener = new URLRequestHttpListener(resourceHandle);
    } else if (requestURL.protocolIsData()) {
        // Data URI scheme
        message = URLRequestDataFactory::createRequestMessage(resourceHandle);
        *listener = new URLRequestDataListener(resourceHandle);
    } else if (requestURL.isLocalFile()) {
        // FILE
        message = URLRequestHttpFactory::createRequestMessage(resourceHandle);
        *listener = new URLRequestHttpListener(resourceHandle);
    }

    if (!message || !(*listener)) {
        if (message) {
            NTF::URLRequestMessage::destruct(message);
            message = 0;
        }

        if (*listener) {
            delete *listener;
            *listener = 0;
        }

        return false;
    }

    *job = NTF::URLRequestJob::create(message, *listener);

    return true;
}

} // namespace WebCore

#endif // #if USE(NTF)
