/*
 * Copyright (C) 2011 Apple Inc. All rights reserved.
 * Copyright (C) 2012 Igalia S.L.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "LayerTreeHostManx.h"

#if USE(TEXTURE_MAPPER_GL)

#include "DrawingAreaImpl.h"
#include "TextureMapperGL.h"
#include "WebPage.h"
#include "WebProcess.h"

#if USE(OPENGL_ES_2)
#include <GLES2/gl2.h>
#else
#include <GL/gl.h>
#endif

#include <WebCore/FocusController.h>
#include <WebCore/Frame.h>
#include <WebCore/FrameView.h>
#include <WebCore/GraphicsLayerTextureMapper.h>
#include <WebCore/Page.h>
#include <WebCore/Settings.h>
#include <manx/util.h>
#include <wtf/CurrentTime.h>
#if USE(REQUEST_ANIMATION_FRAME_DISPLAY_MONITOR)
#include <WebCore/DisplayRefreshMonitor.h>
#endif

#if ENABLE(MANX_INDIRECT_COMPOSITING)
#define BEGIN_DIRECT_COMPOSITE() sce::IndirectCompositor::beginDirectComposite()
#define END_DIRECT_COMPOSITE() sce::IndirectCompositor::endDirectComposite()
#define SET_END_OF_FRAME() sce::IndirectCompositor::setEndOfFrame()
#else
#define BEGIN_DIRECT_COMPOSITE()
#define END_DIRECT_COMPOSITE()
#define SET_END_OF_FRAME() pa_sync()
#endif

using namespace WebCore;

namespace WebKit {

class LayerFlushScheduler {
public:
    typedef WebKit::LayerTreeHostManx Client;

    LayerFlushScheduler()
        : m_scheduled(false)
    {
        m_threadIdentifier = createThread(threadFunc, this, "SceLayerFlushScheduler");
        if (!m_threadIdentifier)
            CRASH();
    }

    void scheduleLayerFlush(Client* client)
    {
        m_clients.add(client);
        m_scheduled = true;
    }

    void cancelPendingLayerFlush(Client* client)
    {
        m_clients.remove(client);
    }

private:
    static void threadFunc(void* that)
    {
        Manx::DisplayRefreshMonitor* monitor = new Manx::DisplayRefreshMonitor;
        if (!monitor)
            CRASH();

        int priority = -1; // Slightly higher than default.
        monitor->start(priority); 
        while (true) {
            monitor->waitVblank();
            displayLinkFired(that);
        }
        monitor->stop();
        delete monitor;
    }

    static void displayLinkFired(void* that)
    {
        if (reinterpret_cast<LayerFlushScheduler*>(that)->m_scheduled)
            callOnMainThread(doLayerFlush, that);
    }

    static void doLayerFlush(void* that)
    {
        reinterpret_cast<LayerFlushScheduler*>(that)->doLayerFlush();
    }

    void doLayerFlush()
    {
        Vector<Client*> activeClients;
        Vector<Client*> inactiveClients;

        for (ClientSet::iterator it = m_clients.begin(); it != m_clients.end(); ++it) {
            Client* client = *it;
            if (!client)
                continue;

            // Do not use client->m_isActive. It may not have been updated yet.
            bool isActive = client->m_webPage->corePage()->focusController()->isActive();
            if (isActive)
                activeClients.append(client);
            else
                inactiveClients.append(client);
        }
        m_clients.clear();
        m_scheduled = false;

        // Flush inactive clients first.
        size_t size = inactiveClients.size();
        for (size_t i = 0; i < size; i++)
            inactiveClients[i]->doLayerFlush();

        // Flush active clients next.
        size = activeClients.size();
        for (size_t i = 0; i < size; i++)
            activeClients[i]->doLayerFlush();
    }

    WTF::ThreadIdentifier m_threadIdentifier;
    volatile bool m_scheduled;
    typedef HashSet<Client*> ClientSet;
    ClientSet m_clients;
};

LayerFlushScheduler* layerFlushScheduler()
{
    static LayerFlushScheduler* scheduler = new LayerFlushScheduler;
    if (!scheduler)
        CRASH();
    return scheduler;
}

}

namespace WebKit {

PassRefPtr<LayerTreeHostManx> LayerTreeHostManx::create(WebPage* webPage)
{
    RefPtr<LayerTreeHostManx> host = adoptRef(new LayerTreeHostManx(webPage));
    host->initialize();
    return host.release();
}

LayerTreeHostManx::LayerTreeHostManx(WebPage* webPage)
    : LayerTreeHost(webPage)
    , m_isValid(true)
    , m_notifyAfterScheduledLayerFlush(false)
    , m_layerFlushSchedulingEnabled(true)
    , m_isActive(true)
    , m_canvasID(sce::Canvas::kInvalidIdValue)
    , m_canvasBuffer(0)
#if ENABLE(MANX_INDIRECT_COMPOSITING)
    , m_indirectCanvas(0)
    , m_indirectCompositor(0)
#endif    
{
    m_canvasBuffer = new CanvasBuffer;
    if (!m_canvasBuffer)
        CRASH();
}

bool LayerTreeHostManx::makeGLContextCurrent()
{
    if (m_offScreenContext)
        return m_offScreenContext->makeCurrent(m_offScreenSurface.get());

    m_offScreenContext = GLPlatformContext::createContext(GraphicsContext3D::RenderOffscreen);

    if (!m_offScreenContext)
        CRASH();

    m_offScreenSurface = GLPlatformSurface::createOffScreenSurface();
    if (!m_offScreenSurface)
        CRASH();

    if (!m_offScreenContext->initialize(m_offScreenSurface.get()))
        CRASH();

    return m_offScreenContext->makeCurrent(m_offScreenSurface.get());
}

void LayerTreeHostManx::initialize()
{
    m_rootLayer = GraphicsLayer::create(graphicsLayerFactory(), this);
    m_rootLayer->setDrawsContent(false);
    m_rootLayer->setSize(m_webPage->size());

    // The non-composited contents are a child of the root layer.
    m_nonCompositedContentLayer = GraphicsLayer::create(graphicsLayerFactory(), this);
    m_nonCompositedContentLayer->setDrawsContent(true);
    m_nonCompositedContentLayer->setContentsOpaque(m_webPage->drawsBackground() && !m_webPage->drawsTransparentBackground());
    m_nonCompositedContentLayer->setSize(m_webPage->size());
    if (m_webPage->corePage()->settings()->acceleratedDrawingEnabled())
        m_nonCompositedContentLayer->setAcceleratesDrawing(true);

#ifndef NDEBUG
    m_rootLayer->setName("LayerTreeHost root layer");
    m_nonCompositedContentLayer->setName("LayerTreeHost non-composited content");
#endif

    m_rootLayer->addChild(m_nonCompositedContentLayer.get());
    m_nonCompositedContentLayer->setNeedsDisplay();

    if (!makeGLContextCurrent()) {
        m_isValid = false;
        return;
    }

    // Acqurire Canvas ID and set its handle to LayerTreeContext.
    sce::Canvas::releaseId(m_canvasID);
    m_canvasID = sce::Canvas::kInvalidIdValue;
    uint32_t canvasID = sce::Canvas::acquireId();
    m_canvasID = canvasID;
    // Force Canvas handle update by setting kInvalidHandle.
    m_layerTreeContext.setCanvasHandle(sce::Canvas::handle(m_canvasID));

    // Initialize canvas buffer with current width and height.
    int width = m_webPage->size().width();
    int height = m_webPage->size().height();
    m_canvasBuffer->initialize(width, height);
    
    m_textureMapper = TextureMapperGL::create();
    static_cast<TextureMapperGL*>(m_textureMapper.get())->setEnableEdgeDistanceAntialiasing(true);
    toTextureMapperLayer(m_rootLayer.get())->setTextureMapper(m_textureMapper.get());

    if (m_webPage->hasPageOverlay()) {
        PageOverlayList& pageOverlays = m_webPage->pageOverlays();
        PageOverlayList::iterator end = pageOverlays.end();
        for (PageOverlayList::iterator it = pageOverlays.begin(); it != end; ++it)
            createPageOverlayLayer(it->get());
    }

    scheduleLayerFlush();
}

LayerTreeHostManx::~LayerTreeHostManx()
{
    ASSERT(!m_isValid);
    ASSERT(!m_rootLayer);
    ASSERT(!m_canvasBuffer);
    ASSERT(!m_indirectCanvas);
    ASSERT(m_canvasID == sce::Canvas::kInvalidIdValue);

    cancelPendingLayerFlush();
}

const LayerTreeContext& LayerTreeHostManx::layerTreeContext()
{
    return m_layerTreeContext;
}

void LayerTreeHostManx::setShouldNotifyAfterNextScheduledLayerFlush(bool notifyAfterScheduledLayerFlush)
{
    m_notifyAfterScheduledLayerFlush = notifyAfterScheduledLayerFlush;
}

void LayerTreeHostManx::setRootCompositingLayer(GraphicsLayer* graphicsLayer)
{
    m_nonCompositedContentLayer->removeAllChildren();

    // Add the accelerated layer tree hierarchy.
    if (graphicsLayer)
        m_nonCompositedContentLayer->addChild(graphicsLayer);

    scheduleLayerFlush();
}

void LayerTreeHostManx::invalidate()
{
    ASSERT(m_isValid);

    cancelPendingLayerFlush();
    m_rootLayer = nullptr;
    m_nonCompositedContentLayer = nullptr;
    m_textureMapper = nullptr;

    // Make m_offScreenContext current so that we can delete resources which are bound to the context.
    if (m_offScreenContext)
        m_offScreenContext->makeCurrent(m_offScreenSurface.get());

    if (m_canvasBuffer) {
        m_canvasBuffer->terminate();
        delete m_canvasBuffer;
        m_canvasBuffer = nullptr;
    }

#if ENABLE(MANX_INDIRECT_COMPOSITING)
    if (m_indirectCanvas) {
        sce::IndirectCanvas::destroy(m_indirectCanvas);
        m_indirectCanvas = nullptr;
    }
    if (m_indirectCompositor) {
        delete m_indirectCompositor;
        m_indirectCompositor = nullptr;
    }
#endif
    if (m_canvasID != sce::Canvas::kInvalidIdValue) {
        sce::Canvas::releaseId(m_canvasID);
        m_canvasID = sce::Canvas::kInvalidIdValue;
    }

    if (m_offScreenContext) {
        m_offScreenContext->releaseCurrent();
        m_offScreenContext->destroy();
    }
    
    m_offScreenSurface = nullptr;
    m_offScreenContext = nullptr;
    m_isValid = false;
}

void LayerTreeHostManx::setNonCompositedContentsNeedDisplay()
{
    m_nonCompositedContentLayer->setNeedsDisplay();

    PageOverlayLayerMap::iterator end = m_pageOverlayLayers.end();
    for (PageOverlayLayerMap::iterator it = m_pageOverlayLayers.begin(); it != end; ++it)
        it->value->setNeedsDisplay();

    scheduleLayerFlush();
}

void LayerTreeHostManx::setNonCompositedContentsNeedDisplayInRect(const IntRect& rect)
{
    m_nonCompositedContentLayer->setNeedsDisplayInRect(rect);

    PageOverlayLayerMap::iterator end = m_pageOverlayLayers.end();
    for (PageOverlayLayerMap::iterator it = m_pageOverlayLayers.begin(); it != end; ++it)
        it->value->setNeedsDisplayInRect(rect);

    scheduleLayerFlush();
}

void LayerTreeHostManx::scrollNonCompositedContents(const IntRect& scrollRect)
{
    setNonCompositedContentsNeedDisplayInRect(scrollRect);
}

void LayerTreeHostManx::sizeDidChange(const IntSize& newSize)
{
    if (m_rootLayer->size() == newSize)
        return;

    if (!makeGLContextCurrent())
        return;

    m_rootLayer->setSize(newSize);

    // If the newSize exposes new areas of the non-composited content a setNeedsDisplay is needed
    // for those newly exposed areas.
    m_nonCompositedContentLayer->setSize(newSize);
    m_nonCompositedContentLayer->setNeedsDisplay();

    PageOverlayLayerMap::iterator end = m_pageOverlayLayers.end();
    for (PageOverlayLayerMap::iterator it = m_pageOverlayLayers.begin(); it != end; ++it)
        it->value->setSize(newSize);

    flushPendingLayerChanges();
    scheduleLayerFlush();

    // Recreate canvas buffer.
    m_canvasBuffer->initialize(newSize.width(), newSize.height());
    // Force canvas handle update by setting kInvalidHandle.
    m_layerTreeContext.setCanvasHandle(sce::Canvas::kInvalidHandle);
}

void LayerTreeHostManx::deviceOrPageScaleFactorChanged()
{
    // Other layers learn of the scale factor change via WebPage::setDeviceScaleFactor.
    m_nonCompositedContentLayer->deviceOrPageScaleFactorChanged();
}

void LayerTreeHostManx::forceRepaint()
{
    scheduleLayerFlush();
}

void LayerTreeHostManx::didInstallPageOverlay(PageOverlay* pageOverlay)
{
    createPageOverlayLayer(pageOverlay);
    scheduleLayerFlush();
}

void LayerTreeHostManx::didUninstallPageOverlay(PageOverlay* pageOverlay)
{
    destroyPageOverlayLayer(pageOverlay);
    scheduleLayerFlush();
}

void LayerTreeHostManx::setPageOverlayNeedsDisplay(PageOverlay* pageOverlay, const IntRect& rect)
{
    GraphicsLayer* layer = m_pageOverlayLayers.get(pageOverlay);
    if (!layer)
        return;

    layer->setNeedsDisplayInRect(rect);
    scheduleLayerFlush();
}

void LayerTreeHostManx::notifyAnimationStarted(const WebCore::GraphicsLayer*, double time)
{
}

void LayerTreeHostManx::notifyFlushRequired(const WebCore::GraphicsLayer*)
{
}

void LayerTreeHostManx::paintContents(const GraphicsLayer* graphicsLayer, GraphicsContext& graphicsContext, GraphicsLayerPaintingPhase, const IntRect& clipRect)
{
    if (graphicsLayer == m_nonCompositedContentLayer) {
        m_webPage->drawRect(graphicsContext, clipRect);
        return;
    }

    PageOverlayLayerMap::iterator end = m_pageOverlayLayers.end();
    for (PageOverlayLayerMap::iterator it = m_pageOverlayLayers.begin(); it != end; ++it) {
        if (it->value == graphicsLayer) {
            m_webPage->drawPageOverlay(it->key, graphicsContext, clipRect);
            break;
        }
    }
}

void LayerTreeHostManx::doLayerFlush()
{
    FUNCTION_MARKER;

    flushAndRenderLayers();

    if (toTextureMapperLayer(m_rootLayer.get())->descendantsOrSelfHaveRunningAnimations())
        layerFlushScheduler()->scheduleLayerFlush(this);
}

bool LayerTreeHostManx::flushPendingLayerChanges()
{
    FUNCTION_MARKER;

    m_rootLayer->flushCompositingStateForThisLayerOnly();
    m_nonCompositedContentLayer->flushCompositingStateForThisLayerOnly();

    PageOverlayLayerMap::iterator end = m_pageOverlayLayers.end();
    for (PageOverlayLayerMap::iterator it = m_pageOverlayLayers.begin(); it != end; ++it)
        it->value->flushCompositingStateForThisLayerOnly();

    return m_webPage->corePage()->mainFrame()->view()->flushCompositingStateIncludingSubframes();
}

void LayerTreeHostManx::compositeLayersToContext()
{
    FUNCTION_MARKER;

    WebCore::Settings* settings = m_webPage->corePage()->settings();
    bool isWebGLEnabled = settings ? settings->webGLEnabled() : false;
    bool isActive = m_webPage->corePage()->focusController()->isActive();
    if (!m_isActive && !isActive && !isWebGLEnabled) {
        if (m_layerFlushSchedulingEnabled)
            layerFlushScheduler()->scheduleLayerFlush(this);
        return;
    }
    m_isActive = isActive;

    if (!makeGLContextCurrent())
        return;

    // Since the Y axis goes up in OpenGL's texture coodinate, vertical flip is required.
    const bool shouldFlipY = true;

    // Store "self" canvas handle.
    sce::Canvas::Handle selfCanvas = sce::Canvas::handle(m_canvasID);
    sce::Canvas::Handle canvasHandle = selfCanvas;

    // The window size may be out of sync with the page size at this point, and getting
    // the viewport parameters incorrect, means that the content will be misplaced. Thus
    // we set the viewport parameters directly from the window size.
    glViewport(0, 0, m_canvasBuffer->width(), m_canvasBuffer->height());

    bool drawsTransparentBackground = m_webPage->drawsTransparentBackground();
    if (drawsTransparentBackground) { // FIXME: Opaque background color may be set.
        glClearColor(0, 0, 0, 0);
        glClear(GL_COLOR_BUFFER_BIT);
    }

    BEGIN_DIRECT_COMPOSITE();
    {
        m_canvasBuffer->activateAsFrameBuffer();

        TextureMapper::PaintFlags flags = 0;
        if (shouldFlipY)
            flags |= TextureMapperGL::PaintingMirrored;
#if ENABLE(MANX_INDIRECT_COMPOSITING)
        if (isActive)
            flags |= TextureMapper::PaintingForIndirectCompositing;
#endif
        m_textureMapper->beginPainting(flags);
        toTextureMapperLayer(m_rootLayer.get())->paint();
        m_textureMapper->endPainting();
        glFlush();

        m_canvasBuffer->activateAsCanvas(m_canvasID);
        m_canvasBuffer->rotateBuffers();
    }
    END_DIRECT_COMPOSITE();    

#if ENABLE(MANX_INDIRECT_COMPOSITING)
    if (m_textureMapper->accelerationMode() == TextureMapper::OpenGLMode) {
        TextureMapperGL* texmapGL = static_cast<TextureMapperGL*>(m_textureMapper.get());
        const Vector<TextureMapperGL::IndirectCompositionRecord>& records = texmapGL->indirectCompositionRecords();

        if (isActive && records.size() > 0) { // Needs to create IndirectCanvas and IndirectCompositor.
            if (!m_indirectCanvas)
                m_indirectCanvas = sce::IndirectCanvas::create();
            if (m_indirectCanvas)
                m_indirectCanvas->setSize(m_webPage->size().width(), m_webPage->size().height());
            if (m_indirectCanvas && !m_indirectCompositor)
                m_indirectCompositor = new sce::IndirectCompositor();

            ASSERT(m_indirectCanvas);
            ASSERT(m_indirectCompositor);
        }

        if (isActive && m_indirectCanvas && m_indirectCompositor && records.size() > 0) {
            m_indirectCompositor->beginComposite();
            // We activate the base canvas here to tell its ID to the IndirectCompositor.
            m_canvasBuffer->activateAsCanvas(m_canvasID);
            m_indirectCompositor->drawBaseCanvas(selfCanvas);
            for (Vector<TextureMapperGL::IndirectCompositionRecord>::const_reverse_iterator it = records.rbegin(); it != records.rend(); ++it) {
                // it->debugPrint();

                int i;
                const TransformationMatrix& m = it->transformation;
                sce::Vec4 vertices[4];

                // (0, 0, 0, 1)
                i = 0;
                vertices[i].f[0] = m.m41();
                vertices[i].f[1] = m.m42();
                vertices[i].f[2] = 0;
                vertices[i].f[3] = m.m44();

                // (0, 1, 0, 1)
                i = 1;
                vertices[i].f[0] = m.m41() + m.m21();
                vertices[i].f[1] = m.m42() + m.m22();
                vertices[i].f[2] = 0;
                vertices[i].f[3] = m.m44() + m.m24();

                // (1, 1, 0, 1)
                i = 3;
                vertices[i].f[0] = m.m41() + m.m11() + m.m21();
                vertices[i].f[1] = m.m42() + m.m12() + m.m22();
                vertices[i].f[2] = 0;
                vertices[i].f[3] = m.m44() + m.m14() + m.m24();

                // (1, 0, 0, 1)
                i = 2;
                vertices[i].f[0] = m.m41() + m.m11();
                vertices[i].f[1] = m.m42() + m.m12();
                vertices[i].f[2] = 0;
                vertices[i].f[3] = m.m44() + m.m14();

                if (shouldFlipY) {
                    vertices[0].f[1] *= -1;
                    vertices[1].f[1] *= -1;
                    vertices[2].f[1] *= -1;
                    vertices[3].f[1] *= -1;
                }

                m_indirectCompositor->drawCanvas(it->texture, vertices, it->opacity);
            }
            m_indirectCompositor->endComposite(m_indirectCanvas, m_canvasID);
        }
    }
#endif

    SET_END_OF_FRAME();

    // Update LayerTreeContext if needed.
    if (canvasHandle != m_layerTreeContext.canvasHandle()) {
        m_layerTreeContext.setCanvasHandle(canvasHandle);
        m_webPage->updateAcceleratedCompositingMode();
    }
}

void LayerTreeHostManx::flushAndRenderLayers()
{
    {
        RefPtr<LayerTreeHostManx> protect(this);
        m_webPage->layoutIfNeeded();

        if (!m_isValid)
            return;
    }

    if (!makeGLContextCurrent())
        return;

    if (!flushPendingLayerChanges())
        return;

    // Our model is very simple. We always composite and render the tree immediately after updating it.
    compositeLayersToContext();

    if (m_notifyAfterScheduledLayerFlush) {
        // Let the drawing area know that we've done a flush of the layer changes.
        static_cast<DrawingAreaImpl*>(m_webPage->drawingArea())->layerHostDidFlushLayers();
        m_notifyAfterScheduledLayerFlush = false;
    }
}

void LayerTreeHostManx::createPageOverlayLayer(PageOverlay* pageOverlay)
{
    OwnPtr<GraphicsLayer> layer = GraphicsLayer::create(graphicsLayerFactory(), this);
#ifndef NDEBUG
    layer->setName("LayerTreeHost page overlay content");
#endif

    layer->setAcceleratesDrawing(m_webPage->corePage()->settings()->acceleratedDrawingEnabled());
    layer->setDrawsContent(true);
    layer->setSize(m_webPage->size());
    layer->setShowDebugBorder(m_webPage->corePage()->settings()->showDebugBorders());
    layer->setShowRepaintCounter(m_webPage->corePage()->settings()->showRepaintCounter());

    m_rootLayer->addChild(layer.get());
    m_pageOverlayLayers.add(pageOverlay, layer.release());
}

void LayerTreeHostManx::destroyPageOverlayLayer(PageOverlay* pageOverlay)
{
    OwnPtr<GraphicsLayer> layer = m_pageOverlayLayers.take(pageOverlay);
    ASSERT(layer);

    layer->removeFromParent();
}

void LayerTreeHostManx::scheduleLayerFlush()
{
    if (!m_layerFlushSchedulingEnabled)
        return;

    layerFlushScheduler()->scheduleLayerFlush(this);
}

void LayerTreeHostManx::setLayerFlushSchedulingEnabled(bool layerFlushingEnabled)
{
    if (m_layerFlushSchedulingEnabled == layerFlushingEnabled)
        return;

    m_layerFlushSchedulingEnabled = layerFlushingEnabled;

    if (m_layerFlushSchedulingEnabled) {
        scheduleLayerFlush();
        return;
    }

    cancelPendingLayerFlush();
}

void LayerTreeHostManx::pageBackgroundTransparencyChanged()
{
    m_nonCompositedContentLayer->setContentsOpaque(m_webPage->drawsBackground() && !m_webPage->drawsTransparentBackground());
}

void LayerTreeHostManx::cancelPendingLayerFlush()
{
    layerFlushScheduler()->cancelPendingLayerFlush(this);
}

void LayerTreeHostManx::setActive(bool)
{
    // Schedule layer flush to update check and update 'active' status.
    scheduleLayerFlush();
}

} // namespace WebKit

#endif
