/*
 * Copyright (C) 2011 Samsung Electronics
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "PageClientImpl.h"

#include "Cursor.h"
#include "DrawingAreaProxyImpl.h"
#include "NativeWebKeyboardEvent.h"
#include "NotImplemented.h"
#include "WebContext.h"
#include "WebContextMenuProxyManx.h"
#include "WebPageProxy.h"
#include "WebPopupMenuProxyManx.h"
#include "WebView.h"

using namespace WebCore;

namespace WebKit {

PageClientImpl::PageClientImpl(WebView* webView)
    : m_webView(webView)
{
}

PageClientImpl::~PageClientImpl()
{
}

// PageClient
PassOwnPtr<DrawingAreaProxy> PageClientImpl::createDrawingAreaProxy()
{
    return DrawingAreaProxyImpl::create(m_webView->page());
}

void PageClientImpl::setViewNeedsDisplay(const WebCore::IntRect& rect)
{
    m_webView->setViewNeedsDisplay(rect);
}

void PageClientImpl::displayView()
{
    notImplemented();
}

bool PageClientImpl::canScrollView()
{
    return false;
}

void PageClientImpl::scrollView(const WebCore::IntRect& scrollRect, const WebCore::IntSize&)
{
    setViewNeedsDisplay(scrollRect);
}

WebCore::IntSize PageClientImpl::viewSize()
{
    return m_webView->viewSize();
}

bool PageClientImpl::isViewWindowActive()
{
    return (m_webView && m_webView->isActive());
}

bool PageClientImpl::isViewFocused()
{
    return (m_webView && m_webView->isFocused());
}

bool PageClientImpl::isViewVisible()
{
    return m_webView->isVisible();
}

bool PageClientImpl::isViewInWindow()
{
    notImplemented();
    return true;
}

void PageClientImpl::processDidCrash()
{
    notImplemented();
}

void PageClientImpl::didRelaunchProcess()
{
    m_webView->didRelaunchProcess();
}

void PageClientImpl::pageClosed()
{
    notImplemented();
}

void PageClientImpl::preferencesDidChange()
{
    notImplemented();
}

void PageClientImpl::toolTipChanged(const String&, const String&)
{
    notImplemented();
}

#if USE(TILED_BACKING_STORE)
void PageClientImpl::pageDidRequestScroll(const WebCore::IntPoint&)
{
    notImplemented();
}

void PageClientImpl::didRenderFrame(const WebCore::IntSize&, const WebCore::IntRect&)
{
    notImplemented();
}

void PageClientImpl::pageTransitionViewportReady()
{
    notImplemented();
}
#endif

void PageClientImpl::didChangeContentsSize(const WebCore::IntSize&)
{
    notImplemented();
}

void PageClientImpl::didChangeEditorState(int eventType, const WebKit::EditorState& editorState)
{
    m_webView->didChangeEditorState(eventType, editorState);
}

void PageClientImpl::didChangeCompositionState(const WebCore::IntRect& compositionRect)
{
    m_webView->didChangeCompositionState(compositionRect);
}

void PageClientImpl::didChangeSelectionState(const String& selectedText, const WebCore::IntRect& selectedRect)
{
    m_webView->didChangeSelectionState(selectedText, selectedRect);
}

#if ENABLE(MANX_CURSOR_NAVIGATION)
void PageClientImpl::setCursorPosition(const WebCore::IntPoint& cursorPosition)
{
    m_webView->setCursorPosition(cursorPosition);
}
#endif

void PageClientImpl::setCursor(const Cursor& cursor)
{
    m_webView->setCursor(cursor);
}

void PageClientImpl::setCursorHiddenUntilMouseMoves(bool)
{
    notImplemented();
}

void PageClientImpl::didChangeViewportProperties(const WebCore::ViewportAttributes&)
{
    notImplemented();
}

void PageClientImpl::registerEditCommand(PassRefPtr<WebEditCommandProxy>, WebPageProxy::UndoOrRedo)
{
    notImplemented();
}

void PageClientImpl::clearAllEditCommands()
{
    notImplemented();
}

bool PageClientImpl::canUndoRedo(WebPageProxy::UndoOrRedo)
{
    notImplemented();
    return false;
}

void PageClientImpl::executeUndoRedo(WebPageProxy::UndoOrRedo)
{
    notImplemented();
}

FloatRect PageClientImpl::convertToDeviceSpace(const FloatRect& viewRect)
{
    notImplemented();
    return viewRect;
}

FloatRect PageClientImpl::convertToUserSpace(const FloatRect& viewRect)
{
    notImplemented();
    return viewRect;
}

IntPoint PageClientImpl::screenToWindow(const IntPoint& point)
{
    notImplemented();
    return point;
}

IntRect PageClientImpl::windowToScreen(const IntRect& rect)
{
    notImplemented();
    return rect;
}

void PageClientImpl::doneWithKeyEvent(const NativeWebKeyboardEvent& event, bool wasEventHandled)
{
    m_webView->doneWithKeyEvent(event, wasEventHandled);
}

#if ENABLE(GESTURE_EVENTS)
void PageClientImpl::doneWithGestureEvent(const WebGestureEvent& event, bool wasEventHandled)
{
    notImplemented();
}
#endif

#if ENABLE(TOUCH_EVENTS)
void PageClientImpl::doneWithTouchEvent(const NativeWebTouchEvent& event, bool wasEventHandled)
{
    notImplemented();
}
#endif

void PageClientImpl::doneWithMouseDownEvent(bool wasEventHandled)
{
    m_webView->doneWithMouseDownEvent(wasEventHandled);
}

void PageClientImpl::doneWithMouseUpEvent(bool wasEventHandled)
{
    m_webView->doneWithMouseUpEvent(wasEventHandled);
}

PassRefPtr<WebPopupMenuProxy> PageClientImpl::createPopupMenuProxy(WebPageProxy* pageProxy)
{
    return WebPopupMenuProxyManx::create(m_webView, pageProxy);
}

PassRefPtr<WebContextMenuProxy> PageClientImpl::createContextMenuProxy(WebPageProxy*)
{
    return WebContextMenuProxyManx::create(m_webView);
}

void PageClientImpl::setFindIndicator(PassRefPtr<FindIndicator>, bool, bool)
{
    notImplemented();
}

#if USE(ACCELERATED_COMPOSITING)
void PageClientImpl::enterAcceleratedCompositingMode(const LayerTreeContext& layerTreeContext)
{
    m_webView->enterAcceleratedCompositingMode(layerTreeContext.canvasHandle());
}

void PageClientImpl::exitAcceleratedCompositingMode()
{
    m_webView->exitAcceleratedCompositingMode();
}

void PageClientImpl::updateAcceleratedCompositingMode(const LayerTreeContext& layerTreeContext)
{
    m_webView->updateAcceleratedCompositingMode(layerTreeContext.canvasHandle());
}
#endif // USE(ACCELERATED_COMPOSITING)

void PageClientImpl::didChangeScrollbarsForMainFrame() const
{
    notImplemented();
}

void PageClientImpl::didCommitLoadForMainFrame(bool)
{
    notImplemented();
}

void PageClientImpl::didFinishLoadingDataForCustomRepresentation(const String&, const CoreIPC::DataReference&)
{
    notImplemented();
}

double PageClientImpl::customRepresentationZoomFactor()
{
    notImplemented();
    return 0;
}

void PageClientImpl::setCustomRepresentationZoomFactor(double)
{
    notImplemented();
}

void PageClientImpl::flashBackingStoreUpdates(const Vector<IntRect>&)
{
    notImplemented();
}

void PageClientImpl::findStringInCustomRepresentation(const String&, FindOptions, unsigned)
{
    notImplemented();
}

void PageClientImpl::countStringMatchesInCustomRepresentation(const String&, FindOptions, unsigned)
{
    notImplemented();
}

void PageClientImpl::handleAuthenticationRequiredRequest(WebFrameProxy* frame, const String& hostname, const String& realm, const String& prefilledUsername, const String& prefilledPassword, PassRefPtr<Messages::WebPageProxy::AuthenticationRequiredRequest::DelayedReply> reply)
{
    m_webView->handleAuthenticationRequiredRequest(frame, hostname, realm, prefilledUsername, prefilledPassword, reply);
}

void PageClientImpl::handleCertificateVerificationRequest(WebFrameProxy* frame, uint32_t error, const String& url, const Vector<CString>& certificates, PassRefPtr<Messages::WebPageProxy::CertificateVerificationRequest::DelayedReply> reply)
{
    m_webView->handleCertificateVerificationRequest(frame, error, url, certificates, reply);
}

void PageClientImpl::runJavaScriptAlert(WebFrameProxy* frame, const String& message, PassRefPtr<Messages::WebPageProxy::RunJavaScriptAlert::DelayedReply> reply)
{
    m_webView->runJavaScriptAlert(frame, message, reply);
}

void PageClientImpl::runJavaScriptComfirm(WebFrameProxy* frame, const String& message,  PassRefPtr<Messages::WebPageProxy::RunJavaScriptConfirm::DelayedReply> reply)
{
    m_webView->runJavaScriptComfirm(frame, message, reply);
}

void PageClientImpl::runJavaScriptPrompt(WebFrameProxy* frame, const String& message, const String& defaultValue, PassRefPtr<Messages::WebPageProxy::RunJavaScriptPrompt::DelayedReply> reply)
{
    m_webView->runJavaScriptPrompt(frame, message, defaultValue, reply);
}

void PageClientImpl::createNewPage(WebPageProxy* page, const WebCore::ResourceRequest& request, const WebCore::WindowFeatures& features, WebEvent::Modifiers modifiers, WebMouseEvent::Button button, PassRefPtr<Messages::WebPageProxy::CreateNewPage::DelayedReply> reply)
{
    m_webView->createNewPage(page, request, features, modifiers, button, reply);
}


} // namespace WebKit
