/*
 * Copyright (C) 2011 Apple Inc. All rights reserved.
 * Copyright (C) 2013 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "LayerTreeContext.h"

#include "ArgumentCoders.h"
#include <wtf/Atomics.h>

#define INVALID_CANVAS_HANDLE 0xFFFFFFFFU

namespace WebKit {

static volatile int s_uniqueID = 0;

LayerTreeContext::LayerTreeContext()
    : canvasHandleID(INVALID_CANVAS_HANDLE)
{
#if USE(COORDINATED_GRAPHICS)
    coordinatedLayerID = 0;
#endif
}

LayerTreeContext::~LayerTreeContext()
{
}

void LayerTreeContext::setCanvasHandle(uint32_t canvasHandle)
{
    canvasHandleID = (uint32_t)WTF::atomicIncrement(&s_uniqueID);
    canvasHandleID <<= 32;
    canvasHandleID |= canvasHandle;
}

void LayerTreeContext::encode(CoreIPC::ArgumentEncoder& encoder) const
{
    encoder << canvasHandleID;
#if USE(COORDINATED_GRAPHICS)
    encoder << coordinatedLayerID;
#endif
}

bool LayerTreeContext::decode(CoreIPC::ArgumentDecoder& decoder, LayerTreeContext& context)
{
    if (!decoder.decode(context.canvasHandleID))
        return false;
#if USE(COORDINATED_GRAPHICS)
    if (!decoder.decode(context.coordinatedLayerID))
        return false;
#endif
    return true;
}

bool LayerTreeContext::isEmpty() const
{
#if USE(COORDINATED_GRAPHICS)
    return !coordinatedLayerID;
#else
    return canvasHandle() == INVALID_CANVAS_HANDLE;
#endif
}

bool operator==(const LayerTreeContext& a, const LayerTreeContext& b)
{
#if USE(COORDINATED_GRAPHICS)
    return a.canvasHandleID == b.canvasHandleID && a.coordinatedLayerID == b.coordinatedLayerID;
#else
    return a.canvasHandleID == b.canvasHandleID;
#endif
}

} // namespace WebKit
