/*
 * Copyright (C) 2010 Apple Inc. All rights reserved.
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WorkQueue.h"

#include <WebCore/NotImplemented.h>
#include <wtf/Threading.h>

// TimerFunction

class TimerFunction : public Manx::WorkQueue::Function {
public:
    TimerFunction(WTF::Function<void()> function) : m_function(function) { }
    virtual ~TimerFunction() { }
    virtual void call() const { m_function(); }
private:
    WTF::Function<void()> m_function;
};

// WorkQueue

void WorkQueue::platformInitialize(const char* name)
{
    m_manxWorkQueue = Manx::WorkQueue::create(name);
    m_manxWorkQueue->startThread();
}

void WorkQueue::platformInvalidate()
{
    m_manxWorkQueue->destroy();
    m_manxWorkQueue = 0;
}

void WorkQueue::dispatch(const Function<void()>& function)
{
    m_manxWorkQueue->dispatchAfterDelay(new TimerFunction(function), 0);
}

void WorkQueue::dispatchAfterDelay(const Function<void()>& function, double delay)
{
    m_manxWorkQueue->dispatchAfterDelay(new TimerFunction(function), delay);
}

void WorkQueue::registerSocketEventHandler(Manx::Pipe::Handle socket, const Function<void()>& function)
{
    m_manxWorkQueue->registerSocketEventHandler(socket, new TimerFunction(function));
}

void WorkQueue::unregisterSocketEventHandler(Manx::Pipe::Handle socket)
{
    m_manxWorkQueue->unregisterSocketEventHandler(socket);
}
