/*
    Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies)
    Copyright (C) 2014 Sony Computer Entertainment Inc.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef GraphicsLayerAcagi_h
#define GraphicsLayerAcagi_h

#if USE(ACCELERATED_COMPOSITING) && USE(ACAGI)

#include "AcagiLayer.h"
#include "AcagiTiledBackingStore.h"
#include "GraphicsLayer.h"
#include "GraphicsLayerClient.h"
#include "Image.h"
#include "PlatformLayerAcagi.h"
#include "Timer.h"

namespace WebCore {

class GraphicsLayerAcagi : public GraphicsLayer, public PlatformLayerAcagi::Client {
public:
    explicit GraphicsLayerAcagi(GraphicsLayerClient*);
    virtual ~GraphicsLayerAcagi();

    void setScrollClient(AcagiLayer::ScrollingClient* client) { m_layer->setScrollClient(client); }
    void setID(uint32_t id) { m_layer->setID(id); }

    // reimps from GraphicsLayer.h
    virtual void setNeedsDisplay();
    virtual void setContentsNeedsDisplay();
    virtual void setNeedsDisplayInRect(const FloatRect&);
    virtual bool setChildren(const Vector<GraphicsLayer*>&);
    virtual void addChild(GraphicsLayer*);
    virtual void addChildAtIndex(GraphicsLayer*, int index);
    virtual void addChildAbove(GraphicsLayer*, GraphicsLayer* sibling);
    virtual void addChildBelow(GraphicsLayer*, GraphicsLayer* sibling);
    virtual bool replaceChild(GraphicsLayer* oldChild, GraphicsLayer* newChild);
    virtual void setMaskLayer(GraphicsLayer*);
    virtual void setPosition(const FloatPoint&);
    virtual void setAnchorPoint(const FloatPoint3D&);
    virtual void setSize(const FloatSize&);
    virtual void setTransform(const TransformationMatrix&);
    virtual void setChildrenTransform(const TransformationMatrix&);
    virtual void setPreserves3D(bool);
    virtual void setMasksToBounds(bool);
    virtual void setDrawsContent(bool);
    virtual void setContentsVisible(bool);
    virtual void setContentsOpaque(bool);
    virtual void setBackfaceVisibility(bool);
    virtual void setOpacity(float);
    virtual void setContentsRect(const IntRect&);
    virtual void setReplicatedByLayer(GraphicsLayer*);
    virtual void setContentsToImage(Image*);
    virtual void setContentsToSolidColor(const Color&);
    Color solidColor() const { return m_solidColor; }
    virtual void setContentsToMedia(PlatformLayer*);
    virtual void setContentsToCanvas(PlatformLayer* canvas) { setContentsToMedia(canvas); }
    virtual void setShowDebugBorder(bool) OVERRIDE;
    virtual void setDebugBorder(const Color&, float width) OVERRIDE;
    virtual void setShowRepaintCounter(bool) OVERRIDE;
    virtual void flushCompositingState(const FloatRect&);
    virtual void flushCompositingStateForThisLayerOnly();
    virtual void setName(const String&);
    virtual PlatformLayer* platformLayer() const { return m_contentsLayer; }

    inline int changeMask() const { return m_changeMask; }

    virtual bool addAnimation(const KeyframeValueList&, const IntSize&, const Animation*, const String&, double);
    virtual void pauseAnimation(const String&, double);
    virtual void removeAnimation(const String&);
    void setAnimations(const GraphicsLayerAnimations&);

    AcagiLayer* layer() const { return m_layer.get(); }

    void didCommitScrollOffset(const IntSize&);
    void setIsScrollable(bool);
    bool isScrollable() const { return m_isScrollable; }

    void setFixedToViewport(bool);
    bool fixedToViewport() const { return m_fixedToViewport; }

    Color debugBorderColor() const { return m_debugBorderColor; }
    float debugBorderWidth() const { return m_debugBorderWidth; }
    void setRepaintCount(int);

    void setActualVisibleRect(const IntRect& actualVisibleRect) { m_actualVisibleRect = actualVisibleRect; }

private:
    virtual void willBeDestroyed();

    void commitLayerChanges();
    void updateDebugBorderAndRepaintCount();
    void updateBackingStoreIfNeeded();
    void prepareBackingStoreIfNeeded();
    bool shouldHaveBackingStore() const;

    virtual void setPlatformLayerNeedsDisplay() OVERRIDE { setContentsNeedsDisplay(); }

    // This set of flags help us defer which properties of the layer have been
    // modified by the compositor, so we can know what to look for in the next flush.
    enum ChangeMask {
        NoChanges =                 0,

        ChildrenChange =            (1L << 1),
        MaskLayerChange =           (1L << 2),
        ReplicaLayerChange =        (1L << 3),

        ContentChange =             (1L << 4),
        ContentsRectChange =        (1L << 5),
        ContentsVisibleChange =     (1L << 6),
        ContentsOpaqueChange =      (1L << 7),

        PositionChange =            (1L << 8),
        AnchorPointChange =         (1L << 9),
        SizeChange =                (1L << 10),
        TransformChange =           (1L << 11),
        ChildrenTransformChange =   (1L << 12),
        Preserves3DChange =         (1L << 13),

        MasksToBoundsChange =       (1L << 14),
        DrawsContentChange =        (1L << 15),
        OpacityChange =             (1L << 16),
        BackfaceVisibilityChange =  (1L << 17),

        BackingStoreChange =        (1L << 18),
        DisplayChange =             (1L << 19),
        ContentsDisplayChange =     (1L << 20),
        BackgroundColorChange =     (1L << 21),

        AnimationChange =           (1L << 22),
        FilterChange =              (1L << 23),

        DebugVisualsChange =        (1L << 24),
        RepaintCountChange =        (1L << 25),

        FixedToViewporChange =      (1L << 26),
        AnimationStarted =          (1L << 27),

        CommittedScrollOffsetChange =     (1L << 28),
        IsScrollableChange =              (1L << 29)
    };
    void notifyChange(ChangeMask);

    OwnPtr<AcagiLayer> m_layer;
    RefPtr<AcagiTiledBackingStore> m_compositedImage;
    NativeImagePtr m_compositedNativeImagePtr;
    RefPtr<AcagiBackingStore> m_backingStore;

    int m_changeMask;
    bool m_needsDisplay;
    bool m_hasOwnBackingStore;
    bool m_fixedToViewport;
    Color m_solidColor;

    Color m_debugBorderColor;
    float m_debugBorderWidth;

    PlatformLayerAcagi* m_contentsLayer;
    FloatRect m_needsDisplayRect;
    FloatRect m_actualVisibleRect;
    GraphicsLayerAnimations m_animations;
    double m_animationStartTime;

    IntSize m_committedScrollOffset;
    bool m_isScrollable;
};

inline static GraphicsLayerAcagi* toGraphicsLayerAcagi(GraphicsLayer* layer)
{
    return static_cast<GraphicsLayerAcagi*>(layer);
}

AcagiLayer* toAcagiLayer(GraphicsLayer*);

}
#endif // USE(ACCELERATED_COMPOSITING) && USE(ACAGI)

#endif // GraphicsLayerAcagi_h
