/*
    Copyright (C) 2012 Nokia Corporation and/or its subsidiary(-ies)
    Copyright (C) 2012, 2013 Company 100, Inc.
    Copyright (C) 2012, 2013 basysKom GmbH

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "config.h"

#if USE(ACCELERATED_COMPOSITING) && USE(TEXTURE_MAPPER)

#include "TextureMapperFPSCounter.h"

#include "TextureMapper.h"
#include <wtf/CurrentTime.h>

#if OS(ORBIS)
#include "OpenGLManx.h"
#endif

namespace WebCore {

TextureMapperFPSCounter::TextureMapperFPSCounter()
    : m_isShowingFPS(false)
    , m_fpsInterval(0)
    , m_fpsTimestamp(0)
    , m_lastFPS(0)
    , m_frameCount(0)
#if PLATFORM(MANX)
    , m_lastSyncRate(0)
    , m_syncCount(0)
#endif
{
    String showFPSEnvironment = getenv("WEBKIT_SHOW_FPS");
    bool ok = false;
    m_fpsInterval = showFPSEnvironment.toDouble(&ok);
    if (ok && m_fpsInterval) {
        m_isShowingFPS = true;
        m_fpsTimestamp = WTF::currentTime();
    }
}

#if PLATFORM(MANX)
static void drawBar(TextureMapper* textureMapper, float value, const Color& color, const FloatRect& barRect, const TransformationMatrix& matrix)
{
    FloatRect rect = barRect;
    textureMapper->drawBorder(color, 1, rect, matrix);

    rect.setWidth(rect.width() * value);
    textureMapper->drawSolidColor(rect, matrix, color);
}
#endif

void TextureMapperFPSCounter::updateFPSAndDisplay(TextureMapper* textureMapper, const FloatPoint& location, const TransformationMatrix& matrix)
{
    if (!m_isShowingFPS)
        return;

    m_frameCount++;
    double delta = WTF::currentTime() - m_fpsTimestamp;
    if (delta >= m_fpsInterval) {
#if PLATFORM(MANX)
        m_lastFPS = int(m_frameCount / delta + 0.5);
        m_lastSyncRate = int(m_syncCount / delta + 0.5);
        m_syncCount = 0;
#else
        m_lastFPS = int(m_frameCount / delta);
#endif
        m_frameCount = 0;
        m_fpsTimestamp += delta;
    }

#if PLATFORM(MANX)
    const float BarWidth = 100;
    const float BarHeight = 8;
    const float Margin = 4;
    const int NumBars = 3;
    textureMapper->drawSolidColor(FloatRect(location, FloatSize(BarWidth + Margin * 2, BarHeight * NumBars + Margin * (NumBars + 1))), matrix, Color(0.f, 0.f, 0.f, 0.8f));

    float x = Margin;
    float y = Margin;
    FloatRect barRect;

    barRect.setLocation(location);
    barRect.moveBy(FloatPoint(x, y));
    barRect.setSize(FloatSize(BarWidth, BarHeight));
    drawBar(textureMapper, m_lastFPS / 60.0, Color(0.f, 1.f, 0.f, .8f), barRect, matrix);
    y += Margin + BarHeight;

    barRect.setLocation(location);
    barRect.moveBy(FloatPoint(x, y));
    barRect.setSize(FloatSize(BarWidth, BarHeight));
    drawBar(textureMapper, m_lastSyncRate / 60.0, Color(0.f, 1.f, 0.f, .8f), barRect, matrix);
    y += Margin + BarHeight;
#else
    textureMapper->drawNumber(m_lastFPS, Color::black, location, matrix);
#endif

#if OS(ORBIS) && EGL_SCE_piglet_memory_info
    EGLMEMORYINFO_t memoryInfo = { };
    memoryInfo.uSize = sizeof(memoryInfo);
    eglMemoryInfo(&memoryInfo);
    if (memoryInfo.uMaxAllowed > 0) {
#if 0
        if  (delta >= m_fpsInterval) {
#define PRINT_MEMINFO(item) printf("%16s %d\n", #item, memoryInfo.item)
            printf("--------------------------------\n");
            PRINT_MEMINFO(uTexture);
            PRINT_MEMINFO(uSurfaces);
            PRINT_MEMINFO(uPrograms);
            PRINT_MEMINFO(uBuffers);
            PRINT_MEMINFO(uCommandBuffers);
            PRINT_MEMINFO(uTotal);
            PRINT_MEMINFO(uMaxAllowed);
        }
#endif

        barRect.setLocation(location);
        barRect.moveBy(FloatPoint(x, y));
        barRect.setSize(FloatSize(BarWidth, BarHeight));
        drawBar(textureMapper, (float)memoryInfo.uTotal / memoryInfo.uMaxAllowed, Color(0.f, 1.f, 0.f, .8f), barRect, matrix);
        textureMapper->drawNumber(memoryInfo.uTotal / 1024 / 1024, Color::black, barRect.location() + FloatPoint(BarWidth + 2, -9), matrix);
    }
#endif
}

} // namespace WebCore

#endif // USE(ACCELERATED_COMPOSITING)
