/*
 * Copyright (C) 2008 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebViewJavaScriptObject.h"

#include "WebViewQueue.h"
#include <stdio.h>
#include <wtf/Vector.h>
#include <wtf/text/CString.h>

JavaScriptArg::JavaScriptArg()
    : type(type)
    , argBool(false)
    , argDouble(0.f)
{
    argString = 0;
}

JavaScriptArg::~JavaScriptArg()
{
}

WebViewJavaScriptObjectDelegate::WebViewJavaScriptObjectDelegate(const char* name, int argCount)
    : argCount(argCount)
{
    funcName = new char[strlen(name) + 1];
    strncpy(funcName, name, strlen(name) + 1);
    args = new JavaScriptArg[argCount];
}

WebViewJavaScriptObjectDelegate::WebViewJavaScriptObjectDelegate(WebViewJavaScriptObjectDelegate* base)
{
    funcName = new char[strlen(base->funcName) + 1];
    strncpy(funcName, base->funcName, strlen(base->funcName) + 1);

    argCount = base->argCount;

    args = new JavaScriptArg[argCount];
    for (int i = 0; i < argCount; i++) {
        switch (base->args[i].type) {
        case JSTYPE_BOOLEAN:
            args[i].type = JSTYPE_BOOLEAN;
            args[i].argBool = base->args[i].argBool;
            break;
        case JSTYPE_NUMBER:
            args[i].type = JSTYPE_NUMBER;
            args[i].argDouble = base->args[i].argDouble;
            break;
        case JSTYPE_STRING:
            args[i].type = JSTYPE_STRING;
            args[i].argString = new char[strlen(base->args[i].argString) + 1];
            strncpy(args[i].argString, base->args[i].argString, strlen(base->args[i].argString) + 1);
            break;
        }
    }
}

WebViewJavaScriptObjectDelegate::~WebViewJavaScriptObjectDelegate()
{
    delete[] funcName;
    delete[] args;
}

WebViewJavaScriptObject::WebViewJavaScriptObject(const char* name, WebViewJavaScriptObjectType type)
    : m_type(type)
    , m_delegate(true)
{
    m_baseUrl = new WTF::CString(name);
    m_childObjects = new WebViewQueue();
}

WebViewJavaScriptObject::WebViewJavaScriptObject(WebViewJavaScriptObject* base)
{
    m_baseUrl = new WTF::CString(base->baseUrl());
    m_type = base->objectType();
    m_delegate = base->delegate();
    m_childObjects = new WebViewQueue();

    WebViewJavaScriptObject* child = 0;
    int index = 0;
    while (child = base->childObjectAtIndex(index)) {
        addChildObject(child);
        index++;
    }
}

WebViewJavaScriptObject::~WebViewJavaScriptObject()
{
    WebViewQueueItem* child = 0;
    while ((child = m_childObjects->pop())) {
        delete (WebViewJavaScriptObject*)child->getId();
        delete child;
    }
    delete m_childObjects;
    delete m_baseUrl;
}

const char* WebViewJavaScriptObject::baseUrl()
{
    return m_baseUrl->data();
}

WebViewJavaScriptObjectType WebViewJavaScriptObject::objectType()
{
    return m_type;
}

WebViewJavaScriptObject* WebViewJavaScriptObject::childObjectAtIndex(int index)
{
    WebViewQueueItem* ptr = m_childObjects->head();

    for (int i = 0; i < index; i++)
        ptr = ptr->next();

    WebViewJavaScriptObject* jsObj = 0;
    if (ptr)
        jsObj = (WebViewJavaScriptObject*)ptr->getId();
    return jsObj;
}

void WebViewJavaScriptObject::addChildObject(const WebViewJavaScriptObject* object)
{
    WebViewQueueItem* item = new WebViewQueueItem((uintptr_t)object);
    m_childObjects->push(item);
}

void WebViewJavaScriptObject::deleteChildObject(const char* name)
{

}

void WebViewJavaScriptObject::setdelegate(bool delegate)
{
    m_delegate = delegate;
}

bool WebViewJavaScriptObject::delegate()
{
    return m_delegate;
}
