///////////////////////////////////////////////////////////////////////////////
// Copyright 2004,2005,2006 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef __CEUHASHTABLE_T_H__
#define __CEUHASHTABLE_T_H__

#include "ceutypes.h"
#include "ceudebugsupport.h"
#include "CECom.h"
//
// Hashtable using Closed-hashing 
// ------------------------------
//
//   Basic Idea
//   ----------
//     If the hash bucket collision is occurred on insertion,
//     this will try the 'next' bucket, instead of making list.
//
//            :                                :
//         +-----+    +----------+          +-----+    +------------+
//      n  |   ------>| already  |       n  |  ------->| already... |
//         +-----+    | existing |          +-----+    +------------+
//      n+1|     |    +----------+  ==>  n+1|  ------+
//         +-----+                          +-----+  |  +--------------+
//      n+2|     |                       n+2|     |  +->|newly inserted|
//         +-----+                          +-----+     +--------------+
//            :                                :
//
//    There are many algorithms to solve the hash collision.
//    This template uses the most simple way, just using the next bucket.
//
//
//   Tombstone
//   ---------
//     To find the element correctly, the bucket that is previously used 
//     and currently not used, has the special value 'tombstone'.
//
//     When there are 3 elements that has the same hash value,
//     and you need to delete the [n+1] one, the tombstone is required
//     to find [n+2].
//
//            :                                :
//         +-----+                          +-----+
//      n  |   ----->elem[n]             n  |   ----->elem[n] 
//         +-----+                          +-----+
//      n+1|   ----->elem[n+1] ====>     n+1|xxxxx|   
//         +-----+                          +-----+
//      n+2|   ----->elem[n+2]           n+2|   ----->elem[n+2]    
//         +-----+                          +-----+
//            :                                :
//
//    But if you insert and delete elements frequently, buckets will be
//    covered with tombstones, and the performance will be close to the
//    linear search.
//
//    The solutions for this problem are:
//      a) periodic rehashing
//      b) try to remove tombstone on deleting element just before the empty bucket
//
//

//  Assumed TOps Class
//  ------------------
//
// class Ops
// {
// public:
// 	   // TKey operations
// 	   static bool isEqual(const TKey& key1, const TKey& key2);
//     static bool isEqual(const TElem* elem, const TKey& key);  // back compat mode
// 	   static const TKey& getKey(const TElem* elem);
// 	   static UINT_PTR getHash(const TKey& key);
//
// 	   // notify for TElem
// 	   static void onInserted(TElem* elem);
// 	   static void onRemoved(TElem* elem);
// };

template <class TElem, class TKey, class TOps>
class CEUHashtableBaseT 
{
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	CEUHashtableBaseT() : _buckets(0), _bucketsSize(0), _numOfEntries(0), _limitNumOfEntries(0), _pAllocator(0) {}

	~CEUHashtableBaseT()
	{
		shutdown();
	}

	bool isInitialized()const { return (_buckets != 0); }
	
	CEHResult init(CEComAllocatorRec* pAllocator, UINT32 initialBucketSize)
	{
		CEHResult hr = CE_SILK_ERR_MEMERR;

		// calculate the actual bucket size
		//   the actual bucket size should be:
		//     - larger than one, because one empty entry is used as a stopper
		//     - power of 2, because bitwise-and is used, instead of modulo
		UINT32 size;
		for (size = 2 ; size < initialBucketSize ; size <<= 1);

		_pAllocator = pAllocator;
		if ( _pAllocator  && _pAllocator->malloc )
		{
			// allocate hash bucket
			_buckets = reinterpret_cast<TElem**>(_pAllocator->malloc(_pAllocator, sizeof(TElem*) * size));
			if (_buckets)
			{
				// clear buckets to empty
				CESysFillMemory(_buckets, 0, sizeof(TElem*)*size);

				// update the bucket size
				_bucketsSize = size;
				_limitNumOfEntries = _bucketsSize - (_bucketsSize >> 2);
				hr = CE_S_OK;
			}
		}

		return hr;
	}

	void shutdown()
	{
		if (_buckets)
		{
			// TODO: should we call TElem's destructor for each bucket?
			_numOfEntries = 0;

			// release buckets
			_deleteBuckets();
			_buckets = 0;
			_bucketsSize = 0;
			_limitNumOfEntries = 0;
		}
	}

	bool find(const TKey& key, TElem*& elemOut)
	{
		UINT_PTR index=0;
		bool isFound = _find(key, index);
		if ( isFound )
		{
			elemOut = _buckets[index];
		}
		return isFound;
	}

	CEHResult insert(TElem* elem)
	{
		CEHResult hr = _insert(elem);
		if (CESucceeded(hr))
		{
			// notify to the element after actual insert
			_onInserted(elem);
		}
		return hr;
	}

	void remove(TElem* elem)
	{
		UINT_PTR index;
		if (_find(_getKey(elem), index))
		{
			CEASSERT(elem == _buckets[index]); 

			// notify to the element before actual remove
			_onRemoved(elem);

			// remove the element actually
			_buckets[index] = reinterpret_cast<TElem*>(0x1);  // tombstone
			CEASSERT(_numOfEntries);
			_numOfEntries--;
		}
		else
		{
			//CEASSERT(0 && "no item removed.");
		}
	}

	UINT32 size () { return _numOfEntries; }
	
	class Iterator
	{
	public:
		Iterator(CEUHashtableBaseT& hashtable) : _idx(0), _hashtable(hashtable) {}

		CEHResult reset()
		{
			_idx = 0;
			return CE_S_OK;
		}

		TElem* nextElement()
		{
			TElem* ret = 0;
			for ( ; _idx < _hashtable._bucketsSize ; _idx++ )
			{
				if ( _hashtable._isUsed(_idx) )
				{
					ret = _hashtable._buckets[_idx];
					break;
				}
			}
			_idx++;
			return ret;
		}

	private:
		UINT_PTR _idx;
		CEUHashtableBaseT& _hashtable;
	};

protected:
	friend class CEUHashtableBaseT::Iterator;

	void _deleteBuckets() 
	{
		if ( _pAllocator && _pAllocator->free )
		{
			_pAllocator->free(_pAllocator, _buckets);
		}
	}

	void _init(CEUHashtableBaseT& other)
	{
		if ( _buckets ) 
		{ 
			_deleteBuckets();
		}
		_buckets = other._buckets;
		_bucketsSize = other._bucketsSize;
		_numOfEntries = other._numOfEntries;
		_limitNumOfEntries = other._limitNumOfEntries;

		other._buckets = 0;
	}

	CEHResult _rehash()
	{
		CEUHashtableBaseT newTable;
		CEHResult hr = newTable.init(_pAllocator, _bucketsSize * 2);
		UINT_PTR i=0;
		for ( i=0; i<_bucketsSize && CESucceeded(hr) ; i++ )
		{
			if ( _isUsed(i) )
			{
				hr = newTable.insert(_buckets[i]);
			}
		}
		if (CESucceeded(hr))
		{
			_init(newTable);
		}
		return hr;
	}

	bool _find(const TKey& key, UINT_PTR& indexOut)
	{
		bool isFound = false;

		// find out the NEVER used (empty only) bucket
		UINT_PTR mask = _bucketsSize - 1;
		UINT_PTR curIdx = _getHash(key) & mask;
		UINT_PTR end = curIdx ? curIdx-1 : mask;

		while (_bucketsSize > curIdx)
		{			
			if (_isUsed(curIdx) && _isEqual(_buckets[curIdx], key))
			{
				isFound = true;
				indexOut = curIdx;
				break;
			}

			if ( !_hasBeenUsed(curIdx) ) 
			{
				break; 
			}

			// go to the next slot
			curIdx = (curIdx+1) & mask;
			if ( curIdx == end ) 
			{
				break;
			}
		}		
		return isFound;
	}

	CEHResult _insert(TElem* elem)
	{
		CEHResult hr = CE_S_OK;
		
		// find out the CURRENTLY not used (empty or tombstone) bucket
		UINT_PTR mask = _bucketsSize - 1;
		UINT_PTR curIdx = _getHash(_getKey(elem)) & mask;
		UINT_PTR end = curIdx ? curIdx-1 : mask;
		while (CESucceeded(hr) && _isUsed(curIdx))
		{
			// check the duplicated key
			if (_isEqual(_buckets[curIdx], _getKey(elem)))
			{
				hr = CE_SILK_ERR_DUPLICATE;
				break;
			}

			// go to the next slot
			// curIdx+1 or 0.
			curIdx = (curIdx+1) & mask;
			if ( curIdx == end ) 
			{
				hr = CE_SILK_ERR_BADARRAYINDEX;
			}
		}

		// set the element to the found bucket
		if (CESucceeded(hr))
		{
			_buckets[curIdx] = elem;
			_numOfEntries++;
		}
		
		if ( hr == CE_SILK_ERR_BADARRAYINDEX || _numOfEntries > _limitNumOfEntries )
		{
			CEHResult rehashErr = _rehash();
			if ( !rehashErr )
			{
				if (  hr == CE_SILK_ERR_BADARRAYINDEX )
				{
					hr = _insert(elem);
				}
			}
			else { hr = rehashErr; }
		}
		return hr;
	}

	///////////////////////////////////
	//  bucket accessor
	///////////////////////////////////

	bool _isUsed(UINT_PTR idx)
	{
		// _bucket[x] == 0 :  empty (not used)
		// _bucket[x] == 1 :  tombstone (previously used and currently not used)
		// others :           currenty used

		// the following mask should be:
		//   0xfffffffe			(for 32bit environment)
		//   0xfffffffffffffffe (for 64bit environment)
		UINT_PTR mask = ~(UINT_PTR_CONST(0x1));
		return (reinterpret_cast<UINT_PTR>(_buckets[idx]) & mask);
	}
	
	///////////////////////////////////
	//  TElem/TKey helper methods
	///////////////////////////////////

	bool _hasBeenUsed(UINT_PTR idx)
	{
		// _bucket[x] == 0 :  empty (not used)
		return (_buckets[idx] != 0);
	}

#if 0
	static bool _isEqual(const TKey& key1, const TKey& key2)
	{
		return TOps::isEqual(key1, key2);
	}
#else // back compat
	static bool _isEqual(const TElem* elem, const TKey& key)
	{
		return TOps::isEqual(elem, key);
	}
#endif

	static const TKey& _getKey(const TElem* elem)
	{
		return TOps::getKey(elem);
	}

	static UINT_PTR _getHash(const TKey& key)
	{
		return TOps::getHash(key);
	}

	static void _onInserted(TElem* elem)
	{
		TOps::onInserted(elem);
	}

	static void _onRemoved(TElem* elem)
	{
		TOps::onRemoved(elem);
	}

private:
	TElem** _buckets;
	UINT32  _bucketsSize;
	UINT32  _numOfEntries;
	UINT32	_limitNumOfEntries;
	CEComAllocatorRec* _pAllocator;
};

///////////////////////////////////////////////////////////////////////
//  Set of unique pointers
///////////////////////////////////////////////////////////////////////

template <class T>
class CEUniqPtrSetOpsT
{
public:
	// TKey operations
	static bool isEqual(const T* elem, const T& key) { return (elem == &key); }
	static const T& getKey(const T* elem) { return *elem; }
	static UINT_PTR getHash(const T& key) { return ((UINT_PTR)(&key) >> 2); }

	// notify for TElem
	static void onInserted(T* elem) {}
	static void onRemoved(T* elem) {}
};

template <class T>
class CEUniqPtrSetT : public CEUHashtableBaseT<T, T, CEUniqPtrSetOpsT<T> >
{
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;
};

///////////////////////////////////////////////////////////////////////
//  Set of unique interface pointers
///////////////////////////////////////////////////////////////////////

template <class T>
class CEUniqIntfPtrSetOpsT : public CEUniqPtrSetOpsT<T>
{
public:
	// notify for TElem
	static void onInserted(T* elem) { elem->_vtbl->_addRef(elem); }
	static void onRemoved(T* elem) { elem->_vtbl->_release(elem); }
};

template <class T>
class CEUniqIntfPtrSetT : public CEUHashtableBaseT<T, T, CEUniqIntfPtrSetOpsT<T> >
{
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;
};


///////////////////////////////////////////////////////////////////////
//  Backward Compatibility Wrapper
///////////////////////////////////////////////////////////////////////

//  Assumed Element Class
//  ---------------------
//
//    class TKey 
//	  {
//		  UINT_PTR getHash() const;
//    };
//    class TElem
//    {
//      public:
//	      const TKey& getKey();
//	      bool isEqual(const TKey& key);
//    };
//

template <class TElem, class TKey>
class CEUBackCompatOpsT
{
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	// TKey operations
	static bool isEqual(const TElem* elem, const TKey& key)
	{
		return elem->isEqual(key);
	}

	static const TKey& getKey(const TElem* elem)
	{
		return elem->getKey();
	}

	static UINT_PTR getHash(const TKey& key)
	{
		return key.getHash();
	}

	// notify for TElem
	static void onInserted(TElem* elem) {}
	static void onRemoved(TElem* elem) {}
};

template <class TElem, class TKey>
class CEUHashtableT : public CEUHashtableBaseT<TElem, TKey, CEUBackCompatOpsT<TElem, TKey> >
{
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

private:
};


#endif // #ifndef __CEUHASHTABLE_T_H__

