///////////////////////////////////////////////////////////////////////////////
// Copyright 2011 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICEUString_h
#define ICEUString_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CECom.h"
#include "CEApiUnknown.h"
#include "ICETextConf.h"
#include "ICEI18nTypes.h"
#include "ceutypes.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------

struct ICEI18nCharsetEncoding;

//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

//! declarations of special state numbers
typedef enum
{
	eCEUTokenizerState_Initial = 0,		//!< initial state
	eCEUTokenizerState_Error = 0xff,	//!< syntax error has occurred
} eCEUTokenizerState;

//! declarations of ICEUString find flags.
typedef enum
{
	eCEUStrFindFlg_None = 0,
	eCEUStrFindFlg_IgnoreCaseSearch = 1<<0,  //! Case insensitive search
	eCEUStrFindFlg_LiteralSearch    = 1<<1,  //! Disable unicode composition character matching.
	eCEUStrFindFlg_BackwardSearch   = 1<<2,  //! Search pattern from end of string.
	eCEUStrFindFlg_AnchoredSearch   = 1<<3,  //! Only search pattern at beggining or end of string.

	eCEUStrFindFlg_ConcatPattern    = 1<<4,  //! Concatinate pattern (default: pattern is concatinated to the leading part).
	eCEUStrFindFlg_ConcatPatternToTrailing  = 1<<5   //! Concatinate pattern to trailing.
} eCEUStrFindFlg;

//! declarations of ICEUString trim flags.
typedef enum
{
	eCEUStrTrimFlg_Left = 1<<0,   
	eCEUStrTrimFlg_Right = 1<<1,
	eCEUStrTrimFlg_Both = (eCEUStrTrimFlg_Right|eCEUStrTrimFlg_Left)
} eCEUStrTrimFlg;

//! declarations of action flags on the state transition
typedef enum
{
	eCEUTokenizerActionFlag_None	   = 0,
	eCEUTokenizerActionFlag_StartToken = (1UL << 0),	//!< start gathering chars for token
	eCEUTokenizerActionFlag_EndToken   = (1UL << 1),	//!< end gathering chars for token
	eCEUTokenizerActionFlag_Notify	   = (1UL << 2)		//!< call "notifyProc" callback function
} eCEUTokenizerActionFlag;

//! declarations of predefined char kinds
typedef enum
{
	eCEUCharKind_Generic_Illegal = 0,
	eCEUCharKind_Generic_EOF,
	eCEUCharKind_Generic_NonAscii,
	eCEUCharKind_Generic_Whitespace,
	eCEUCharKind_Generic_UpperAlpha,
	eCEUCharKind_Generic_LowerAlpha,
	eCEUCharKind_Generic_Digit,
	eCEUCharKind_Generic_LAST = eCEUCharKind_Generic_Digit		
} eCEUCharKind_Generic;

typedef enum
{
	eCEUStringRadix_MIN = 2,
	eCEUStringRadix_MAX = 36
} eCEUStringRadix;

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//! state transition rule
typedef struct CEUTokenizerRule
{
	UINT32	charKindMask;	//!< char kind mask to fire this rule
	UINT8	next;			//!< next state
	UINT8	actionFlags;	//!< action to be executed on fire
	UINT8	tokenType;		//!< arbitrary data to be passed to "tokenProc" callback function
	UINT8	notifyFlags;	//!< arbitrary data to be passed to "notifyProc" callback function
} CEUTokenizerRule;

typedef struct CEUTokenizerState
{
	CEUTokenizerRule*	rules;
	UINT32				numOfRules;
} CEUTokenizerState;

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEUString ICEUString
 * @{
 */

/*!
 * ID of ICEUString
 */
#define CEComIID_ICEUString 0x99c193e2


/*!
* ICEUString
*/
typedef struct ICEUString
{
	const struct ICEUString_vtbl* _vtbl;
} ICEUString;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUString	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUString_queryInterface) (ICEUString* iUString, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUString_addRef) (ICEUString* iUString);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUString_release) (ICEUString* iUString);


	/*!
	* This method change this string to immutable.
	*/

typedef CEHResult (*iCEUString_toImmutable) (ICEUString* iUString);


	/*!
	* Returns true if this string is immutable.
	*/

typedef CEHResult (*iCEUString_isImmutable) (ICEUString* iUString, bool *const bImmutableOut);

typedef CEHResult (*iCEUString_getHashcode) (ICEUString* iUString, UINT32 *const hashcodeOut);


	/*!
	* This method print the string.
	* On the linux platform, this method output to stderr.
	* On the Windows platform, this method output to debug output. (with OutputDebugStr()).
	*/

typedef CEHResult (*iCEUString_debugPrint) (ICEUString* iUString);


	/*!
	* This method print the string and line separator string.
	* On the linux platform, this method output to stderr.
	* On the Windows platform, this method output to debug output. (with OutputDebugStr()).
	*/

typedef CEHResult (*iCEUString_debugPrintln) (ICEUString* iUString);


	/*!
	* This method returns null terminated UTF16CHAR array object representing
	* this string's value for debugging use only. This method returns empty 
	* UTF16CHAR array on a release build. It you need to get the string data bytes,
	* use getBytesWithAlloc(). And this method works in the watch window in the VS.NET debugger.
	*
	* \param[in]	iUString	Specifies interface pointer
	*
	* \return  null terminated UTF16CHAR array object representing this string's value. Do not free this return value.
	* \see getBytesWithAlloc().
	*/
typedef CEHResult (*iCEUString_dbgToString) (ICEUString* iUString, const UTF16CHAR* *const pStr16Out);


	/*!
	* getNumOfChars32() returns a number of characters that this string holds.  
	*  \param[in] iUString   Specifies interface pointer
	*  \param[out] numOfCharsOut   Specifies the pointer to the variable to get the number of characters that this string has. 
	*  \return error code 
	* This method returns a number of UTF32 characaters that this string holds, 
	* So, in the case of UTF16CHAR array, if the array contain the surrogate pairs,  
	* the returned number-of-utf32chars is smaller than actual string datas array length.  
	*/
typedef CEHResult (*iCEUString_getNumOfChars32) (ICEUString* iUString, UINT32* pNumOfCharsOut);


	/*!
	* initialize string. This method allocates internal string object and 
	* allocates buffer to copy \a pAnyCharStr and convert bytes from \a srcEncoding to UTF16.
	*
	* \param[in] iUString  Specifies interface pointer
	* \param[in] pAnyCharStr  null terminated source byte array.
	* \param[in] srcEncoding. see eICEI18nEncoding.
	* \retrurn error code.
	*
	* \note const eICEI18nEncoding srcEncoding=eICEI18nEncoding_ascii, const UINT32 anyCharStrSizeInBytes=UINT32_UNKNOWN_SIZE
	*/
typedef CEHResult (*iCEUString_initWithByteArray) (ICEUString* iUString, const UCHAR8* pAnyCharStr, const eICEI18nEncoding srcEncoding, const UINT32 anyCharStrSizeInBytes);

typedef CEHResult (*iCEUString_initWithByteArrayEx) (ICEUString* iUString, const UCHAR8* pAnyCharStr, const struct ICEI18nCharsetEncoding* srcEncoding, const UINT32 anyCharStrSizeInBytes);


	/*!
	* initialize string. This method allocates internal string object and allocates buffer to copy \a str.
	*
	* \param[in] iUString  Specifies interface pointer
	* \param[in] str  source string.
	* \retrurn error code.
	*/
typedef CEHResult (*iCEUString_initWithString) (ICEUString* iUString, const struct ICEUString* pStr);


	/*!
	* initialize string. This method allocates internal string object and allocates buffer to copy \a pUtf16Str.
	* 
	* \param[in] iUStringSpecifies interface pointer
	* \param[in] pUtf16Str.
	* \param[in] size of \a pUtf16Str in bytes.
	* \retrurn error code.
	*/
typedef CEHResult (*iCEUString_initWithUTF16Array) (ICEUString* iUString, const UTF16CHAR* pUtf16Str, const UINT32 sizeInBytes);


	/*!
	* initialize string. This method allocates internal string object and allocates buffer to copy \a c32.
	* 
	* \param[in] iUString Specifies interface pointer
	* \param[in] c32 (UTF32CHAR).
	* \retrurn error code.
	*/
typedef CEHResult (*iCEUString_initWithUTF32CHAR) (ICEUString* iUString, const UTF32CHAR c32);


	/*!
	* initialize string. This method allocates allocates internal string object and hold a reference to the 
	* pCStringLiteral. Specify global or static variable UCHAR8 pointer. Otherwise, this string  will 
	* be initialized as unstable(may crash later).
	* 
	* \param[in] iUStringSpecifies interface pointer
	* \param[in] pCStringLiteral.
	* \retrurn error code.
	*/
typedef CEHResult (*iCEUString_initWithCStringLiteral) (ICEUString* iUString, const CHAR8* pCStringLiteral);


	/*!
	* This method returns whether this string is empty or not.
	*/

typedef CEHResult (*iCEUString_isEmpty) (ICEUString* iUString, bool *const pIsEmptyOut);


	/*!
	* This method makes this string empty
	*/

typedef CEHResult (*iCEUString_clear) (ICEUString* iUString);


	/*!
	* This method appends the specified CEUString in the argument to this string.
	* 
	* \param[in] iUString  Specifies interface pointer
	* \param[in] str  Specifies the string to be appended to this string.
	* \return error code
	*/
typedef CEHResult (*iCEUString_append) (ICEUString* iUString, const struct ICEUString* pStrToAppend);


	/*!
	* This method appends the specified C string literal in the argument to this string.
	* 
	* \param[in] iUString  Specifies interface pointer
	* \param[in] pCStringLiteral  
	* \return error code
	*/
typedef CEHResult (*iCEUString_appendCStringLiteral) (ICEUString* iUString, const CHAR8* pCStringLiteral);


	/*!
	* This method trims leading and trailing whitespaces in this string.
	* 
	* \param [in] flg: Combination of eCEUStrTrimFlg. 
	* \param [in] numLeft: Number of characters to trim in left part. Default value is UINT32_UNKNOWN_SIZE. All white space in left part.
	* \param [in] numRight: Number of characters to trim in right part. Default value is UINT32_UNKNOWN_SIZE. All white space in right part.
	* <pre>
	* ex..
	* CEUString str;
	* str.initWithCStringLiteral("  ab  ");
	* // str.trim();  ==> "ab" 
	* // str.trim(eCEUStrigTrimFlg_Both, 0, 1);  ==> "  ab " : One white space sharacter at right part is removed.
	* // str.trim(eCEUStrigTrimFlg_Left, 10, 1);  ==> "ab  " : All (two) white space characters at left part is removed.
	* </pre>
	* \param[in]	iUString	Specifies interface pointer
	*
	* \return error code
	* \see eCEUStrTrimFlg
	* 
	* NOTE: All characters that have codes less than or equal to 0x20 (the space character) are considered to be white space. 
	* eCEUStrTrimFlg flg = eCEUStrTrimFlg_Both, UINT32 numLeft=UINT32_UNKNOWN_SIZE, UINT32 numRight=UINT32_UNKNOWN_SIZE
	*/
typedef CEHResult (*iCEUString_trimWithOptions) (ICEUString* iUString, eCEUStrTrimFlg flg, UINT32 numLeft, UINT32 numRight);

typedef CEHResult (*iCEUString_trim) (ICEUString* iUString);


	/*!
	* This method replace all 'from' character to 'to' character in this string.
	* 
	* \param[in] iUString  Specifies interface pointer
	* \param[in] from  Specifies the character to be replaced from.
	* \param[in] to  Specifies the character to be replaced to.
	* \return error code
	*/
typedef CEHResult (*iCEUString_replace) (ICEUString* iUString, const UCHAR8 pFrom, const UCHAR8 pTo);


	/*!
	* This method makes a comparison at the beginning of the string, 
	* determines whether it matches this current instance, and returns a 
	* bool representation of their relationship. 
	* 
	* \param[in] iUString Specifies interface pointer
	* \param[in] str 
	* \param[out] b  result.
	* \param[in] ignoreCase: false: case sensitive comparison. 
	*
	* \return error code
	*
	* \note const bool ignoreCase=false
	*/
typedef CEHResult (*iCEUString_startsWith) (ICEUString* iUString, const struct ICEUString* pStr, const bool ignoreCase, bool *const pBRet);


	/*!
	* This method makes a comparison at the beginning of the string, 
	* determines whether it matches this current instance, and returns a 
	* bool representation of their relationship. 
	* 
	* \param[in] iUString Specifies interface pointer
	* \param[in] pCStrintLiteral 
	* \param[out] b  result.
	* \param[in] ignoreCase: false: case sensitive comparison. 
	* \return error code
	*
	* \note const bool ignoreCase=false
	*/
typedef CEHResult (*iCEUString_startsWithCStringLiteral) (ICEUString* iUString, const CHAR8* pCStringLiteral, const bool ignoreCase, bool *const pBRet);


	/*!
	* This method makes a comparison at the end of the string, 
	* determines whether it matches this current instance, and returns a 
	* bool representation of their relationship. 
	* 
	* \param[in] iUString Specifies interface pointer
	* \param[in] str 
	* \param[out] b  result.
	* \param[in] ignoreCase: false: case sensitive comparison. 
	* \return error code
	*
	* \note const bool ignoreCase=false
	*/
typedef CEHResult (*iCEUString_endsWith) (ICEUString* iUString, const struct ICEUString* pStr, const bool ignoreCase, bool *const pBRet);

typedef CEHResult (*iCEUString_endsWithCStringLiteral) (ICEUString* iUString, const CHAR8* pCStringLiteral, const bool ignoreCase, bool *const pBRet);


	/*!
	* This method makes a comparison of the string, 
	* determines whether it matches this current instance, and returns a 
	* bool representation of their relationship. 
	* 
	* \param[in] iUString Specifies interface pointer
	* \param[in] str 
	* \param[out] b  result.
	* \param[in] ignoreCase: false: case sensitive comparison. 
	* \return error code
	*
	*\ note const bool ignoreCase=false
	*/
typedef CEHResult (*iCEUString_equals) (ICEUString* iUString, const struct ICEUString* pStr, const bool ignoreCase, bool *const pBRet);

typedef CEHResult (*iCEUString_equalsCStringLiteral) (ICEUString* iUString, const CHAR8* pCStringLiteral, const bool ignoreCase, bool *const pBRet);

typedef CEHResult (*iCEUString_equalsUTF16Array) (ICEUString* iUString, const UTF16CHAR* pCharArray16, UINT32 numOfChars16, const bool ignoreCase, bool *const pBRet);


	/*!
	* If this string and specified \a istr are of equal length and match in every character, the \a bMismatchedOut is less than 0,
	* Otherwise, the \a bMismatchedOut is a non-negative integer, the index within this string at which the two string fail to match. 
	*
	* If \a optMatchedOut is specified, matched substring is returned.
	* 
	* \param[in] iUStringSpecifies interface pointer
	* \param[in] istr
	* \param[out] resultOut  result.
	* \param[out] optMatchedOut  matched substring. 
	* \return error code
	*
	*/
typedef CEHResult (*iCEUString_mismatch) (ICEUString* iUString, const struct ICEUString* istr, INT32 *const resultOut, struct ICEUString* *const optMatchedOut);

typedef CEHResult (*iCEUString_compareTo) (ICEUString* iUString, const struct ICEUString* istr, INT32 *const resultOut);


	/*!
	* This method find specified \a pattern string from this string.
	* 
	* \param[in] iUStringSpecifies interface pointer
	* \param[in] pPattern
	* \param[out] headingOut. (specify 0 to ignore the result)
	* \param[out] metchedOut. (specify 0 to ignore the result)
	* \param[out] trailingOut. (specify 0 to ignore the result)
	* \param[in]  eceuStrFindFlgMask: the following options may be specified in mask by combining them with the C bitwise OR operator:
	* <br><pre>
	* eCEUStrFindFlg_IgnoreCaseSearch   // Case insensitive search
	* eCEUStrFindFlg_LiteralSearch      // Disable unicode composition character matching.
	* eCEUStrFindFlg_BackwardSearch     // Search pattern from end of string.
	* eCEUStrFindFlg_AnchoredSearch     // Only search pattern at beggining or end of string.
	* eCEUStrFindFlg_ConcatPattern      // Concatinate pattern to leading.
	* eCEUStrFindFlg_ConcatPatternToTrailing    // Concatinate pattern to trailing.
	* </pre>
	* \return error code
	* \see eCEUStrFindFlg
	*/
typedef CEHResult (*iCEUString_findString) (ICEUString* iUString, const struct ICEUString* pPattern, struct ICEUString* *const headingOut, struct ICEUString* *const matchedStrOut, struct ICEUString* *const trailingOut, const UINT8 eceuStrFindFlgMask);

typedef CEHResult (*iCEUString_findCStringLiteral) (ICEUString* iUString, const CHAR8* pCStringLiteralPattern, struct ICEUString* *const headingOut, struct ICEUString* *const matchedStrOut, struct ICEUString* *const trailingOut, const UINT8 eceuStrFindFlgMask);

typedef CEHResult (*iCEUString_findChar) (ICEUString* iUString, const UTF32CHAR c, struct ICEUString* *const headingOut,  UTF32CHAR *const matchedCharOut,  struct ICEUString* *const trailingOut, const UINT8 eceuStrFindFlgMask);

typedef CEHResult (*iCEUString_findChars) (ICEUString* iUString, const struct ICEUString* pChars, struct ICEUString* *const headingOut,  UTF32CHAR *const matchedCharOut, struct ICEUString* *const trailingOut, const UINT8 eceuStrFindFlgMask);

typedef CEHResult (*iCEUString_findCharsStringLiteral) (ICEUString* iUString, const CHAR8* pCharsCStringLiteral,  struct ICEUString* *const headingOut, UTF32CHAR *const matchedCharOut, struct ICEUString* *const trailingOut, const UINT8 eceuStrFindFlgMask);


	/*!
	* This method convert string data bytes to the specified \a encoding and copy it. This method allocate 
	* the buffer.
	* 
	* \param[in] iUStringSpecifies interface pointer
	* \param[in] encoding
	* 
	* \param[in] pAllocator  CEComAllocatorRec structure. Specify malloc, free, calloc, realloc members before invoke this method.
	* 
	* \param[out] bytes	Pointer to a new buffer that receives a null-terminated string data that 
	* is converted to the \a encoding of this string. When the new buffer is no longer 
	* needed, The caller of this method must free this new buffer by using CEComAllocator.free().
	* Otherwise, this buffer will be leaked.
	* 
	* \param[out] bytesLenOut	If the method succeeds, this value is changed to the <b>length in bytes</b> of 
	* the data copied (without null character) to the buffer. 
	* 
	* \see CEComAllocatorRec
	*/
typedef CEHResult (*iCEUString_getBytesWithAlloc) (ICEUString* iUString, const eICEI18nEncoding encoding, const CEComAllocatorRec* pAllocator, UCHAR8* *const pBytesOut, UINT32 *const pBytesLenOut);

typedef CEHResult (*iCEUString_getBytesWithAllocEx) (ICEUString* iUString, const struct ICEI18nCharsetEncoding* encoding, const CEComAllocatorRec* pAllocator, UCHAR8* *const pBytesOut, UINT32 *const pBytesLenOut);


	/*!
	* This method returns UTF16CHAR array pointer and number of UTF16CHARs of this string object.
	* Since, this method returns direct pointer to the internal characters, use carefully.
	*
	* \param [out] pCharArray16Out pointer to the UTF16CHAR array of this string or 0.
	* \param [out] numOfChars16Out number of utf16 chars
	*
	* \param[in]	iUString	Specifies interface pointer
	*
	* \return pointer to the UTF16CHAR array of this string or 0.
	*
	* \note UINT32 *const numOfChars16Out=0
	*/
typedef CEHResult (*iCEUString_getCharArray16) (ICEUString* iUString, const UTF16CHAR* *const pCharArray16Out, UINT32 *const pNumOfChars16Out);


	/*!
	* convert this string to integer.
	*
	* \param [out] pInt32Out pointer to the INT32 value.
	* \param [out] radix from eCEUStringRadix_MIN through eCEUStringRadix_MAX (from 2 through 32).
	*
	* \param[in]	iUString	Specifies interface pointer
	*
	* \return JGSS_SUCCESS: 
	*		  JGSS_ERR_BADARGS: 
	*/
typedef CEHResult (*iCEUString_toInt32) (ICEUString* iUString, INT32 *pInt32Out, INT32 radix);

typedef CEHResult (*iCEUString_toInt16) (ICEUString* iUString, INT16 *pInt16Out, INT32 radix);

/*!
 * V-table of ICEUString
 */
struct ICEUString_vtbl
{
	iCEUString_queryInterface	_queryInterface;	//!< Query interface.
	iCEUString_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUString_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUString_toImmutable	_toImmutable;
	iCEUString_isImmutable	_isImmutable;
	iCEUString_getHashcode	_getHashcode;
	iCEUString_debugPrint	_debugPrint;
	iCEUString_debugPrintln	_debugPrintln;
	iCEUString_dbgToString	_dbgToString;
	iCEUString_getNumOfChars32	_getNumOfChars32;
	iCEUString_initWithByteArray	_initWithByteArray;
	iCEUString_initWithByteArrayEx	_initWithByteArrayEx;
	iCEUString_initWithString	_initWithString;
	iCEUString_initWithUTF16Array	_initWithUTF16Array;
	iCEUString_initWithUTF32CHAR	_initWithUTF32CHAR;
	iCEUString_initWithCStringLiteral	_initWithCStringLiteral;
	iCEUString_isEmpty	_isEmpty;
	iCEUString_clear	_clear;
	iCEUString_append	_append;
	iCEUString_appendCStringLiteral	_appendCStringLiteral;
	iCEUString_trimWithOptions	_trimWithOptions;
	iCEUString_trim	_trim;
	iCEUString_replace	_replace;
	iCEUString_startsWith	_startsWith;
	iCEUString_startsWithCStringLiteral	_startsWithCStringLiteral;
	iCEUString_endsWith	_endsWith;
	iCEUString_endsWithCStringLiteral	_endsWithCStringLiteral;
	iCEUString_equals	_equals;
	iCEUString_equalsCStringLiteral	_equalsCStringLiteral;
	iCEUString_equalsUTF16Array	_equalsUTF16Array;
	iCEUString_mismatch	_mismatch;
	iCEUString_compareTo	_compareTo;
	iCEUString_findString	_findString;
	iCEUString_findCStringLiteral	_findCStringLiteral;
	iCEUString_findChar	_findChar;
	iCEUString_findChars	_findChars;
	iCEUString_findCharsStringLiteral	_findCharsStringLiteral;
	iCEUString_getBytesWithAlloc	_getBytesWithAlloc;
	iCEUString_getBytesWithAllocEx	_getBytesWithAllocEx;
	iCEUString_getCharArray16	_getCharArray16;
	iCEUString_toInt32	_toInt32;
	iCEUString_toInt16	_toInt16;
};

/*! @}
 * end of ICEUString
 */

/*! \defgroup ICEUStringReader ICEUStringReader
 * @{
 */

/*!
 * ID of ICEUStringReader
 */
#define CEComIID_ICEUStringReader 0xc31601c7


/*!
* ICEUStringReader 
*/
typedef struct ICEUStringReader
{
	const struct ICEUStringReader_vtbl* _vtbl;
} ICEUStringReader;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUStringReader	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUStringReader_queryInterface) (ICEUStringReader* iUStringReader, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUStringReader_addRef) (ICEUStringReader* iUStringReader);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUStringReader_release) (ICEUStringReader* iUStringReader);


	/*!
	* This method opens a reader of string.
	* 
	* \param[in] iUStringReader Specifies interface pointer
	* \param[in] str source string.
	* \return error code
	*/
typedef CEHResult (*iCEUStringReader_open) (ICEUStringReader* iUStringReader, const struct ICEUString* pStr);


	/*!
	* This method skips skips a specified number of character, counting from the current stream position.
	*/

typedef CEHResult (*iCEUStringReader_skip) (ICEUStringReader* iUStringReader, const UINT32 numOfChar);


	/*!
	* This method read a ucs4 character from this string reader.
	* 
	* \param[in] iUStringReaderSpecifies interface pointer
	* \param[out] c32
	* \return error code
	*/
typedef CEHResult (*iCEUStringReader_read) (ICEUStringReader* iUStringReader, UTF32CHAR *const pC32);


	/*!
	* This method unread a ucs4 character to this string reader.
	* Moves the buffer position on character backwards.
	* 
	* So that the next call to "read" will read it again. "unread" is guaranteed to work only 
	* if "read" was the last operation on this object. Thus "unread" cannot be called twice. 
	* If this rule is violated, a checked error may result.
	* 
	* \param[in] iUStringReaderSpecifies interface pointer
	* \param[in] c32
	* \return error code
	*/
typedef CEHResult (*iCEUStringReader_unread) (ICEUStringReader* iUStringReader, const UTF32CHAR c32);


	/*!
	* This method returns a current position of this stream.
	* 
	* <pre>
	* CEUStringReader in;
	* UINT32 offset32=0;
	* UTF32CHAR c32=0;
	* if ( ! in.open(str) ) {
	* in.tell(offset);   // offset => 0
	* in.read(c32);     
	* in.tell(offset);   // offset => 1
	* in.unread(c32);     
	* in.tell(offset);   // offset => 0
	* }
	* </pre>
	* \param[in] iUStringReader Specifies interface pointer
	* \param[out] offset 
	* \return error code
	*/
typedef CEHResult (*iCEUStringReader_tell) (ICEUStringReader* iUStringReader, UINT32 *const pOffsetOut);


	/*!
	* Return "true" if this reader  is at end-of-file(string).
	* 
	* \param[in]	iUStringReader	Specifies interface pointer
	*
	* \return error code
	*/
typedef CEHResult (*iCEUStringReader_eof) (ICEUStringReader* iUStringReader, bool* pIsEofOut);


	/*!
	* Return "true" if this reader  is at end-of-file(string).
	* 
	* \param[in]	iUStringReader	Specifies interface pointer
	*
	* \return error code
	*/
typedef CEHResult (*iCEUStringReader_clone) (ICEUStringReader* iUStringReader, struct ICEUStringReader* *const pClonedOut);


	/*!
	* close this reader.
	* \param[in]	iUStringReader	Specifies interface pointer
	*
	* \return JGSS_SUCCESS
	*/
typedef CEHResult (*iCEUStringReader_close) (ICEUStringReader* iUStringReader);

/*!
 * V-table of ICEUStringReader
 */
struct ICEUStringReader_vtbl
{
	iCEUStringReader_queryInterface	_queryInterface;	//!< Query interface.
	iCEUStringReader_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUStringReader_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUStringReader_open	_open;
	iCEUStringReader_skip	_skip;
	iCEUStringReader_read	_read;
	iCEUStringReader_unread	_unread;
	iCEUStringReader_tell	_tell;
	iCEUStringReader_eof	_eof;
	iCEUStringReader_clone	_clone;
	iCEUStringReader_close	_close;
};

/*! @}
 * end of ICEUStringReader
 */

/*! \defgroup ICEUStringWriter ICEUStringWriter
 * @{
 */

/*!
 * ID of ICEUStringWriter
 */
#define CEComIID_ICEUStringWriter 0x9285c961


/*!
* ICEUStringWriter
*/
typedef struct ICEUStringWriter
{
	const struct ICEUStringWriter_vtbl* _vtbl;
} ICEUStringWriter;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUStringWriter	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUStringWriter_queryInterface) (ICEUStringWriter* iUStringWriter, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUStringWriter_addRef) (ICEUStringWriter* iUStringWriter);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUStringWriter_release) (ICEUStringWriter* iUStringWriter);


	/*!
	* Determine if the buffer of writer is empty.
	* 
	* \param[in]	iUStringWriter	Specifies interface pointer
	*
	* \return true if the buffer of writer is empty otherwise false. 
	*/
typedef CEHResult (*iCEUStringWriter_isEmpty) (ICEUStringWriter* iUStringWriter, bool *const pIsEmptyOut);


	/*!
	* Determine if the writer is closed.
	* 
	* \param[in]	iUStringWriter	Specifies interface pointer
	*
	* \return true if the writer is closed otherwise false. 
	*/
typedef CEHResult (*iCEUStringWriter_isClosed) (ICEUStringWriter* iUStringWriter, bool *const pIsClosedOut);


	/*!
	* This method opens a writer with \a buff.
	* 
	* \param[in] iUStringWriter Specifies interface pointer
	* \param[in] buff buffer to write characters.
	* \param[in] bytesLen length in bytes of \a buff.
	* \return error code
	*/
typedef CEHResult (*iCEUStringWriter_openWithBytes) (ICEUStringWriter* iUStringWriter, UCHAR8* pBuff, const UINT32 bytesLen);


	/*!
	* This method opens a writer.
	* 
	* \param[in] iUStringWriter Specifies interface pointer
	* \param[in] initialBytes initial internal buffer size in bytes.
	* \param[in] expandBytes expand buffer byte size in bytes.
	* \return error code
	*
	* \note const UINT32 initialBytes=CEU_BAOS_INITIAL_BYTES, const UINT32 expandBytes=CEU_BAOS_EXPAND_BYTES
	*/
typedef CEHResult (*iCEUStringWriter_openWithBufferOptions) (ICEUStringWriter* iUStringWriter, const UINT32 initialBytes, const UINT32 expandBytes);

typedef CEHResult (*iCEUStringWriter_open) (ICEUStringWriter* iUStringWriter);


	/*!
	* This method write \a str into the internal buffer.
	* 
	* \param[in] iUStringWriter Specifies interface pointer
	* \param[in] str string to write.
	* \param[out] bytesWrittenOut number of bytes written is returned. 
	* (ex..) <pre>
	* CEUStringWriter out;
	* if ( !out.open() ) {
	* CEUString str;
	* if ( !str.initWithCStringLiteral("abc") ) {
	* UINT32 bytesWritten=0;
	* if (!out.writeString(str, bytesWritten)) {
	* assert( bytesWritten == sizeof(UTF16CHAR)*3 );
	* .....
	* </pre>
	* \return error code
	*/
typedef CEHResult (*iCEUStringWriter_writeString) (ICEUStringWriter* iUStringWriter, const struct ICEUString* pStr, UINT32 *const pBytesWrittenOut);


	/*!
	* This method write \a c32(ucs4 character) into the internal buffer.
	* 
	* \param[in] iUStringWriter Specifies interface pointer
	* \param[in] c32 utf32 character to write.
	* \param[out] bytesWrittenOut number of bytes written is returned.
	* \return error code
	*/
typedef CEHResult (*iCEUStringWriter_writeUTF32) (ICEUStringWriter* iUStringWriter, const UTF32CHAR c32, UINT32 *const pBytesWrittenOut);


	/*!
	* This method write \a c8(ascii or single byte character) into the internal buffer.
	* 
	* \param[in] iUStringWriter Specifies interface pointer
	* \param[in] c8 single byte character to write.
	* \param[out] bytesWrittenOut number of bytes written.
	* \param[in] srcEncoding eICEI18nEncoding of \a c8.
	* \return error code
	*
	* \note eICEI18nEncoding srcEncoding=eICEI18nEncoding_us_ascii
	*/
typedef CEHResult (*iCEUStringWriter_writeByte) (ICEUStringWriter* iUStringWriter, const UCHAR8 c8, UINT32 *const pBytesWrittenOut, eICEI18nEncoding srcEncoding);

typedef CEHResult (*iCEUStringWriter_writeByteEx) (ICEUStringWriter* iUStringWriter, const UCHAR8 c8, UINT32 *const pBytesWrittenOut, struct ICEI18nCharsetEncoding* srcEncoding);


	/*!
	* This method write \a uchar8str into the internal buffer.
	* 
	* \param[in] iUStringWriter Specifies interface pointer
	* \param[in] uchar8str byte array to write.
	* \param[in,out] bytesLenRef specify length in bytes of \a uchar8str, If method is succedded, number of bytes written is returned.
	* \param[in] srcEncoding eICEI18nEncoding of \a uchar8str.
	* \return error code
	*
	* \note eICEI18nEncoding srcEncoding=eICEI18nEncoding_ascii
	*/
typedef CEHResult (*iCEUStringWriter_writeByteArray) (ICEUStringWriter* iUStringWriter, const UCHAR8* pUchar8str, UINT32 *const pBytesLenRef, eICEI18nEncoding srcEncoding);

typedef CEHResult (*iCEUStringWriter_writeByteArrayEx) (ICEUStringWriter* iUStringWriter, const UCHAR8* pUchar8str, UINT32 *const pBytesLenRef, struct ICEI18nCharsetEncoding* srcEncoding);


	/*!
	* This method write \a utf16char into the internal buffer.
	* 
	* \param[in] iUStringWriter Specifies interface pointer
	* \param[in] utf16char utf16 character to write.
	* \param[out] bytesWrittenOut number of bytes written is returned.
	* \return error code
	*/
typedef CEHResult (*iCEUStringWriter_writeUTF16) (ICEUStringWriter* iUStringWriter, const UTF16CHAR utf16char, UINT32 *const pBytesWrittenOut);


	/*!
	* This method write \a utf16str into the internal buffer.
	* 
	* \param[in] iUStringWriter Specifies interface pointer
	* \param[in] utf16char utf16 character to write.
	* \param[in, out] bytesLenRef specify length in bytes of \a utf16str, If method is succedded, number of bytes written is returned.
	* \return error code
	*/
typedef CEHResult (*iCEUStringWriter_writeUTF16Array) (ICEUStringWriter* iUStringWriter, const UTF16CHAR* pUtf16str, UINT32 *const pBytesLenRef);


	/*!
	* This method initialize \a newStrOut with current internal buffer data and close this writer.
	* 
	* \param[in] iUStringWriterSpecifies interface pointer
	* \param[out] newStrOut
	* \return error code
	*/
typedef CEHResult (*iCEUStringWriter_close) (ICEUStringWriter* iUStringWriter, struct ICEUString* *const pNewStrOut);


	/*!
	* This method returns internal data buffer and number of bytes.
	* 
	* \param[in]	iUStringWriter	Specifies interface pointer
	*
	* \return error code
	*/
typedef CEHResult (*iCEUStringWriter_getBuffer) (ICEUStringWriter* iUStringWriter, const UINT8* *const pBufferOut, UINT32 *const pNumOfBytesOut);

/*!
 * V-table of ICEUStringWriter
 */
struct ICEUStringWriter_vtbl
{
	iCEUStringWriter_queryInterface	_queryInterface;	//!< Query interface.
	iCEUStringWriter_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUStringWriter_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUStringWriter_isEmpty	_isEmpty;
	iCEUStringWriter_isClosed	_isClosed;
	iCEUStringWriter_openWithBytes	_openWithBytes;
	iCEUStringWriter_openWithBufferOptions	_openWithBufferOptions;
	iCEUStringWriter_open	_open;
	iCEUStringWriter_writeString	_writeString;
	iCEUStringWriter_writeUTF32	_writeUTF32;
	iCEUStringWriter_writeByte	_writeByte;
	iCEUStringWriter_writeByteEx	_writeByteEx;
	iCEUStringWriter_writeByteArray	_writeByteArray;
	iCEUStringWriter_writeByteArrayEx	_writeByteArrayEx;
	iCEUStringWriter_writeUTF16	_writeUTF16;
	iCEUStringWriter_writeUTF16Array	_writeUTF16Array;
	iCEUStringWriter_close	_close;
	iCEUStringWriter_getBuffer	_getBuffer;
};

/*! @}
 * end of ICEUStringWriter
 */

/*! \defgroup ICEUTokenizerCallbacks ICEUTokenizerCallbacks
 * @{
 */

/*!
 * ID of ICEUTokenizerCallbacks
 */
#define CEComIID_ICEUTokenizerCallbacks 0x0f3ac318


/*!
* ICEUTokenizerCallbacks
*
* A ICEUTokenizer callback interface.
*/
typedef struct ICEUTokenizerCallbacks
{
	const struct ICEUTokenizerCallbacks_vtbl* _vtbl;
} ICEUTokenizerCallbacks;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUTokenizerCallbacks	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUTokenizerCallbacks_queryInterface) (ICEUTokenizerCallbacks* iUTokenizerCallbacks, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUTokenizerCallbacks_addRef) (ICEUTokenizerCallbacks* iUTokenizerCallbacks);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUTokenizerCallbacks_release) (ICEUTokenizerCallbacks* iUTokenizerCallbacks);

typedef CEHResult (*iCEUTokenizerCallbacks_init) (ICEUTokenizerCallbacks* iUTokenizerCallbacks, ICEUTokenizerCallbacks_tokenProc tp, ICEUTokenizerCallbacks_notifyProc np);

typedef CEHResult (*iCEUTokenizerCallbacks_tokenProc) (ICEUTokenizerCallbacks* iUTokenizerCallbacks, struct ICEUString* token, UINT8 tokenType, void* userData);

typedef CEHResult (*iCEUTokenizerCallbacks_notifyProc) (ICEUTokenizerCallbacks* iUTokenizerCallbacks, UTF32CHAR c32, UINT8 notifyFlags, void* userData);

/*!
 * V-table of ICEUTokenizerCallbacks
 */
struct ICEUTokenizerCallbacks_vtbl
{
	iCEUTokenizerCallbacks_queryInterface	_queryInterface;	//!< Query interface.
	iCEUTokenizerCallbacks_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUTokenizerCallbacks_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUTokenizerCallbacks_init	_init;
	iCEUTokenizerCallbacks_tokenProc	_tokenProc;
	iCEUTokenizerCallbacks_notifyProc	_notifyProc;
};

/*! @}
 * end of ICEUTokenizerCallbacks
 */

/*! \defgroup ICEUTokenizer ICEUTokenizer
 * @{
 */

/*!
 * ID of ICEUTokenizer
 */
#define CEComIID_ICEUTokenizer 0x190cfdae


/*!
* ICEUTokenizer
*/
typedef struct ICEUTokenizer
{
	const struct ICEUTokenizer_vtbl* _vtbl;
} ICEUTokenizer;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUTokenizer	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUTokenizer_queryInterface) (ICEUTokenizer* iUTokenizer, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUTokenizer_addRef) (ICEUTokenizer* iUTokenizer);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUTokenizer_release) (ICEUTokenizer* iUTokenizer);


	/*! 
	* initialize the tokenizer, and setup the state transition rule to tokenize
	*/

typedef CEHResult (*iCEUTokenizer_init) (ICEUTokenizer* iUTokenizer, UINT16 numOfStates, const CEUTokenizerState* stateSet, const UINT8* charKindTable );


	/*! 
	 * open the tokenizer with the specified callbacks and userdata
	 */

typedef CEHResult (*iCEUTokenizer_open) (ICEUTokenizer* iUTokenizer, const struct ICEUTokenizerCallbacks* callbacks, void* userData);


	/*! 
	 * write the specified string to the tokenizer and let it parse 
	 */

typedef CEHResult (*iCEUTokenizer_writeString) (ICEUTokenizer* iUTokenizer, const struct ICEUString* str);


	/*! 
	 * flush the gathering token, and finilize the tokenizer for the given callbacks and userdata
	 */

typedef CEHResult (*iCEUTokenizer_close) (ICEUTokenizer* iUTokenizer);

/*!
 * V-table of ICEUTokenizer
 */
struct ICEUTokenizer_vtbl
{
	iCEUTokenizer_queryInterface	_queryInterface;	//!< Query interface.
	iCEUTokenizer_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUTokenizer_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUTokenizer_init	_init;
	iCEUTokenizer_open	_open;
	iCEUTokenizer_writeString	_writeString;
	iCEUTokenizer_close	_close;
};

/*! @}
 * end of ICEUTokenizer
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEUStringRef CEComICEUStringRef
 * @{
 */

class CEComICEUStringRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUStringRef() : _iUString(0)
	{
#if defined(_DEBUG) || !defined(NDEBUG)
		dbgToString();
#endif //_DEBUG || !NDEBUG
	}
	CEComICEUStringRef(ICEUString* iOther) : _iUString(0)
	{
		if (iOther)
		{
			_iUString = iOther;
			_iUString->_vtbl->_addRef(_iUString);
		}
	}

	CEComICEUStringRef(const CEComICEUStringRef& other) : _iUString(0)
	{
		if (other._iUString)
		{
			_iUString = other._iUString;
			_iUString->_vtbl->_addRef(_iUString);
		}
	}

	~CEComICEUStringRef()
	{
		if (_iUString)
		{
			ICEUString* tmp = _iUString;
			_iUString = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUString* iUString;
				void* _ptr;
			} uIntf;
			uIntf.iUString = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUString, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUString)
				{
					ICEUString* tmp = _iUString;
					_iUString = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUString = uIntf.iUString;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUString* object() const	{ return _iUString; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUString* iOther)
	{
		if (_iUString)
		{
			ICEUString* tmp = _iUString;
			_iUString = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUString = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUString* detach()
	{
		ICEUString* iIntf = _iUString;
		_iUString = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUString* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUString;
		if (_iUString)
		{
			_iUString->_vtbl->_addRef(_iUString);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUString*() const	{ return _iUString; }
	FORCEINLINE_WITHOUT_DEBUG ICEUString& operator*() const	{ return *_iUString; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUString == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUString* iOther) const	{ return (_iUString != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUString* iOther) const	{ return (_iUString == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUStringRef& operator=(const CEComICEUStringRef& other)	{ return operator=(other._iUString); }

	CEComICEUStringRef& operator=(const ICEUString* iOther)
	{
		if (_iUString != iOther)
		{
			if (_iUString)
			{
				ICEUString* tmp = _iUString;
				_iUString = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUString = const_cast<ICEUString*>(iOther);
			if (_iUString)
			{
				_iUString->_vtbl->_addRef(_iUString);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUString** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUStringRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUString && "has a com object reference. clear first.");
		return & _iUString; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUStringRef& other)
	{
		bool result = false;
		if (_iUString)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUString);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUString ? _iUString->_vtbl->_queryInterface(_iUString, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUString) { _iUString->_vtbl->_addRef(_iUString); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUString) { _iUString->_vtbl->_release(_iUString); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult toImmutable()	{ return _iUString ? _iUString->_vtbl->_toImmutable(_iUString) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isImmutable(bool *const bImmutableOut)	{ return _iUString ? _iUString->_vtbl->_isImmutable(_iUString, bImmutableOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getHashcode(UINT32 *const hashcodeOut)	{ return _iUString ? _iUString->_vtbl->_getHashcode(_iUString, hashcodeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult debugPrint()	{ return _iUString ? _iUString->_vtbl->_debugPrint(_iUString) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult debugPrintln()	{ return _iUString ? _iUString->_vtbl->_debugPrintln(_iUString) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult dbgToString(const UTF16CHAR* *const pStr16Out)	{ return _iUString ? _iUString->_vtbl->_dbgToString(_iUString, pStr16Out) : CE_SILK_ERR_UNINITIALIZED; }

#if defined(_DEBUG) || !defined(NDEBUG)
	const UTF16CHAR* dbgToString() { const UTF16CHAR* ret=0; if (_iUString) { dbgToString(&ret); } return ret; }
#endif // _DEBUG || !NDEBUG

	FORCEINLINE_WITHOUT_DEBUG CEHResult getNumOfChars32(UINT32* pNumOfCharsOut)	{ return _iUString ? _iUString->_vtbl->_getNumOfChars32(_iUString, pNumOfCharsOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initWithByteArray(const UCHAR8* pAnyCharStr, const eICEI18nEncoding srcEncoding, const UINT32 anyCharStrSizeInBytes)	{ return _iUString ? _iUString->_vtbl->_initWithByteArray(_iUString, pAnyCharStr, srcEncoding, anyCharStrSizeInBytes) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initWithByteArrayEx(const UCHAR8* pAnyCharStr, const struct ICEI18nCharsetEncoding* srcEncoding, const UINT32 anyCharStrSizeInBytes)	{ return _iUString ? _iUString->_vtbl->_initWithByteArrayEx(_iUString, pAnyCharStr, srcEncoding, anyCharStrSizeInBytes) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initWithString(const struct ICEUString* pStr)	{ return _iUString ? _iUString->_vtbl->_initWithString(_iUString, pStr) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initWithUTF16Array(const UTF16CHAR* pUtf16Str, const UINT32 sizeInBytes)	{ return _iUString ? _iUString->_vtbl->_initWithUTF16Array(_iUString, pUtf16Str, sizeInBytes) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initWithUTF32CHAR(const UTF32CHAR c32)	{ return _iUString ? _iUString->_vtbl->_initWithUTF32CHAR(_iUString, c32) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initWithCStringLiteral(const CHAR8* pCStringLiteral)	{ return _iUString ? _iUString->_vtbl->_initWithCStringLiteral(_iUString, pCStringLiteral) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isEmpty(bool *const pIsEmptyOut)	{ return _iUString ? _iUString->_vtbl->_isEmpty(_iUString, pIsEmptyOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult clear()	{ return _iUString ? _iUString->_vtbl->_clear(_iUString) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult append(const struct ICEUString* pStrToAppend)	{ return _iUString ? _iUString->_vtbl->_append(_iUString, pStrToAppend) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult appendCStringLiteral(const CHAR8* pCStringLiteral)	{ return _iUString ? _iUString->_vtbl->_appendCStringLiteral(_iUString, pCStringLiteral) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult trimWithOptions(eCEUStrTrimFlg flg, UINT32 numLeft, UINT32 numRight)	{ return _iUString ? _iUString->_vtbl->_trimWithOptions(_iUString, flg, numLeft, numRight) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult trim()	{ return _iUString ? _iUString->_vtbl->_trim(_iUString) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult replace(const UCHAR8 pFrom, const UCHAR8 pTo)	{ return _iUString ? _iUString->_vtbl->_replace(_iUString, pFrom, pTo) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult startsWith(const struct ICEUString* pStr, const bool ignoreCase, bool *const pBRet)	{ return _iUString ? _iUString->_vtbl->_startsWith(_iUString, pStr, ignoreCase, pBRet) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult startsWithCStringLiteral(const CHAR8* pCStringLiteral, const bool ignoreCase, bool *const pBRet)	{ return _iUString ? _iUString->_vtbl->_startsWithCStringLiteral(_iUString, pCStringLiteral, ignoreCase, pBRet) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult endsWith(const struct ICEUString* pStr, const bool ignoreCase, bool *const pBRet)	{ return _iUString ? _iUString->_vtbl->_endsWith(_iUString, pStr, ignoreCase, pBRet) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult endsWithCStringLiteral(const CHAR8* pCStringLiteral, const bool ignoreCase, bool *const pBRet)	{ return _iUString ? _iUString->_vtbl->_endsWithCStringLiteral(_iUString, pCStringLiteral, ignoreCase, pBRet) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult equals(const struct ICEUString* pStr, const bool ignoreCase, bool *const pBRet)	{ return _iUString ? _iUString->_vtbl->_equals(_iUString, pStr, ignoreCase, pBRet) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult equalsCStringLiteral(const CHAR8* pCStringLiteral, const bool ignoreCase, bool *const pBRet)	{ return _iUString ? _iUString->_vtbl->_equalsCStringLiteral(_iUString, pCStringLiteral, ignoreCase, pBRet) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult equalsUTF16Array(const UTF16CHAR* pCharArray16, UINT32 numOfChars16, const bool ignoreCase, bool *const pBRet)	{ return _iUString ? _iUString->_vtbl->_equalsUTF16Array(_iUString, pCharArray16, numOfChars16, ignoreCase, pBRet) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult mismatch(const struct ICEUString* istr, INT32 *const resultOut, struct ICEUString* *const optMatchedOut)	{ return _iUString ? _iUString->_vtbl->_mismatch(_iUString, istr, resultOut, optMatchedOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult compareTo(const struct ICEUString* istr, INT32 *const resultOut)	{ return _iUString ? _iUString->_vtbl->_compareTo(_iUString, istr, resultOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult findString(const struct ICEUString* pPattern, struct ICEUString* *const headingOut, struct ICEUString* *const matchedStrOut, struct ICEUString* *const trailingOut, const UINT8 eceuStrFindFlgMask)	{ return _iUString ? _iUString->_vtbl->_findString(_iUString, pPattern, headingOut, matchedStrOut, trailingOut, eceuStrFindFlgMask) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult findCStringLiteral(const CHAR8* pCStringLiteralPattern, struct ICEUString* *const headingOut, struct ICEUString* *const matchedStrOut, struct ICEUString* *const trailingOut, const UINT8 eceuStrFindFlgMask)	{ return _iUString ? _iUString->_vtbl->_findCStringLiteral(_iUString, pCStringLiteralPattern, headingOut, matchedStrOut, trailingOut, eceuStrFindFlgMask) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult findChar(const UTF32CHAR c, struct ICEUString* *const headingOut,  UTF32CHAR *const matchedCharOut,  struct ICEUString* *const trailingOut, const UINT8 eceuStrFindFlgMask)	{ return _iUString ? _iUString->_vtbl->_findChar(_iUString, c, headingOut, matchedCharOut, trailingOut, eceuStrFindFlgMask) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult findChars(const struct ICEUString* pChars, struct ICEUString* *const headingOut,  UTF32CHAR *const matchedCharOut, struct ICEUString* *const trailingOut, const UINT8 eceuStrFindFlgMask)	{ return _iUString ? _iUString->_vtbl->_findChars(_iUString, pChars, headingOut, matchedCharOut, trailingOut, eceuStrFindFlgMask) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult findCharsStringLiteral(const CHAR8* pCharsCStringLiteral,  struct ICEUString* *const headingOut, UTF32CHAR *const matchedCharOut, struct ICEUString* *const trailingOut, const UINT8 eceuStrFindFlgMask)	{ return _iUString ? _iUString->_vtbl->_findCharsStringLiteral(_iUString, pCharsCStringLiteral, headingOut, matchedCharOut, trailingOut, eceuStrFindFlgMask) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getBytesWithAlloc(const eICEI18nEncoding encoding, const CEComAllocatorRec* pAllocator, UCHAR8* *const pBytesOut, UINT32 *const pBytesLenOut)	{ return _iUString ? _iUString->_vtbl->_getBytesWithAlloc(_iUString, encoding, pAllocator, pBytesOut, pBytesLenOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getBytesWithAllocEx(const struct ICEI18nCharsetEncoding* encoding, const CEComAllocatorRec* pAllocator, UCHAR8* *const pBytesOut, UINT32 *const pBytesLenOut)	{ return _iUString ? _iUString->_vtbl->_getBytesWithAllocEx(_iUString, encoding, pAllocator, pBytesOut, pBytesLenOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCharArray16(const UTF16CHAR* *const pCharArray16Out, UINT32 *const pNumOfChars16Out)	{ return _iUString ? _iUString->_vtbl->_getCharArray16(_iUString, pCharArray16Out, pNumOfChars16Out) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult toInt32(INT32 *pInt32Out, INT32 radix)	{ return _iUString ? _iUString->_vtbl->_toInt32(_iUString, pInt32Out, radix) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult toInt16(INT16 *pInt16Out, INT32 radix)	{ return _iUString ? _iUString->_vtbl->_toInt16(_iUString, pInt16Out, radix) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEUString* _iUString;
};

/*! @}
 * end of CEComICEUStringRef
 */

/*! \defgroup CEComICEUStringReaderRef CEComICEUStringReaderRef
 * @{
 */

class CEComICEUStringReaderRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUStringReaderRef() : _iUStringReader(0) {}
	CEComICEUStringReaderRef(ICEUStringReader* iOther) : _iUStringReader(0)
	{
		if (iOther)
		{
			_iUStringReader = iOther;
			_iUStringReader->_vtbl->_addRef(_iUStringReader);
		}
	}

	CEComICEUStringReaderRef(const CEComICEUStringReaderRef& other) : _iUStringReader(0)
	{
		if (other._iUStringReader)
		{
			_iUStringReader = other._iUStringReader;
			_iUStringReader->_vtbl->_addRef(_iUStringReader);
		}
	}

	~CEComICEUStringReaderRef()
	{
		if (_iUStringReader)
		{
			ICEUStringReader* tmp = _iUStringReader;
			_iUStringReader = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUStringReader* iUStringReader;
				void* _ptr;
			} uIntf;
			uIntf.iUStringReader = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUStringReader, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUStringReader)
				{
					ICEUStringReader* tmp = _iUStringReader;
					_iUStringReader = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUStringReader = uIntf.iUStringReader;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUStringReader* object() const	{ return _iUStringReader; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUStringReader* iOther)
	{
		if (_iUStringReader)
		{
			ICEUStringReader* tmp = _iUStringReader;
			_iUStringReader = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUStringReader = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUStringReader* detach()
	{
		ICEUStringReader* iIntf = _iUStringReader;
		_iUStringReader = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUStringReader* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUStringReader;
		if (_iUStringReader)
		{
			_iUStringReader->_vtbl->_addRef(_iUStringReader);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUStringReader*() const	{ return _iUStringReader; }
	FORCEINLINE_WITHOUT_DEBUG ICEUStringReader& operator*() const	{ return *_iUStringReader; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUStringReader == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUStringReader* iOther) const	{ return (_iUStringReader != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUStringReader* iOther) const	{ return (_iUStringReader == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUStringReaderRef& operator=(const CEComICEUStringReaderRef& other)	{ return operator=(other._iUStringReader); }

	CEComICEUStringReaderRef& operator=(const ICEUStringReader* iOther)
	{
		if (_iUStringReader != iOther)
		{
			if (_iUStringReader)
			{
				ICEUStringReader* tmp = _iUStringReader;
				_iUStringReader = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUStringReader = const_cast<ICEUStringReader*>(iOther);
			if (_iUStringReader)
			{
				_iUStringReader->_vtbl->_addRef(_iUStringReader);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUStringReader** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUStringReaderRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUStringReader && "has a com object reference. clear first.");
		return & _iUStringReader; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUStringReaderRef& other)
	{
		bool result = false;
		if (_iUStringReader)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUStringReader);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUStringReader ? _iUStringReader->_vtbl->_queryInterface(_iUStringReader, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUStringReader) { _iUStringReader->_vtbl->_addRef(_iUStringReader); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUStringReader) { _iUStringReader->_vtbl->_release(_iUStringReader); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult open(const struct ICEUString* pStr)	{ return _iUStringReader ? _iUStringReader->_vtbl->_open(_iUStringReader, pStr) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult skip(const UINT32 numOfChar)	{ return _iUStringReader ? _iUStringReader->_vtbl->_skip(_iUStringReader, numOfChar) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult read(UTF32CHAR *const pC32)	{ return _iUStringReader ? _iUStringReader->_vtbl->_read(_iUStringReader, pC32) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unread(const UTF32CHAR c32)	{ return _iUStringReader ? _iUStringReader->_vtbl->_unread(_iUStringReader, c32) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult tell(UINT32 *const pOffsetOut)	{ return _iUStringReader ? _iUStringReader->_vtbl->_tell(_iUStringReader, pOffsetOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult eof(bool* pIsEofOut)	{ return _iUStringReader ? _iUStringReader->_vtbl->_eof(_iUStringReader, pIsEofOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult clone(struct ICEUStringReader* *const pClonedOut)	{ return _iUStringReader ? _iUStringReader->_vtbl->_clone(_iUStringReader, pClonedOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult close()	{ return _iUStringReader ? _iUStringReader->_vtbl->_close(_iUStringReader) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEUStringReader* _iUStringReader;
};

/*! @}
 * end of CEComICEUStringReaderRef
 */

/*! \defgroup CEComICEUStringWriterRef CEComICEUStringWriterRef
 * @{
 */

class CEComICEUStringWriterRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUStringWriterRef() : _iUStringWriter(0) {}
	CEComICEUStringWriterRef(ICEUStringWriter* iOther) : _iUStringWriter(0)
	{
		if (iOther)
		{
			_iUStringWriter = iOther;
			_iUStringWriter->_vtbl->_addRef(_iUStringWriter);
		}
	}

	CEComICEUStringWriterRef(const CEComICEUStringWriterRef& other) : _iUStringWriter(0)
	{
		if (other._iUStringWriter)
		{
			_iUStringWriter = other._iUStringWriter;
			_iUStringWriter->_vtbl->_addRef(_iUStringWriter);
		}
	}

	~CEComICEUStringWriterRef()
	{
		if (_iUStringWriter)
		{
			ICEUStringWriter* tmp = _iUStringWriter;
			_iUStringWriter = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUStringWriter* iUStringWriter;
				void* _ptr;
			} uIntf;
			uIntf.iUStringWriter = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUStringWriter, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUStringWriter)
				{
					ICEUStringWriter* tmp = _iUStringWriter;
					_iUStringWriter = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUStringWriter = uIntf.iUStringWriter;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUStringWriter* object() const	{ return _iUStringWriter; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUStringWriter* iOther)
	{
		if (_iUStringWriter)
		{
			ICEUStringWriter* tmp = _iUStringWriter;
			_iUStringWriter = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUStringWriter = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUStringWriter* detach()
	{
		ICEUStringWriter* iIntf = _iUStringWriter;
		_iUStringWriter = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUStringWriter* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUStringWriter;
		if (_iUStringWriter)
		{
			_iUStringWriter->_vtbl->_addRef(_iUStringWriter);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUStringWriter*() const	{ return _iUStringWriter; }
	FORCEINLINE_WITHOUT_DEBUG ICEUStringWriter& operator*() const	{ return *_iUStringWriter; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUStringWriter == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUStringWriter* iOther) const	{ return (_iUStringWriter != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUStringWriter* iOther) const	{ return (_iUStringWriter == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUStringWriterRef& operator=(const CEComICEUStringWriterRef& other)	{ return operator=(other._iUStringWriter); }

	CEComICEUStringWriterRef& operator=(const ICEUStringWriter* iOther)
	{
		if (_iUStringWriter != iOther)
		{
			if (_iUStringWriter)
			{
				ICEUStringWriter* tmp = _iUStringWriter;
				_iUStringWriter = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUStringWriter = const_cast<ICEUStringWriter*>(iOther);
			if (_iUStringWriter)
			{
				_iUStringWriter->_vtbl->_addRef(_iUStringWriter);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUStringWriter** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUStringWriterRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUStringWriter && "has a com object reference. clear first.");
		return & _iUStringWriter; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUStringWriterRef& other)
	{
		bool result = false;
		if (_iUStringWriter)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUStringWriter);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_queryInterface(_iUStringWriter, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUStringWriter) { _iUStringWriter->_vtbl->_addRef(_iUStringWriter); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUStringWriter) { _iUStringWriter->_vtbl->_release(_iUStringWriter); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isEmpty(bool *const pIsEmptyOut)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_isEmpty(_iUStringWriter, pIsEmptyOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isClosed(bool *const pIsClosedOut)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_isClosed(_iUStringWriter, pIsClosedOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult openWithBytes(UCHAR8* pBuff, const UINT32 bytesLen)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_openWithBytes(_iUStringWriter, pBuff, bytesLen) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult openWithBufferOptions(const UINT32 initialBytes, const UINT32 expandBytes)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_openWithBufferOptions(_iUStringWriter, initialBytes, expandBytes) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult open()	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_open(_iUStringWriter) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult writeString(const struct ICEUString* pStr, UINT32 *const pBytesWrittenOut)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_writeString(_iUStringWriter, pStr, pBytesWrittenOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult writeUTF32(const UTF32CHAR c32, UINT32 *const pBytesWrittenOut)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_writeUTF32(_iUStringWriter, c32, pBytesWrittenOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult writeByte(const UCHAR8 c8, UINT32 *const pBytesWrittenOut, eICEI18nEncoding srcEncoding)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_writeByte(_iUStringWriter, c8, pBytesWrittenOut, srcEncoding) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult writeByteEx(const UCHAR8 c8, UINT32 *const pBytesWrittenOut, struct ICEI18nCharsetEncoding* srcEncoding)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_writeByteEx(_iUStringWriter, c8, pBytesWrittenOut, srcEncoding) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult writeByteArray(const UCHAR8* pUchar8str, UINT32 *const pBytesLenRef, eICEI18nEncoding srcEncoding)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_writeByteArray(_iUStringWriter, pUchar8str, pBytesLenRef, srcEncoding) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult writeByteArrayEx(const UCHAR8* pUchar8str, UINT32 *const pBytesLenRef, struct ICEI18nCharsetEncoding* srcEncoding)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_writeByteArrayEx(_iUStringWriter, pUchar8str, pBytesLenRef, srcEncoding) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult writeUTF16(const UTF16CHAR utf16char, UINT32 *const pBytesWrittenOut)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_writeUTF16(_iUStringWriter, utf16char, pBytesWrittenOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult writeUTF16Array(const UTF16CHAR* pUtf16str, UINT32 *const pBytesLenRef)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_writeUTF16Array(_iUStringWriter, pUtf16str, pBytesLenRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult close(struct ICEUString* *const pNewStrOut)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_close(_iUStringWriter, pNewStrOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getBuffer(const UINT8* *const pBufferOut, UINT32 *const pNumOfBytesOut)	{ return _iUStringWriter ? _iUStringWriter->_vtbl->_getBuffer(_iUStringWriter, pBufferOut, pNumOfBytesOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEUStringWriter* _iUStringWriter;
};

/*! @}
 * end of CEComICEUStringWriterRef
 */

/*! \defgroup CEComICEUTokenizerCallbacksRef CEComICEUTokenizerCallbacksRef
 * @{
 */

class CEComICEUTokenizerCallbacksRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUTokenizerCallbacksRef() : _iUTokenizerCallbacks(0) {}
	CEComICEUTokenizerCallbacksRef(ICEUTokenizerCallbacks* iOther) : _iUTokenizerCallbacks(0)
	{
		if (iOther)
		{
			_iUTokenizerCallbacks = iOther;
			_iUTokenizerCallbacks->_vtbl->_addRef(_iUTokenizerCallbacks);
		}
	}

	CEComICEUTokenizerCallbacksRef(const CEComICEUTokenizerCallbacksRef& other) : _iUTokenizerCallbacks(0)
	{
		if (other._iUTokenizerCallbacks)
		{
			_iUTokenizerCallbacks = other._iUTokenizerCallbacks;
			_iUTokenizerCallbacks->_vtbl->_addRef(_iUTokenizerCallbacks);
		}
	}

	~CEComICEUTokenizerCallbacksRef()
	{
		if (_iUTokenizerCallbacks)
		{
			ICEUTokenizerCallbacks* tmp = _iUTokenizerCallbacks;
			_iUTokenizerCallbacks = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUTokenizerCallbacks* iUTokenizerCallbacks;
				void* _ptr;
			} uIntf;
			uIntf.iUTokenizerCallbacks = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUTokenizerCallbacks, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUTokenizerCallbacks)
				{
					ICEUTokenizerCallbacks* tmp = _iUTokenizerCallbacks;
					_iUTokenizerCallbacks = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUTokenizerCallbacks = uIntf.iUTokenizerCallbacks;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUTokenizerCallbacks* object() const	{ return _iUTokenizerCallbacks; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUTokenizerCallbacks* iOther)
	{
		if (_iUTokenizerCallbacks)
		{
			ICEUTokenizerCallbacks* tmp = _iUTokenizerCallbacks;
			_iUTokenizerCallbacks = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUTokenizerCallbacks = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUTokenizerCallbacks* detach()
	{
		ICEUTokenizerCallbacks* iIntf = _iUTokenizerCallbacks;
		_iUTokenizerCallbacks = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUTokenizerCallbacks* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUTokenizerCallbacks;
		if (_iUTokenizerCallbacks)
		{
			_iUTokenizerCallbacks->_vtbl->_addRef(_iUTokenizerCallbacks);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUTokenizerCallbacks*() const	{ return _iUTokenizerCallbacks; }
	FORCEINLINE_WITHOUT_DEBUG ICEUTokenizerCallbacks& operator*() const	{ return *_iUTokenizerCallbacks; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUTokenizerCallbacks == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUTokenizerCallbacks* iOther) const	{ return (_iUTokenizerCallbacks != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUTokenizerCallbacks* iOther) const	{ return (_iUTokenizerCallbacks == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUTokenizerCallbacksRef& operator=(const CEComICEUTokenizerCallbacksRef& other)	{ return operator=(other._iUTokenizerCallbacks); }

	CEComICEUTokenizerCallbacksRef& operator=(const ICEUTokenizerCallbacks* iOther)
	{
		if (_iUTokenizerCallbacks != iOther)
		{
			if (_iUTokenizerCallbacks)
			{
				ICEUTokenizerCallbacks* tmp = _iUTokenizerCallbacks;
				_iUTokenizerCallbacks = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUTokenizerCallbacks = const_cast<ICEUTokenizerCallbacks*>(iOther);
			if (_iUTokenizerCallbacks)
			{
				_iUTokenizerCallbacks->_vtbl->_addRef(_iUTokenizerCallbacks);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUTokenizerCallbacks** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUTokenizerCallbacksRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUTokenizerCallbacks && "has a com object reference. clear first.");
		return & _iUTokenizerCallbacks; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUTokenizerCallbacksRef& other)
	{
		bool result = false;
		if (_iUTokenizerCallbacks)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUTokenizerCallbacks);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUTokenizerCallbacks ? _iUTokenizerCallbacks->_vtbl->_queryInterface(_iUTokenizerCallbacks, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUTokenizerCallbacks) { _iUTokenizerCallbacks->_vtbl->_addRef(_iUTokenizerCallbacks); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUTokenizerCallbacks) { _iUTokenizerCallbacks->_vtbl->_release(_iUTokenizerCallbacks); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult init(ICEUTokenizerCallbacks_tokenProc tp, ICEUTokenizerCallbacks_notifyProc np)	{ return _iUTokenizerCallbacks ? _iUTokenizerCallbacks->_vtbl->_init(_iUTokenizerCallbacks, tp, np) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult tokenProc(struct ICEUString* token, UINT8 tokenType, void* userData)	{ return _iUTokenizerCallbacks ? _iUTokenizerCallbacks->_vtbl->_tokenProc(_iUTokenizerCallbacks, token, tokenType, userData) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult notifyProc(UTF32CHAR c32, UINT8 notifyFlags, void* userData)	{ return _iUTokenizerCallbacks ? _iUTokenizerCallbacks->_vtbl->_notifyProc(_iUTokenizerCallbacks, c32, notifyFlags, userData) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEUTokenizerCallbacks* _iUTokenizerCallbacks;
};

/*! @}
 * end of CEComICEUTokenizerCallbacksRef
 */

/*! \defgroup CEComICEUTokenizerRef CEComICEUTokenizerRef
 * @{
 */

class CEComICEUTokenizerRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUTokenizerRef() : _iUTokenizer(0) {}
	CEComICEUTokenizerRef(ICEUTokenizer* iOther) : _iUTokenizer(0)
	{
		if (iOther)
		{
			_iUTokenizer = iOther;
			_iUTokenizer->_vtbl->_addRef(_iUTokenizer);
		}
	}

	CEComICEUTokenizerRef(const CEComICEUTokenizerRef& other) : _iUTokenizer(0)
	{
		if (other._iUTokenizer)
		{
			_iUTokenizer = other._iUTokenizer;
			_iUTokenizer->_vtbl->_addRef(_iUTokenizer);
		}
	}

	~CEComICEUTokenizerRef()
	{
		if (_iUTokenizer)
		{
			ICEUTokenizer* tmp = _iUTokenizer;
			_iUTokenizer = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUTokenizer* iUTokenizer;
				void* _ptr;
			} uIntf;
			uIntf.iUTokenizer = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUTokenizer, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUTokenizer)
				{
					ICEUTokenizer* tmp = _iUTokenizer;
					_iUTokenizer = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUTokenizer = uIntf.iUTokenizer;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUTokenizer* object() const	{ return _iUTokenizer; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUTokenizer* iOther)
	{
		if (_iUTokenizer)
		{
			ICEUTokenizer* tmp = _iUTokenizer;
			_iUTokenizer = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUTokenizer = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUTokenizer* detach()
	{
		ICEUTokenizer* iIntf = _iUTokenizer;
		_iUTokenizer = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUTokenizer* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUTokenizer;
		if (_iUTokenizer)
		{
			_iUTokenizer->_vtbl->_addRef(_iUTokenizer);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUTokenizer*() const	{ return _iUTokenizer; }
	FORCEINLINE_WITHOUT_DEBUG ICEUTokenizer& operator*() const	{ return *_iUTokenizer; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUTokenizer == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUTokenizer* iOther) const	{ return (_iUTokenizer != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUTokenizer* iOther) const	{ return (_iUTokenizer == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUTokenizerRef& operator=(const CEComICEUTokenizerRef& other)	{ return operator=(other._iUTokenizer); }

	CEComICEUTokenizerRef& operator=(const ICEUTokenizer* iOther)
	{
		if (_iUTokenizer != iOther)
		{
			if (_iUTokenizer)
			{
				ICEUTokenizer* tmp = _iUTokenizer;
				_iUTokenizer = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUTokenizer = const_cast<ICEUTokenizer*>(iOther);
			if (_iUTokenizer)
			{
				_iUTokenizer->_vtbl->_addRef(_iUTokenizer);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUTokenizer** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUTokenizerRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUTokenizer && "has a com object reference. clear first.");
		return & _iUTokenizer; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUTokenizerRef& other)
	{
		bool result = false;
		if (_iUTokenizer)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUTokenizer);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUTokenizer ? _iUTokenizer->_vtbl->_queryInterface(_iUTokenizer, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUTokenizer) { _iUTokenizer->_vtbl->_addRef(_iUTokenizer); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUTokenizer) { _iUTokenizer->_vtbl->_release(_iUTokenizer); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult init(UINT16 numOfStates, const CEUTokenizerState* stateSet, const UINT8* charKindTable )	{ return _iUTokenizer ? _iUTokenizer->_vtbl->_init(_iUTokenizer, numOfStates, stateSet, charKindTable) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult open(const struct ICEUTokenizerCallbacks* callbacks, void* userData)	{ return _iUTokenizer ? _iUTokenizer->_vtbl->_open(_iUTokenizer, callbacks, userData) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult writeString(const struct ICEUString* str)	{ return _iUTokenizer ? _iUTokenizer->_vtbl->_writeString(_iUTokenizer, str) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult close()	{ return _iUTokenizer ? _iUTokenizer->_vtbl->_close(_iUTokenizer) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEUTokenizer* _iUTokenizer;
};

/*! @}
 * end of CEComICEUTokenizerRef
 */

#endif // __cplusplus

#endif
