///////////////////////////////////////////////////////////////////////////////
// Copyright 2011 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICEDXMath_h
#define ICEDXMath_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"
#include "CEUITypes.h"
#include "CEDXMathTypes.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------


//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

typedef enum
{
	eCEDXRegionFormat_Win32HRGN, //!< Win32 GDI HRGN
} eCEDXRegionFormat;

typedef enum
{
	eCEDXRegionOp_FIRST = 0,

	eCEDXRegionOp_Union = eCEDXRegionOp_FIRST,
	eCEDXRegionOp_Intersect,
	eCEDXRegionOp_Subtract,

	eCEDXRegionOp_LAST = eCEDXRegionOp_Subtract
} eCEDXRegionOp;

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEDXImmutableRegion ICEDXImmutableRegion
 * @{
 */

/*!
 * ID of ICEDXImmutableRegion
 */
#define CEComIID_ICEDXImmutableRegion 0xd4c60d5b


/*!
 * ICEDXImmutableRegion
 */
typedef struct ICEDXImmutableRegion
{
	const struct ICEDXImmutableRegion_vtbl* _vtbl;
} ICEDXImmutableRegion;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXImmutableRegion	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXImmutableRegion_queryInterface) (ICEDXImmutableRegion* iDXImmutableRegion, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXImmutableRegion_addRef) (ICEDXImmutableRegion* iDXImmutableRegion);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXImmutableRegion_release) (ICEDXImmutableRegion* iDXImmutableRegion);

	//! Return whether this region is empty or not

typedef CEHResult (*iCEDXImmutableRegion_isEmpty) (ICEDXImmutableRegion* iDXImmutableRegion, bool *const isEmptyOut);

	//! Get the extents of this region

typedef CEHResult (*iCEDXImmutableRegion_getExtents) (ICEDXImmutableRegion* iDXImmutableRegion, CERect *const extentsOut);

	//! Check whether the given region is same as this

typedef CEHResult (*iCEDXImmutableRegion_isEqual) (ICEDXImmutableRegion* iDXImmutableRegion, struct ICEDXImmutableRegion* iRgn, bool *const isEqualOut);

	//! Check whether the specified point is in this region

typedef CEHResult (*iCEDXImmutableRegion_containPoint) (ICEDXImmutableRegion* iDXImmutableRegion, const CEPointBase* point, bool *const containOut);

	//! Check whether the specified rect is contained in this region

typedef CEHResult (*iCEDXImmutableRegion_containRect) (ICEDXImmutableRegion* iDXImmutableRegion, const CERect* rect, bool *const containOut);

	//! Lock the series of sub-rects in this region

typedef CEHResult (*iCEDXImmutableRegion_lockRect) (ICEDXImmutableRegion* iDXImmutableRegion, const CERect* rect, UINT32 flags, CERectSet *const lockedRectOut);

	//! Unlock the this region

typedef CEHResult (*iCEDXImmutableRegion_unlockRect) (ICEDXImmutableRegion* iDXImmutableRegion);

	//! Clone the given region 

typedef CEHResult (*iCEDXImmutableRegion_clone) (ICEDXImmutableRegion* iDXImmutableRegion, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the translated region

typedef CEHResult (*iCEDXImmutableRegion_getTranslatedRegion) (ICEDXImmutableRegion* iDXImmutableRegion, INT32 x, INT32 y, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the inverted region within the given rect

typedef CEHResult (*iCEDXImmutableRegion_getInvertedRegion) (ICEDXImmutableRegion* iDXImmutableRegion, const CERect* rect, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the union of this region and the given rect

typedef CEHResult (*iCEDXImmutableRegion_getUnionRect) (ICEDXImmutableRegion* iDXImmutableRegion, const CERect* rect, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the union between this region and the given one

typedef CEHResult (*iCEDXImmutableRegion_getUnionRegion) (ICEDXImmutableRegion* iDXImmutableRegion, struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the intersection of this region and the given rect

typedef CEHResult (*iCEDXImmutableRegion_getIntersectRect) (ICEDXImmutableRegion* iDXImmutableRegion, const CERect* rect, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the intersection between this region and the given one

typedef CEHResult (*iCEDXImmutableRegion_getIntersectRegion) (ICEDXImmutableRegion* iDXImmutableRegion, struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the subtract the given rect from this region 

typedef CEHResult (*iCEDXImmutableRegion_getSubtractRect) (ICEDXImmutableRegion* iDXImmutableRegion, const CERect* rect, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the subtract the given region from this

typedef CEHResult (*iCEDXImmutableRegion_getSubtractRegion) (ICEDXImmutableRegion* iDXImmutableRegion, struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut);

	//! apply operator between this and given rect, and return result as new region

typedef CEHResult (*iCEDXImmutableRegion_applyWithRect) (ICEDXImmutableRegion* iDXImmutableRegion, eCEDXRegionOp op, const CERect* rect, struct ICEDXRegion* *const iRgnOut);

	//! apply operator between this and given ICEDXImmutableRegion, return result as new region

typedef CEHResult (*iCEDXImmutableRegion_applyWithImmutableRegion) (ICEDXImmutableRegion* iDXImmutableRegion, eCEDXRegionOp op, struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut);

	//! apply operator between this and given ICEDXImmutableRegion, return result as new region

typedef CEHResult (*iCEDXImmutableRegion_applyWithRegion) (ICEDXImmutableRegion* iDXImmutableRegion, eCEDXRegionOp op, struct ICEDXRegion* iRgn, struct ICEDXRegion* *const iRgnOut);

/*!
 * V-table of ICEDXImmutableRegion
 */
struct ICEDXImmutableRegion_vtbl
{
	iCEDXImmutableRegion_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXImmutableRegion_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXImmutableRegion_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXImmutableRegion_isEmpty	_isEmpty;
	iCEDXImmutableRegion_getExtents	_getExtents;
	iCEDXImmutableRegion_isEqual	_isEqual;
	iCEDXImmutableRegion_containPoint	_containPoint;
	iCEDXImmutableRegion_containRect	_containRect;
	iCEDXImmutableRegion_lockRect	_lockRect;
	iCEDXImmutableRegion_unlockRect	_unlockRect;
	iCEDXImmutableRegion_clone	_clone;
	iCEDXImmutableRegion_getTranslatedRegion	_getTranslatedRegion;
	iCEDXImmutableRegion_getInvertedRegion	_getInvertedRegion;
	iCEDXImmutableRegion_getUnionRect	_getUnionRect;
	iCEDXImmutableRegion_getUnionRegion	_getUnionRegion;
	iCEDXImmutableRegion_getIntersectRect	_getIntersectRect;
	iCEDXImmutableRegion_getIntersectRegion	_getIntersectRegion;
	iCEDXImmutableRegion_getSubtractRect	_getSubtractRect;
	iCEDXImmutableRegion_getSubtractRegion	_getSubtractRegion;
	iCEDXImmutableRegion_applyWithRect	_applyWithRect;
	iCEDXImmutableRegion_applyWithImmutableRegion	_applyWithImmutableRegion;
	iCEDXImmutableRegion_applyWithRegion	_applyWithRegion;
};

/*! @}
 * end of ICEDXImmutableRegion
 */

/*! \defgroup ICEDXRegion ICEDXRegion
 * @{
 */

/*!
 * ID of ICEDXRegion
 */
#define CEComIID_ICEDXRegion 0xf8f4c440


/*!
 * ICEDXRegion
 */
typedef struct ICEDXRegion
{
	const struct ICEDXRegion_vtbl* _vtbl;
} ICEDXRegion;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXRegion	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXRegion_queryInterface) (ICEDXRegion* iDXRegion, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXRegion_addRef) (ICEDXRegion* iDXRegion);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXRegion_release) (ICEDXRegion* iDXRegion);

	//! Return whether this region is empty or not

typedef CEHResult (*iCEDXRegion_isEmpty) (ICEDXRegion* iDXRegion, bool *const isEmptyOut);

	//! Get the extents of this region

typedef CEHResult (*iCEDXRegion_getExtents) (ICEDXRegion* iDXRegion, CERect *const extentsOut);

	//! Check whether the given region is same as this

typedef CEHResult (*iCEDXRegion_isEqual) (ICEDXRegion* iDXRegion, struct ICEDXImmutableRegion* iRgn, bool *const isEqualOut);

	//! Check whether the specified point is in this region

typedef CEHResult (*iCEDXRegion_containPoint) (ICEDXRegion* iDXRegion, const CEPointBase* point, bool *const containOut);

	//! Check whether the specified rect is contained in this region

typedef CEHResult (*iCEDXRegion_containRect) (ICEDXRegion* iDXRegion, const CERect* rect, bool *const containOut);

	//! Lock the series of sub-rects in this region

typedef CEHResult (*iCEDXRegion_lockRect) (ICEDXRegion* iDXRegion, const CERect* rect, UINT32 flags, CERectSet *const lockedRectOut);

	//! Unlock the this region

typedef CEHResult (*iCEDXRegion_unlockRect) (ICEDXRegion* iDXRegion);

	//! Clone the given region 

typedef CEHResult (*iCEDXRegion_clone) (ICEDXRegion* iDXRegion, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the translated region

typedef CEHResult (*iCEDXRegion_getTranslatedRegion) (ICEDXRegion* iDXRegion, INT32 x, INT32 y, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the inverted region within the given rect

typedef CEHResult (*iCEDXRegion_getInvertedRegion) (ICEDXRegion* iDXRegion, const CERect* rect, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the union of this region and the given rect

typedef CEHResult (*iCEDXRegion_getUnionRect) (ICEDXRegion* iDXRegion, const CERect* rect, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the union between this region and the given one

typedef CEHResult (*iCEDXRegion_getUnionRegion) (ICEDXRegion* iDXRegion, struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the intersection of this region and the given rect

typedef CEHResult (*iCEDXRegion_getIntersectRect) (ICEDXRegion* iDXRegion, const CERect* rect, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the intersection between this region and the given one

typedef CEHResult (*iCEDXRegion_getIntersectRegion) (ICEDXRegion* iDXRegion, struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the subtract the given rect from this region 

typedef CEHResult (*iCEDXRegion_getSubtractRect) (ICEDXRegion* iDXRegion, const CERect* rect, struct ICEDXRegion* *const iRgnOut);

	//! Calculate the subtract the given region from this

typedef CEHResult (*iCEDXRegion_getSubtractRegion) (ICEDXRegion* iDXRegion, struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut);

	//! apply operator between this and given rect, and return result as new region

typedef CEHResult (*iCEDXRegion_applyWithRect) (ICEDXRegion* iDXRegion, eCEDXRegionOp op, const CERect* rect, struct ICEDXRegion* *const iRgnOut);

	//! apply operator between this and given ICEDXImmutableRegion, return result as new region

typedef CEHResult (*iCEDXRegion_applyWithImmutableRegion) (ICEDXRegion* iDXRegion, eCEDXRegionOp op, struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut);

	//! apply operator between this and given ICEDXImmutableRegion, return result as new region

typedef CEHResult (*iCEDXRegion_applyWithRegion) (ICEDXRegion* iDXRegion, eCEDXRegionOp op, struct ICEDXRegion* iRgn, struct ICEDXRegion* *const iRgnOut);

	//! Clear the region's rectangles

typedef CEHResult (*iCEDXRegion_clear) (ICEDXRegion* iDXRegion);

	//! Calculate union of pAddRect with region object.
	//! \param[in] pAddRect	adding rectangle

typedef CEHResult (*iCEDXRegion_calcUnion) (ICEDXRegion* iDXRegion, const CERect *pAddRect);

	//! Get number of calculated rectangles in region.
	//! Operation of this API is same as "getRegionScans(NULL,0,pRectNumOut)".
	//! \param[out] pRectNumOut	the number of rectangles in the region.
	//! this API returns row rect data. sometimes they are so many rects.
	//! if you want to get some gathered rects, you have to use 'getNumOfRegionScansInScreen'

typedef CEHResult (*iCEDXRegion_getNumOfRegionScans) (ICEDXRegion* iDXRegion, UINT32 *const pRectNum);

	//! Get calculated rectangles in region.
	//! \param[out] pRects		the pointer of CERect which getRegionScans filled with.
	//!							If pRect is null, only pRectNum is returned.
	//! \param		numOfRects  the number of rectangle in pRects.
	//!                         numOfRects is treated as the maximun count of return rect,
	//!                         so, if numOfRects is smaller than retaining rect count,
	//!                         getRegionScans returns only the specified number of rect in pRect 
	//!                         and the numOfRects value is set to pRectNumOut. 
	//! \param[out] pRectNumOut	the number of rectangles in the region.
	//! this API returns row rect data. sometimes they are so many rects.
	//! if you want to get some gathered rects, you have to use 'getRegionScansInScreen'

typedef CEHResult (*iCEDXRegion_getRegionScans) (ICEDXRegion* iDXRegion, CERect *const  pRects, UINT32 numOfRects, UINT32 *const pRectNumOut);

	//! Get number of calculated rectangles in region.
	//! Some near Rect is brought together in their bbox.
	//! 'near' is judged by position in screen

typedef CEHResult (*iCEDXRegion_getNumOfRegionScansInScreen) (ICEDXRegion* iDXRegion, INT32 const screenWidth, INT32 const screenHeight, UINT32 *const pRectNumOut);

	//! Get calculated rectangles in region.
	//! Some near Rect is brought together in their bbox. 
	//! 'near' is judged by position in screen

typedef CEHResult (*iCEDXRegion_getRegionScansInScreen) (ICEDXRegion* iDXRegion, INT32 const screenWidth, INT32 const screenHeight, CERect *const  pRects, UINT32 numOfRects, UINT32 *const pRectNumOut);

	//! Translate this region

typedef CEHResult (*iCEDXRegion_translate) (ICEDXRegion* iDXRegion, INT32 x, INT32 y);

	//! Invert this region within the given rect

typedef CEHResult (*iCEDXRegion_invert) (ICEDXRegion* iDXRegion, const CERect* rect);

	//! Calculate the union of this region and the given rect

typedef CEHResult (*iCEDXRegion_unionRect) (ICEDXRegion* iDXRegion, const CERect* rect);

	//! Calculate the union between this region and the given one

typedef CEHResult (*iCEDXRegion_unionRegion) (ICEDXRegion* iDXRegion, struct ICEDXImmutableRegion* iRgn);

	//! Calculate the intersection of this region and the given rect

typedef CEHResult (*iCEDXRegion_intersectRect) (ICEDXRegion* iDXRegion, const CERect* rect);

	//! Calculate the intersection between this region and the given one

typedef CEHResult (*iCEDXRegion_intersectRegion) (ICEDXRegion* iDXRegion, struct ICEDXImmutableRegion* iRgn);

	//! Calculate the subtract the given rect from this region 

typedef CEHResult (*iCEDXRegion_subtractRect) (ICEDXRegion* iDXRegion, const CERect* rect);

	//! Calculate the subtract the given region from this

typedef CEHResult (*iCEDXRegion_subtractRegion) (ICEDXRegion* iDXRegion, struct ICEDXImmutableRegion* iRgn);

	//! apply operator between this and given rect, and modify self with its result

typedef CEHResult (*iCEDXRegion_applySelfWithRect) (ICEDXRegion* iDXRegion, eCEDXRegionOp op, const CERect* rect);

	//! apply operator between this and given ICEDXImmutableRegion, and modify self with its result

typedef CEHResult (*iCEDXRegion_applySelfWithImmutableRegion) (ICEDXRegion* iDXRegion, eCEDXRegionOp op, struct ICEDXImmutableRegion* iRgn);

	//! apply operator between this and given ICEDXImmutableRegion, and modify self with its result

typedef CEHResult (*iCEDXRegion_applySelfWithRegion) (ICEDXRegion* iDXRegion, eCEDXRegionOp op, struct ICEDXRegion* iRgn);

	//! initialize this from other format region data

typedef CEHResult (*iCEDXRegion_initWithNativeRegion) (ICEDXRegion* iDXRegion, eCEDXRegionFormat format, void* regionData);

	//! initialize this from given rect

typedef CEHResult (*iCEDXRegion_initWithRect) (ICEDXRegion* iDXRegion, const CERect* rect);

	//! initialize this with given region

typedef CEHResult (*iCEDXRegion_initWithRegion) (ICEDXRegion* iDXRegion, struct ICEDXImmutableRegion* iRgn);

	//! initialize this with the result of set operation between region and rect

typedef CEHResult (*iCEDXRegion_initWithRegionRect) (ICEDXRegion* iDXRegion, eCEDXRegionOp binaryOp, struct ICEDXImmutableRegion* iRgn, const CERect* rect);

	//! initialize this with the result of set operation between regions

typedef CEHResult (*iCEDXRegion_initWithRegionRegion) (ICEDXRegion* iDXRegion, eCEDXRegionOp binaryOp, struct ICEDXImmutableRegion* iRgn1, struct ICEDXImmutableRegion* iRgn2);

/*!
 * V-table of ICEDXRegion
 */
struct ICEDXRegion_vtbl
{
	iCEDXRegion_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXRegion_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXRegion_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXRegion_isEmpty	_isEmpty;
	iCEDXRegion_getExtents	_getExtents;
	iCEDXRegion_isEqual	_isEqual;
	iCEDXRegion_containPoint	_containPoint;
	iCEDXRegion_containRect	_containRect;
	iCEDXRegion_lockRect	_lockRect;
	iCEDXRegion_unlockRect	_unlockRect;
	iCEDXRegion_clone	_clone;
	iCEDXRegion_getTranslatedRegion	_getTranslatedRegion;
	iCEDXRegion_getInvertedRegion	_getInvertedRegion;
	iCEDXRegion_getUnionRect	_getUnionRect;
	iCEDXRegion_getUnionRegion	_getUnionRegion;
	iCEDXRegion_getIntersectRect	_getIntersectRect;
	iCEDXRegion_getIntersectRegion	_getIntersectRegion;
	iCEDXRegion_getSubtractRect	_getSubtractRect;
	iCEDXRegion_getSubtractRegion	_getSubtractRegion;
	iCEDXRegion_applyWithRect	_applyWithRect;
	iCEDXRegion_applyWithImmutableRegion	_applyWithImmutableRegion;
	iCEDXRegion_applyWithRegion	_applyWithRegion;
	iCEDXRegion_clear	_clear;
	iCEDXRegion_calcUnion	_calcUnion;
	iCEDXRegion_getNumOfRegionScans	_getNumOfRegionScans;
	iCEDXRegion_getRegionScans	_getRegionScans;
	iCEDXRegion_getNumOfRegionScansInScreen	_getNumOfRegionScansInScreen;
	iCEDXRegion_getRegionScansInScreen	_getRegionScansInScreen;
	iCEDXRegion_translate	_translate;
	iCEDXRegion_invert	_invert;
	iCEDXRegion_unionRect	_unionRect;
	iCEDXRegion_unionRegion	_unionRegion;
	iCEDXRegion_intersectRect	_intersectRect;
	iCEDXRegion_intersectRegion	_intersectRegion;
	iCEDXRegion_subtractRect	_subtractRect;
	iCEDXRegion_subtractRegion	_subtractRegion;
	iCEDXRegion_applySelfWithRect	_applySelfWithRect;
	iCEDXRegion_applySelfWithImmutableRegion	_applySelfWithImmutableRegion;
	iCEDXRegion_applySelfWithRegion	_applySelfWithRegion;
	iCEDXRegion_initWithNativeRegion	_initWithNativeRegion;
	iCEDXRegion_initWithRect	_initWithRect;
	iCEDXRegion_initWithRegion	_initWithRegion;
	iCEDXRegion_initWithRegionRect	_initWithRegionRect;
	iCEDXRegion_initWithRegionRegion	_initWithRegionRegion;
};

/*! @}
 * end of ICEDXRegion
 */

/*! \defgroup ICEDXMath ICEDXMath
 * @{
 */

/*!
 * ID of ICEDXMath
 */
#define CEComIID_ICEDXMath 0xd495b84d


/*!
 * ICEDXMath
 */
typedef struct ICEDXMath
{
	const struct ICEDXMath_vtbl* _vtbl;
} ICEDXMath;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXMath	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXMath_queryInterface) (ICEDXMath* iDXMath, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXMath_addRef) (ICEDXMath* iDXMath);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXMath_release) (ICEDXMath* iDXMath);

	//! Creates an identity matrix.
	//! \param[in, out] pOut	Pointer to the structure that is the result of the operation. 

typedef CEHResult (*iCEDXMath_matrixIdentity44) (ICEDXMath* iDXMath, CEDXMatrix44* pOut);

typedef CEHResult (*iCEDXMath_matrixIdentity33) (ICEDXMath* iDXMath, CEDXMatrix33* pOut);

	//! Out = M1 * M2

typedef CEHResult (*iCEDXMath_matrixMultiply44) (ICEDXMath* iDXMath, CEDXMatrix44* pOut, CEDXMatrix44* pM1, CEDXMatrix44* pM2);

typedef CEHResult (*iCEDXMath_matrixMultiply33) (ICEDXMath* iDXMath, CEDXMatrix33* pOut, CEDXMatrix33* pM1, CEDXMatrix33* pM2);

	//! Create an translation matrix.
	//!
	//! \param[out] pOut		Pointer to the structure that is the result of the operation. 
	//! \param[in] offsetX		Real number that specifies the horizontal component of translation.
	//! \param[in] offsetY		Real number that specifies the horizontal component of translation.

typedef CEHResult (*iCEDXMath_matrixTranslate33) (ICEDXMath* iDXMath, CEDXMatrix33* pOut, float tX, float tY);

	//! Create an scaling matrix.

typedef CEHResult (*iCEDXMath_matrixScale33) (ICEDXMath* iDXMath, CEDXMatrix33* pOut, float sX, float sY);

	//! Create an rotation matrix.

typedef CEHResult (*iCEDXMath_matrixRotate33) (ICEDXMath* iDXMath, CEDXMatrix33* pOut, float angle);

	//! Builds a customized, left-handed orthogonal projection matrix.
	//! This function uses the following formula to compute the returned matrix. 
	//!
	//! 2/(r-l)      0            0           0
	//! 0            2/(t-b)      0           0
	//! 0            0            1/(zf-zn)   0
	//! (l+r)/(l-r)  (t+b)/(b-t)  zn/(zn-zf)  1
	//! 
	//! \param[in,out] pOut		Pointer to the structure that is the result of the operation. 
	//! \param[in] l			Minimum x-value of view volume. 
	//! \param[in] r			Maximum x-value of view volume. 
	//! \param[in] b			Minimum y-value of view volume. 
	//! \param[in] t			Maximum y-value of view volume. 
	//! \param[in] zn			Minimum z-value of the view volume. 
	//! \param[in] zf			Maximum z-value of the view volume. 

typedef CEHResult (*iCEDXMath_matrixOrthoOffCenterLH) (ICEDXMath* iDXMath, CEDXMatrix44* pOut, float l, float r, float b, float t, float zn, float zf);

	//! Region object factory

typedef CEHResult (*iCEDXMath_createRegion) (ICEDXMath* iDXMath, struct ICEDXRegion* *const piRegionOut);

/*!
 * V-table of ICEDXMath
 */
struct ICEDXMath_vtbl
{
	iCEDXMath_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXMath_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXMath_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXMath_matrixIdentity44	_matrixIdentity44;
	iCEDXMath_matrixIdentity33	_matrixIdentity33;
	iCEDXMath_matrixMultiply44	_matrixMultiply44;
	iCEDXMath_matrixMultiply33	_matrixMultiply33;
	iCEDXMath_matrixTranslate33	_matrixTranslate33;
	iCEDXMath_matrixScale33	_matrixScale33;
	iCEDXMath_matrixRotate33	_matrixRotate33;
	iCEDXMath_matrixOrthoOffCenterLH	_matrixOrthoOffCenterLH;
	iCEDXMath_createRegion	_createRegion;
};

/*! @}
 * end of ICEDXMath
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEDXImmutableRegionRef CEComICEDXImmutableRegionRef
 * @{
 */

class CEComICEDXImmutableRegionRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXImmutableRegionRef() : _iDXImmutableRegion(0) {}
	CEComICEDXImmutableRegionRef(ICEDXImmutableRegion* iOther) : _iDXImmutableRegion(0)
	{
		if (iOther)
		{
			_iDXImmutableRegion = iOther;
			_iDXImmutableRegion->_vtbl->_addRef(_iDXImmutableRegion);
		}
	}

	CEComICEDXImmutableRegionRef(const CEComICEDXImmutableRegionRef& other) : _iDXImmutableRegion(0)
	{
		if (other._iDXImmutableRegion)
		{
			_iDXImmutableRegion = other._iDXImmutableRegion;
			_iDXImmutableRegion->_vtbl->_addRef(_iDXImmutableRegion);
		}
	}

	~CEComICEDXImmutableRegionRef()
	{
		if (_iDXImmutableRegion)
		{
			ICEDXImmutableRegion* tmp = _iDXImmutableRegion;
			_iDXImmutableRegion = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXImmutableRegion* iDXImmutableRegion;
				void* _ptr;
			} uIntf;
			uIntf.iDXImmutableRegion = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXImmutableRegion, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXImmutableRegion)
				{
					ICEDXImmutableRegion* tmp = _iDXImmutableRegion;
					_iDXImmutableRegion = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXImmutableRegion = uIntf.iDXImmutableRegion;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXImmutableRegion* object() const	{ return _iDXImmutableRegion; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXImmutableRegion* iOther)
	{
		if (_iDXImmutableRegion)
		{
			ICEDXImmutableRegion* tmp = _iDXImmutableRegion;
			_iDXImmutableRegion = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXImmutableRegion = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXImmutableRegion* detach()
	{
		ICEDXImmutableRegion* iIntf = _iDXImmutableRegion;
		_iDXImmutableRegion = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXImmutableRegion* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXImmutableRegion;
		if (_iDXImmutableRegion)
		{
			_iDXImmutableRegion->_vtbl->_addRef(_iDXImmutableRegion);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXImmutableRegion*() const	{ return _iDXImmutableRegion; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXImmutableRegion& operator*() const	{ return *_iDXImmutableRegion; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXImmutableRegion == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXImmutableRegion* iOther) const	{ return (_iDXImmutableRegion != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXImmutableRegion* iOther) const	{ return (_iDXImmutableRegion == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXImmutableRegionRef& operator=(const CEComICEDXImmutableRegionRef& other)	{ return operator=(other._iDXImmutableRegion); }

	CEComICEDXImmutableRegionRef& operator=(const ICEDXImmutableRegion* iOther)
	{
		if (_iDXImmutableRegion != iOther)
		{
			if (_iDXImmutableRegion)
			{
				ICEDXImmutableRegion* tmp = _iDXImmutableRegion;
				_iDXImmutableRegion = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXImmutableRegion = const_cast<ICEDXImmutableRegion*>(iOther);
			if (_iDXImmutableRegion)
			{
				_iDXImmutableRegion->_vtbl->_addRef(_iDXImmutableRegion);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXImmutableRegion** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXImmutableRegionRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXImmutableRegion && "has a com object reference. clear first.");
		return & _iDXImmutableRegion; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXImmutableRegionRef& other)
	{
		bool result = false;
		if (_iDXImmutableRegion)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXImmutableRegion);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_queryInterface(_iDXImmutableRegion, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXImmutableRegion) { _iDXImmutableRegion->_vtbl->_addRef(_iDXImmutableRegion); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXImmutableRegion) { _iDXImmutableRegion->_vtbl->_release(_iDXImmutableRegion); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isEmpty(bool *const isEmptyOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_isEmpty(_iDXImmutableRegion, isEmptyOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getExtents(CERect *const extentsOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_getExtents(_iDXImmutableRegion, extentsOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isEqual(struct ICEDXImmutableRegion* iRgn, bool *const isEqualOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_isEqual(_iDXImmutableRegion, iRgn, isEqualOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult containPoint(const CEPointBase* point, bool *const containOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_containPoint(_iDXImmutableRegion, point, containOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult containRect(const CERect* rect, bool *const containOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_containRect(_iDXImmutableRegion, rect, containOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult lockRect(const CERect* rect, UINT32 flags, CERectSet *const lockedRectOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_lockRect(_iDXImmutableRegion, rect, flags, lockedRectOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unlockRect()	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_unlockRect(_iDXImmutableRegion) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult clone(struct ICEDXRegion* *const iRgnOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_clone(_iDXImmutableRegion, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getTranslatedRegion(INT32 x, INT32 y, struct ICEDXRegion* *const iRgnOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_getTranslatedRegion(_iDXImmutableRegion, x, y, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getInvertedRegion(const CERect* rect, struct ICEDXRegion* *const iRgnOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_getInvertedRegion(_iDXImmutableRegion, rect, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getUnionRect(const CERect* rect, struct ICEDXRegion* *const iRgnOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_getUnionRect(_iDXImmutableRegion, rect, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getUnionRegion(struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_getUnionRegion(_iDXImmutableRegion, iRgn, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getIntersectRect(const CERect* rect, struct ICEDXRegion* *const iRgnOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_getIntersectRect(_iDXImmutableRegion, rect, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getIntersectRegion(struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_getIntersectRegion(_iDXImmutableRegion, iRgn, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getSubtractRect(const CERect* rect, struct ICEDXRegion* *const iRgnOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_getSubtractRect(_iDXImmutableRegion, rect, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getSubtractRegion(struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_getSubtractRegion(_iDXImmutableRegion, iRgn, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult applyWithRect(eCEDXRegionOp op, const CERect* rect, struct ICEDXRegion* *const iRgnOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_applyWithRect(_iDXImmutableRegion, op, rect, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult applyWithImmutableRegion(eCEDXRegionOp op, struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_applyWithImmutableRegion(_iDXImmutableRegion, op, iRgn, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult applyWithRegion(eCEDXRegionOp op, struct ICEDXRegion* iRgn, struct ICEDXRegion* *const iRgnOut)	{ return _iDXImmutableRegion ? _iDXImmutableRegion->_vtbl->_applyWithRegion(_iDXImmutableRegion, op, iRgn, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXImmutableRegion* _iDXImmutableRegion;
};

/*! @}
 * end of CEComICEDXImmutableRegionRef
 */

/*! \defgroup CEComICEDXRegionRef CEComICEDXRegionRef
 * @{
 */

class CEComICEDXRegionRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXRegionRef() : _iDXRegion(0) {}
	CEComICEDXRegionRef(ICEDXRegion* iOther) : _iDXRegion(0)
	{
		if (iOther)
		{
			_iDXRegion = iOther;
			_iDXRegion->_vtbl->_addRef(_iDXRegion);
		}
	}

	CEComICEDXRegionRef(const CEComICEDXRegionRef& other) : _iDXRegion(0)
	{
		if (other._iDXRegion)
		{
			_iDXRegion = other._iDXRegion;
			_iDXRegion->_vtbl->_addRef(_iDXRegion);
		}
	}

	~CEComICEDXRegionRef()
	{
		if (_iDXRegion)
		{
			ICEDXRegion* tmp = _iDXRegion;
			_iDXRegion = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXRegion* iDXRegion;
				void* _ptr;
			} uIntf;
			uIntf.iDXRegion = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXRegion, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXRegion)
				{
					ICEDXRegion* tmp = _iDXRegion;
					_iDXRegion = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXRegion = uIntf.iDXRegion;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXRegion* object() const	{ return _iDXRegion; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXRegion* iOther)
	{
		if (_iDXRegion)
		{
			ICEDXRegion* tmp = _iDXRegion;
			_iDXRegion = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXRegion = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXRegion* detach()
	{
		ICEDXRegion* iIntf = _iDXRegion;
		_iDXRegion = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXRegion* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXRegion;
		if (_iDXRegion)
		{
			_iDXRegion->_vtbl->_addRef(_iDXRegion);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXRegion*() const	{ return _iDXRegion; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXRegion& operator*() const	{ return *_iDXRegion; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXRegion == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXRegion* iOther) const	{ return (_iDXRegion != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXRegion* iOther) const	{ return (_iDXRegion == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXRegionRef& operator=(const CEComICEDXRegionRef& other)	{ return operator=(other._iDXRegion); }

	CEComICEDXRegionRef& operator=(const ICEDXRegion* iOther)
	{
		if (_iDXRegion != iOther)
		{
			if (_iDXRegion)
			{
				ICEDXRegion* tmp = _iDXRegion;
				_iDXRegion = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXRegion = const_cast<ICEDXRegion*>(iOther);
			if (_iDXRegion)
			{
				_iDXRegion->_vtbl->_addRef(_iDXRegion);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXRegion** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXRegionRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXRegion && "has a com object reference. clear first.");
		return & _iDXRegion; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXRegionRef& other)
	{
		bool result = false;
		if (_iDXRegion)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXRegion);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_queryInterface(_iDXRegion, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXRegion) { _iDXRegion->_vtbl->_addRef(_iDXRegion); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXRegion) { _iDXRegion->_vtbl->_release(_iDXRegion); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isEmpty(bool *const isEmptyOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_isEmpty(_iDXRegion, isEmptyOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getExtents(CERect *const extentsOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getExtents(_iDXRegion, extentsOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isEqual(struct ICEDXImmutableRegion* iRgn, bool *const isEqualOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_isEqual(_iDXRegion, iRgn, isEqualOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult containPoint(const CEPointBase* point, bool *const containOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_containPoint(_iDXRegion, point, containOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult containRect(const CERect* rect, bool *const containOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_containRect(_iDXRegion, rect, containOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult lockRect(const CERect* rect, UINT32 flags, CERectSet *const lockedRectOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_lockRect(_iDXRegion, rect, flags, lockedRectOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unlockRect()	{ return _iDXRegion ? _iDXRegion->_vtbl->_unlockRect(_iDXRegion) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult clone(struct ICEDXRegion* *const iRgnOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_clone(_iDXRegion, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getTranslatedRegion(INT32 x, INT32 y, struct ICEDXRegion* *const iRgnOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getTranslatedRegion(_iDXRegion, x, y, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getInvertedRegion(const CERect* rect, struct ICEDXRegion* *const iRgnOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getInvertedRegion(_iDXRegion, rect, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getUnionRect(const CERect* rect, struct ICEDXRegion* *const iRgnOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getUnionRect(_iDXRegion, rect, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getUnionRegion(struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getUnionRegion(_iDXRegion, iRgn, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getIntersectRect(const CERect* rect, struct ICEDXRegion* *const iRgnOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getIntersectRect(_iDXRegion, rect, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getIntersectRegion(struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getIntersectRegion(_iDXRegion, iRgn, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getSubtractRect(const CERect* rect, struct ICEDXRegion* *const iRgnOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getSubtractRect(_iDXRegion, rect, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getSubtractRegion(struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getSubtractRegion(_iDXRegion, iRgn, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult applyWithRect(eCEDXRegionOp op, const CERect* rect, struct ICEDXRegion* *const iRgnOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_applyWithRect(_iDXRegion, op, rect, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult applyWithImmutableRegion(eCEDXRegionOp op, struct ICEDXImmutableRegion* iRgn, struct ICEDXRegion* *const iRgnOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_applyWithImmutableRegion(_iDXRegion, op, iRgn, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult applyWithRegion(eCEDXRegionOp op, struct ICEDXRegion* iRgn, struct ICEDXRegion* *const iRgnOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_applyWithRegion(_iDXRegion, op, iRgn, iRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult clear()	{ return _iDXRegion ? _iDXRegion->_vtbl->_clear(_iDXRegion) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult calcUnion(const CERect *pAddRect)	{ return _iDXRegion ? _iDXRegion->_vtbl->_calcUnion(_iDXRegion, pAddRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getNumOfRegionScans(UINT32 *const pRectNum)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getNumOfRegionScans(_iDXRegion, pRectNum) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getRegionScans(CERect *const  pRects, UINT32 numOfRects, UINT32 *const pRectNumOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getRegionScans(_iDXRegion, pRects, numOfRects, pRectNumOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getNumOfRegionScansInScreen(INT32 const screenWidth, INT32 const screenHeight, UINT32 *const pRectNumOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getNumOfRegionScansInScreen(_iDXRegion, screenWidth, screenHeight, pRectNumOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getRegionScansInScreen(INT32 const screenWidth, INT32 const screenHeight, CERect *const  pRects, UINT32 numOfRects, UINT32 *const pRectNumOut)	{ return _iDXRegion ? _iDXRegion->_vtbl->_getRegionScansInScreen(_iDXRegion, screenWidth, screenHeight, pRects, numOfRects, pRectNumOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult translate(INT32 x, INT32 y)	{ return _iDXRegion ? _iDXRegion->_vtbl->_translate(_iDXRegion, x, y) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult invert(const CERect* rect)	{ return _iDXRegion ? _iDXRegion->_vtbl->_invert(_iDXRegion, rect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unionRect(const CERect* rect)	{ return _iDXRegion ? _iDXRegion->_vtbl->_unionRect(_iDXRegion, rect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unionRegion(struct ICEDXImmutableRegion* iRgn)	{ return _iDXRegion ? _iDXRegion->_vtbl->_unionRegion(_iDXRegion, iRgn) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult intersectRect(const CERect* rect)	{ return _iDXRegion ? _iDXRegion->_vtbl->_intersectRect(_iDXRegion, rect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult intersectRegion(struct ICEDXImmutableRegion* iRgn)	{ return _iDXRegion ? _iDXRegion->_vtbl->_intersectRegion(_iDXRegion, iRgn) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult subtractRect(const CERect* rect)	{ return _iDXRegion ? _iDXRegion->_vtbl->_subtractRect(_iDXRegion, rect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult subtractRegion(struct ICEDXImmutableRegion* iRgn)	{ return _iDXRegion ? _iDXRegion->_vtbl->_subtractRegion(_iDXRegion, iRgn) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult applySelfWithRect(eCEDXRegionOp op, const CERect* rect)	{ return _iDXRegion ? _iDXRegion->_vtbl->_applySelfWithRect(_iDXRegion, op, rect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult applySelfWithImmutableRegion(eCEDXRegionOp op, struct ICEDXImmutableRegion* iRgn)	{ return _iDXRegion ? _iDXRegion->_vtbl->_applySelfWithImmutableRegion(_iDXRegion, op, iRgn) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult applySelfWithRegion(eCEDXRegionOp op, struct ICEDXRegion* iRgn)	{ return _iDXRegion ? _iDXRegion->_vtbl->_applySelfWithRegion(_iDXRegion, op, iRgn) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initWithNativeRegion(eCEDXRegionFormat format, void* regionData)	{ return _iDXRegion ? _iDXRegion->_vtbl->_initWithNativeRegion(_iDXRegion, format, regionData) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initWithRect(const CERect* rect)	{ return _iDXRegion ? _iDXRegion->_vtbl->_initWithRect(_iDXRegion, rect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initWithRegion(struct ICEDXImmutableRegion* iRgn)	{ return _iDXRegion ? _iDXRegion->_vtbl->_initWithRegion(_iDXRegion, iRgn) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initWithRegionRect(eCEDXRegionOp binaryOp, struct ICEDXImmutableRegion* iRgn, const CERect* rect)	{ return _iDXRegion ? _iDXRegion->_vtbl->_initWithRegionRect(_iDXRegion, binaryOp, iRgn, rect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initWithRegionRegion(eCEDXRegionOp binaryOp, struct ICEDXImmutableRegion* iRgn1, struct ICEDXImmutableRegion* iRgn2)	{ return _iDXRegion ? _iDXRegion->_vtbl->_initWithRegionRegion(_iDXRegion, binaryOp, iRgn1, iRgn2) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXRegion* _iDXRegion;
};

/*! @}
 * end of CEComICEDXRegionRef
 */

/*! \defgroup CEComICEDXMathRef CEComICEDXMathRef
 * @{
 */

class CEComICEDXMathRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXMathRef() : _iDXMath(0) {}
	CEComICEDXMathRef(ICEDXMath* iOther) : _iDXMath(0)
	{
		if (iOther)
		{
			_iDXMath = iOther;
			_iDXMath->_vtbl->_addRef(_iDXMath);
		}
	}

	CEComICEDXMathRef(const CEComICEDXMathRef& other) : _iDXMath(0)
	{
		if (other._iDXMath)
		{
			_iDXMath = other._iDXMath;
			_iDXMath->_vtbl->_addRef(_iDXMath);
		}
	}

	~CEComICEDXMathRef()
	{
		if (_iDXMath)
		{
			ICEDXMath* tmp = _iDXMath;
			_iDXMath = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXMath* iDXMath;
				void* _ptr;
			} uIntf;
			uIntf.iDXMath = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXMath, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXMath)
				{
					ICEDXMath* tmp = _iDXMath;
					_iDXMath = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXMath = uIntf.iDXMath;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXMath* object() const	{ return _iDXMath; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXMath* iOther)
	{
		if (_iDXMath)
		{
			ICEDXMath* tmp = _iDXMath;
			_iDXMath = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXMath = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXMath* detach()
	{
		ICEDXMath* iIntf = _iDXMath;
		_iDXMath = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXMath* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXMath;
		if (_iDXMath)
		{
			_iDXMath->_vtbl->_addRef(_iDXMath);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXMath*() const	{ return _iDXMath; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXMath& operator*() const	{ return *_iDXMath; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXMath == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXMath* iOther) const	{ return (_iDXMath != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXMath* iOther) const	{ return (_iDXMath == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXMathRef& operator=(const CEComICEDXMathRef& other)	{ return operator=(other._iDXMath); }

	CEComICEDXMathRef& operator=(const ICEDXMath* iOther)
	{
		if (_iDXMath != iOther)
		{
			if (_iDXMath)
			{
				ICEDXMath* tmp = _iDXMath;
				_iDXMath = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXMath = const_cast<ICEDXMath*>(iOther);
			if (_iDXMath)
			{
				_iDXMath->_vtbl->_addRef(_iDXMath);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXMath** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXMathRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXMath && "has a com object reference. clear first.");
		return & _iDXMath; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXMathRef& other)
	{
		bool result = false;
		if (_iDXMath)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXMath);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXMath ? _iDXMath->_vtbl->_queryInterface(_iDXMath, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXMath) { _iDXMath->_vtbl->_addRef(_iDXMath); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXMath) { _iDXMath->_vtbl->_release(_iDXMath); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult matrixIdentity44(CEDXMatrix44* pOut)	{ return _iDXMath ? _iDXMath->_vtbl->_matrixIdentity44(_iDXMath, pOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult matrixIdentity33(CEDXMatrix33* pOut)	{ return _iDXMath ? _iDXMath->_vtbl->_matrixIdentity33(_iDXMath, pOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult matrixMultiply44(CEDXMatrix44* pOut, CEDXMatrix44* pM1, CEDXMatrix44* pM2)	{ return _iDXMath ? _iDXMath->_vtbl->_matrixMultiply44(_iDXMath, pOut, pM1, pM2) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult matrixMultiply33(CEDXMatrix33* pOut, CEDXMatrix33* pM1, CEDXMatrix33* pM2)	{ return _iDXMath ? _iDXMath->_vtbl->_matrixMultiply33(_iDXMath, pOut, pM1, pM2) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult matrixTranslate33(CEDXMatrix33* pOut, float tX, float tY)	{ return _iDXMath ? _iDXMath->_vtbl->_matrixTranslate33(_iDXMath, pOut, tX, tY) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult matrixScale33(CEDXMatrix33* pOut, float sX, float sY)	{ return _iDXMath ? _iDXMath->_vtbl->_matrixScale33(_iDXMath, pOut, sX, sY) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult matrixRotate33(CEDXMatrix33* pOut, float angle)	{ return _iDXMath ? _iDXMath->_vtbl->_matrixRotate33(_iDXMath, pOut, angle) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult matrixOrthoOffCenterLH(CEDXMatrix44* pOut, float l, float r, float b, float t, float zn, float zf)	{ return _iDXMath ? _iDXMath->_vtbl->_matrixOrthoOffCenterLH(_iDXMath, pOut, l, r, b, t, zn, zf) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createRegion(struct ICEDXRegion* *const piRegionOut)	{ return _iDXMath ? _iDXMath->_vtbl->_createRegion(_iDXMath, piRegionOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXMath* _iDXMath;
};

/*! @}
 * end of CEComICEDXMathRef
 */

#endif // __cplusplus

#endif
