///////////////////////////////////////////////////////////////////////////////
// Copyright 2011 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICEDXContext2D_h
#define ICEDXContext2D_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "ICEDXDevice.h"
#include "CETLTypes.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------

struct ICEPFGlyphLoader;
struct ICECompressedImage;
struct ICEDX2DDisplayList;
struct ICEVGContext;

//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

//! context2D's state
														//!< the default value is eCEDXCtx2DBlendMode_SrcOver_opaque.
														//!< if Context2D is applied for 8bpp surface,
														//!< specified blend mode is ignored.
														//!< the default value is eCEDXCtx2DChromaKeyIndex_None.
														//!< if Context2D is applied for direct color surface,
														//!< specified chroma key index is ignored.
														//!< whose render target's pixel format is P8 (indexed).
														//!< the value is clamped into [0, 255] on drawing.
														//!< the default value is 0.
														//!< whose render target's pixel format is P8 (indexed).
														//!< the value is clamped into [0, 255] on drawing.
														//!< the default value is 0.
typedef enum
{
	eCEDXCtx2DState_FIRST					= 0,
	eCEDXCtx2DState_BlendMode				= 0,		//!< the value must be eCEDXCtx2DBlendMode.

	eCEDXCtx2DState_ChromaKeyIndex			= 1,		//!< the value must be eCEDXCtx2DChromaKeyIndex_None(0xFFFFFFFF) or 0-255.
	eCEDXCtx2DState_StrokePaint				= 2,		//!< current storoke "paint". the default value is eCEDXCtx2DPaintType_SolidColor.
	eCEDXCtx2DState_FillPaint				= 3,		//!< current fill "paint". the default value is eCEDXCtx2DPaintType_SolidColor.
	eCEDXCtx2DState_StrokeColorIndex		= 4,		//!< stroke color's palette index. only valid for Context2D 
	eCEDXCtx2DState_FillColorIndex			= 5,		//!< fill color's palette index. only valid for Context2D 
	eCEDXCtx2DState_LAST					= 5
} eCEDXCtx2DState;

//! context2D's state (color)
typedef enum
{
	eCEDXCtx2DColorState_FIRST				= 0,
	eCEDXCtx2DColorState_StrokeColor		= 0,		//!< the default value is opaque black.
	eCEDXCtx2DColorState_FillColor			= 1,		//!< the default value is opaque black.
	eCEDXCtx2DColorState_LAST				= 1
} eCEDXCtx2DColorState;

//! context2D's state (real value)
														//!< expressed in units of the context2D. 
														//!< By default, the units are in pixels. 
														//!< It must be gerater than 0.
														//!< The line surrounds the center of the path, 
														//!< with half of the total width on either side. 
														//!< the default value is 0.0.
														//!< if value is 0.0, actual line width dpends on device.
														//!< the line starts, in units of the context. 
														//!< For example, passing a value of 3 means the line 
														//!< is drawn with the dash pattern starting at 3 units 
														//!< in from its beginning. 
														//!< Passing a value of 0 draws a line starting with 
														//!< the beginning of a dash pattern.
														//!< the default value is 0.
typedef enum
{
	eCEDXCtx2DRealState_FIRST				= 0,

	eCEDXCtx2DRealState_LineWidth			= 0,		//!< The line width is the total width of the line, 

	eCEDXCtx2DRealState_LineDashPhase		= 1,		//!< A value that specifies how far into the dash pattern 

	eCEDXCtx2DRealState_LAST				= 1
} eCEDXCtx2DRealState;

//! context2D's state (real value's array)
														//!< the painted segments and unpainted segments, 
														//!< respectively, of the dash pattern - or NULL 
														//!< for no dash pattern.
														//!< For example, passing an array with the values 
														//!< [2,3] sets a dash pattern that alternates between
														//!< a 2-unit-long painted segment and a 3-unit-long
														//!< unpainted segment.
														//!< Passing the values [1,3,4,2] sets the pattern
														//!< to a 1-unit painted segment, a 3-unit unpainted
														//!< segment, a 4-unit painted segment, and a 2-unit
														//!< unpainted segment.
														//!< the default value is NULL.
														//!< this color combination is applied before blending, controlled
														//!< by eCEDXCtx2DState_BlendMode.
														//!< See http://3dshaders.com/s2006/KhronosSpecs/openvg-1.0.pdf Section 11.3 
														//!< Also see the description of CCM of context2D.
														//!< the default value is NULL, which means the following identity matrix:
														//!<  (1.0f, 0.0f, 0.0f, 0.0f,
														//!<   0.0f, 1.0f, 0.0f, 0.0f,
														//!<   0.0f, 0.0f, 1.0f, 0.0f,
														//!<   0.0f, 0.0f, 0.0f, 1.0f,
														//!<   0.0f, 0.0f, 0.0f, 0.0f)	
typedef enum
{
	eCEDXCtx2DRealvState_FIRST				= 0,

	eCEDXCtx2DRealvState_LineDashPattern	= 0,		//!< An array of values that specify the lengths of 

	eCEDXCtx2DRealvState_ColorMatrix		= 1,		//!< An array of values in the current color matrix (CCM).
	eCEDXCtx2DRealvState_LAST				= 1
} eCEDXCtx2DRealvState;

//! context2D's state (pointer)
														//!< the default value is NULL, which means
														//!< solid color paint using the color
														//!< specified with eCEDXCtx2DColorState_StrokeColor.
														//!< the default value is NULL, which means
														//!< solid color paint using the color
														//!< specified with eCEDXCtx2DColorState_FillColor.
typedef enum
{
	eCEDXCtx2DPtrState_FIRST				= 0,

	eCEDXCtx2DPtrState_StrokePaint			= 0,		//!< paint object used when stroking a path.

	eCEDXCtx2DPtrState_FillPaint			= 1,		//!< paint object used when filling a path.

	eCEDXCtx2DPtrState_LAST					= 1
} eCEDXCtx2DPtrState;

//! context2D's state (integer value's array)
														//!< are used. the array specified must contain 32 integers whose range is [0, 255].
														//!< - the glyph luminance [0, 7]   is mapped onto value[0].
														//!< - the glyph luminance [8, 15]  is mapped onto value[1].
														//!< - the glyph luminance [16, 23] is mapped onto value[2].
														//!<     :
														//!<     :
														//!< - the glyph luminance [248, 255] is mapped onto value[31].
typedef enum
{
	eCEDXCtx2DIntvState_FIRST				= 0,

	eCEDXCtx2DIntvState_GlyphColorIndices	= 0,		//!< when drawing glyphs on 8bpp render target, the specified glyph color indices
	eCEDXCtx2DIntvState_LAST				= 0,
} eCEDXCtx2DIntvState;

//! blend mode for context2D
//! this comes from SVG's composition operator
//! See http://www.w3.org/TR/2004/WD-SVG12-20041027/rendering.html#comp-op-prop
//! if Context2D is applied for 8bpp surface,
//! specified blend mode is ignored.
//! on 8bpp, use the chroma-key if you assume pixel that disregards a specified color or index. 
typedef enum
{
	eCEDXCtx2DBlendMode_Clear,			//!< unsupported
	eCEDXCtx2DBlendMode_Src,
	eCEDXCtx2DBlendMode_Src_opaque,
	eCEDXCtx2DBlendMode_Dst,			//!< unsupported
	eCEDXCtx2DBlendMode_SrcOver,
	eCEDXCtx2DBlendMode_SrcOver_opaque,
	eCEDXCtx2DBlendMode_DstOver,		//!< unsupported
	eCEDXCtx2DBlendMode_SrcIn,			//!< unsupported
	eCEDXCtx2DBlendMode_DstIn,			//!< unsupported
	eCEDXCtx2DBlendMode_SrcOut,			//!< unsupported
	eCEDXCtx2DBlendMode_DstOut,			//!< unsupported
	eCEDXCtx2DBlendMode_SrcAtop,		//!< unsupported
	eCEDXCtx2DBlendMode_DstAtop,		//!< unsupported
	eCEDXCtx2DBlendMode_Xor,			//!< unsupported
	eCEDXCtx2DBlendMode_Plus,			//!< unsupported
	eCEDXCtx2DBlendMode_Multiply,		//!< unsupported
	eCEDXCtx2DBlendMode_Screen,			//!< unsupported
	eCEDXCtx2DBlendMode_Overlay,		//!< unsupported
	eCEDXCtx2DBlendMode_Darken,			//!< unsupported
	eCEDXCtx2DBlendMode_Lighten,		//!< unsupported
	eCEDXCtx2DBlendMode_ColorDodge,		//!< unsupported
	eCEDXCtx2DBlendMode_ColorBurn,		//!< unsupported
	eCEDXCtx2DBlendMode_HardLight,		//!< unsupported
	eCEDXCtx2DBlendMode_SoftLight,		//!< unsupported
	eCEDXCtx2DBlendMode_Difference,		//!< unsupported
	eCEDXCtx2DBlendMode_Exclusion		//!< unsupported
} eCEDXCtx2DBlendMode;

//! disable chroma key
//! when you want to enable chroma key, you should specify the index.
typedef enum
{
	eCEDXCtx2DChromaKeyIndex_None = 0xFFFF
} eCEDXCtx2DChromaKeyIndex;

//! context2D's paint type of stroke and fill
typedef enum
{
	eCEDXCtx2DPaintType_SolidColor		= 0,
	eCEDXCtx2DPaintType_PatternFill		= 1,
	eCEDXCtx2DPaintType_GradientColor	= 2 
} eCEDXCtx2DPaintType;

//! context2D's wrap modes of pattern paint
//! this enumurations comes from GDI+
//! http://msdn.microsoft.com/library/default.asp?url=/library/en-us/gdicpp/GDIPlus/GDIPlusReference/Enumerations/WrapMode.asp
typedef enum
{
	eCEDX2DWrapMode_Tile		= 0,	// specifies tiling without flipping. 
	eCEDX2DWrapMode_TileFlipX	= 1,	// species that tiles are flipped horizontally as you move 
										// from one tile to the next in a row.
	eCEDX2DWrapMode_TileFlipY	= 2,	// specifies that tiles are flipped vertically as you move 
										// from one tile to the next in a column.
	eCEDX2DWrapMode_TileFlipXY	= 3,	// specifies that tiles are flipped horizontally as you move along a row 
										// and flipped vertically as you move along a column. 
	eCEDX2DWrapMode_Clamp		= 4		// specifies that no tiling takes place.
} eCEDX2DWrapMode;

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEDXVGContext ICEDXVGContext
 * @{
 */

/*!
 * ID of ICEDXVGContext
 */
#define CEComIID_ICEDXVGContext 0x0f3a4c87


/*!
 * ICEDXVGContext
 * deprecated.
 */
typedef struct ICEDXVGContext
{
	const struct ICEDXVGContext_vtbl* _vtbl;
} ICEDXVGContext;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXVGContext	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXVGContext_queryInterface) (ICEDXVGContext* iDXVGContext, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXVGContext_addRef) (ICEDXVGContext* iDXVGContext);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXVGContext_release) (ICEDXVGContext* iDXVGContext);

typedef CEHResult (*iCEDXVGContext_initByVGContext) (ICEDXVGContext* iDXVGContext, struct ICEVGContext* vgc);

typedef CEHResult (*iCEDXVGContext_shutdownByVGContext) (ICEDXVGContext* iDXVGContext, struct ICEVGContext* vgc);

typedef CEHResult (*iCEDXVGContext_getVGContext) (ICEDXVGContext* iDXVGContext, struct ICEVGContext* *const vgcOut);

/*!
 * V-table of ICEDXVGContext
 */
struct ICEDXVGContext_vtbl
{
	iCEDXVGContext_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXVGContext_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXVGContext_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXVGContext_initByVGContext	_initByVGContext;
	iCEDXVGContext_shutdownByVGContext	_shutdownByVGContext;
	iCEDXVGContext_getVGContext	_getVGContext;
};

/*! @}
 * end of ICEDXVGContext
 */

/*! \defgroup ICEDXCtx2DPaint ICEDXCtx2DPaint
 * @{
 */

/*!
 * ID of ICEDXCtx2DPaint
 */
#define CEComIID_ICEDXCtx2DPaint 0x238ba5bb


/*!
 * ICEDXPaint
 */
typedef struct ICEDXCtx2DPaint
{
	const struct ICEDXCtx2DPaint_vtbl* _vtbl;
} ICEDXCtx2DPaint;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXCtx2DPaint	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXCtx2DPaint_queryInterface) (ICEDXCtx2DPaint* iDXCtx2DPaint, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXCtx2DPaint_addRef) (ICEDXCtx2DPaint* iDXCtx2DPaint);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXCtx2DPaint_release) (ICEDXCtx2DPaint* iDXCtx2DPaint);

typedef CEHResult (*iCEDXCtx2DPaint_getType) (ICEDXCtx2DPaint* iDXCtx2DPaint, eCEDXCtx2DPaintType* pPaintTypeOut);

/*!
 * V-table of ICEDXCtx2DPaint
 */
struct ICEDXCtx2DPaint_vtbl
{
	iCEDXCtx2DPaint_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXCtx2DPaint_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXCtx2DPaint_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXCtx2DPaint_getType	_getType;
};

/*! @}
 * end of ICEDXCtx2DPaint
 */

/*! \defgroup ICEDXCtx2DPatternPaint ICEDXCtx2DPatternPaint
 * @{
 */

/*!
 * ID of ICEDXCtx2DPatternPaint
 */
#define CEComIID_ICEDXCtx2DPatternPaint 0x6e76576f


/*!
 * ICEDXPatternPaint
 */
typedef struct ICEDXCtx2DPatternPaint
{
	const struct ICEDXCtx2DPatternPaint_vtbl* _vtbl;
} ICEDXCtx2DPatternPaint;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXCtx2DPatternPaint	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXCtx2DPatternPaint_queryInterface) (ICEDXCtx2DPatternPaint* iDXCtx2DPatternPaint, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXCtx2DPatternPaint_addRef) (ICEDXCtx2DPatternPaint* iDXCtx2DPatternPaint);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXCtx2DPatternPaint_release) (ICEDXCtx2DPatternPaint* iDXCtx2DPatternPaint);

typedef CEHResult (*iCEDXCtx2DPatternPaint_getType) (ICEDXCtx2DPatternPaint* iDXCtx2DPatternPaint, eCEDXCtx2DPaintType* pPaintTypeOut);

	//! set the pattern image.

typedef CEHResult (*iCEDXCtx2DPatternPaint_setImage) (ICEDXCtx2DPatternPaint* iDXCtx2DPatternPaint, struct ICEDXSurface* iImage, struct ICEDXPalette* iPalette);

	//! get the pattern image.

typedef CEHResult (*iCEDXCtx2DPatternPaint_getImage) (ICEDXCtx2DPatternPaint* iDXCtx2DPatternPaint, struct ICEDXSurface* *const piImage, struct ICEDXPalette* *const piPalette);

	//! set the wrap mode of this pattern paint.

typedef CEHResult (*iCEDXCtx2DPatternPaint_setWrapMode) (ICEDXCtx2DPatternPaint* iDXCtx2DPatternPaint, eCEDX2DWrapMode mode);

	//! get the wrap mode currently set for this pattern paint.

typedef CEHResult (*iCEDXCtx2DPatternPaint_getWrapMode) (ICEDXCtx2DPatternPaint* iDXCtx2DPatternPaint, eCEDX2DWrapMode *const modeOut);

	//! reset the transformation matrix of this pattern paint to the identy matrix.

typedef CEHResult (*iCEDXCtx2DPatternPaint_resetTransform) (ICEDXCtx2DPatternPaint* iDXCtx2DPatternPaint);

	//! set the transformation matrix of this pattern paint.

typedef CEHResult (*iCEDXCtx2DPatternPaint_setTransform) (ICEDXCtx2DPatternPaint* iDXCtx2DPatternPaint, CEDXMatrix33* pMatrix);

	//! get the transformation matrix of this pattern paint.

typedef CEHResult (*iCEDXCtx2DPatternPaint_getTransform) (ICEDXCtx2DPatternPaint* iDXCtx2DPatternPaint, CEDXMatrix33* const pMatOut);

/*!
 * V-table of ICEDXCtx2DPatternPaint
 */
struct ICEDXCtx2DPatternPaint_vtbl
{
	iCEDXCtx2DPatternPaint_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXCtx2DPatternPaint_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXCtx2DPatternPaint_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXCtx2DPatternPaint_getType	_getType;
	iCEDXCtx2DPatternPaint_setImage	_setImage;
	iCEDXCtx2DPatternPaint_getImage	_getImage;
	iCEDXCtx2DPatternPaint_setWrapMode	_setWrapMode;
	iCEDXCtx2DPatternPaint_getWrapMode	_getWrapMode;
	iCEDXCtx2DPatternPaint_resetTransform	_resetTransform;
	iCEDXCtx2DPatternPaint_setTransform	_setTransform;
	iCEDXCtx2DPatternPaint_getTransform	_getTransform;
};

/*! @}
 * end of ICEDXCtx2DPatternPaint
 */

/*! \defgroup ICEDXCtx2DDisplayList ICEDXCtx2DDisplayList
 * @{
 */

/*!
 * ID of ICEDXCtx2DDisplayList
 */
#define CEComIID_ICEDXCtx2DDisplayList 0x706777fc


/*!
 * display list 
 * - generated by ICEDXContext2Df::beginRecord / endRecord
 *                ICEDXContext2Di::beginRecord / endRecord
 *   so the created display list may contain both float-based and int-based commands
 *
 * - executed  by ICEDXContext2Df::execDisplayList
 *   usually ICEDXContext2Df can interpret both float-base and int-based commands
 *
 * - executed  by ICEDXContext2Di::execDisplayList
 *   ICEDXContext2Di can interpret int-based commands only
 */
typedef struct ICEDXCtx2DDisplayList
{
	const struct ICEDXCtx2DDisplayList_vtbl* _vtbl;
} ICEDXCtx2DDisplayList;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXCtx2DDisplayList	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXCtx2DDisplayList_queryInterface) (ICEDXCtx2DDisplayList* iDXCtx2DDisplayList, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXCtx2DDisplayList_addRef) (ICEDXCtx2DDisplayList* iDXCtx2DDisplayList);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXCtx2DDisplayList_release) (ICEDXCtx2DDisplayList* iDXCtx2DDisplayList);

/*!
 * V-table of ICEDXCtx2DDisplayList
 */
struct ICEDXCtx2DDisplayList_vtbl
{
	iCEDXCtx2DDisplayList_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXCtx2DDisplayList_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXCtx2DDisplayList_release	_release;	//!< Decrease the reference count of the specified interface pointer.
};

/*! @}
 * end of ICEDXCtx2DDisplayList
 */

/*! \defgroup ICEDXContext2D ICEDXContext2D
 * @{
 */

/*!
 * ID of ICEDXContext2D
 */
#define CEComIID_ICEDXContext2D 0xb998f519


/*!
 * ICEDXContext2D
 * Context object which manages 2D drawing on DX Device.
 *
 * Principle <br>
 * --------- <br>
 *
 * The APIs are conceptually similar to Apple's Quartz2D, since it is the most 
 * efficient as the low-level APIs, compared with GDI+, Java's Graphics2D, and OpenVG.
 *
 * See http://developer.apple.com/documentation/GraphicsImaging/Reference/CGContext/index.html
 *
 * Coordinate system <br>
 * ----------------- <br>
 *
 * - Default
 *   The origin is at the upper-left corner of the render target.
 *   +X goes rightward.  +Y goes downward. 
 *   Unit is in pixels.
 * <pre>
 *    +-----------> X
 *    |
 *    |
 *    V
 *    Y
 * </pre>
 * - Transforms
 *   Just like Quartz2D, context 2D has the currnet transformation matrix (CTM).
 *   All the coordinates (including clipping regions) are affected by this CTM.
 * 
 *   Context object has two coordinate systems, 'target coordinate system'
 *   and 'user coordinate system'. Context object determines an initial
 *   'target coordinate system' and an initial 'user coordinate system' 
 *   where the two coordinates systems are identical.
 *
 *   You can modify the initial user coordinate system by operating on CTM.
 *   After you create a context object, the CTM is the identity matrix.
 *   You can use transformation functions to modify the CTM and, as a result,
 *   modify drawing in user coordinate system.
 *  
 *   All points or paths are placed on the currently user coordinate system.
 *   Therefore, the CTM will be applied when the APIs are executed.
 *
 *
 * Sub-pixel accuracy <br>
 * ------------------ <br>
 *
 * Pixel centers have coordinates that are half way between integer values. 
 * This definition avoids any inconsistent results between aliased and antialiased rendering. 
 * See http://java.sun.com/j2se/1.5.0/docs/api/java/awt/Graphics2D.html "Rendering Compatibility Issues".
 *
 * See the following figure.
 * <pre>
 *    0.0   1.0   2.0   3.0
 * 0.0 +-----+-----+-----+----> X
 *     |     |     |     |
 *     |  O  |  O  |  O  |                 "O" represents pixel centers.
 *     |     |     |     |
 * 1.0 +-----+-----+-----+
 *     |     |     |     |
 *     |  O  |  O  |  O  |
 *     |     |     |     |
 * 2.0 +-----+-----+-----+
 *     |
 *     V
 *     Y
 * </pre>
 * Pen <br>
 * --- <br>
 *
 * Attributes regarding the current "pen" are held as Context2D's states such as
 * eCEDXCtx2DRealState_LineWidth. 
 * Note that there is no explicit "pen" or "stroke" object.
 *
 * Paint <br>
 * ----- <br>
 *
 * Cenceptually "paint" (or "brush") can be categorized into three cases:
 *
 * (1) solid color paint
 *        setting the context's current paint to be NULL means using solid paint
 *        implicitly.  the stroke color and fill color are held as
 *        eCEDXCtx2DColorState_StrokeColor and eCEDXCtx2DColorState_FillColor.
 *
 * (2) gradient paint
 *        set the context's current paint to be a gradient paint.
 *
 * (3) pattern paint
 *        set the context's current paint to be a pattern paint.
 *
 * Setting "paint" lastly is adopted, therefore "paint" is exclusive.
 *
 * Color modulation<br>
 * ----------------<br>
 *
 * A fragment color (Rf Gf Bf Af) (e.g. a pixel in a bitmap or the current stroke/fill color) is modulated
 * using the current color matrix (CCM), and later will be fed into the blending phase.
 * The color modulation is defined in OpenVG (http://3dshaders.com/s2006/KhronosSpecs/openvg-1.0.pdf Section 11.3).
 *
 *                                  | m11 m12 m13 m14 |    
 *  (Rs Gs Bs As) = (Rf Gf Bf Af) * | m21 m22 m23 m24 | +  | m51 m52 m53 m54 |
 *                                  | m31 m32 m33 m34 |
 *                                  | m41 m42 m43 m44 |
 *
 *  the array specified by eCEDXCtx2DRealvState_ColorMatrix contains the values in the order :
 *  (m11 m12 m13 m14 m21 m22 m23 m24 m31 m32 m33 m34 m41 m42 m43 m44 m51 m52 m53 m54).
 *
 * Character drawing <br>
 * ----------------- <br>
 *
 * Character drawing (drawChar / drawChars) are considered as a special case of "filling a polygon".
 * Therefore the character color is specified as eCEDXCtx2DColorState_FillColor.
 *
 * Layouted Text drawing <br>
 * ---------------------- <br>
 *
 * Context2D supports layouted text drawing (drawText / drawString / drawMAString) using ICETLLayout internally.
 * In this case, all the layouting attributes are controlled via setStateTSR, setStateTSC, setStateTLR, setStateTLB().
 * Note that the text color is specified via setStateTSC(eCETLStyleAttribute_Color), NOT
 * setStateC(eCEDXCtx2DColorState_FillColor).
 *
 * Clipping Path <br>
 * -------------- <br>
 * 
 * context2D maintains a clipping path as one of its states.
 * - clearClip() clears the clipping path.
 * - clipToRect() sets the clipping path to the intersection of the current clipping path 
 *   with the area defined by the specified rectangle.
 * - Note that clipping path is affected by the CTM when the API (such as clipToRect() ) 
 *   is called. Be careful getClipBoundingBox() returns the bounding box in the 
 *   user coordinate system defined by the current CTM.
 *
 * Clipping path described with integer-based rectangle is interpreted as the following
 * figure describes. See "Sub-pixel accuracy" section above also.
 * <pre>
 *    0.0   1.0   2.0   3.0   4.0   5.0   6.0
 * 0.0 +-----+-----+-----+-----+-----+-----+---> X
 *     |     |     |     |     |     |     |
 *     |  x  |  x  |  x  |  x  |  x  |  x  |
 *     |     |     |     |     |     |     |
 * 1.0 +-----+-----+-----+-----+-----+-----+
 *     |     |     |     |     |     |     |
 *     |  x  |  O  |  O  |  O  |  O  |  x  |    the clip path is a rectangle (x = 1.0f, y = 1.0f, w = 4.0f, h = 2.0f)
 *     |     |     |     |     |     |     |
 * 2.0 +-----+-----+-----+-----+-----+-----+    "x" are pixels out of the clipping path.
 *     |     |     |     |     |     |     |
 *     |  x  |  O  |  O  |  O  |  O  |  x  |    "O" are pixels inside the clipping path.
 *     |     |     |     |     |     |     |
 * 3.0 +-----+-----+-----+-----+-----+-----+
 *     |     |     |     |     |     |     |
 *     |  x  |  x  |  x  |  x  |  x  |  x  |
 *     |     |     |     |     |     |     |
 * 4.0 +-----+-----+-----+-----+-----+-----+
 *     |
 *     V
 *     Y
 * </pre>
 * Display List (command recording) <br>
 * -------------------------------- <br>
 *
 * When you want to "record" all the context2D commands and execute later, you can
 * create display lists. The typical usecase is: 
 * - the thread for recording generates a display list, and hands it to another thread
 *   in charge of painting.
 * - the thread for painting executes the dipslay list arbitrary times to realize 60 fps, for example.
 *
 * recorder thread's code is:
 * <pre>
 *     CEComICEDXContext2DRef iCtx2DRef;
 *     CEComICEDXCtx2DDisplayList iDListRef;
 *     iBackBuf.createRecContext2D(&iCtx2DRef);
 *
 *     iCtx2DRef.beginRecord();
 *         iCtx2DRef.drawImage(...);
 *         iCtx2DRef.drawImage(...);
 *     iCtx2DRef.endRecord(&iDListRef);
 * </pre>
 * executer thread's code is:
 * <pre>
 *     CEComICEDXContext2DRef iCtx2DRef;
 *     iBackBuf.beginPaint(&iCtx2DRef);
 *         iCtx2DRef.execDisplayList(iDListRef);
 *     iBackBuf.endPaint();
 * </pre>
 * Note that you have to use two isolated context2D objects for thread-safety.
 */
typedef struct ICEDXContext2D
{
	const struct ICEDXContext2D_vtbl* _vtbl;
} ICEDXContext2D;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXContext2D	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXContext2D_queryInterface) (ICEDXContext2D* iDXContext2D, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXContext2D_addRef) (ICEDXContext2D* iDXContext2D);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXContext2D_release) (ICEDXContext2D* iDXContext2D);

	//! get the factory

typedef CEHResult (*iCEDXContext2D_getFactory) (ICEDXContext2D* iDXContext2D, struct ICEDX2DFactory* *const piFactoryOut);

	//! get DXDevice

typedef CEHResult (*iCEDXContext2D_getDevice) (ICEDXContext2D* iDXContext2D, struct ICEDXDevice* *const piDevice);

	//! get the DX surface on which  this context draws.

typedef CEHResult (*iCEDXContext2D_getSurface) (ICEDXContext2D* iDXContext2D, struct ICEDXSurface* *const piSurfaceOut);

	//! get the context2d is for index color or not.

typedef CEHResult (*iCEDXContext2D_isForIndexColor) (ICEDXContext2D* iDXContext2D, bool* const pOut);

	//! set an enum state

typedef CEHResult (*iCEDXContext2D_setState) (ICEDXContext2D* iDXContext2D, eCEDXCtx2DState state, const UINT32 val);

	//! get an enum state

typedef CEHResult (*iCEDXContext2D_getState) (ICEDXContext2D* iDXContext2D, eCEDXCtx2DState state, UINT32 *const pValOut);

	//! set a color state

typedef CEHResult (*iCEDXContext2D_setStateC) (ICEDXContext2D* iDXContext2D, eCEDXCtx2DColorState state, const CERGBAColor* pColor);

	//! get a color state

typedef CEHResult (*iCEDXContext2D_getStateC) (ICEDXContext2D* iDXContext2D, eCEDXCtx2DColorState state, CERGBAColor *const pColorOut);

	//! set a real value state (float version)

typedef CEHResult (*iCEDXContext2D_setStateF) (ICEDXContext2D* iDXContext2D, eCEDXCtx2DRealState state, const float val);

	//! get a real value state (float version)

typedef CEHResult (*iCEDXContext2D_getStateF) (ICEDXContext2D* iDXContext2D, eCEDXCtx2DRealState state, float *const pValOut);

	//! set a real-value-array state (float version)

typedef CEHResult (*iCEDXContext2D_setStateFv) (ICEDXContext2D* iDXContext2D, eCEDXCtx2DRealvState state, const float* pVals, UINT32 count);

	//! get a real-value-array state (float version)

typedef CEHResult (*iCEDXContext2D_getStateFv) (ICEDXContext2D* iDXContext2D, eCEDXCtx2DRealvState state, float *const pValsOut, UINT32 *const countOut);

	//! set a pointer state.

typedef CEHResult (*iCEDXContext2D_setStateP) (ICEDXContext2D* iDXContext2D, eCEDXCtx2DPtrState state, void* val);

	//! get a pointer state.

typedef CEHResult (*iCEDXContext2D_getStateP) (ICEDXContext2D* iDXContext2D, eCEDXCtx2DPtrState state, void* *const pValOut);

	//! set a integer-value-array state

typedef CEHResult (*iCEDXContext2D_setStateIv) (ICEDXContext2D* iDXContext2D, eCEDXCtx2DIntvState state, const INT32* pVals, UINT32 count);

	//! set a text style state (CETLReal)

typedef CEHResult (*iCEDXContext2D_setStateTSR) (ICEDXContext2D* iDXContext2D, eCETLStyleAttribute attr, const CETLReal val);

	//! get a text style state (CETLReal)

typedef CEHResult (*iCEDXContext2D_getStateTSR) (ICEDXContext2D* iDXContext2D, eCETLStyleAttribute attr, CETLReal* const pValOut);

	//! set a text style state (CERGBAColor)

typedef CEHResult (*iCEDXContext2D_setStateTSC) (ICEDXContext2D* iDXContext2D, eCETLStyleAttribute attr, const CERGBAColor* pVal);

	//! get a text style state (CERGBAColor)

typedef CEHResult (*iCEDXContext2D_getStateTSC) (ICEDXContext2D* iDXContext2D, eCETLStyleAttribute attr, CERGBAColor* const pValOut);

	//! set a text layout state (CETLReal)

typedef CEHResult (*iCEDXContext2D_setStateTLR) (ICEDXContext2D* iDXContext2D, eCETLLayoutAttribute attr, const CETLReal val);

	//! get a text layout state (CETLReal)

typedef CEHResult (*iCEDXContext2D_getStateTLR) (ICEDXContext2D* iDXContext2D, eCETLLayoutAttribute attr, CETLReal* const pValOut);

	//! set a text layout state (bool)

typedef CEHResult (*iCEDXContext2D_setStateTLB) (ICEDXContext2D* iDXContext2D, eCETLLayoutAttribute attr, const bool val);

	//! get a text layout state (bool)

typedef CEHResult (*iCEDXContext2D_getStateTLB) (ICEDXContext2D* iDXContext2D, eCETLLayoutAttribute attr, bool* const pValOut);

	//! get the horizontal resolution, in dots per inch, 
	//! of the display device associated with this context object.

typedef CEHResult (*iCEDXContext2D_getDpiX) (ICEDXContext2D* iDXContext2D, float *pDpiX);

	//! get the vertical resolution, in dots per inch, 
	//! of the display device associated with this context object.

typedef CEHResult (*iCEDXContext2D_getDpiY) (ICEDXContext2D* iDXContext2D, float *pDpiY);

	//! Changes the origin of the user coordinate system in a context.
	//! \param[in] tx	The amount to displace the x-axis of the coordinate space, in units of the user space, of the specified context.
	//! \param[in] ty	The amount to displace the y-axis of the coordinate space, in units of the user space, of the specified context.

typedef CEHResult (*iCEDXContext2D_translateCTM) (ICEDXContext2D* iDXContext2D, float tx, float ty);

	//! Rotates the user coordinate system in a context
	//! \param[in] angle	The angle, in radians, by which to rotate the coordinate space of the specified context.

typedef CEHResult (*iCEDXContext2D_rotateCTM) (ICEDXContext2D* iDXContext2D, float angle);

	//! Changes the scale of the user coordinate system in a context.
	//! \param[in] sx	The factor by which to scale the x-axis of the coordinate space of the specified context.
	//! \param[in] sy	The factor by which to scale the y-axis of the coordinate space of the specified context.

typedef CEHResult (*iCEDXContext2D_scaleCTM) (ICEDXContext2D* iDXContext2D, float sx, float sy);

	//! Returns the current transformation matrix.
	//! \param[out]	pMatOut		fill the current CTM.

typedef CEHResult (*iCEDXContext2D_getCTM) (ICEDXContext2D* iDXContext2D, CEDXMatrix33* const pMatOut);

	//! push a copy of the current graphics state on the top of a context's graphics state stack.
	//!
	//! Note that not all aspects of the current drawing environment are elements of the graphics state. 
	//! For example, the current path is not considered part of the graphics state and is therefore not 
	//! saved when you call this function. The graphics state parameters that are saved are:
	//! - clip region
	//! - state parameters specified via setState / setStatec / setStater / setStaterv / setStetep

typedef CEHResult (*iCEDXContext2D_saveState) (ICEDXContext2D* iDXContext2D);

	//! set the current graphics state to the state most recently saved.

typedef CEHResult (*iCEDXContext2D_restoreState) (ICEDXContext2D* iDXContext2D);

	//! start recording commands into a new display list.
	//! all commands between beginRecord and endRecord are NOT executed and
	//! recorded into a display list, which will be retrieved by endRecord().

typedef CEHResult (*iCEDXContext2D_beginRecord) (ICEDXContext2D* iDXContext2D);

	//! end recording commands and return the generated display list.
	//! \return JGSS_ERR_OPERATION_FAILED when beginRecord() is not called properly beforehand.

typedef CEHResult (*iCEDXContext2D_endRecord) (ICEDXContext2D* iDXContext2D, struct ICEDX2DDisplayList* *const piDListOut);

	//! execute commands in the display list.

typedef CEHResult (*iCEDXContext2D_execDisplayList) (ICEDXContext2D* iDXContext2D, struct ICEDX2DDisplayList* iDList);

	//! clear the scissoring rectangle.

typedef CEHResult (*iCEDXContext2D_clearScissorRect) (ICEDXContext2D* iDXContext2D);

	//! sets the scissoring rectangle.
	//! \param[in] pRect	scissoring rectangle

typedef CEHResult (*iCEDXContext2D_setScissorRect) (ICEDXContext2D* iDXContext2D, const CERect* pRect);

	//! returns the scissoring rectangle in the user coordinate system.
	//! \param[out] pRectOut	scissoring rectangle.

typedef CEHResult (*iCEDXContext2D_getScissorRect) (ICEDXContext2D* iDXContext2D, CERect*  pRectOut);

	//! clear the clipping path.
	//! sets the clipping path to the intersection of the current clipping path 
	//! with the area defined by the specified rectangle.
	//! \param[in] pRect	clipping rectangle
	//! returns the bounding box of the clipping path represented in the user coordinate system.
	//! \param[out] pRectOut	fill the bounding box of the clipping path.
	//! begin painting
	//! \param[in] iUpdateRgn  specifies the region to repaint in the render target.
	//! \remarks If iUpdateRgn is NULL, the repaint region is the entire render target.

typedef CEHResult (*iCEDXContext2D_beginPaint) (ICEDXContext2D* iDXContext2D, struct ICEDXImmutableRegion* iUpdateRgn);

	//! end painting

typedef CEHResult (*iCEDXContext2D_endPaint) (ICEDXContext2D* iDXContext2D);

	//! get update region

typedef CEHResult (*iCEDXContext2D_getUpdateRegion) (ICEDXContext2D* iDXContext2D, struct ICEDXImmutableRegion* *const iUpdateRgnOut);

	//! stroke a line.
	//! \param[in] pStartPos	starting position
	//! \param[in] pEndPos		ending position

typedef CEHResult (*iCEDXContext2D_strokeLineF) (ICEDXContext2D* iDXContext2D, const CEPointF*    pStartPos, const CEPointF*    pEndPos);

typedef CEHResult (*iCEDXContext2D_strokeLineI) (ICEDXContext2D* iDXContext2D, const CEPointBase* pStartPos, const CEPointBase* pEndPos);

	//! stroke a line along the path of the specified rectangle.

typedef CEHResult (*iCEDXContext2D_strokeRectF) (ICEDXContext2D* iDXContext2D, const CERectF* pRect);

typedef CEHResult (*iCEDXContext2D_strokeRectI) (ICEDXContext2D* iDXContext2D, const CERect*  pRect);

	//! paint the area contained within a rectangle.

typedef CEHResult (*iCEDXContext2D_fillRectF) (ICEDXContext2D* iDXContext2D, const CERectF* pRect);

typedef CEHResult (*iCEDXContext2D_fillRectI) (ICEDXContext2D* iDXContext2D, const CERect*  pRect);

	//! stroke a line along the path of the specified round-cornered rectangle.

typedef CEHResult (*iCEDXContext2D_strokeRoundRectF) (ICEDXContext2D* iDXContext2D, const CERectF* pRect, float arcWidth, float arcHeight);

typedef CEHResult (*iCEDXContext2D_strokeRoundRectI) (ICEDXContext2D* iDXContext2D, const CERect*  pRect, INT32 arcWidth, INT32 arcHeight);

	//! paint the area contained within a round-cornered rectangle.

typedef CEHResult (*iCEDXContext2D_fillRoundRectF) (ICEDXContext2D* iDXContext2D, const CERectF* pRect, float arcWidth, float arcHeight);

typedef CEHResult (*iCEDXContext2D_fillRoundRectI) (ICEDXContext2D* iDXContext2D, const CERect* pRect,  INT32 arcWidth, INT32 arcHeight);

	//! stroke a line along the path of the specified ellipse.

typedef CEHResult (*iCEDXContext2D_strokeEllipseF) (ICEDXContext2D* iDXContext2D, const CERectF* pBbox);

typedef CEHResult (*iCEDXContext2D_strokeEllipseI) (ICEDXContext2D* iDXContext2D, const CERect*  pBbox);

	//! paint the area contained within an ellipse.

typedef CEHResult (*iCEDXContext2D_fillEllipseF) (ICEDXContext2D* iDXContext2D, const CERectF* pBbox);

typedef CEHResult (*iCEDXContext2D_fillEllipseI) (ICEDXContext2D* iDXContext2D, const CERect*  pBbox);

	//! stroke a line along the path of the specified polygon.
	//! \param[in]  points		an array of points.
	//!							If the first and last coordinates in the points array are not identical, 
	//!							a line is drawn between them to close the polygon.
	//! \param[in]  count		the number of points.

typedef CEHResult (*iCEDXContext2D_strokePolygonF) (ICEDXContext2D* iDXContext2D, const CEPointF*    points, UINT32 count);

typedef CEHResult (*iCEDXContext2D_strokePolygonI) (ICEDXContext2D* iDXContext2D, const CEPointBase* points, UINT32 count);

	//! paint the area contained within a polygon.
	//! \param[in]  points		an array of points.
	//!							If the last point and the first point do not coincide,
	//!							they specify the last side of the polygon.
	//! \param[in]  count		the number of points.

typedef CEHResult (*iCEDXContext2D_fillPolygonF) (ICEDXContext2D* iDXContext2D, const CEPointF*    points, UINT32 count);

typedef CEHResult (*iCEDXContext2D_fillPolygonI) (ICEDXContext2D* iDXContext2D, const CEPointBase* points, UINT32 count);

	//! stroke a line along the path of the specified arc
	//! The resulting arc begins at startAngle and extends for arcAngle radians.
	//! Angles are interpreted such that radian 0 is at the 3 ofclock position.
	//! A positive value indicates a counter-clockwise rotation white a negative value indicates a clockwise rotation.

typedef CEHResult (*iCEDXContext2D_strokeArcF) (ICEDXContext2D* iDXContext2D, const CERectF* pBbox, float startAngle, float arcAngle);

typedef CEHResult (*iCEDXContext2D_strokeArcI) (ICEDXContext2D* iDXContext2D, const CERect*  pBbox, float startAngle, float arcAngle);

	//! paint the area contained within an arc
	//! The resulting arc begins at startAngle and extends for arcAngle radians.
	//! Angles are interpreted such that radian 0 is at the 3 ofclock position.
	//! A positive value indicates a counter-clockwise rotation white a negative value indicates a clockwise rotation.

typedef CEHResult (*iCEDXContext2D_fillArcF) (ICEDXContext2D* iDXContext2D, const CERectF* pBbox, float startAngle, float arcAngle);

typedef CEHResult (*iCEDXContext2D_fillArcI) (ICEDXContext2D* iDXContext2D, const CERect*  pBbox, float startAngle, float arcAngle);

	//! draw a specified portion of an image at a specified location.
	//! \param[in] pDstRect		Rectangle, measured in pixels, that specifies the bounds 
	//!							of the rendered image. The portion of the image specified by 
	//!							pSrcRect is scaled to fill the rectangle specified by pDstRect.
	//! \param[in] iImage		image to be drawn.
	//! \param[in] iPalette		image palette. NULL means no palette.
	//! \param[in] pSrcRect		Rectangle, measured in pixels, that specifies the portion of 
	//!							the image to be drawn. NULL means the whole image.

typedef CEHResult (*iCEDXContext2D_drawImageF) (ICEDXContext2D* iDXContext2D, const CERectF* pDstRect, struct ICEDXSurface* iImage, struct ICEDXPalette* iPalette, const CERectF* pSrcRect);

typedef CEHResult (*iCEDXContext2D_drawImageI) (ICEDXContext2D* iDXContext2D, const CERect*  pDstRect, struct ICEDXSurface* iImage, struct ICEDXPalette* iPalette, const CERect*  pSrcRect);

	//! draw a specified portion of an compressed-image at a specified location.
	//! \param[in] pDstRect		Rectangle, measured in pixels, that specifies the bounds 
	//!							of the rendered image. The portion of the image specified by 
	//!							pSrcRect is scaled to fill the rectangle specified by pDstRect.
	//! \param[in] iImage		image to be drawn.
	//! \param[in] pSrcRect		Rectangle, measured in pixels, that specifies the portion of 
	//!							the image to be drawn. NULL means the whole image.

typedef CEHResult (*iCEDXContext2D_drawCompressedImageF) (ICEDXContext2D* iDXContext2D, const CERectF* pDstRect, struct ICECompressedImage* iCompressedImage, const CERectF* pSrcRect);

typedef CEHResult (*iCEDXContext2D_drawCompressedImageI) (ICEDXContext2D* iDXContext2D, const CERect*  pDstRect, struct ICECompressedImage* iCompressedImage, const CERect*  pSrcRect);

	//! Copy any rectangle area to a specified location.
	//! \param[in] pDstPoint	dstination point. if size of pDstRect, _width and_height, and pSrcRect isn't equal value, this method can be scaling copy. 
	//! \param[in] pSrcSurface	Src surface is copied to Context2D's dst surface. If it is NULL, copy area is done on the same surface.
	//! \param[in] pSrcRect		Rectangle, measured in pixels, that specifies the portion of 
	//!							the image to be drawn.

typedef CEHResult (*iCEDXContext2D_copyAreaF) (ICEDXContext2D* iDXContext2D, const CERectF* pDstRect, struct ICEDXSurface* pSrcSurface, const CERectF* pSrcRect);

typedef CEHResult (*iCEDXContext2D_copyAreaI) (ICEDXContext2D* iDXContext2D, const CERect*  pDstRect, struct ICEDXSurface* pSrcSurface, const CERect*  pSrcRect);


	/*!
	*  Draw a character.
	* 
	*  Note that the context2D's coordinate system tends to be
	*  "pixel" oriented, while the coordinate system defined by the text
	*  layout system is strictly "point" oriented. The implementation of
	*  this method has to take care of this difference.
	* 
	*  Note that the "origin" of the glyph tends to be near the lower left corner.
	*  See the following figure:
	*  \param[in] iDXContext2D				Specifies interface pointer
	*  \param[in] pPos				the position where the origin of the glyph is settled on this
	* 								context2D's render target.
	*  \param[in] iGlyphLoader		glyph loader
	*  \param[in] c32				UTF32 code point to represent a character.
	* 
	* <pre>
	*       +---------------+
	*       |  ********     |
	*       | **      **    |
	*       |**        **   |
	*       |**   ***  **   |
	*       | ** *   ***    |
	*       |  *********    |
	*       X---------**----+        X position is the origin.
	*       |         **  **|
	*       |          ***  |
	*       +---------------+
	* </pre>
	*/
typedef CEHResult (*iCEDXContext2D_drawCharF) (ICEDXContext2D* iDXContext2D, const CEPointF*    pPos, const struct ICEPFGlyphLoader* iGlyphLoader, const UTF32CHAR c32, const UINT32 glyphIndex, bool bUseC32);

typedef CEHResult (*iCEDXContext2D_drawCharI) (ICEDXContext2D* iDXContext2D, const CEPointBase* pPos, const struct ICEPFGlyphLoader* iGlyphLoader, const UTF32CHAR c32);

typedef CEHResult (*iCEDXContext2D_drawCharsF) (ICEDXContext2D* iDXContext2D, const CEPointF*    ppPos, const struct ICEPFGlyphLoader* iGlyphLoader, const UTF32CHAR *c32, UINT32 size);

typedef CEHResult (*iCEDXContext2D_drawCharsI) (ICEDXContext2D* iDXContext2D, const CEPointBase* ppPos, const struct ICEPFGlyphLoader* iGlyphLoader, const UTF32CHAR *c32, UINT32 size);

	//! draw text in a specified rectangle.
	//! note that the text layout is controlled using setStateTSR, setStateTSC, setStateTLR, setStateTLB().
	//! for more details, see ICETLLayout.
	//! \param pBbox		the bounding box in which the characters is drawn.
	//! \param pStr			text data
	//! \param textLen		the number of characters to draw. -1 means "until the end of the specified string, specified by NULL".

typedef CEHResult (*iCEDXContext2D_drawTextF) (ICEDXContext2D* iDXContext2D, const CERectF* pBbox, const UTF16CHAR* pStr, INT32 textLen);

typedef CEHResult (*iCEDXContext2D_drawTextI) (ICEDXContext2D* iDXContext2D, const CERect* pBbox,  const UTF16CHAR* pStr, INT32 textLen);

	//! draw a string in a specified rectangle.
	//! note that the text layout is controlled using setStateTSR, setStateTSC, setStateTLR, setStateTLB().
	//! for more details, see ICETLLayout.
	//! \param pBbox		the bounding box in which the text is drawn.
	//! \param iStr			text data
	//! \param textStart	index of the first character to draw in the specified text. 0 means the first character.
	//! \param textLen		the number of characters to draw. -1 means "until the end of the specified string".

typedef CEHResult (*iCEDXContext2D_drawStringF) (ICEDXContext2D* iDXContext2D, const CERectF* pBbox, struct ICEUString* iStr, UINT32 textStart, INT32 textLen);

typedef CEHResult (*iCEDXContext2D_drawStringI) (ICEDXContext2D* iDXContext2D, const CERect* pBbox,  struct ICEUString* iStr, UINT32 textStart, INT32 textLen);

	//! draw a mutable attributed string in a specified rectangle.
	//! note that the text layout is controlled using setStateTSR, setStateTSC, setStateTLR, setStateTLB().
	//! for more details, see ICETLLayout.
	//! \param pBbox		the bounding box in which the text is drawn.
	//! \param iStr			text data
	//! \param textStart	index of the first character to draw in the specified text. 0 means the first character.
	//! \param textLen		the number of characters to draw. -1 means "until the end of the specified string".

typedef CEHResult (*iCEDXContext2D_drawMAStringF) (ICEDXContext2D* iDXContext2D, const CERectF* pBbox, struct ICEMAString* iStr, UINT32 textStart, INT32 textLen);

typedef CEHResult (*iCEDXContext2D_drawMAStringI) (ICEDXContext2D* iDXContext2D, const CERect* pBbox,  struct ICEMAString* iStr, UINT32 textStart, INT32 textLen);

typedef CEHResult (*iCEDXContext2D_getPixelFormat) (ICEDXContext2D* iDXContext2D, CEPixelFormat* const pPxlFmtOut);

/*!
 * V-table of ICEDXContext2D
 */
struct ICEDXContext2D_vtbl
{
	iCEDXContext2D_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXContext2D_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXContext2D_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXContext2D_getFactory	_getFactory;
	iCEDXContext2D_getDevice	_getDevice;
	iCEDXContext2D_getSurface	_getSurface;
	iCEDXContext2D_isForIndexColor	_isForIndexColor;
	iCEDXContext2D_setState	_setState;
	iCEDXContext2D_getState	_getState;
	iCEDXContext2D_setStateC	_setStateC;
	iCEDXContext2D_getStateC	_getStateC;
	iCEDXContext2D_setStateF	_setStateF;
	iCEDXContext2D_getStateF	_getStateF;
	iCEDXContext2D_setStateFv	_setStateFv;
	iCEDXContext2D_getStateFv	_getStateFv;
	iCEDXContext2D_setStateP	_setStateP;
	iCEDXContext2D_getStateP	_getStateP;
	iCEDXContext2D_setStateIv	_setStateIv;
	iCEDXContext2D_setStateTSR	_setStateTSR;
	iCEDXContext2D_getStateTSR	_getStateTSR;
	iCEDXContext2D_setStateTSC	_setStateTSC;
	iCEDXContext2D_getStateTSC	_getStateTSC;
	iCEDXContext2D_setStateTLR	_setStateTLR;
	iCEDXContext2D_getStateTLR	_getStateTLR;
	iCEDXContext2D_setStateTLB	_setStateTLB;
	iCEDXContext2D_getStateTLB	_getStateTLB;
	iCEDXContext2D_getDpiX	_getDpiX;
	iCEDXContext2D_getDpiY	_getDpiY;
	iCEDXContext2D_translateCTM	_translateCTM;
	iCEDXContext2D_rotateCTM	_rotateCTM;
	iCEDXContext2D_scaleCTM	_scaleCTM;
	iCEDXContext2D_getCTM	_getCTM;
	iCEDXContext2D_saveState	_saveState;
	iCEDXContext2D_restoreState	_restoreState;
	iCEDXContext2D_beginRecord	_beginRecord;
	iCEDXContext2D_endRecord	_endRecord;
	iCEDXContext2D_execDisplayList	_execDisplayList;
	iCEDXContext2D_clearScissorRect	_clearScissorRect;
	iCEDXContext2D_setScissorRect	_setScissorRect;
	iCEDXContext2D_getScissorRect	_getScissorRect;
	iCEDXContext2D_beginPaint	_beginPaint;
	iCEDXContext2D_endPaint	_endPaint;
	iCEDXContext2D_getUpdateRegion	_getUpdateRegion;
	iCEDXContext2D_strokeLineF	_strokeLineF;
	iCEDXContext2D_strokeLineI	_strokeLineI;
	iCEDXContext2D_strokeRectF	_strokeRectF;
	iCEDXContext2D_strokeRectI	_strokeRectI;
	iCEDXContext2D_fillRectF	_fillRectF;
	iCEDXContext2D_fillRectI	_fillRectI;
	iCEDXContext2D_strokeRoundRectF	_strokeRoundRectF;
	iCEDXContext2D_strokeRoundRectI	_strokeRoundRectI;
	iCEDXContext2D_fillRoundRectF	_fillRoundRectF;
	iCEDXContext2D_fillRoundRectI	_fillRoundRectI;
	iCEDXContext2D_strokeEllipseF	_strokeEllipseF;
	iCEDXContext2D_strokeEllipseI	_strokeEllipseI;
	iCEDXContext2D_fillEllipseF	_fillEllipseF;
	iCEDXContext2D_fillEllipseI	_fillEllipseI;
	iCEDXContext2D_strokePolygonF	_strokePolygonF;
	iCEDXContext2D_strokePolygonI	_strokePolygonI;
	iCEDXContext2D_fillPolygonF	_fillPolygonF;
	iCEDXContext2D_fillPolygonI	_fillPolygonI;
	iCEDXContext2D_strokeArcF	_strokeArcF;
	iCEDXContext2D_strokeArcI	_strokeArcI;
	iCEDXContext2D_fillArcF	_fillArcF;
	iCEDXContext2D_fillArcI	_fillArcI;
	iCEDXContext2D_drawImageF	_drawImageF;
	iCEDXContext2D_drawImageI	_drawImageI;
	iCEDXContext2D_drawCompressedImageF	_drawCompressedImageF;
	iCEDXContext2D_drawCompressedImageI	_drawCompressedImageI;
	iCEDXContext2D_copyAreaF	_copyAreaF;
	iCEDXContext2D_copyAreaI	_copyAreaI;
	iCEDXContext2D_drawCharF	_drawCharF;
	iCEDXContext2D_drawCharI	_drawCharI;
	iCEDXContext2D_drawCharsF	_drawCharsF;
	iCEDXContext2D_drawCharsI	_drawCharsI;
	iCEDXContext2D_drawTextF	_drawTextF;
	iCEDXContext2D_drawTextI	_drawTextI;
	iCEDXContext2D_drawStringF	_drawStringF;
	iCEDXContext2D_drawStringI	_drawStringI;
	iCEDXContext2D_drawMAStringF	_drawMAStringF;
	iCEDXContext2D_drawMAStringI	_drawMAStringI;
	iCEDXContext2D_getPixelFormat	_getPixelFormat;
};

/*! @}
 * end of ICEDXContext2D
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEDXVGContextRef CEComICEDXVGContextRef
 * @{
 */

class CEComICEDXVGContextRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXVGContextRef() : _iDXVGContext(0) {}
	CEComICEDXVGContextRef(ICEDXVGContext* iOther) : _iDXVGContext(0)
	{
		if (iOther)
		{
			_iDXVGContext = iOther;
			_iDXVGContext->_vtbl->_addRef(_iDXVGContext);
		}
	}

	CEComICEDXVGContextRef(const CEComICEDXVGContextRef& other) : _iDXVGContext(0)
	{
		if (other._iDXVGContext)
		{
			_iDXVGContext = other._iDXVGContext;
			_iDXVGContext->_vtbl->_addRef(_iDXVGContext);
		}
	}

	~CEComICEDXVGContextRef()
	{
		if (_iDXVGContext)
		{
			ICEDXVGContext* tmp = _iDXVGContext;
			_iDXVGContext = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXVGContext* iDXVGContext;
				void* _ptr;
			} uIntf;
			uIntf.iDXVGContext = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXVGContext, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXVGContext)
				{
					ICEDXVGContext* tmp = _iDXVGContext;
					_iDXVGContext = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXVGContext = uIntf.iDXVGContext;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXVGContext* object() const	{ return _iDXVGContext; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXVGContext* iOther)
	{
		if (_iDXVGContext)
		{
			ICEDXVGContext* tmp = _iDXVGContext;
			_iDXVGContext = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXVGContext = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXVGContext* detach()
	{
		ICEDXVGContext* iIntf = _iDXVGContext;
		_iDXVGContext = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXVGContext* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXVGContext;
		if (_iDXVGContext)
		{
			_iDXVGContext->_vtbl->_addRef(_iDXVGContext);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXVGContext*() const	{ return _iDXVGContext; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXVGContext& operator*() const	{ return *_iDXVGContext; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXVGContext == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXVGContext* iOther) const	{ return (_iDXVGContext != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXVGContext* iOther) const	{ return (_iDXVGContext == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXVGContextRef& operator=(const CEComICEDXVGContextRef& other)	{ return operator=(other._iDXVGContext); }

	CEComICEDXVGContextRef& operator=(const ICEDXVGContext* iOther)
	{
		if (_iDXVGContext != iOther)
		{
			if (_iDXVGContext)
			{
				ICEDXVGContext* tmp = _iDXVGContext;
				_iDXVGContext = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXVGContext = const_cast<ICEDXVGContext*>(iOther);
			if (_iDXVGContext)
			{
				_iDXVGContext->_vtbl->_addRef(_iDXVGContext);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXVGContext** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXVGContextRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXVGContext && "has a com object reference. clear first.");
		return & _iDXVGContext; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXVGContextRef& other)
	{
		bool result = false;
		if (_iDXVGContext)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXVGContext);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXVGContext ? _iDXVGContext->_vtbl->_queryInterface(_iDXVGContext, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXVGContext) { _iDXVGContext->_vtbl->_addRef(_iDXVGContext); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXVGContext) { _iDXVGContext->_vtbl->_release(_iDXVGContext); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initByVGContext(struct ICEVGContext* vgc)	{ return _iDXVGContext ? _iDXVGContext->_vtbl->_initByVGContext(_iDXVGContext, vgc) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult shutdownByVGContext(struct ICEVGContext* vgc)	{ return _iDXVGContext ? _iDXVGContext->_vtbl->_shutdownByVGContext(_iDXVGContext, vgc) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getVGContext(struct ICEVGContext* *const vgcOut)	{ return _iDXVGContext ? _iDXVGContext->_vtbl->_getVGContext(_iDXVGContext, vgcOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXVGContext* _iDXVGContext;
};

/*! @}
 * end of CEComICEDXVGContextRef
 */

/*! \defgroup CEComICEDXCtx2DPaintRef CEComICEDXCtx2DPaintRef
 * @{
 */

class CEComICEDXCtx2DPaintRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXCtx2DPaintRef() : _iDXCtx2DPaint(0) {}
	CEComICEDXCtx2DPaintRef(ICEDXCtx2DPaint* iOther) : _iDXCtx2DPaint(0)
	{
		if (iOther)
		{
			_iDXCtx2DPaint = iOther;
			_iDXCtx2DPaint->_vtbl->_addRef(_iDXCtx2DPaint);
		}
	}

	CEComICEDXCtx2DPaintRef(const CEComICEDXCtx2DPaintRef& other) : _iDXCtx2DPaint(0)
	{
		if (other._iDXCtx2DPaint)
		{
			_iDXCtx2DPaint = other._iDXCtx2DPaint;
			_iDXCtx2DPaint->_vtbl->_addRef(_iDXCtx2DPaint);
		}
	}

	~CEComICEDXCtx2DPaintRef()
	{
		if (_iDXCtx2DPaint)
		{
			ICEDXCtx2DPaint* tmp = _iDXCtx2DPaint;
			_iDXCtx2DPaint = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXCtx2DPaint* iDXCtx2DPaint;
				void* _ptr;
			} uIntf;
			uIntf.iDXCtx2DPaint = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXCtx2DPaint, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXCtx2DPaint)
				{
					ICEDXCtx2DPaint* tmp = _iDXCtx2DPaint;
					_iDXCtx2DPaint = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXCtx2DPaint = uIntf.iDXCtx2DPaint;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXCtx2DPaint* object() const	{ return _iDXCtx2DPaint; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXCtx2DPaint* iOther)
	{
		if (_iDXCtx2DPaint)
		{
			ICEDXCtx2DPaint* tmp = _iDXCtx2DPaint;
			_iDXCtx2DPaint = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXCtx2DPaint = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXCtx2DPaint* detach()
	{
		ICEDXCtx2DPaint* iIntf = _iDXCtx2DPaint;
		_iDXCtx2DPaint = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXCtx2DPaint* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXCtx2DPaint;
		if (_iDXCtx2DPaint)
		{
			_iDXCtx2DPaint->_vtbl->_addRef(_iDXCtx2DPaint);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXCtx2DPaint*() const	{ return _iDXCtx2DPaint; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXCtx2DPaint& operator*() const	{ return *_iDXCtx2DPaint; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXCtx2DPaint == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXCtx2DPaint* iOther) const	{ return (_iDXCtx2DPaint != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXCtx2DPaint* iOther) const	{ return (_iDXCtx2DPaint == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXCtx2DPaintRef& operator=(const CEComICEDXCtx2DPaintRef& other)	{ return operator=(other._iDXCtx2DPaint); }

	CEComICEDXCtx2DPaintRef& operator=(const ICEDXCtx2DPaint* iOther)
	{
		if (_iDXCtx2DPaint != iOther)
		{
			if (_iDXCtx2DPaint)
			{
				ICEDXCtx2DPaint* tmp = _iDXCtx2DPaint;
				_iDXCtx2DPaint = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXCtx2DPaint = const_cast<ICEDXCtx2DPaint*>(iOther);
			if (_iDXCtx2DPaint)
			{
				_iDXCtx2DPaint->_vtbl->_addRef(_iDXCtx2DPaint);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXCtx2DPaint** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXCtx2DPaintRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXCtx2DPaint && "has a com object reference. clear first.");
		return & _iDXCtx2DPaint; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXCtx2DPaintRef& other)
	{
		bool result = false;
		if (_iDXCtx2DPaint)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXCtx2DPaint);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXCtx2DPaint ? _iDXCtx2DPaint->_vtbl->_queryInterface(_iDXCtx2DPaint, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXCtx2DPaint) { _iDXCtx2DPaint->_vtbl->_addRef(_iDXCtx2DPaint); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXCtx2DPaint) { _iDXCtx2DPaint->_vtbl->_release(_iDXCtx2DPaint); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getType(eCEDXCtx2DPaintType* pPaintTypeOut)	{ return _iDXCtx2DPaint ? _iDXCtx2DPaint->_vtbl->_getType(_iDXCtx2DPaint, pPaintTypeOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXCtx2DPaint* _iDXCtx2DPaint;
};

/*! @}
 * end of CEComICEDXCtx2DPaintRef
 */

/*! \defgroup CEComICEDXCtx2DPatternPaintRef CEComICEDXCtx2DPatternPaintRef
 * @{
 */

class CEComICEDXCtx2DPatternPaintRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXCtx2DPatternPaintRef() : _iDXCtx2DPatternPaint(0) {}
	CEComICEDXCtx2DPatternPaintRef(ICEDXCtx2DPatternPaint* iOther) : _iDXCtx2DPatternPaint(0)
	{
		if (iOther)
		{
			_iDXCtx2DPatternPaint = iOther;
			_iDXCtx2DPatternPaint->_vtbl->_addRef(_iDXCtx2DPatternPaint);
		}
	}

	CEComICEDXCtx2DPatternPaintRef(const CEComICEDXCtx2DPatternPaintRef& other) : _iDXCtx2DPatternPaint(0)
	{
		if (other._iDXCtx2DPatternPaint)
		{
			_iDXCtx2DPatternPaint = other._iDXCtx2DPatternPaint;
			_iDXCtx2DPatternPaint->_vtbl->_addRef(_iDXCtx2DPatternPaint);
		}
	}

	~CEComICEDXCtx2DPatternPaintRef()
	{
		if (_iDXCtx2DPatternPaint)
		{
			ICEDXCtx2DPatternPaint* tmp = _iDXCtx2DPatternPaint;
			_iDXCtx2DPatternPaint = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXCtx2DPatternPaint* iDXCtx2DPatternPaint;
				void* _ptr;
			} uIntf;
			uIntf.iDXCtx2DPatternPaint = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXCtx2DPatternPaint, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXCtx2DPatternPaint)
				{
					ICEDXCtx2DPatternPaint* tmp = _iDXCtx2DPatternPaint;
					_iDXCtx2DPatternPaint = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXCtx2DPatternPaint = uIntf.iDXCtx2DPatternPaint;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXCtx2DPatternPaint* object() const	{ return _iDXCtx2DPatternPaint; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXCtx2DPatternPaint* iOther)
	{
		if (_iDXCtx2DPatternPaint)
		{
			ICEDXCtx2DPatternPaint* tmp = _iDXCtx2DPatternPaint;
			_iDXCtx2DPatternPaint = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXCtx2DPatternPaint = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXCtx2DPatternPaint* detach()
	{
		ICEDXCtx2DPatternPaint* iIntf = _iDXCtx2DPatternPaint;
		_iDXCtx2DPatternPaint = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXCtx2DPatternPaint* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXCtx2DPatternPaint;
		if (_iDXCtx2DPatternPaint)
		{
			_iDXCtx2DPatternPaint->_vtbl->_addRef(_iDXCtx2DPatternPaint);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXCtx2DPatternPaint*() const	{ return _iDXCtx2DPatternPaint; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXCtx2DPatternPaint& operator*() const	{ return *_iDXCtx2DPatternPaint; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXCtx2DPatternPaint == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXCtx2DPatternPaint* iOther) const	{ return (_iDXCtx2DPatternPaint != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXCtx2DPatternPaint* iOther) const	{ return (_iDXCtx2DPatternPaint == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXCtx2DPatternPaintRef& operator=(const CEComICEDXCtx2DPatternPaintRef& other)	{ return operator=(other._iDXCtx2DPatternPaint); }

	CEComICEDXCtx2DPatternPaintRef& operator=(const ICEDXCtx2DPatternPaint* iOther)
	{
		if (_iDXCtx2DPatternPaint != iOther)
		{
			if (_iDXCtx2DPatternPaint)
			{
				ICEDXCtx2DPatternPaint* tmp = _iDXCtx2DPatternPaint;
				_iDXCtx2DPatternPaint = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXCtx2DPatternPaint = const_cast<ICEDXCtx2DPatternPaint*>(iOther);
			if (_iDXCtx2DPatternPaint)
			{
				_iDXCtx2DPatternPaint->_vtbl->_addRef(_iDXCtx2DPatternPaint);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXCtx2DPatternPaint** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXCtx2DPatternPaintRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXCtx2DPatternPaint && "has a com object reference. clear first.");
		return & _iDXCtx2DPatternPaint; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXCtx2DPatternPaintRef& other)
	{
		bool result = false;
		if (_iDXCtx2DPatternPaint)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXCtx2DPatternPaint);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXCtx2DPatternPaint ? _iDXCtx2DPatternPaint->_vtbl->_queryInterface(_iDXCtx2DPatternPaint, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXCtx2DPatternPaint) { _iDXCtx2DPatternPaint->_vtbl->_addRef(_iDXCtx2DPatternPaint); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXCtx2DPatternPaint) { _iDXCtx2DPatternPaint->_vtbl->_release(_iDXCtx2DPatternPaint); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getType(eCEDXCtx2DPaintType* pPaintTypeOut)	{ return _iDXCtx2DPatternPaint ? _iDXCtx2DPatternPaint->_vtbl->_getType(_iDXCtx2DPatternPaint, pPaintTypeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setImage(struct ICEDXSurface* iImage, struct ICEDXPalette* iPalette)	{ return _iDXCtx2DPatternPaint ? _iDXCtx2DPatternPaint->_vtbl->_setImage(_iDXCtx2DPatternPaint, iImage, iPalette) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getImage(struct ICEDXSurface* *const piImage, struct ICEDXPalette* *const piPalette)	{ return _iDXCtx2DPatternPaint ? _iDXCtx2DPatternPaint->_vtbl->_getImage(_iDXCtx2DPatternPaint, piImage, piPalette) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setWrapMode(eCEDX2DWrapMode mode)	{ return _iDXCtx2DPatternPaint ? _iDXCtx2DPatternPaint->_vtbl->_setWrapMode(_iDXCtx2DPatternPaint, mode) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getWrapMode(eCEDX2DWrapMode *const modeOut)	{ return _iDXCtx2DPatternPaint ? _iDXCtx2DPatternPaint->_vtbl->_getWrapMode(_iDXCtx2DPatternPaint, modeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult resetTransform()	{ return _iDXCtx2DPatternPaint ? _iDXCtx2DPatternPaint->_vtbl->_resetTransform(_iDXCtx2DPatternPaint) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setTransform(CEDXMatrix33* pMatrix)	{ return _iDXCtx2DPatternPaint ? _iDXCtx2DPatternPaint->_vtbl->_setTransform(_iDXCtx2DPatternPaint, pMatrix) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getTransform(CEDXMatrix33* const pMatOut)	{ return _iDXCtx2DPatternPaint ? _iDXCtx2DPatternPaint->_vtbl->_getTransform(_iDXCtx2DPatternPaint, pMatOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXCtx2DPatternPaint* _iDXCtx2DPatternPaint;
};

/*! @}
 * end of CEComICEDXCtx2DPatternPaintRef
 */

/*! \defgroup CEComICEDXCtx2DDisplayListRef CEComICEDXCtx2DDisplayListRef
 * @{
 */

class CEComICEDXCtx2DDisplayListRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXCtx2DDisplayListRef() : _iDXCtx2DDisplayList(0) {}
	CEComICEDXCtx2DDisplayListRef(ICEDXCtx2DDisplayList* iOther) : _iDXCtx2DDisplayList(0)
	{
		if (iOther)
		{
			_iDXCtx2DDisplayList = iOther;
			_iDXCtx2DDisplayList->_vtbl->_addRef(_iDXCtx2DDisplayList);
		}
	}

	CEComICEDXCtx2DDisplayListRef(const CEComICEDXCtx2DDisplayListRef& other) : _iDXCtx2DDisplayList(0)
	{
		if (other._iDXCtx2DDisplayList)
		{
			_iDXCtx2DDisplayList = other._iDXCtx2DDisplayList;
			_iDXCtx2DDisplayList->_vtbl->_addRef(_iDXCtx2DDisplayList);
		}
	}

	~CEComICEDXCtx2DDisplayListRef()
	{
		if (_iDXCtx2DDisplayList)
		{
			ICEDXCtx2DDisplayList* tmp = _iDXCtx2DDisplayList;
			_iDXCtx2DDisplayList = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXCtx2DDisplayList* iDXCtx2DDisplayList;
				void* _ptr;
			} uIntf;
			uIntf.iDXCtx2DDisplayList = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXCtx2DDisplayList, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXCtx2DDisplayList)
				{
					ICEDXCtx2DDisplayList* tmp = _iDXCtx2DDisplayList;
					_iDXCtx2DDisplayList = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXCtx2DDisplayList = uIntf.iDXCtx2DDisplayList;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXCtx2DDisplayList* object() const	{ return _iDXCtx2DDisplayList; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXCtx2DDisplayList* iOther)
	{
		if (_iDXCtx2DDisplayList)
		{
			ICEDXCtx2DDisplayList* tmp = _iDXCtx2DDisplayList;
			_iDXCtx2DDisplayList = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXCtx2DDisplayList = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXCtx2DDisplayList* detach()
	{
		ICEDXCtx2DDisplayList* iIntf = _iDXCtx2DDisplayList;
		_iDXCtx2DDisplayList = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXCtx2DDisplayList* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXCtx2DDisplayList;
		if (_iDXCtx2DDisplayList)
		{
			_iDXCtx2DDisplayList->_vtbl->_addRef(_iDXCtx2DDisplayList);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXCtx2DDisplayList*() const	{ return _iDXCtx2DDisplayList; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXCtx2DDisplayList& operator*() const	{ return *_iDXCtx2DDisplayList; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXCtx2DDisplayList == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXCtx2DDisplayList* iOther) const	{ return (_iDXCtx2DDisplayList != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXCtx2DDisplayList* iOther) const	{ return (_iDXCtx2DDisplayList == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXCtx2DDisplayListRef& operator=(const CEComICEDXCtx2DDisplayListRef& other)	{ return operator=(other._iDXCtx2DDisplayList); }

	CEComICEDXCtx2DDisplayListRef& operator=(const ICEDXCtx2DDisplayList* iOther)
	{
		if (_iDXCtx2DDisplayList != iOther)
		{
			if (_iDXCtx2DDisplayList)
			{
				ICEDXCtx2DDisplayList* tmp = _iDXCtx2DDisplayList;
				_iDXCtx2DDisplayList = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXCtx2DDisplayList = const_cast<ICEDXCtx2DDisplayList*>(iOther);
			if (_iDXCtx2DDisplayList)
			{
				_iDXCtx2DDisplayList->_vtbl->_addRef(_iDXCtx2DDisplayList);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXCtx2DDisplayList** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXCtx2DDisplayListRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXCtx2DDisplayList && "has a com object reference. clear first.");
		return & _iDXCtx2DDisplayList; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXCtx2DDisplayListRef& other)
	{
		bool result = false;
		if (_iDXCtx2DDisplayList)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXCtx2DDisplayList);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXCtx2DDisplayList ? _iDXCtx2DDisplayList->_vtbl->_queryInterface(_iDXCtx2DDisplayList, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXCtx2DDisplayList) { _iDXCtx2DDisplayList->_vtbl->_addRef(_iDXCtx2DDisplayList); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXCtx2DDisplayList) { _iDXCtx2DDisplayList->_vtbl->_release(_iDXCtx2DDisplayList); } }


private:
	ICEDXCtx2DDisplayList* _iDXCtx2DDisplayList;
};

/*! @}
 * end of CEComICEDXCtx2DDisplayListRef
 */

/*! \defgroup CEComICEDXContext2DRef CEComICEDXContext2DRef
 * @{
 */

class CEComICEDXContext2DRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXContext2DRef() : _iDXContext2D(0) {}
	CEComICEDXContext2DRef(ICEDXContext2D* iOther) : _iDXContext2D(0)
	{
		if (iOther)
		{
			_iDXContext2D = iOther;
			_iDXContext2D->_vtbl->_addRef(_iDXContext2D);
		}
	}

	CEComICEDXContext2DRef(const CEComICEDXContext2DRef& other) : _iDXContext2D(0)
	{
		if (other._iDXContext2D)
		{
			_iDXContext2D = other._iDXContext2D;
			_iDXContext2D->_vtbl->_addRef(_iDXContext2D);
		}
	}

	~CEComICEDXContext2DRef()
	{
		if (_iDXContext2D)
		{
			ICEDXContext2D* tmp = _iDXContext2D;
			_iDXContext2D = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXContext2D* iDXContext2D;
				void* _ptr;
			} uIntf;
			uIntf.iDXContext2D = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXContext2D, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXContext2D)
				{
					ICEDXContext2D* tmp = _iDXContext2D;
					_iDXContext2D = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXContext2D = uIntf.iDXContext2D;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXContext2D* object() const	{ return _iDXContext2D; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXContext2D* iOther)
	{
		if (_iDXContext2D)
		{
			ICEDXContext2D* tmp = _iDXContext2D;
			_iDXContext2D = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXContext2D = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXContext2D* detach()
	{
		ICEDXContext2D* iIntf = _iDXContext2D;
		_iDXContext2D = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXContext2D* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXContext2D;
		if (_iDXContext2D)
		{
			_iDXContext2D->_vtbl->_addRef(_iDXContext2D);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXContext2D*() const	{ return _iDXContext2D; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXContext2D& operator*() const	{ return *_iDXContext2D; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXContext2D == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXContext2D* iOther) const	{ return (_iDXContext2D != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXContext2D* iOther) const	{ return (_iDXContext2D == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXContext2DRef& operator=(const CEComICEDXContext2DRef& other)	{ return operator=(other._iDXContext2D); }

	CEComICEDXContext2DRef& operator=(const ICEDXContext2D* iOther)
	{
		if (_iDXContext2D != iOther)
		{
			if (_iDXContext2D)
			{
				ICEDXContext2D* tmp = _iDXContext2D;
				_iDXContext2D = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXContext2D = const_cast<ICEDXContext2D*>(iOther);
			if (_iDXContext2D)
			{
				_iDXContext2D->_vtbl->_addRef(_iDXContext2D);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXContext2D** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXContext2DRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXContext2D && "has a com object reference. clear first.");
		return & _iDXContext2D; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXContext2DRef& other)
	{
		bool result = false;
		if (_iDXContext2D)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXContext2D);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_queryInterface(_iDXContext2D, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXContext2D) { _iDXContext2D->_vtbl->_addRef(_iDXContext2D); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXContext2D) { _iDXContext2D->_vtbl->_release(_iDXContext2D); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFactory(struct ICEDX2DFactory* *const piFactoryOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getFactory(_iDXContext2D, piFactoryOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDevice(struct ICEDXDevice* *const piDevice)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getDevice(_iDXContext2D, piDevice) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getSurface(struct ICEDXSurface* *const piSurfaceOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getSurface(_iDXContext2D, piSurfaceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isForIndexColor(bool* const pOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_isForIndexColor(_iDXContext2D, pOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setState(eCEDXCtx2DState state, const UINT32 val)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_setState(_iDXContext2D, state, val) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getState(eCEDXCtx2DState state, UINT32 *const pValOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getState(_iDXContext2D, state, pValOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setStateC(eCEDXCtx2DColorState state, const CERGBAColor* pColor)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_setStateC(_iDXContext2D, state, pColor) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStateC(eCEDXCtx2DColorState state, CERGBAColor *const pColorOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getStateC(_iDXContext2D, state, pColorOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setStateF(eCEDXCtx2DRealState state, const float val)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_setStateF(_iDXContext2D, state, val) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStateF(eCEDXCtx2DRealState state, float *const pValOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getStateF(_iDXContext2D, state, pValOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setStateFv(eCEDXCtx2DRealvState state, const float* pVals, UINT32 count)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_setStateFv(_iDXContext2D, state, pVals, count) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStateFv(eCEDXCtx2DRealvState state, float *const pValsOut, UINT32 *const countOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getStateFv(_iDXContext2D, state, pValsOut, countOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setStateP(eCEDXCtx2DPtrState state, void* val)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_setStateP(_iDXContext2D, state, val) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStateP(eCEDXCtx2DPtrState state, void* *const pValOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getStateP(_iDXContext2D, state, pValOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setStateIv(eCEDXCtx2DIntvState state, const INT32* pVals, UINT32 count)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_setStateIv(_iDXContext2D, state, pVals, count) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setStateTSR(eCETLStyleAttribute attr, const CETLReal val)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_setStateTSR(_iDXContext2D, attr, val) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStateTSR(eCETLStyleAttribute attr, CETLReal* const pValOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getStateTSR(_iDXContext2D, attr, pValOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setStateTSC(eCETLStyleAttribute attr, const CERGBAColor* pVal)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_setStateTSC(_iDXContext2D, attr, pVal) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStateTSC(eCETLStyleAttribute attr, CERGBAColor* const pValOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getStateTSC(_iDXContext2D, attr, pValOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setStateTLR(eCETLLayoutAttribute attr, const CETLReal val)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_setStateTLR(_iDXContext2D, attr, val) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStateTLR(eCETLLayoutAttribute attr, CETLReal* const pValOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getStateTLR(_iDXContext2D, attr, pValOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setStateTLB(eCETLLayoutAttribute attr, const bool val)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_setStateTLB(_iDXContext2D, attr, val) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStateTLB(eCETLLayoutAttribute attr, bool* const pValOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getStateTLB(_iDXContext2D, attr, pValOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDpiX(float *pDpiX)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getDpiX(_iDXContext2D, pDpiX) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDpiY(float *pDpiY)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getDpiY(_iDXContext2D, pDpiY) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult translateCTM(float tx, float ty)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_translateCTM(_iDXContext2D, tx, ty) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult rotateCTM(float angle)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_rotateCTM(_iDXContext2D, angle) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult scaleCTM(float sx, float sy)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_scaleCTM(_iDXContext2D, sx, sy) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCTM(CEDXMatrix33* const pMatOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getCTM(_iDXContext2D, pMatOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult saveState()	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_saveState(_iDXContext2D) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult restoreState()	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_restoreState(_iDXContext2D) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult beginRecord()	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_beginRecord(_iDXContext2D) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult endRecord(struct ICEDX2DDisplayList* *const piDListOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_endRecord(_iDXContext2D, piDListOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult execDisplayList(struct ICEDX2DDisplayList* iDList)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_execDisplayList(_iDXContext2D, iDList) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult clearScissorRect()	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_clearScissorRect(_iDXContext2D) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setScissorRect(const CERect* pRect)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_setScissorRect(_iDXContext2D, pRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getScissorRect(CERect*  pRectOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getScissorRect(_iDXContext2D, pRectOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult beginPaint(struct ICEDXImmutableRegion* iUpdateRgn)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_beginPaint(_iDXContext2D, iUpdateRgn) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult endPaint()	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_endPaint(_iDXContext2D) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getUpdateRegion(struct ICEDXImmutableRegion* *const iUpdateRgnOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getUpdateRegion(_iDXContext2D, iUpdateRgnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokeLineF(const CEPointF*    pStartPos, const CEPointF*    pEndPos)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_strokeLineF(_iDXContext2D, pStartPos, pEndPos) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokeLineI(const CEPointBase* pStartPos, const CEPointBase* pEndPos)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_strokeLineI(_iDXContext2D, pStartPos, pEndPos) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokeRectF(const CERectF* pRect)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_strokeRectF(_iDXContext2D, pRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokeRectI(const CERect*  pRect)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_strokeRectI(_iDXContext2D, pRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fillRectF(const CERectF* pRect)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_fillRectF(_iDXContext2D, pRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fillRectI(const CERect*  pRect)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_fillRectI(_iDXContext2D, pRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokeRoundRectF(const CERectF* pRect, float arcWidth, float arcHeight)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_strokeRoundRectF(_iDXContext2D, pRect, arcWidth, arcHeight) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokeRoundRectI(const CERect*  pRect, INT32 arcWidth, INT32 arcHeight)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_strokeRoundRectI(_iDXContext2D, pRect, arcWidth, arcHeight) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fillRoundRectF(const CERectF* pRect, float arcWidth, float arcHeight)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_fillRoundRectF(_iDXContext2D, pRect, arcWidth, arcHeight) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fillRoundRectI(const CERect* pRect,  INT32 arcWidth, INT32 arcHeight)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_fillRoundRectI(_iDXContext2D, pRect, arcWidth, arcHeight) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokeEllipseF(const CERectF* pBbox)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_strokeEllipseF(_iDXContext2D, pBbox) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokeEllipseI(const CERect*  pBbox)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_strokeEllipseI(_iDXContext2D, pBbox) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fillEllipseF(const CERectF* pBbox)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_fillEllipseF(_iDXContext2D, pBbox) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fillEllipseI(const CERect*  pBbox)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_fillEllipseI(_iDXContext2D, pBbox) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokePolygonF(const CEPointF*    points, UINT32 count)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_strokePolygonF(_iDXContext2D, points, count) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokePolygonI(const CEPointBase* points, UINT32 count)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_strokePolygonI(_iDXContext2D, points, count) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fillPolygonF(const CEPointF*    points, UINT32 count)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_fillPolygonF(_iDXContext2D, points, count) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fillPolygonI(const CEPointBase* points, UINT32 count)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_fillPolygonI(_iDXContext2D, points, count) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokeArcF(const CERectF* pBbox, float startAngle, float arcAngle)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_strokeArcF(_iDXContext2D, pBbox, startAngle, arcAngle) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokeArcI(const CERect*  pBbox, float startAngle, float arcAngle)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_strokeArcI(_iDXContext2D, pBbox, startAngle, arcAngle) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fillArcF(const CERectF* pBbox, float startAngle, float arcAngle)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_fillArcF(_iDXContext2D, pBbox, startAngle, arcAngle) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fillArcI(const CERect*  pBbox, float startAngle, float arcAngle)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_fillArcI(_iDXContext2D, pBbox, startAngle, arcAngle) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawImageF(const CERectF* pDstRect, struct ICEDXSurface* iImage, struct ICEDXPalette* iPalette, const CERectF* pSrcRect)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawImageF(_iDXContext2D, pDstRect, iImage, iPalette, pSrcRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawImageI(const CERect*  pDstRect, struct ICEDXSurface* iImage, struct ICEDXPalette* iPalette, const CERect*  pSrcRect)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawImageI(_iDXContext2D, pDstRect, iImage, iPalette, pSrcRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawCompressedImageF(const CERectF* pDstRect, struct ICECompressedImage* iCompressedImage, const CERectF* pSrcRect)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawCompressedImageF(_iDXContext2D, pDstRect, iCompressedImage, pSrcRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawCompressedImageI(const CERect*  pDstRect, struct ICECompressedImage* iCompressedImage, const CERect*  pSrcRect)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawCompressedImageI(_iDXContext2D, pDstRect, iCompressedImage, pSrcRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult copyAreaF(const CERectF* pDstRect, struct ICEDXSurface* pSrcSurface, const CERectF* pSrcRect)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_copyAreaF(_iDXContext2D, pDstRect, pSrcSurface, pSrcRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult copyAreaI(const CERect*  pDstRect, struct ICEDXSurface* pSrcSurface, const CERect*  pSrcRect)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_copyAreaI(_iDXContext2D, pDstRect, pSrcSurface, pSrcRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawCharF(const CEPointF*    pPos, const struct ICEPFGlyphLoader* iGlyphLoader, const UTF32CHAR c32, const UINT32 glyphIndex, bool bUseC32)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawCharF(_iDXContext2D, pPos, iGlyphLoader, c32, glyphIndex, bUseC32) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawCharI(const CEPointBase* pPos, const struct ICEPFGlyphLoader* iGlyphLoader, const UTF32CHAR c32)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawCharI(_iDXContext2D, pPos, iGlyphLoader, c32) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawCharsF(const CEPointF*    ppPos, const struct ICEPFGlyphLoader* iGlyphLoader, const UTF32CHAR *c32, UINT32 size)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawCharsF(_iDXContext2D, ppPos, iGlyphLoader, c32, size) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawCharsI(const CEPointBase* ppPos, const struct ICEPFGlyphLoader* iGlyphLoader, const UTF32CHAR *c32, UINT32 size)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawCharsI(_iDXContext2D, ppPos, iGlyphLoader, c32, size) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawTextF(const CERectF* pBbox, const UTF16CHAR* pStr, INT32 textLen)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawTextF(_iDXContext2D, pBbox, pStr, textLen) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawTextI(const CERect* pBbox,  const UTF16CHAR* pStr, INT32 textLen)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawTextI(_iDXContext2D, pBbox, pStr, textLen) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawStringF(const CERectF* pBbox, struct ICEUString* iStr, UINT32 textStart, INT32 textLen)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawStringF(_iDXContext2D, pBbox, iStr, textStart, textLen) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawStringI(const CERect* pBbox,  struct ICEUString* iStr, UINT32 textStart, INT32 textLen)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawStringI(_iDXContext2D, pBbox, iStr, textStart, textLen) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawMAStringF(const CERectF* pBbox, struct ICEMAString* iStr, UINT32 textStart, INT32 textLen)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawMAStringF(_iDXContext2D, pBbox, iStr, textStart, textLen) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawMAStringI(const CERect* pBbox,  struct ICEMAString* iStr, UINT32 textStart, INT32 textLen)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_drawMAStringI(_iDXContext2D, pBbox, iStr, textStart, textLen) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getPixelFormat(CEPixelFormat* const pPxlFmtOut)	{ return _iDXContext2D ? _iDXContext2D->_vtbl->_getPixelFormat(_iDXContext2D, pPxlFmtOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXContext2D* _iDXContext2D;
};

/*! @}
 * end of CEComICEDXContext2DRef
 */

#endif // __cplusplus

#endif
