///////////////////////////////////////////////////////////////////////////////
// Copyright 2010 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef __CESysHeap_h__
#define __CESysHeap_h__

#include "CESysDefs.h"
#include "CESysHandle.h"

////////////////////////////////////////////////////////////////
// C-linkage API
////////////////////////////////////////////////////////////////

#ifdef __cplusplus
extern "C"
{
#endif

	/////////////////////////////////////////////
	// heap config
	/////////////////////////////////////////////

	typedef enum
	{
		eCESysHeapStat_UsedBytes = 0,	//!< sum of allocated block size (in bytes)
	} eCESysHeapStat;

	CE_DLL_EXPORT INT32 CESysHeapGetStat(eCESysHeapStat name);

	CE_DLL_EXPORT bool CESysHeapStatControl(const char* cmd, ...);

	typedef enum
	{
		eCESysHeapLimit_MaxHeapSize = 1,  //!< total heap size (in bytes)
		eCESysHeapLimit_MaxPurgeableSize,
	} eCESysHeapLimit;

	CE_DLL_EXPORT bool  CESysHeapSetLimit(eCESysHeapLimit name, INT32 value);
	CE_DLL_EXPORT INT32 CESysHeapGetLimit(eCESysHeapLimit name);

	CE_DLL_EXPORT bool CESysHeapSetAddrRange(void* addr, size_t size);
	CE_DLL_EXPORT bool CESysHeapUnsetAddrRange(void* addr, size_t size);

	/////////////////////////////////////////////
	// normal heap
	/////////////////////////////////////////////

	CE_DLL_EXPORT void*  CESysMalloc(size_t size);
	CE_DLL_EXPORT void*  CESysRealloc(void* mblk, size_t size);
	CE_DLL_EXPORT void*  CESysCalloc(size_t num, size_t size);
	CE_DLL_EXPORT void   CESysFree(void* mblk);
	CE_DLL_EXPORT size_t CESysMSize(const void* mblk);

	CE_DLL_EXPORT void*  CESysAlignedMalloc(size_t size, size_t alignment);
	CE_DLL_EXPORT void   CESysAlignedFree(void* mblk, size_t size);
//	CE_DLL_EXPORT size_t CESysAlignedMSize(const void* mblk);
	

	/////////////////////////////////////////////
	// client-specific heap
	/////////////////////////////////////////////

	///////////////////////
	// eCESysHeapClientId
	///////////////////////
	#define CESYS_HEAP_CLIENT(name)		  eCESysHeapClientId_##name, 
	#define CESYS_HEAP_CLIENT_API(name)	  eCESysHeapClientId_##name, 

	typedef enum
	{
		#include "CESysHeapClientDefs.h"

		eCESysHeapClientId_MAX
	} CESysHeapClientId;

	#undef CESYS_HEAP_CLIENT
	#undef CESYS_HEAP_CLIENT_API


	////////////////////////////////
	// malloc API with client-id
	////////////////////////////////

	CE_DLL_EXPORT void*  CESysMallocEx(CESysHeapClientId id, size_t size);
	CE_DLL_EXPORT void*  CESysReallocEx(CESysHeapClientId id, void* mblk, size_t size);
	CE_DLL_EXPORT void*  CESysCallocEx(CESysHeapClientId id, size_t num, size_t size);
	CE_DLL_EXPORT void   CESysFreeEx(CESysHeapClientId id, void* mblk);
	CE_DLL_EXPORT size_t CESysMSizeEx(CESysHeapClientId id, const void* mblk);

	CE_DLL_EXPORT void*  CESysAlignedMallocEx(CESysHeapClientId id, size_t size, size_t alignment);
	CE_DLL_EXPORT void   CESysAlignedFreeEx(CESysHeapClientId id, void* mblk, size_t size);

	CE_DLL_EXPORT char*  CESysStrDup(CESysHeapClientId id, const char* s);


	////////////////////////////////
	// purgeable memblock
	////////////////////////////////

	typedef struct _CESysMemBlock* CESysMemBlockHandle;

	CE_DLL_EXPORT CESysMemBlockHandle CESysMemBlockCreate(CESysHeapClientId id, size_t size);
	CE_DLL_EXPORT void CESysMemBlockDestroy(CESysMemBlockHandle hMem);
	CE_DLL_EXPORT void* CESysMemBlockLock(CESysMemBlockHandle hMem);
	CE_DLL_EXPORT void CESysMemBlockUnlock(CESysMemBlockHandle hMem);
	CE_DLL_EXPORT void CESysMemBlockSetPurgeable(CESysMemBlockHandle hMem);
	CE_DLL_EXPORT void CESysMemBlockUnsetPurgeable(CESysMemBlockHandle hMem);


	///////////////////////
	// heap-specific APIs
	///////////////////////

	#define CESYS_HEAP_CLIENT(name)
	#define CESYS_HEAP_CLIENT_API(name) \
		CE_DLL_EXPORT void*  CESys##name##Malloc(size_t size); \
		CE_DLL_EXPORT void*  CESys##name##Realloc(void* mblk, size_t size); \
		CE_DLL_EXPORT void*  CESys##name##Calloc(size_t num, size_t size); \
		CE_DLL_EXPORT void   CESys##name##Free(void* mblk); \
		CE_DLL_EXPORT size_t CESys##name##MSize(const void* mblk); \
		CE_DLL_EXPORT char*  CESys##name##StrDup(const char* s);

	#include "CESysHeapClientDefs.h"

	#undef CESYS_HEAP_CLIENT
	#undef CESYS_HEAP_CLIENT_API


	/////////////////////////////////////////////
	// allocator macros
	/////////////////////////////////////////////

	#define CEMALLOC(sz)			CESysMalloc(sz)
	#define CEREALLOC(p,sz)			CESysRealloc((p), (sz))
	#define CECALLOC(nElem, szElem) CESysCalloc((nElem), (szElem))
	#define CEFREE(p)				CESysFree(p)
	#define CEMSIZE(p)				CESysMSize(p)

	#define CEMALLOCEX(id, sz)				CESysMallocEx((eCESysHeapClientId_##id), (sz))
	#define CEREALLOCEX(id, p, sz)			CESysReallocEx((eCESysHeapClientId_##id), (p), (sz))
	#define CECALLOCEX(id, nElem, szElem)	CESysCallocEx((eCESysHeapClientId_##id), (nElem), (szElem))
	#define CEFREEEX(id, p)					CESysFreeEx((eCESysHeapClientId_##id), (p))
	#define CEMSIZEEX(id, p)				CESysMSizeEx((eCESysHeapClientId_##id), (p))

	#define CE_ALIGNED_MALLOC(sz, align)	CESysAlignedMalloc((sz), (align))
	#define CE_ALIGNED_FREE(p, sz)				CESysAlignedFree((p), (sz))

	#define CE_ALIGNED_MALLOCEX(id, sz, align)	CESysAlignedMallocEx((eCESysHeapClientId_##id),(sz),(align))
	#define CE_ALIGNED_FREEEX(id, p, sz)			CESysAlignedFreeEx((eCESysHeapClientId_##id),(p),(sz))

	#define CESTRDUPEX(id, s)	CESysStrDup((eCESysHeapClientId_##id),(s))

	#define CEMBLK_CREATE(id, sz)		CESysMemBlockCreate((eCESysHeapClientId_##id), (sz))
	#define CEMBLK_DESTROY(h)			CESysMemBlockDestroy(h)
	#define CEMBLK_LOCK(h, type)		reinterpret_cast<type *>(CESysMemBlockLock(h))
	#define CEMBLK_UNLOCK(h)			CESysMemBlockUnlock(h)
	#define CEMBLK_SET_PURGEABLE(h)		CESysMemBlockSetPurgeable(h)
	#define CEMBLK_UNSET_PURGEABLE(h)	CESysMemBlockUnsetPurgeable(h)



#ifdef __cplusplus
}
#endif


////////////////////////////////////////////////////////////////
// new/delete operator override in class.
//   Declare this in the public section in class header.
////////////////////////////////////////////////////////////////
#ifdef __cplusplus
	#define CEALLOCATORS \
		void* operator new(size_t size)    { return CEMALLOCEX(CENew, size); } \
		void operator delete(void* p)      { CEFREEEX(CENew, p); } \
		void* operator new[] (size_t size) { return CEMALLOCEX(CENew, size); } \
		void operator delete[] (void* p)   { CEFREEEX(CENew, p); }

	#define CEALLOCATORSEX(id) \
		void* operator new(size_t size)    { return CEMALLOCEX(id, size); } \
		void operator delete(void* p)      { CEFREEEX(id, p); } \
		void* operator new[] (size_t size) { return CEMALLOCEX(id, size); } \
		void operator delete[] (void* p)   { CEFREEEX(id, p); }
#endif

#endif // __CESysHeap_h__
