///////////////////////////////////////////////////////////////////////////////
// Copyright 2004,2005,2006 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef __CESysDList_H__
#define __CESysDList_H__

#include "CESysDefs.h"

//! Double-linked circular list (C structure-version)
//!
//! Design Principle
//! ----------------
//! This structure and companion functions are supposed to be used
//! to form circular list over modules. This is why pure C structure
//! is used.
//!
//! Usage
//! ------
//! Embed this structure as the first member of your structure as follows:
//!
//! struct MyData {
//!     CEUDLList _list;
//!     int   _valueA;
//!     float _valueB;
//! };
//!
//! You can static-cast your data structure to CEUDLList and use it as follows:
//!
//! MyData* pMyData = ....;
//! if(CESysDListIsEmpty((CEUDLList*)pMyData))
//! {
//!     // do something.
//! }
//! 

#ifdef __cplusplus
extern "C"
{
#endif

	typedef struct _CESysDListEntry* CESysDListEntryPtr;

	typedef struct _CESysDListEntry
	{
		CESysDListEntryPtr next;
		CESysDListEntryPtr prev;
	} CESysDListEntry;

	typedef struct _CESysDListHeader
	{
		CESysDListEntry headtail;
	} CESysDListHeader;

	//! initialize the double-linked list
	CE_DLL_EXPORT void CESysDListInitializeHead(CESysDListHeader* list);
	
	//! initialize the entry of the double-linked list
	CE_DLL_EXPORT void CESysDListInitializeEntry(CESysDListEntry* entry);

	//! determine whether the double-linked list is empty or not
	CE_DLL_EXPORT bool CESysDListIsEmpty(CESysDListHeader* list);


	//! Insert the new entry before the referenceEntry in the list
	//!
	//!  [Before]                        [After]
	//!
	//!            next                            next            next
	//!  entry[i] -----> entry[j]  ===>  entry[i] -----> newEntry -----> entry[j]
	//!           <-----  ^                       <-----          <-----  ^
	//!            prev   |                        prev            prev   |
	//!                   |                                               |
	//!               referenceEntry                                 referenceEntry
	//!  
	CE_DLL_EXPORT void CESysDListInsertBefore(CESysDListEntry* newEntry, CESysDListEntry* referenceEntry);

	//! Insert the new entry after the referenceEntry in the list
	//!
	//!  [Before]                        [After]
	//!
	//!            next                            next            next
	//!  entry[i] -----> entry[j]  ===>  entry[i] -----> newEntry -----> entry[j]
	//!    ^      <-----                   ^      <-----          <-----  
	//!    |       prev                    |       prev            prev   
	//!    |                               |
	//!  referenceEntry                  referenceEntry
	//!  
	CE_DLL_EXPORT void CESysDListInsertAfter(CESysDListEntry* newEntry, CESysDListEntry* referenceEntry);

	//! Remove the specified entry from the list
	//!
	//!  [Before]                                    [After]
	//!
	//!            next        next                            next  
	//!  entry[i] -----> this -----> entry[j]  ===>  entry[i] -----> entry[j]
	//!           <-----      <-----                          <-----
	//!            prev        prev                            prev
	//!  
	CE_DLL_EXPORT void CESysDListRemoveEntry(CESysDListEntry* entry);

	//! add the entry to the list from the "front" side
	CE_DLL_EXPORT void CESysDListPushFront(CESysDListHeader* list, CESysDListEntry* entry);

	//! add the entry to the list from the "back" side
	CE_DLL_EXPORT void CESysDListPushBack(CESysDListHeader* list, CESysDListEntry* entry);

	//! Remove the entry from the list of the "front" side
	//! \return The removed entry is returned if the list is not empty. Otherwise, NULL is returned. 
	CE_DLL_EXPORT CESysDListEntry* CESysDListPopFront(CESysDListHeader* list);

	//! remove the entry from the list of the "back" side
	//! \return The removed entry is returned if the list is not empty. Otherwise, NULL is returned. 
	CE_DLL_EXPORT CESysDListEntry* CESysDListPopBack(CESysDListHeader* list);

#ifdef __cplusplus
};
#endif


	

#endif // __CESysDList_H__



