///////////////////////////////////////////////////////////////////////////////
// Copyright 2004,2005,2006,2007 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef CEComSingleIntfObjectT_h
#define CEComSingleIntfObjectT_h
#include "CECom.h"
#include "CEApiMacro.h"
#include "CEApiUnknown.h"
#include "ceudebugsupport.h"
#include "ceuhashtableT.h"
#include "CEApiErrorImpl.h"
#include "CEComAtomicSupport.h"


//! this template class is to define the base class that implements the single COM interface.
//!
//! TIntf     specifies the interface type name (e.g. ICEFoo)
//! TIntfVtbl specifies the vtable structure name corresponding to TIntf (e.g. ICEFoo_vtbl)
//! IID_TIntf specifies the IID of the interface
//! TAllocator specifies the memory manager class. 
//!
//!
template <class TIntf, class TIntfVtbl, UINT32 IID_TIntf>
class CEComSingleIntfObjectT
{
	//! constructor to setup the COM's vptr and reset the refcount
	//!
	//! \note This class should not be instanciated directly.
	//!       You need to define the subclass.
protected:
	CEComSingleIntfObjectT(const TIntfVtbl* vptr) : _iUnknownVptr(vptr), _refCount(0)  {}
	virtual ~CEComSingleIntfObjectT(){}

public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	/////////////////////////////////////////////////////////////////////////
	//  converting between this pointer and the COM interface pointer
	/////////////////////////////////////////////////////////////////////////

	static CEComSingleIntfObjectT* toInstance(TIntf* iIntf)
	{
		return IPTR_TO_INSTANCE(CEComSingleIntfObjectT, iIntf, _iUnknownVptr);
	}

	TIntf* toInterface()
	{
	 	return INSTANCE_TO_IPTR(TIntf, CEComSingleIntfObjectT, this, _iUnknownVptr);
	}

protected:
	/////////////////////////////////////////////////////////////////////////
	//  ICEUnknown methods implementation
	/////////////////////////////////////////////////////////////////////////

	static CEHResult __queryInterface(TIntf* iIntf, const UINT32 iId, void* *const iOut)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iOut)
		{
			switch (iId)
			{
			case IID_TIntf:
				{
					// Return the interface pointer casted to the specified IID
					// (But this template is for the single interface object,
					//  so the casting is just a copy of the given pointer.)
					*iOut = iIntf;

					// increment the refcount on return of the interface pointer
					CEComAtomicIncrement(toInstance(iIntf)->_refCount);
					hr = CE_S_OK;
					break;
				}
			default:
				{
					hr = CE_SILK_ERR_WRONGTYPE;
					break;
				}
			}
		}
		return hr;
	}

	static void __addRef(TIntf* iIntf)
	{
		JGSS_ASSERT(iIntf && "iIntf is 0");

		CEComAtomicIncrement(toInstance(iIntf)->_refCount);
	}

	static void __release(TIntf* iIntf)
	{
		JGSS_ASSERT(iIntf && "iIntf is 0");

		if (!CEComAtomicDecrement(toInstance(iIntf)->_refCount))
		{
			// NOTE: If the deferred finalize is required, 
			//       this code should be reconsidered. 
			delete toInstance(iIntf);
		}
	}

protected:
	// vptr of the interface
	const TIntfVtbl* _iUnknownVptr;

	// reference count
	CEATOMIC _refCount;
};

///
/// \note Usage of this template class
///
/*

template <class TImpl>
class CEFooT : public CEComSingleIntfObjectT<TImpl, ICEFoo, ICEFoo_vtbl, CEComIID_ICEFoo>
{
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	CEFooT(...) : 
	  CEComSingleIntfObjectT<TImpl, ICEFoo, ICEFoo_vtbl, CEComIID_ICEFoo>(&_vtbl) {}

private:
	// actual 
	TImpl	_impl;

	static TIntfVtbl _vtbl;
};
*/


class CEComObject
{
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	CEComObject():_refCount(0) {}
	virtual ~CEComObject(){}

	INT32 AddRef()	
	{
		return CEComAtomicIncrement(_refCount);
	}

	INT32 Release()	
	{
		INT32 ret = CEComAtomicDecrement(_refCount);
		if (ret <= 0) _Release();
		return ret;
	}

	INT32 RefCount() 
	{ 
		return CEComAtomicCount(_refCount);
	}

	virtual bool operator < (const CEComObject& /*objOther*/)const
	{
		return false;
	}
	virtual bool operator ==(const CEComObject& /*objOther*/)const
	{
		return false;
	}

protected:
	virtual void _Release() 
	{
		delete this; 
	}

	CEATOMIC _refCount;
};

//! this template class is to define the base class that implements the class object (with class cache).
//!
//! TClasFactory specify the class factory. 
//!
//! class SomeModuleEnv {
//! public:
//!     static CEComAbstractClassFactoryT* createClassFactory(UINT32 clsid, UINT32 iid, ICEClassFactory*& iCfOut);
//!     static bool isValidGuid(UINT32 clsid, UINT32 iid);
//!     static CEATOMIC incLoadCount();
//!     static CEATOMIC decLoadCount();
//!     static INT32 getLoadCount();
//! };
//!
template<class TModuleEnv>
class CEComAbstractClassFactoryT : public CEComObject
{
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	CEComAbstractClassFactoryT(UINT32 clsid) : CEComObject(), _key(clsid) { TModuleEnv::incLoadCount(); }

	~CEComAbstractClassFactoryT() { TModuleEnv::decLoadCount(); }

	static CEHResult getClassFactory(UINT32 clsid, UINT32 iid, void** ppv)
	{
		CEHResult hr = CE_SILK_ERR_BADARGS;
		*ppv = 0;

		if ( TModuleEnv::isValidGuid(clsid, iid) )
		{
			CEComAbstractClassFactoryT* item = TModuleEnv::createClassFactory(clsid, iid, ppv);
			if ( item )
			{
				hr = CE_S_OK;
			}
		}

		return hr;
	}

	static bool canUnloadNow() 
	{
		return (0 == TModuleEnv::getLoadCount());
	}

	class Object : public CEComObject
	{
	public:
		Object(CEComAbstractClassFactoryT* pClazz) : CEComObject() 
		{
			TModuleEnv::incLoadCount();
		}

		~Object() 
		{
			TModuleEnv::decLoadCount();
		}
		
		CEComAbstractClassFactoryT* getClassFactory() const 
		{
			return NULL;
		}
	};

public:

	UINT32 getClsid() { return _key.getClsid(); }

	class Key
	{
	public:
		Key(UINT32 clsid) { _clsid = clsid; }
		UINT32 getHash() const 
		{
			return _clsid;
		}

		UINT32 getClsid() const { return _clsid; }
	private:
		UINT32  _clsid;
	};
		  
	const Key& getKey()const
	{
		return _key;
	}
	bool isEqual(const Key& key)const
	{
		return (key.getClsid() == _key.getClsid());
	}

	static void globalShutdown(bool acquireGlobalLock=true)
	{
		if (acquireGlobalLock)
		{
			CEComGlobalLock();
		}

		if (acquireGlobalLock)
		{
			CEComGlobalUnlock();
		}
	}

private:
	Key _key;
};

#endif // CEComSingleIntfObjectT_h




