///////////////////////////////////////////////////////////////////////////////
// Copyright 2011 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef CEApiFile_h
#define CEApiFile_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CESysDefs.h"
#include "CEApiUnknown.h"
#include "CEApiFileLockStatus.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------


//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEFile ICEFile
 * @{
 */

/*!
 * ID of ICEFile
 */
#define CEComIID_ICEFile 0xeec72aa8


/*!
 * ICEFile
 */
typedef struct ICEFile
{
	const struct ICEFile_vtbl* _vtbl;
} ICEFile;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iFile	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFile_queryInterface) (ICEFile* iFile, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFile_addRef) (ICEFile* iFile);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFile_release) (ICEFile* iFile);


/*!
 * <b>Summary:</b>
 * TODO<br>
 *
 * \param[in]	iFile	Specifies interface pointer
 * \param	buf	TODO
 * \param	nbyte	TODO
 * \param	numReadOut	TODO
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFile_read) (ICEFile* iFile, void* buf, unsigned int nbyte, int *const numReadOut);


/*!
 * <b>Summary:</b>
 * TODO<br>
 *
 * \param[in]	iFile	Specifies interface pointer
 * \param	buf	TODO
 * \param	nbyte	TODO
 * \param	numWriteOut	TODO
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFile_write) (ICEFile* iFile, const void* buf, unsigned int nbyte, int *const numWriteOut);


/*!
 * <b>Summary:</b>
 * TODO<br>
 *
 * \param[in]	iFile	Specifies interface pointer
 * \param	offset	TODO
 * \param	whence	TODO
 * \param	positionOut	TODO
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFile_lseek) (ICEFile* iFile, ce_off_t offset, int whence, ce_off_t *const positionOut);


/*!
 * <b>Summary:</b>
 * TODO<br>
 *
 * \param[in]	iFile	Specifies interface pointer
 * \param	cmd	TODO
 * \param	length	TODO
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFile_lockf) (ICEFile* iFile, int cmd, ce_off_t length);


/*!
 * <b>Summary:</b>
 * TODO<br>
 *
 * \param[in]	iFile	Specifies interface pointer
 * \param	length	TODO
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFile_ftruncate) (ICEFile* iFile, ce_off_t length);


/*!
 * <b>Summary:</b>
 * TODO<br>
 *
 * \param[in]	iFile	Specifies interface pointer
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFile_fsync) (ICEFile* iFile);


/*!
 * <b>Summary:</b>
 * TODO<br>
 *
 * \param[in]	iFile	Specifies interface pointer
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFile_fdatasync) (ICEFile* iFile);

/*!
 * V-table of ICEFile
 */
struct ICEFile_vtbl
{
	iCEFile_queryInterface	_queryInterface;	//!< Query interface.
	iCEFile_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEFile_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEFile_read	_read;	//!< TODO
	iCEFile_write	_write;	//!< TODO
	iCEFile_lseek	_lseek;	//!< TODO
	iCEFile_lockf	_lockf;	//!< TODO
	iCEFile_ftruncate	_ftruncate;	//!< TODO
	iCEFile_fsync	_fsync;	//!< TODO
	iCEFile_fdatasync	_fdatasync;	//!< TODO
};

/*! @}
 * end of ICEFile
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEFileRef CEComICEFileRef
 * @{
 */

class CEComICEFileRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEFileRef() : _iFile(0) {}
	CEComICEFileRef(ICEFile* iOther) : _iFile(0)
	{
		if (iOther)
		{
			_iFile = iOther;
			_iFile->_vtbl->_addRef(_iFile);
		}
	}

	CEComICEFileRef(const CEComICEFileRef& other) : _iFile(0)
	{
		if (other._iFile)
		{
			_iFile = other._iFile;
			_iFile->_vtbl->_addRef(_iFile);
		}
	}

	~CEComICEFileRef()
	{
		if (_iFile)
		{
			ICEFile* tmp = _iFile;
			_iFile = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEFile* iFile;
				void* _ptr;
			} uIntf;
			uIntf.iFile = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEFile, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iFile)
				{
					ICEFile* tmp = _iFile;
					_iFile = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iFile = uIntf.iFile;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEFile* object() const	{ return _iFile; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEFile* iOther)
	{
		if (_iFile)
		{
			ICEFile* tmp = _iFile;
			_iFile = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iFile = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEFile* detach()
	{
		ICEFile* iIntf = _iFile;
		_iFile = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEFile* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iFile;
		if (_iFile)
		{
			_iFile->_vtbl->_addRef(_iFile);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEFile*() const	{ return _iFile; }
	FORCEINLINE_WITHOUT_DEBUG ICEFile& operator*() const	{ return *_iFile; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iFile == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEFile* iOther) const	{ return (_iFile != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEFile* iOther) const	{ return (_iFile == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEFileRef& operator=(const CEComICEFileRef& other)	{ return operator=(other._iFile); }

	CEComICEFileRef& operator=(const ICEFile* iOther)
	{
		if (_iFile != iOther)
		{
			if (_iFile)
			{
				ICEFile* tmp = _iFile;
				_iFile = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iFile = const_cast<ICEFile*>(iOther);
			if (_iFile)
			{
				_iFile->_vtbl->_addRef(_iFile);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEFile** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEFileRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iFile && "has a com object reference. clear first.");
		return & _iFile; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEFileRef& other)
	{
		bool result = false;
		if (_iFile)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iFile);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iFile ? _iFile->_vtbl->_queryInterface(_iFile, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iFile) { _iFile->_vtbl->_addRef(_iFile); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iFile) { _iFile->_vtbl->_release(_iFile); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult read(void* buf, unsigned int nbyte, int *const numReadOut)	{ return _iFile ? _iFile->_vtbl->_read(_iFile, buf, nbyte, numReadOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult write(const void* buf, unsigned int nbyte, int *const numWriteOut)	{ return _iFile ? _iFile->_vtbl->_write(_iFile, buf, nbyte, numWriteOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult lseek(ce_off_t offset, int whence, ce_off_t *const positionOut)	{ return _iFile ? _iFile->_vtbl->_lseek(_iFile, offset, whence, positionOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult lockf(int cmd, ce_off_t length)	{ return _iFile ? _iFile->_vtbl->_lockf(_iFile, cmd, length) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult ftruncate(ce_off_t length)	{ return _iFile ? _iFile->_vtbl->_ftruncate(_iFile, length) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fsync()	{ return _iFile ? _iFile->_vtbl->_fsync(_iFile) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fdatasync()	{ return _iFile ? _iFile->_vtbl->_fdatasync(_iFile) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEFile* _iFile;
};

/*! @}
 * end of CEComICEFileRef
 */

#endif // __cplusplus

#endif
