///////////////////////////////////////////////////////////////////////////////
// Copyright 2011 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef CEApiFSFileSystem_h
#define CEApiFSFileSystem_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiFSFile.h"
#include "CEApiFSPathname.h"
#include "ICEUString.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------


//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

typedef enum
{
	eCEFileOpenFlag_Read		= 0x0001,		//!< Read access.
	eCEFileOpenFlag_Write		= 0x0002,		//!< Write access.
	eCEFileOpenFlag_ReadWrite	= 0x0003,		//!< Read and write access. eCEFileOpenFlag_Read|eCEFileOpenFlag_Write
	eCEFileOpenFlag_Create		= 0x0010,		//!< Create file if it does not exist.
	eCEFileOpenFlag_Truncate	= 0x0020,		//!< Truncate file after open. Require write access.

	eCEFileOpenFlag_AccessMask	= 0x000f,		//!< Read/Write flag.
	eCEFileOpenFlag_CreateMask	= 0x00f0,		//!< Create option flag.
} eCEFileOpenFlag;

typedef enum
{
	eCEFileStatusFlag_Exist		= 0x0001,		//!< File or directory is exist.
	eCEFileStatusFlag_Directory	= 0x0002,		//!< Given pathname is directory.
} eCEFileStatusFlag;

typedef enum
{
	eCEDirectoryEnumerationFlag_skipDot			= 1 << 0,		//!< Skip "."
	eCEDirectoryEnumerationFlag_skipDotDot		= 1 << 1,		//!< Skip ".."
	eCEDirectoryEnumerationFlag_skipDirectory	= 1 << 2,		//!< Skip directory (enumerate only files)
	eCEDirectoryEnumerationFlag_skipHidden		= 1 << 3,		//!< Skip hidden files and directories.
	eCEDirectoryEnumerationFlag_skipFile		= 1 << 4,		//!< Skip file (enumerate only directories and symbolic links)
} eCEDirectoryEnumerationFlag;

typedef enum
{
	eCEFileSystemFeatures_streamRead		= 1 << 0,	//!< FileSystem class has stream-read feature.
	eCEFileSystemFeatures_streamWrite		= 1 << 1,	//!< FileSystem class has stream-write feature.
	eCEFileSystemFeatures_blockRead			= 1 << 2,	//!< FileSystem class has block-read feature.
	eCEFileSystemFeatures_blockWrite		= 1 << 3,	//!< FileSystem class has block-write feature.
	eCEFileSystemFeatures_lockFile			= 1 << 4,	//!< FileSystem class has file lock feature.
	eCEFileSystemFeatures_lockFileSystem	= 1 << 5,	//!< FileSystem class has filesystem lock feature.
	eCEFileSystemFeatures_driveLetter		= 1 << 6,	//!< FileSystem class has drive letter.
	eCEFileSystemFeatures_format			= 1 << 7,	//!< FileSystem class has format feature.
	eCEFileSystemFeatures_attachRequired	= 1 << 8,	//!< This class requires base I/O interface.
} eCEFileSystemFeatures;

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEFSDirectoryEnumerator ICEFSDirectoryEnumerator
 * @{
 */

/*!
 * ID of ICEFSDirectoryEnumerator
 */
#define CEComIID_ICEFSDirectoryEnumerator 0x07703b09


/*!
 * ICEFSDirectoryEnumerator
 *
 *
 *   This interface enumerates files/directories in the specified directory.
 *   Filename will return by ICEFSPathname.
 *
 *   Interface is created by ICEFSFileSystem::createDirectoryEnumerator,
 *   and detroyed when reference count will be 0. (i.e. ICEFSDirectoryEnumerator::_release)
 *
 *   This interface does not guarantee the return value
 *   when you/someone create/rename/remove files/directories whithin enumeration.
 *   If they were done, return value will be uncertain.
 *
 */
typedef struct ICEFSDirectoryEnumerator
{
	const struct ICEFSDirectoryEnumerator_vtbl* _vtbl;
} ICEFSDirectoryEnumerator;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iFSDirectoryEnumerator	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSDirectoryEnumerator_queryInterface) (ICEFSDirectoryEnumerator* iFSDirectoryEnumerator, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSDirectoryEnumerator_addRef) (ICEFSDirectoryEnumerator* iFSDirectoryEnumerator);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSDirectoryEnumerator_release) (ICEFSDirectoryEnumerator* iFSDirectoryEnumerator);


/*! 
 * <b>Summary:</b>
 * Return if the directory has more files/directories.
 *
 * \param[in]	iFSDirectoryEnumerator	Specifies interface pointer
 * \param[out]	hasMoreFlagOut	If more files/directories exist, return true.  Else, false.
 *
 * \return Error status.  If directory has no more files/directories, JGSS_SUCCESS will be returned and hasMoreFlagOut will be false.
 * 
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * Return if the directory has more files/directories.
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSDirectoryEnumerator_hasMoreFile) (ICEFSDirectoryEnumerator* iFSDirectoryEnumerator, bool* hasMoreFlagOut);


/*! 
 * <b>Summary:</b>
 * Get next file/directory by ICEFSPathname object.
 *
 * \param[in]	iFSDirectoryEnumerator	Specifies interface pointer
 * \param[out]	pathnameOut	Found filename.  This pathname contains directory name which was specified at ICEFSFileSysten::createDirectoryEnumerator.  This does not guarantee the pathname is full-path or not.
 *
 * \return Error status.  If directory has no more files/directories, JGSS_ERR_NOTFOUND will be returned.
 * 
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSDirectoryEnumerator_getNext) (ICEFSDirectoryEnumerator* iFSDirectoryEnumerator, struct ICEFSPathname** pathnameOut);

/*!
 * V-table of ICEFSDirectoryEnumerator
 */
struct ICEFSDirectoryEnumerator_vtbl
{
	iCEFSDirectoryEnumerator_queryInterface	_queryInterface;	//!< Query interface.
	iCEFSDirectoryEnumerator_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEFSDirectoryEnumerator_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEFSDirectoryEnumerator_hasMoreFile	_hasMoreFile;
	iCEFSDirectoryEnumerator_getNext	_getNext;
};

/*! @}
 * end of ICEFSDirectoryEnumerator
 */

/*! \defgroup ICEFSFileSystem ICEFSFileSystem
 * @{
 */

/*!
 * ID of ICEFSFileSystem
 */
#define CEComIID_ICEFSFileSystem 0x2744c90e


/*!
 * ICEFileSystem
 *
 *
 *   This interface is for the file/directory operation.
 *   The interface is returned when the library is initialized.
 *
 *   TODO: Current ICEFSFileSystem cannot be singleton object.
 *         Need to make some singleton object and ICEFSFileSystem will be made from that object by QI.
 *
 */
typedef struct ICEFSFileSystem
{
	const struct ICEFSFileSystem_vtbl* _vtbl;
} ICEFSFileSystem;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSFileSystem_queryInterface) (ICEFSFileSystem* iFSFileSystem, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSFileSystem_addRef) (ICEFSFileSystem* iFSFileSystem);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSFileSystem_release) (ICEFSFileSystem* iFSFileSystem);


/*! 
 * <b>Summary:</b>
 * Set library-dependent option
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	option	ICEUnknown pointer which has library-dependent option.
 *
 * \return Error status.
 * 
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSFileSystem_setLibraryDependentOption) (ICEFSFileSystem* iFSFileSystem, struct ICEUnknown* option);


/*! 
 * <b>Summary:</b>
 * Get feature list this library has.
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[out]	featureOut	Bit-or of eCEFileSystemFeatures.
 *
 * \return Error status.
 * 
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSFileSystem_getFeatureList) (ICEFSFileSystem* iFSFileSystem, UINT32* featureOut);


/*! 
 * <b>Summary:</b>
 * Open/Create a file given a descriptor and flags
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[out]	iFileOut	Interface pointer to ICEFSFile.
 * \param[in]	pathname	Specifies path name of file by ICEFSPathname.
 * \param[in]	flags	File open flags.  This is bit-or of eCEFileOpenFlag.
 * \param[in]	mode	File permission by POSIX.  (I think this arg is notessential. Can I remove this arg and always use default value?)
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * You need to make directory the file will be placed before call this interface.
 * If the directory does not exist, this function will be fail.
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSFileSystem_open) (ICEFSFileSystem* iFSFileSystem, struct ICEFSFile* *const iFileOut, struct ICEFSPathname* pathname, int flags, int mode);


/*! 
 * <b>Summary:</b>
 * Open a file given ID and flags
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[out]	iFileOut	Interface pointer to ICEFSFile.
 * \param[in]	id	Specifies id of file.
 * \param[in]	flags	File open flags.  This is bit-or of eCEFileOpenFlag.
 * \param[in]	mode	File permission by POSIX.  (I think this arg is notessential. Can I remove this arg?)
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSFileSystem_openByID) (ICEFSFileSystem* iFSFileSystem, struct ICEFSFile* *const iFileOut, UINT32 id, int flags, int mode);


/*!
 * <b>Summary:</b>
 * Remove specified file.
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	path	Specifies path name of file by ICEFSPathname.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * If specified path is not a regurar file, this function will be fail.
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFileSystem_unlink) (ICEFSFileSystem* iFSFileSystem, struct ICEFSPathname* pathname);


/*!
 * <b>Summary:</b>
 * Make specified directory.
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	path	Specifies directory name of file by ICEFSPathname.
 * \param[in]	mode	File permission by POSIX.  (I think this arg is notessential. Can I remove this arg and always use default value?)
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * You need to make directory the file will be placed before call this interface.
 * If the directory does not exist, this function will be fail.
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFileSystem_mkdir) (ICEFSFileSystem* iFSFileSystem, struct ICEFSPathname* pathname, UINT32 mode);


/*!
 * <b>Summary:</b>
 * Remove specified directory.
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	path	Specifies directory name of file by ICEFSPathname.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * If specified path is not a directory, this function will be fail.
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFileSystem_rmdir) (ICEFSFileSystem* iFSFileSystem, struct ICEFSPathname* pathname);


/*!
 * <b>Summary:</b>
 * Get file status and file size from the specified pathname.
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	path	Specifies path name of file by ICEFSPathname.
 * \param[out]	status	Bit-or of eCEFileStatusFlag
 * \param[out]	fileSize	File size.  This will be returned when the file exists and is a regurar file.
 *
 * \return Error status.  If file does not exist, JGSS_ERR_NOTFOUND is returned.
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFileSystem_getStatus) (ICEFSFileSystem* iFSFileSystem, struct ICEFSPathname* pathname, int* status, UINT32* fileSize);


/*!
 * <b>Summary:</b>
 * Get file status and file size from the specified pathname.
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	id	Specifies id of file.
 * \param[out]	status	Bit-or of eCEFileStatusFlag
 * \param[out]	fileSize	File size.  This will be returned when the file exists and is a regurar file.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFileSystem_getStatusByID) (ICEFSFileSystem* iFSFileSystem, UINT32 id, int* status, UINT32* fileSize);


/*!
 * <b>Summary:</b>
 * Rename the file.
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	fromPath	Specifies path name of file by ICEFSPathname.
 * \param[in]	toPath	Specifies path name of file by ICEFSPathname.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFileSystem_rename) (ICEFSFileSystem* iFSFileSystem, struct ICEFSPathname* fromPath, struct ICEFSPathname* toPath);


/*!
 * <b>Summary:</b>
 * TODO<br>
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	pathname	Specifies path name of file by ICEFSPathname.  This pathname can be relative or absolete path.
 * \param[out]	pathnameOut	Absolete path made from pathnaem.
 *
 * \return Error status.
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * Currently, this function recoginizes only directory.  (Because it uses chdir as implementation).
 * This is not a desirable interface.  I will change this function to be able to get regurar file as arg.
 * (This was because CEFileSystem require to get full pathname only of the directory).
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFileSystem_getDirFullPath) (ICEFSFileSystem* iFSFileSystem, struct ICEFSPathname* pathname, struct ICEFSPathname** pathnameOut);


/*!
 * <b>Summary:</b>
 * Get the list of drive letter of the system.
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[out]	driveList	Bitfield of the existing drive.  Each drive has 1 bit.  A: is 0x00000001, B: is 0x00000002, ...  For example, 0x00000005 means that drive "A:" and "C:" exist.
 *
 * \return Error status. If system does not have notion of drive, JGSS_ERR_NOTFOUND returned.
 *
 * <b>Description:</b>
 * Get the list of drive letter of the system.  You can get the flag if the system has "drive" feature by getFeatureList().
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFileSystem_getDriveList) (ICEFSFileSystem* iFSFileSystem, UINT32* driveList);


/*!
 * <b>Summary:</b>
 * Get the drive information as string like "Fixed Drive (C)"
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	drive	Specifies the drive letter.  1 means "A:", 2 means "B:", ...
 * \param[out]	driveInfo	Description string of the drive like "Fixed Drive (C)"
 *
 * \return Error status
 *
 * <b>Description:</b>
 * Get the information of each drive letter.  You can get the flag if the system has "drive" feature by getFeatureList().
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFileSystem_getDriveInfo) (ICEFSFileSystem* iFSFileSystem, UINT32 drive, struct ICEUString** driveInfo);


/*!
 * <b>Summary:</b>
 * Create the enemurator of the specified directory.
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	pathname	Specifies directory name of file by ICEFSPathname.
 * \param[in]	enumerateOption	Enumeration flags.  This is bit-or of eCEDirectoryEnumerationFlag
 * \param[out]	directoryEnumeratorOut	Interface pointer to ICEFSDirectoryEnumerator.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * Description of ICEFSDirectoryEnumerator.
 *
 */
typedef CEHResult (*iCEFSFileSystem_createDirectoryEnumerator) (ICEFSFileSystem* iFSFileSystem, struct ICEFSPathname* pathname, UINT32 enumerateOption, struct ICEFSDirectoryEnumerator** directoryEnumeratorOut);


/*!
 * <b>Summary:</b>
 * Attach ICEFSFile as a baseline I/O interface.
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	iFile	Baseline File I/O interface by ICEFSFile.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * This interface connect this interface to another File I/O inteface.
 *   +-------------+    +----------------+     +--------------------+
 *   | Appliaction |----o This Interface |-----o Attached Interface |
 *   +-------------+    +----------------+     +--------------------+
 * This will be used especially some structured file format library.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFileSystem_attachFile) (ICEFSFileSystem* iFSFileSystem, struct ICEFSFile* iFile);


/*!
 * <b>Summary:</b>
 * Format file system.
 *
 * \param[in]	iFSFileSystem	Specifies interface pointer
 * \param[in]	maxFileNum	Max number of file the filesystem can have.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * Format file system.  This will be used especially some structured file format library with attach()-ed file.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFileSystem_format) (ICEFSFileSystem* iFSFileSystem, UINT32 maxFileNum);

/*!
 * V-table of ICEFSFileSystem
 */
struct ICEFSFileSystem_vtbl
{
	iCEFSFileSystem_queryInterface	_queryInterface;	//!< Query interface.
	iCEFSFileSystem_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEFSFileSystem_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEFSFileSystem_setLibraryDependentOption	_setLibraryDependentOption;
	iCEFSFileSystem_getFeatureList	_getFeatureList;
	iCEFSFileSystem_open	_open;
	iCEFSFileSystem_openByID	_openByID;
	iCEFSFileSystem_unlink	_unlink;
	iCEFSFileSystem_mkdir	_mkdir;
	iCEFSFileSystem_rmdir	_rmdir;
	iCEFSFileSystem_getStatus	_getStatus;
	iCEFSFileSystem_getStatusByID	_getStatusByID;
	iCEFSFileSystem_rename	_rename;
	iCEFSFileSystem_getDirFullPath	_getDirFullPath;	//!< TODO
	iCEFSFileSystem_getDriveList	_getDriveList;
	iCEFSFileSystem_getDriveInfo	_getDriveInfo;
	iCEFSFileSystem_createDirectoryEnumerator	_createDirectoryEnumerator;
	iCEFSFileSystem_attachFile	_attachFile;
	iCEFSFileSystem_format	_format;
};

/*! @}
 * end of ICEFSFileSystem
 */

/*! \defgroup ICEFSFileSystemFactory ICEFSFileSystemFactory
 * @{
 */

/*!
 * ID of ICEFSFileSystemFactory
 */
#define CEComIID_ICEFSFileSystemFactory 0x53a05751


/*!
 * ICEFSFileSystemFactory
 *
 *
 *
 * This interface creates ICEFSFileSystem instance or return (already-created) instance.
 * 2 or more filesystem object can exist at the same time on some library.  On such library, it creates new object.
 * If the library can create only 1 object, this function returns the (already-created) object.
 *
 */
typedef struct ICEFSFileSystemFactory
{
	const struct ICEFSFileSystemFactory_vtbl* _vtbl;
} ICEFSFileSystemFactory;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iFSFileSystemFactory	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSFileSystemFactory_queryInterface) (ICEFSFileSystemFactory* iFSFileSystemFactory, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSFileSystemFactory_addRef) (ICEFSFileSystemFactory* iFSFileSystemFactory);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSFileSystemFactory_release) (ICEFSFileSystemFactory* iFSFileSystemFactory);

/*!
 * V-table of ICEFSFileSystemFactory
 */
struct ICEFSFileSystemFactory_vtbl
{
	iCEFSFileSystemFactory_queryInterface	_queryInterface;	//!< Query interface.
	iCEFSFileSystemFactory_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEFSFileSystemFactory_release	_release;	//!< Decrease the reference count of the specified interface pointer.
};

/*! @}
 * end of ICEFSFileSystemFactory
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEFSDirectoryEnumeratorRef CEComICEFSDirectoryEnumeratorRef
 * @{
 */

class CEComICEFSDirectoryEnumeratorRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEFSDirectoryEnumeratorRef() : _iFSDirectoryEnumerator(0) {}
	CEComICEFSDirectoryEnumeratorRef(ICEFSDirectoryEnumerator* iOther) : _iFSDirectoryEnumerator(0)
	{
		if (iOther)
		{
			_iFSDirectoryEnumerator = iOther;
			_iFSDirectoryEnumerator->_vtbl->_addRef(_iFSDirectoryEnumerator);
		}
	}

	CEComICEFSDirectoryEnumeratorRef(const CEComICEFSDirectoryEnumeratorRef& other) : _iFSDirectoryEnumerator(0)
	{
		if (other._iFSDirectoryEnumerator)
		{
			_iFSDirectoryEnumerator = other._iFSDirectoryEnumerator;
			_iFSDirectoryEnumerator->_vtbl->_addRef(_iFSDirectoryEnumerator);
		}
	}

	~CEComICEFSDirectoryEnumeratorRef()
	{
		if (_iFSDirectoryEnumerator)
		{
			ICEFSDirectoryEnumerator* tmp = _iFSDirectoryEnumerator;
			_iFSDirectoryEnumerator = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEFSDirectoryEnumerator* iFSDirectoryEnumerator;
				void* _ptr;
			} uIntf;
			uIntf.iFSDirectoryEnumerator = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEFSDirectoryEnumerator, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iFSDirectoryEnumerator)
				{
					ICEFSDirectoryEnumerator* tmp = _iFSDirectoryEnumerator;
					_iFSDirectoryEnumerator = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iFSDirectoryEnumerator = uIntf.iFSDirectoryEnumerator;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEFSDirectoryEnumerator* object() const	{ return _iFSDirectoryEnumerator; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEFSDirectoryEnumerator* iOther)
	{
		if (_iFSDirectoryEnumerator)
		{
			ICEFSDirectoryEnumerator* tmp = _iFSDirectoryEnumerator;
			_iFSDirectoryEnumerator = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iFSDirectoryEnumerator = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEFSDirectoryEnumerator* detach()
	{
		ICEFSDirectoryEnumerator* iIntf = _iFSDirectoryEnumerator;
		_iFSDirectoryEnumerator = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEFSDirectoryEnumerator* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iFSDirectoryEnumerator;
		if (_iFSDirectoryEnumerator)
		{
			_iFSDirectoryEnumerator->_vtbl->_addRef(_iFSDirectoryEnumerator);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEFSDirectoryEnumerator*() const	{ return _iFSDirectoryEnumerator; }
	FORCEINLINE_WITHOUT_DEBUG ICEFSDirectoryEnumerator& operator*() const	{ return *_iFSDirectoryEnumerator; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iFSDirectoryEnumerator == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEFSDirectoryEnumerator* iOther) const	{ return (_iFSDirectoryEnumerator != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEFSDirectoryEnumerator* iOther) const	{ return (_iFSDirectoryEnumerator == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEFSDirectoryEnumeratorRef& operator=(const CEComICEFSDirectoryEnumeratorRef& other)	{ return operator=(other._iFSDirectoryEnumerator); }

	CEComICEFSDirectoryEnumeratorRef& operator=(const ICEFSDirectoryEnumerator* iOther)
	{
		if (_iFSDirectoryEnumerator != iOther)
		{
			if (_iFSDirectoryEnumerator)
			{
				ICEFSDirectoryEnumerator* tmp = _iFSDirectoryEnumerator;
				_iFSDirectoryEnumerator = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iFSDirectoryEnumerator = const_cast<ICEFSDirectoryEnumerator*>(iOther);
			if (_iFSDirectoryEnumerator)
			{
				_iFSDirectoryEnumerator->_vtbl->_addRef(_iFSDirectoryEnumerator);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEFSDirectoryEnumerator** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEFSDirectoryEnumeratorRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iFSDirectoryEnumerator && "has a com object reference. clear first.");
		return & _iFSDirectoryEnumerator; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEFSDirectoryEnumeratorRef& other)
	{
		bool result = false;
		if (_iFSDirectoryEnumerator)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iFSDirectoryEnumerator);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iFSDirectoryEnumerator ? _iFSDirectoryEnumerator->_vtbl->_queryInterface(_iFSDirectoryEnumerator, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iFSDirectoryEnumerator) { _iFSDirectoryEnumerator->_vtbl->_addRef(_iFSDirectoryEnumerator); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iFSDirectoryEnumerator) { _iFSDirectoryEnumerator->_vtbl->_release(_iFSDirectoryEnumerator); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult hasMoreFile(bool* hasMoreFlagOut)	{ return _iFSDirectoryEnumerator ? _iFSDirectoryEnumerator->_vtbl->_hasMoreFile(_iFSDirectoryEnumerator, hasMoreFlagOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getNext(struct ICEFSPathname** pathnameOut)	{ return _iFSDirectoryEnumerator ? _iFSDirectoryEnumerator->_vtbl->_getNext(_iFSDirectoryEnumerator, pathnameOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEFSDirectoryEnumerator* _iFSDirectoryEnumerator;
};

/*! @}
 * end of CEComICEFSDirectoryEnumeratorRef
 */

/*! \defgroup CEComICEFSFileSystemRef CEComICEFSFileSystemRef
 * @{
 */

class CEComICEFSFileSystemRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEFSFileSystemRef() : _iFSFileSystem(0) {}
	CEComICEFSFileSystemRef(ICEFSFileSystem* iOther) : _iFSFileSystem(0)
	{
		if (iOther)
		{
			_iFSFileSystem = iOther;
			_iFSFileSystem->_vtbl->_addRef(_iFSFileSystem);
		}
	}

	CEComICEFSFileSystemRef(const CEComICEFSFileSystemRef& other) : _iFSFileSystem(0)
	{
		if (other._iFSFileSystem)
		{
			_iFSFileSystem = other._iFSFileSystem;
			_iFSFileSystem->_vtbl->_addRef(_iFSFileSystem);
		}
	}

	~CEComICEFSFileSystemRef()
	{
		if (_iFSFileSystem)
		{
			ICEFSFileSystem* tmp = _iFSFileSystem;
			_iFSFileSystem = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEFSFileSystem* iFSFileSystem;
				void* _ptr;
			} uIntf;
			uIntf.iFSFileSystem = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEFSFileSystem, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iFSFileSystem)
				{
					ICEFSFileSystem* tmp = _iFSFileSystem;
					_iFSFileSystem = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iFSFileSystem = uIntf.iFSFileSystem;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEFSFileSystem* object() const	{ return _iFSFileSystem; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEFSFileSystem* iOther)
	{
		if (_iFSFileSystem)
		{
			ICEFSFileSystem* tmp = _iFSFileSystem;
			_iFSFileSystem = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iFSFileSystem = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEFSFileSystem* detach()
	{
		ICEFSFileSystem* iIntf = _iFSFileSystem;
		_iFSFileSystem = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEFSFileSystem* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iFSFileSystem;
		if (_iFSFileSystem)
		{
			_iFSFileSystem->_vtbl->_addRef(_iFSFileSystem);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEFSFileSystem*() const	{ return _iFSFileSystem; }
	FORCEINLINE_WITHOUT_DEBUG ICEFSFileSystem& operator*() const	{ return *_iFSFileSystem; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iFSFileSystem == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEFSFileSystem* iOther) const	{ return (_iFSFileSystem != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEFSFileSystem* iOther) const	{ return (_iFSFileSystem == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEFSFileSystemRef& operator=(const CEComICEFSFileSystemRef& other)	{ return operator=(other._iFSFileSystem); }

	CEComICEFSFileSystemRef& operator=(const ICEFSFileSystem* iOther)
	{
		if (_iFSFileSystem != iOther)
		{
			if (_iFSFileSystem)
			{
				ICEFSFileSystem* tmp = _iFSFileSystem;
				_iFSFileSystem = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iFSFileSystem = const_cast<ICEFSFileSystem*>(iOther);
			if (_iFSFileSystem)
			{
				_iFSFileSystem->_vtbl->_addRef(_iFSFileSystem);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEFSFileSystem** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEFSFileSystemRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iFSFileSystem && "has a com object reference. clear first.");
		return & _iFSFileSystem; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEFSFileSystemRef& other)
	{
		bool result = false;
		if (_iFSFileSystem)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iFSFileSystem);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_queryInterface(_iFSFileSystem, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iFSFileSystem) { _iFSFileSystem->_vtbl->_addRef(_iFSFileSystem); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iFSFileSystem) { _iFSFileSystem->_vtbl->_release(_iFSFileSystem); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setLibraryDependentOption(struct ICEUnknown* option)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_setLibraryDependentOption(_iFSFileSystem, option) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFeatureList(UINT32* featureOut)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_getFeatureList(_iFSFileSystem, featureOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult open(struct ICEFSFile* *const iFileOut, struct ICEFSPathname* pathname, int flags, int mode)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_open(_iFSFileSystem, iFileOut, pathname, flags, mode) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult openByID(struct ICEFSFile* *const iFileOut, UINT32 id, int flags, int mode)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_openByID(_iFSFileSystem, iFileOut, id, flags, mode) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unlink(struct ICEFSPathname* pathname)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_unlink(_iFSFileSystem, pathname) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult mkdir(struct ICEFSPathname* pathname, UINT32 mode)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_mkdir(_iFSFileSystem, pathname, mode) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult rmdir(struct ICEFSPathname* pathname)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_rmdir(_iFSFileSystem, pathname) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStatus(struct ICEFSPathname* pathname, int* status, UINT32* fileSize)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_getStatus(_iFSFileSystem, pathname, status, fileSize) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStatusByID(UINT32 id, int* status, UINT32* fileSize)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_getStatusByID(_iFSFileSystem, id, status, fileSize) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult rename(struct ICEFSPathname* fromPath, struct ICEFSPathname* toPath)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_rename(_iFSFileSystem, fromPath, toPath) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDirFullPath(struct ICEFSPathname* pathname, struct ICEFSPathname** pathnameOut)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_getDirFullPath(_iFSFileSystem, pathname, pathnameOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDriveList(UINT32* driveList)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_getDriveList(_iFSFileSystem, driveList) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDriveInfo(UINT32 drive, struct ICEUString** driveInfo)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_getDriveInfo(_iFSFileSystem, drive, driveInfo) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createDirectoryEnumerator(struct ICEFSPathname* pathname, UINT32 enumerateOption, struct ICEFSDirectoryEnumerator** directoryEnumeratorOut)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_createDirectoryEnumerator(_iFSFileSystem, pathname, enumerateOption, directoryEnumeratorOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult attachFile(struct ICEFSFile* iFile)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_attachFile(_iFSFileSystem, iFile) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult format(UINT32 maxFileNum)	{ return _iFSFileSystem ? _iFSFileSystem->_vtbl->_format(_iFSFileSystem, maxFileNum) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEFSFileSystem* _iFSFileSystem;
};

/*! @}
 * end of CEComICEFSFileSystemRef
 */

/*! \defgroup CEComICEFSFileSystemFactoryRef CEComICEFSFileSystemFactoryRef
 * @{
 */

class CEComICEFSFileSystemFactoryRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEFSFileSystemFactoryRef() : _iFSFileSystemFactory(0) {}
	CEComICEFSFileSystemFactoryRef(ICEFSFileSystemFactory* iOther) : _iFSFileSystemFactory(0)
	{
		if (iOther)
		{
			_iFSFileSystemFactory = iOther;
			_iFSFileSystemFactory->_vtbl->_addRef(_iFSFileSystemFactory);
		}
	}

	CEComICEFSFileSystemFactoryRef(const CEComICEFSFileSystemFactoryRef& other) : _iFSFileSystemFactory(0)
	{
		if (other._iFSFileSystemFactory)
		{
			_iFSFileSystemFactory = other._iFSFileSystemFactory;
			_iFSFileSystemFactory->_vtbl->_addRef(_iFSFileSystemFactory);
		}
	}

	~CEComICEFSFileSystemFactoryRef()
	{
		if (_iFSFileSystemFactory)
		{
			ICEFSFileSystemFactory* tmp = _iFSFileSystemFactory;
			_iFSFileSystemFactory = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEFSFileSystemFactory* iFSFileSystemFactory;
				void* _ptr;
			} uIntf;
			uIntf.iFSFileSystemFactory = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEFSFileSystemFactory, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iFSFileSystemFactory)
				{
					ICEFSFileSystemFactory* tmp = _iFSFileSystemFactory;
					_iFSFileSystemFactory = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iFSFileSystemFactory = uIntf.iFSFileSystemFactory;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEFSFileSystemFactory* object() const	{ return _iFSFileSystemFactory; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEFSFileSystemFactory* iOther)
	{
		if (_iFSFileSystemFactory)
		{
			ICEFSFileSystemFactory* tmp = _iFSFileSystemFactory;
			_iFSFileSystemFactory = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iFSFileSystemFactory = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEFSFileSystemFactory* detach()
	{
		ICEFSFileSystemFactory* iIntf = _iFSFileSystemFactory;
		_iFSFileSystemFactory = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEFSFileSystemFactory* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iFSFileSystemFactory;
		if (_iFSFileSystemFactory)
		{
			_iFSFileSystemFactory->_vtbl->_addRef(_iFSFileSystemFactory);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEFSFileSystemFactory*() const	{ return _iFSFileSystemFactory; }
	FORCEINLINE_WITHOUT_DEBUG ICEFSFileSystemFactory& operator*() const	{ return *_iFSFileSystemFactory; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iFSFileSystemFactory == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEFSFileSystemFactory* iOther) const	{ return (_iFSFileSystemFactory != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEFSFileSystemFactory* iOther) const	{ return (_iFSFileSystemFactory == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEFSFileSystemFactoryRef& operator=(const CEComICEFSFileSystemFactoryRef& other)	{ return operator=(other._iFSFileSystemFactory); }

	CEComICEFSFileSystemFactoryRef& operator=(const ICEFSFileSystemFactory* iOther)
	{
		if (_iFSFileSystemFactory != iOther)
		{
			if (_iFSFileSystemFactory)
			{
				ICEFSFileSystemFactory* tmp = _iFSFileSystemFactory;
				_iFSFileSystemFactory = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iFSFileSystemFactory = const_cast<ICEFSFileSystemFactory*>(iOther);
			if (_iFSFileSystemFactory)
			{
				_iFSFileSystemFactory->_vtbl->_addRef(_iFSFileSystemFactory);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEFSFileSystemFactory** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEFSFileSystemFactoryRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iFSFileSystemFactory && "has a com object reference. clear first.");
		return & _iFSFileSystemFactory; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEFSFileSystemFactoryRef& other)
	{
		bool result = false;
		if (_iFSFileSystemFactory)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iFSFileSystemFactory);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iFSFileSystemFactory ? _iFSFileSystemFactory->_vtbl->_queryInterface(_iFSFileSystemFactory, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iFSFileSystemFactory) { _iFSFileSystemFactory->_vtbl->_addRef(_iFSFileSystemFactory); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iFSFileSystemFactory) { _iFSFileSystemFactory->_vtbl->_release(_iFSFileSystemFactory); } }


private:
	ICEFSFileSystemFactory* _iFSFileSystemFactory;
};

/*! @}
 * end of CEComICEFSFileSystemFactoryRef
 */

#endif // __cplusplus

#endif
