# -*- tab-width: 8 -*- 
########################################################################################################
### idl2h.pl
########################################################################################################

use strict;

my(%IDL) = {};
my(%EMPTY_HASH) = {};

### require the IDL parser.
require 'idlfuncs.pl';

{
    if (!@ARGV)
    {
	print(STDERR "Error: Specify input file name.\n");
	print(STDERR "----------------------------------------------------------\n");
	print(STDERR "Useage: idl2h.pl [options] input.idl\n");
	print(STDERR "Options:\n");
	print(STDERR "  -Iinclude_dir\tSpecify including path of IDL file.\n");
	print(STDERR "  -o output.h\tSpecify output file name.\n");
	exit;
    }

    my($IDL_FILE) = pop(@ARGV);
    if (!(-r $IDL_FILE))
    {
	die("Error: Can not open file '$IDL_FILE' as input\n");
    }

    my($PARAM);
    my(@INCLUDEPATHS);
    my($H_FILE);
    my($RULE_FILE_FLAG) = 0;
    my($RULE_FILE_OUTDIR);
    my($CWDIR);

    push(@INCLUDEPATHS, getPath($IDL_FILE));

    while (@ARGV)
    {
	$PARAM = shift(@ARGV);

	### if -I option
	if ($PARAM =~ /^(-I)([\/\.\w].*)$/)
	{
	    push(@INCLUDEPATHS, $2);
	}

	### if -o option
	elsif ($PARAM =~ /^(-o)$/)
	{
	    $PARAM = shift(@ARGV);
	    if ($PARAM)
	    {
		$H_FILE = $PARAM;
	    }
	    else
	    {
		die("Error: Specify output file name after '-o' option\n");
	    }
	}
	
	### if -MMD option
	elsif ($PARAM =~ /^(-MMD)$/)
	{
	    $RULE_FILE_FLAG = 1;
	}

	### if -cwd option
	elsif ($PARAM =~ /^(-cwd)$/)
	{
	    $PARAM = shift(@ARGV);
	    if ($PARAM)
	    {
		$CWDIR = $PARAM;
	    }
	    else
	    {
		die("Error: Specify current directory after '-cwd' option\n");
	    }
	}
	elsif ($PARAM =~ /^(-WMMD)$/)
	{
	    $RULE_FILE_FLAG = 1;
	    $PARAM = shift(@ARGV);
	    if ($PARAM)
	    {
		$RULE_FILE_OUTDIR = $PARAM;
	    }
	    else
	    {
		die("Error: Specify output directory path after '-WMMD' option\n");
	    }
	}
	elsif ($PARAM =~ /^-/)
	{
	    die("Error: Unknown option '$PARAM'\n");
	}
	else
	{
	    die("Error: Unknown parameter '$PARAM'\n");
	}

    }

    ### read the IDL file. readIdlFile() is in the idlfuncs.pl
    &readIdlFile(\%IDL, $IDL_FILE, @INCLUDEPATHS);

    if (!$H_FILE)
    {
	### set output file name by default
	$H_FILE = &getFileName(&deletePath($IDL_FILE), 'idl', 'h');
    }

    ### write header file
    &writeHeaderFile($IDL_FILE, $H_FILE);

    ### write rule file
    if ($RULE_FILE_FLAG)
    {
	if (!$RULE_FILE_OUTDIR)
	{
	    &writeRuleFileFromIDL(\%IDL, $IDL_FILE, $H_FILE);
	}
	else
	{
	    &writeDepMakeFileFromIDLWin(\%IDL, $IDL_FILE, $H_FILE, $CWDIR, $RULE_FILE_OUTDIR);
	}
    }
}

sub writeHeaderFile
{
    my($IDL_FULLPATH, $H_FILE) = @_;

#    &debugMode(2);
    &debugMode(0);

    ### delete path for databese.
    my($IDL_FILE) = &deletePath($IDL_FULLPATH);

    if ($H_FILE)
    {
	open(FPW, ">$H_FILE") || die("Error: Can not open file $H_FILE as output\n");
    }

    &writeHeader($H_FILE);
    &writeString("\n");

    &writeDefine();

    &writeString("//----------------------------------------------------------------\n");
    &writeString("// Includes\n");
    &writeString("//----------------------------------------------------------------\n");
    &writeString("\n");

    &writeScopedIncludeList(@{$IDL{$IDL_FILE}->{'include'}});
    &writeString("\n");

    &writeString("#ifdef __cplusplus\n");
    &writeString("extern \"C\" {\n");
    &writeString("#endif\n");
    &writeString("\n");
    
    &writeString("//----------------------------------------------------------------\n");
    &writeString("// Prototypes\n");
    &writeString("//----------------------------------------------------------------\n");
    &writeString("\n");
    &writePrototypeList(@{$IDL{$IDL_FILE}->{'prototype'}});
    &writeString("\n");

    &writeString("//----------------------------------------------------------------\n");
    &writeString("// Enumerators\n");
    &writeString("//----------------------------------------------------------------\n");
    &writeString("\n");
    &writeStructEnumeratorList('enum', $IDL{$IDL_FILE});

    &writeString("//----------------------------------------------------------------\n");
    &writeString("// Structures\n");
    &writeString("//----------------------------------------------------------------\n");
    &writeString("\n");
    &writeStructEnumeratorList('struct', $IDL{$IDL_FILE});

    &writeString("//----------------------------------------------------------------\n");
    &writeString("// Interfaces\n");
    &writeString("//----------------------------------------------------------------\n");
    &writeString("\n");
    &writeInterfaceList(@{$IDL{$IDL_FILE}->{'interface'}});

    &writeString("#ifdef __cplusplus\n");
    &writeString("} // end of extern \"C\"\n");
    &writeString("#endif\n");
    &writeString("\n");

    # when @pragma(external_ref_def) is specified, no interface wrapper is generated.
    # instead, the specified header file is included, assuming the header file has
    # hand-coded definition of the interface wrapper such as CEComICESymbolRef.
    &writeString("//----------------------------------------------------------------\n");
    &writeString("// Interface Wrappers\n");
    &writeString("//----------------------------------------------------------------\n");
    &writeString("\n");
    my($EXT_REF_DEF_PRAGMA) = &isIncludedInPragmaList('external_ref_def', @{$IDL{$IDL_FILE}->{'pragma'}});
    if($EXT_REF_DEF_PRAGMA)
    {
	my($EXT_HEADER_FILENAME);
	$EXT_REF_DEF_PRAGMA =~ /external_ref_def\((\".*\")\)/;
	$EXT_HEADER_FILENAME = $1;

	# special code for hand-made interface wrapper.
	&writeString("#include $EXT_HEADER_FILENAME\n");
	&writeString("\n");
    }
    else
    {
	&writeInterfaceWrapper(@{$IDL{$IDL_FILE}->{'interface'}});
    }

    &writeString("#endif\n");

    close(FPW);

    &debugMode(0);
}

sub writeInterfaceList
{
    my(@INTERFACES) = @_;
    my($INTERFACE_REF);
    foreach $INTERFACE_REF (@INTERFACES)
    {
	my($NAME) = $INTERFACE_REF->{'name'};
	my(%INTERFACE) = %$INTERFACE_REF;

	&writeString("/*! \\defgroup $NAME $NAME\n");
	&writeString(" * @\{\n");
	&writeString(" */\n");
	&writeString("\n");

	&writeIID($NAME);
	&writeInterface($NAME, $INTERFACE_REF->{'comment'});
	&writeFunctionListWithInheritance(\%IDL, 'C_FUNCTION_HEADER', \%EMPTY_HASH, $NAME, 0, %INTERFACE);
	&writeVtable($NAME, %INTERFACE);

        &writeString("/*! @\}\n");
        &writeString(" * end of $NAME\n");
        &writeString(" */\n");
	&writeString("\n");
    }
}

sub writeIID
{
    my($NAME) = @_;
    my($MD5) = &getMD5Last32($NAME);

    &writeString("/*!\n");
    &writeString(" * ID of $NAME\n");
    &writeString(" */\n");
    my($DEFINE) = &getIIDName($NAME);
    &writeString('#define ' . "$DEFINE $MD5\n");
    &writeString("\n");
}

sub writeInterface
{
    my($NAME, $COMMENT) = @_;

    if ($COMMENT)
    {
	&writeString($COMMENT);
    }
    else
    {
	&writeString("/*!\n");
	&writeString(" * $NAME\n");
	&writeString(" */\n");
    }
    &writeString("typedef struct $NAME\n");
    &writeString("{\n");
    my($VTBL) = &getVtableName($NAME);
    &writeString("\tconst struct $VTBL* _vtbl;\n");
    &writeString("} $NAME;\n");
    &writeString("\n");
}

sub writeVtable
{
    my($TOPLEVELNAME, %INTERFACE) = @_;
    if (%INTERFACE)
    {
	&writeString("/*!\n");
	&writeString(" * V-table of $TOPLEVELNAME\n");
	&writeString(" */\n");
	my($VTBL) = &getVtableName($TOPLEVELNAME);
	&writeString("struct $VTBL\n");
	&writeString("{\n");
	&writeFunctionListWithInheritance(\%IDL, 'C_VTABLE_HEADER', \%EMPTY_HASH, $TOPLEVELNAME, 0, %INTERFACE);
	&writeString("};\n");
	&writeString("\n");
    }
}

sub writeInterfaceWrapper
{
    my(@INTERFACES) = @_;

    &writeString("#ifdef __cplusplus\n");
    &writeString("\n");

    my($INTERFACE_REF);
    foreach $INTERFACE_REF (@INTERFACES)
    {
	my($NAME) = &getInterfaceWrapperName($INTERFACE_REF->{'name'});
	my(%INTERFACE) = %$INTERFACE_REF;

	&writeString("/*! \\defgroup $NAME $NAME\n");
	&writeString(" * @\{\n");
	&writeString(" */\n");
	&writeString("\n");
	
	&writeString("class $NAME\n");
	&writeString("{\n");
	&writeString("public:\n");
	&writeConstructor($INTERFACE_REF->{'name'}, @{$INTERFACE_REF->{'function'}});
	&writeSpecialMethod($INTERFACE_REF->{'name'});
	&writeOperator($INTERFACE_REF->{'name'});
	&writeInterfaceWrapperFunction(%INTERFACE);

	&writeString("private:\n");
	my($VALUE) = &getPrivateInterfaceValueName($INTERFACE_REF->{'name'});
	&writeString("\t$INTERFACE_REF->{'name'}* $VALUE;\n");
	&writeString("};\n");
	&writeString("\n");

        &writeString("/*! @\}\n");
        &writeString(" * end of $NAME\n");
        &writeString(" */\n");
	&writeString("\n");
    }

    &writeString("#endif // __cplusplus\n");
    &writeString("\n");
}

sub writeConstructor
{
    my($INTERFACE, @FUNCTIONS) = @_;
    my($CLASS) = &getInterfaceWrapperName($INTERFACE);
    my($VALUE) = &getPrivateInterfaceValueName($INTERFACE);

    &writeString("\t//----------------------------------------------------------------\n");
    &writeString("\t// constructor / destructor.\n");
    &writeString("\t//----------------------------------------------------------------\n");

	# write dbgToString() exp in constructor for debug build.
	my($FUNCTION_REF);
	my($haveDbgToString) = 0;

	foreach $FUNCTION_REF (@FUNCTIONS)
    {
		my(%FUNCTION) = %$FUNCTION_REF;
		
		if ( $FUNCTION{'name'} eq "dbgToString" )
		{
			$haveDbgToString = 1;
		}
    }
	
	if ( $haveDbgToString == 1 )
	{
		&writeString("\t$CLASS() : $VALUE(0)\n\t{\n#if defined(_DEBUG) || !defined(NDEBUG)\n\t\tdbgToString();\n#endif //_DEBUG || !NDEBUG\n\t}\n");
	}
    else
	{
		&writeString("\t$CLASS() : $VALUE(0) {}\n");
	}

    &writeString("\t$CLASS($INTERFACE* iOther) : $VALUE(0)\n");
    &writeString("\t{\n");
    &writeString("\t\tif (iOther)\n");
    &writeString("\t\t{\n");
    &writeString("\t\t\t$VALUE = iOther;\n");
    &writeString("\t\t\t$VALUE->_vtbl->_addRef($VALUE);\n");
    &writeString("\t\t}\n");
    &writeString("\t}\n");
    &writeString("\n");

    &writeString("\t$CLASS(const $CLASS& other) : $VALUE(0)\n");
    &writeString("\t{\n");
    &writeString("\t\tif (other.$VALUE)\n");
    &writeString("\t\t{\n");
    &writeString("\t\t\t$VALUE = other.$VALUE;\n");
    &writeString("\t\t\t$VALUE->_vtbl->_addRef($VALUE);\n");
    &writeString("\t\t}\n");
    &writeString("\t}\n");
    &writeString("\n");

    my($TMP_VALUE) = 'tmp';
    &writeString("\t~$CLASS()\n");
    &writeString("\t{\n");
    &writeString("\t\tif ($VALUE)\n");
    &writeString("\t\t{\n");
    &writeString("\t\t\t$INTERFACE* $TMP_VALUE = $VALUE;\n");
    &writeString("\t\t\t$VALUE = 0;\n");
    &writeString("\t\t\t$TMP_VALUE->_vtbl->_release($TMP_VALUE);\n");
    &writeString("\t\t}\n");
    &writeString("\t}\n");
    &writeString("\n");
}

sub writeSpecialMethod
{
    my($INTERFACE) = @_;
    my($PRIVATE_VALUE) = &getPrivateInterfaceValueName($INTERFACE);
    my($VALUE) = &getInterfaceValueName($INTERFACE);
    my($IID) = getIIDName($INTERFACE);

    my($FORCE_INLINE_GCC) = &getGccInlineOption();

    &writeString("\t//----------------------------------------------------------------\n");
    &writeString("\t// initialize instance which uses queryInterface().\n");
    &writeString("\t//   Warning: this method does not increment the reference count.\n");
    &writeString("\t//----------------------------------------------------------------\n");
    &writeString("\tCEHResult initByQueryInterface(void* iIn)\n");
    &writeString("\t{\n");
    &writeString("\t\tCEHResult hr = CE_SILK_ERR_OPERATION_FAILED;\n");
    &writeString("\t\tif (iIn)\n");
    &writeString("\t\t{\n");
    &writeString("\t\t\t// explicit type-punning to notify aliasing to compiler\n");
    &writeString("\t\t\tunion\n");
    &writeString("\t\t\t{\n");
    &writeString("\t\t\t\t$INTERFACE* $VALUE;\n");
    &writeString("\t\t\t\tvoid* _ptr;\n");
    &writeString("\t\t\t} uIntf;\n");
    &writeString("\t\t\tuIntf.$VALUE = 0;\n");
    &writeString("\t\t\thr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), $IID, &uIntf._ptr);\n");
    &writeString("\t\t\tif (CESucceeded(hr))\n");
    &writeString("\t\t\t{\n");
    &writeString("\t\t\t\tif ($PRIVATE_VALUE)\n");
    &writeString("\t\t\t\t{\n");
    my($TMP_VALUE) = 'tmp';
    &writeString("\t\t\t\t\t$INTERFACE* $TMP_VALUE = $PRIVATE_VALUE;\n");
    &writeString("\t\t\t\t\t$PRIVATE_VALUE = 0;\n");
    &writeString("\t\t\t\t\t$TMP_VALUE->_vtbl->_release($TMP_VALUE);\n");
    &writeString("\t\t\t\t}\n");
    &writeString("\t\t\t\t$PRIVATE_VALUE = uIntf.$VALUE;\n");
    &writeString("\t\t\t}\n");
    &writeString("\t\t}\n");
    &writeString("\t\telse\n");
    &writeString("\t\t{\n");
    &writeString("\t\t\thr = CE_SILK_ERR_BADARGS;\n");
    &writeString("\t\t}\n");
    &writeString("\t\treturn hr;\n");
    &writeString("\t}\n");
    &writeString("\n");

    &writeString("\t//----------------------------------------------------------------\n");
    &writeString("\t// get the interface.\n");
    &writeString("\t//   Warning: this method does not change the reference count.\n");
    &writeString("\t//----------------------------------------------------------------\n");
    &writeString("\t$FORCE_INLINE_GCC $INTERFACE* object() const\t{ return $PRIVATE_VALUE; }\n");
    &writeString("\n");

    &writeString("\t//----------------------------------------------------------------\n");
    &writeString("\t// attach this smart pointer to an existing interface.\n");
    &writeString("\t//   Warning: this method does not change the reference count.\n");
    &writeString("\t//----------------------------------------------------------------\n");
	&writeString("\tvoid attach($INTERFACE* iOther)\n");
	&writeString("\t{\n");
	&writeString("\t\tif ($PRIVATE_VALUE)\n");
    &writeString("\t\t{\n");
    my($TMP_VALUE) = 'tmp';
    &writeString("\t\t\t$INTERFACE* $TMP_VALUE = $PRIVATE_VALUE;\n");
    &writeString("\t\t\t$PRIVATE_VALUE = 0;\n");
    &writeString("\t\t\t$TMP_VALUE->_vtbl->_release($TMP_VALUE);\n");
    &writeString("\t\t}\n");
	&writeString("\t\t$PRIVATE_VALUE = iOther;\n");
	&writeString("\t}\n");
    &writeString("\n");
	
    &writeString("\t//----------------------------------------------------------------\n");
    &writeString("\t// detach the interface pointer from this.\n");
    &writeString("\t//   Warning: this method does not change the reference count.\n");
    &writeString("\t//----------------------------------------------------------------\n");
	&writeString("\t$INTERFACE* detach()\n");
	&writeString("\t{\n");
	&writeString("\t\t$INTERFACE* iIntf = $PRIVATE_VALUE;\n");
	&writeString("\t\t$PRIVATE_VALUE = 0;\n");
	&writeString("\t\treturn iIntf;\n");
	&writeString("\t}\n");
    &writeString("\n");
	
    &writeString("\t//----------------------------------------------------------------\n");
    &writeString("\t// copy this to an existing interface pointer holder.\n");
    &writeString("\t//----------------------------------------------------------------\n");
	&writeString("\tCEHResult copyTo($INTERFACE* *const iIntfOut)\n");
	&writeString("\t{\n");
	&writeString("\t\tif (!iIntfOut)\n");
	&writeString("\t\t{\n");
	&writeString("\t\t\treturn CE_SILK_ERR_BADARGS;\n");
	&writeString("\t\t}\n");
	&writeString("\t\t*iIntfOut = $PRIVATE_VALUE;\n");
	&writeString("\t\tif ($PRIVATE_VALUE)\n");
	&writeString("\t\t{\n");
	&writeString("\t\t\t$PRIVATE_VALUE->_vtbl->_addRef($PRIVATE_VALUE);\n");
	&writeString("\t\t}\n");
	&writeString("\t\treturn CE_S_OK;\n");
	&writeString("\t}\n");
    &writeString("\n");
	
}

sub writeOperator
{
    ## put it after return type
    my($FORCE_INLINE_GCC) = &getGccInlineOption();

    my($INTERFACE) = @_;
    my($CLASS) = &getInterfaceWrapperName($INTERFACE);
    my($VALUE) = &getPrivateInterfaceValueName($INTERFACE);

    &writeString("\t//----------------------------------------------------------------\n");
    &writeString("\t// operator overwrite.\n");
    &writeString("\t//----------------------------------------------------------------\n");
    &writeString("\t$FORCE_INLINE_GCC operator $INTERFACE*() const\t{ return $VALUE; }\n");
    &writeString("\t$FORCE_INLINE_GCC $INTERFACE& operator*() const\t{ return *$VALUE; }\n");
    &writeString("\t$FORCE_INLINE_GCC bool operator!() const\t{ return ($VALUE == 0); }\n");
    &writeString("\t$FORCE_INLINE_GCC bool operator!=($INTERFACE* iOther) const\t{ return ($VALUE != iOther); }\n");
    &writeString("\t$FORCE_INLINE_GCC bool operator==($INTERFACE* iOther) const\t{ return ($VALUE == iOther); }\n");
    &writeString("\t$FORCE_INLINE_GCC $CLASS& operator=(const $CLASS& other)\t{ return operator=(other.$VALUE); }\n");
    &writeString("\n");

    &writeString("\t$CLASS& operator=(const $INTERFACE* iOther)\n");
    &writeString("\t{\n");
    &writeString("\t\tif ($VALUE != iOther)\n");
    &writeString("\t\t{\n");
    &writeString("\t\t\tif ($VALUE)\n");
    &writeString("\t\t\t{\n");
    my($TMP_VALUE) = 'tmp';
    &writeString("\t\t\t\t$INTERFACE* $TMP_VALUE = $VALUE;\n");
    &writeString("\t\t\t\t$VALUE = 0;\n");
    &writeString("\t\t\t\t$TMP_VALUE->_vtbl->_release($TMP_VALUE);\n");
    &writeString("\t\t\t}\n");
    &writeString("\n");
    &writeString("\t\t\t$VALUE = const_cast<$INTERFACE*>(iOther);\n");
    &writeString("\t\t\tif ($VALUE)\n");
    &writeString("\t\t\t{\n");
    &writeString("\t\t\t\t$VALUE->_vtbl->_addRef($VALUE);\n");
    &writeString("\t\t\t}\n");
    &writeString("\t\t}\n");
    &writeString("\t\treturn *this;\n");
    &writeString("\t}\n");
    &writeString("\n");


    ## operator &
    &writeString("\t$FORCE_INLINE_GCC $INTERFACE** operator&() \n");
    &writeString("\t{\n");
    &writeString("\t\t// operator& must be used for [out] pointer.\n");
    &writeString("\t\t// But, If this object has a reference to the com object, \n");
    &writeString("\t\t// the reference will be leaked. So, clear the reference first:\n");
    &writeString("\t\t// \n");
    &writeString("\t\t// $CLASS foo;\n");
    &writeString("\t\t// clazz.createInstance(&foo);  //OK\n");
    &writeString("\t\t// clazz.createInstance(&foo);  //NG (assert).\n");
    &writeString("\t\t// foo = 0;\n");
    &writeString("\t\t// clazz.createInstance(&foo);  //OK\n");
    &writeString("\t\t//CEASSERT(!$VALUE && \"has a com object reference. clear first.\");\n");
    &writeString("\t\treturn & $VALUE; \n");
    &writeString("\t}\n");
    
    ### new & delete
    &writeString("protected:\n");
    &writeString("\tvoid* operator new(size_t) throw()\t{ return 0; }\n");
    &writeString("\tvoid operator delete(void*)\t{}\n");
    &writeString("\tvoid* operator new[](size_t) throw()\t{ return 0; }\n");
    &writeString("#if (__GNUC__ == 2)\n");
    &writeString("public:\n");
    &writeString("#endif\n");
    &writeString("\tvoid operator delete[](void*)\t{}\n");
    &writeString("\n");

    ###

    &writeString("public:\n");
#    &writeString("\tbool compareICEUnknown($CLASS& other) { return $VALUE == other.$VALUE; }\n");
    &writeString("\tbool compareICEUnknown($CLASS& other)\n");
    &writeString("\t{\n");
    &writeString("\t\tbool result = false;\n");
    &writeString("\t\tif ($VALUE)\n");
    &writeString("\t\t{\n");
    &writeString("\t\t\tCEComICEUnknownRef unknown;\n");
    &writeString("\t\t\tCEHResult hr = unknown.initByQueryInterface($VALUE);\n");
    &writeString("\t\t\tif(CESucceeded(hr))\n");
    &writeString("\t\t\t{\n");
    &writeString("\t\t\t\tCEComICEUnknownRef otherUnknown;\n");
    &writeString("\t\t\t\thr = otherUnknown.initByQueryInterface(other);\n");
    &writeString("\t\t\t\tif (CESucceeded(hr))\n");
    &writeString("\t\t\t\t{\n");
    &writeString("\t\t\t\t\tresult = (unknown == otherUnknown);\n");
    &writeString("\t\t\t\t}\n");
    &writeString("\t\t\t}\n");
    &writeString("\t\t}\n");
    &writeString("\t\telse\n");
    &writeString("\t\t{\n");
    &writeString("\t\t\tresult = (other == NULL);\n");    
    &writeString("\t\t}\n");
    &writeString("\t\treturn result;\n");
    &writeString("\t}\n");
}

sub writeInterfaceWrapperFunction
{
    my(%INTERFACE) = @_;
    &writeString("public:\n");
    &writeString("\t//----------------------------------------------------------------\n");
    &writeString("\t// interface methods.\n");
    &writeString("\t//----------------------------------------------------------------\n");
    
    &writeFunctionListWithInheritance(\%IDL, 'CREF_FUNCTION_IMPL', \%EMPTY_HASH, $INTERFACE{'name'}, 0, %INTERFACE);
    &writeString("\n");
}

# check if a specified string is included in the pragma list.
# \param  $STRING       string such as 'external_ref_def'. this string could match the 
#                       functor part of a pragma.
# \param  $LIST_REF     list  e.g. ('external_ref_def("ICESymbol_RefDef_only.h")', no_warning)
# \return  matched pragma's whole body. e.g.  'external_ref_def("ICESymbol_RefDef_only.h")'
sub isIncludedInPragmaList
{
    my($STRING, @LIST) = @_;
    my($RET) = 0;
    my($ELM);

    foreach $ELM (@LIST)
    {
	if($ELM =~ /$STRING/)
	{
	    $RET = $ELM;
	}
    }

    # return
    $RET;
}
