///////////////////////////////////////////////////////////////////////////////
// Copyright 2011 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICEUIFont_h
#define ICEUIFont_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"
#include "CEUITypes.h"
#include "ICEUString.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------

struct ICESymbol;
struct ICEPFFace;
struct ICEPFGlyphLoader;
struct ICEI18nCharMapper;
struct ICEI18nTextBreakTester;
struct CEI18nBiDiCCodeStack;
struct CEPFFaceInfo;
struct ICEFSFileSystem;
struct ICEFSPathname;
struct ICEUICompositeFont;
struct CEPFFontStyle;
struct ICEUIPlatformFont;
struct ICEDXRegion;
struct CEPFGlyphAdvance;
struct CEPFGlyphBBox;

//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

typedef enum
{
	CEUIFontValueType_UnderlineThickness,
	CEUIFontValueType_PixelSize,
	CEUIFontValueType_FontHeight,
	CEUIFontValueType_Ascent,
	CEUIFontValueType_Descent,
	CEUIFontValueType_WhiteSpaceWidth,
	CEUIFontValueType_xHeight,
	CEUIFontValueType_PointSize,
	CEUIFontValueType_UnderlinePosition
} CEUIFontValueType;

typedef enum
{
	eCEUIGlyphLE_Unknown,
	eCEUIGlyphLE_Simple,
	eCEUIGlyphLE_WhiteSpace,
	eCEUIGlyphLE_CTL
} eCEUIGlyphLE;

typedef enum
{
	eCEUIFontVariant_normal = 0,
	eCEUIFontVariant_smallCaps
} eCEUIFontVariant;

typedef enum
{
	// platform font rendering options bits.
	eCEUIFontServiceOptionsBit_NONE					= 0,
	eCEUIFontServiceOptionsBit_PF_ALWAYS_AAOUTLINE	= (1<<0),
	eCEUIFontServiceOptionsBit_PF_MONOCHROME		= (1<<1),
	eCEUIFontServiceOptionsBit_PF_USE_FONTINFOCACHE  = (1<<2),
	eCEUIFontServiceOptionsBit_PF_MASK				= 0x0000ffff,

	// composite font options bits.
	eCEUIFontServiceOptionsBit_CF_CTLSUPPORT		= (1<<16), 
	eCEUIFontServiceOptionsBit_CF_MASK				= 0xffff0000,
} eCEUIFontServiceOptionsBit;

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

typedef struct CEUITextExtents
{
    float bearingX;
    float bearingY;
    float width;
    float height;
    float advanceX;
    float advanceY;
} CEUITextExtents;

typedef struct CEUIVariantOption
{
	eCEUIFontVariant variant;
} CEUIVariantOption;

typedef struct CEUIWhiteSpaceOption
{
	CEUTextMeasurement whiteSpaceWidth;
	CEUTextMeasurement letterSpacing;
	CEUTextMeasurement wordSpacing; 
} CEUIWhiteSpaceOption;

//! specify range (in character offset) of string to draw.
typedef struct CEUIFontStrRangeOption
{
	UINT32 startOffset16;
	UINT32 numOfChars16;
} CEUIFontStrRangeOption;


/*!
* CEUIFontHiLiteOption
*
* <pre>
*
* // ex...
* const UTF16CHAR* pC16s = {'a', 'b', 'c', 'd', 'e', 'f'. 'g'};
* UINT32 numOfC16s = 7;
*
* // HiLite "ab" and "cde"
* CEUIFontStrRangeOption ranges[] = {{0, 3}, {2, 3}};	
* CERGBAColor bg, fg;  // initialize colors.
*
* CEUIFontHiLiteOption hiliteOption;
* JG_FMEMSET(&hiliteOption, 0, sizeof(&hiliteOption);
* hiliteOption.numOfPortions = 2;
* hiliteOption.pRanges = ranges;
*
* CEUIFontOption fontOption;
* JG_FMEMSET(&fontOption, 0, sizeof(fontOption);
* fontOption.pHilitOption = &hiliteOption;
* gc.drawUTF16CHARArray(posToDraw, pC16s, numOfC16s, &fontOption);
*
* </pre>
*
* \param numOfPortions[in]: number of HiLited text portions.
* \param pStartOffset16s[in]: pointer to the array of start character index of each portions.
* \param pNumOfChars16s[in]: pointer to the array of number of characters of each portions.
* \param rectColor[in]: HiLite rectangle color to fill HiLite portion.
* \param textColor[in]: text color to draw HiLite portion.
* \param strokeColor[in]: text stroke color to draw HiLite portion.
*/
typedef struct CEUIFontHiLiteOption
{
	UINT32				numOfRanges;
	CEUIFontStrRangeOption* pRanges;
	CERGBAColor		rectColor;
	CERGBAColor		textColor;
	CERGBAColor		strokeColor;
} CEUIFontHiLiteOption;


/*!
* direction option
*/

	/*! 
	* eCEI18nBiDiDirection_Unknown: the text is processed by BiDi processor and detected direction is returned.
	* eCEI18nBiDiDirection_Mixed: the text is processed by BiDi processor.
	* eCEI18nBiDiDirection_LTR, eCEI18nBiDiDirection_RTL: use specified direction to process text. In this case, paragraphLevel must have same direction value with thie member.
	*/

	/*!
	* specify buffer for the CEI18nBiDiCCodeStack pointer. returned value must be freed by using ICEI18nBidiUtil::deleteCCodeStack().
	*/
typedef struct CEUIDirectionOption
{
	eCEI18nBiDiLevel		paragraphLevel;

	eCEI18nBiDiDirection	direction;
	CEI18nBiDiCCodeStack*	optLeadCCodeStackIn;

	CEI18nBiDiCCodeStack**	optTrailCCodeStackOut;
} CEUIDirectionOption;


/*!
* tab stop option.
*/

	/*!
	* tab stop width
	*/
typedef struct CEUITabOption
{
	INT32 tabStopWidth;
	INT32 leftX;
} CEUITabOption;

typedef struct CEUIBiDiDirectionInfo
{
	UINT8 hasLtrChar : 1;
	UINT8 hasRtlChar : 1;
	UINT8 hasWeakOrNeutralChar : 1;
} CEUIBiDiDirectionInfo;

typedef struct CEUIGlyphStrokeOption
{
	CERGBAColor strokeColor; 
	UINT8 fill : 1;
	UINT8 stroke : 1;
} CEUIGlyphStrokeOption;

typedef struct CEUIFontPFArrayOption
{
	UINT32		nPFs;				//! [in] size of pPFArray. [out] number of items in pPFArray.
	ICEUIPlatformFont**	pPFArray;	//! [out] buffer to return.
} CEUIFontPFArrayOption;


/*!
* CEUIFontOption
*/
typedef struct CEUIFontOption
{
	ICEI18nCharMapper*		iCharMapper;   /*! ICEI18nCharMapper interface pointer */
	CEUIFontHiLiteOption*	pHiLiteOption; /*! pointer to the CEUIFontHiLiteOption. specify zero to normal text rendering. */
	ICEI18nTextBreakTester*	iTextBreakTester; /*! ICEI18nTextBreakIterator interface pointer */
	CEUIDirectionOption*	directionOption; /*! pointer to the CEUIDirectionOption. specify zero to render text without consider direction. */
	CEUITabOption*			tabOption; /*! pointer to the CEUITabOption. specify zero to render text with default tab to space conversion */
	CEUIBiDiDirectionInfo*	bidiDirectionProbe; /*! pointer to the CEUIBiDiDirectionInfo */
	CEUIVariantOption*		variantOption; /*! pointer to the CEUIVariantOption*/
	CEUIWhiteSpaceOption*   whiteSpaceOption; /*! pointer to the CEUIWhiteSpaceOption*/
	CEUIGlyphStrokeOption*  strokeOption; /*! pointer to the CEUIGlyphStrokeOption */
	CEUIFontStrRangeOption*	strRangeOption;  /*! pointer to the CUIFontRangeOption */
	CEUIFontPFArrayOption*	pfArrayOption;	/*! pointer to the CEUIFontPFOption. (actually required pfs to process input string). */
	CEUIFontPFArrayOption*	normalPfArrayOption;	/*! pointer to the CEUIFontPFOption. (required normal variant pfs to process input string). */
} CEUIFontOption;

typedef struct CEUIFontServiceInitArg
{
	ICEFSFileSystem* fileSystem;
	ICEUString* istrPlatformFontLibName;
	ICEFSPathname* fontDir;
	ICEFSPathname* fontConfig;
	ICEFSPathname* fontInfoCacheDir;

	UINT32 eCEUIFontServiceOptionsBits;

	INT32 gEnvLogPixelsX;	//! graphics environment logical pixels
	INT32 gEnvLogPixelsY;   //! graphics environment logical pixels
} CEUIFontServiceInitArg;

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEUIFontFamily ICEUIFontFamily
 * @{
 */

/*!
 * ID of ICEUIFontFamily
 */
#define CEComIID_ICEUIFontFamily 0xc2ba681d


/*!
* ICEUIFontFamily
*/
typedef struct ICEUIFontFamily
{
	const struct ICEUIFontFamily_vtbl* _vtbl;
} ICEUIFontFamily;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUIFontFamily	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUIFontFamily_queryInterface) (ICEUIFontFamily* iUIFontFamily, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUIFontFamily_addRef) (ICEUIFontFamily* iUIFontFamily);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUIFontFamily_release) (ICEUIFontFamily* iUIFontFamily);

	//! returns font-family-list-name. ex.. "MS GOCHIC,SANS-SERIF"

typedef CEHResult (*iCEUIFontFamily_getName) (ICEUIFontFamily* iUIFontFamily, struct ICEUString* *const listNameOut);


	/*! 
	* returns array of font-family names. 
	* if familiesOut is zero, returns number of family names. 
	*
	* ex...
	* UINT32 num = 0;
	*
	* // get number of items.
	* fontFamilyRef.getFamilies(0, &num);
	*
	* ICEUIFontFamily** families = new ICEUIFontFamily*[num];
	*
	* {
	*	// get all familiy names. returned all items are addRef(ed).
	*	fontFamilyRef.getFamilies(families, &num);
	*
	*	for ( UINT32 i=0; i<num; i++ )
	*	{
	*		CEComICEUIFontFamilyRef familyRef;
	*		familyRef.atach(families[i]);  
	*
	*		familyRef.getListName(...); // do something.
	*
	*		familyRef = 0;				// decrease refCount.
	*	}
	*
	*	// and delete allocated buffer.
	*	delete [] families;
	* }
	*/

typedef CEHResult (*iCEUIFontFamily_getFamilies) (ICEUIFontFamily* iUIFontFamily, struct ICEUIFontFamily* *const familiesOut, UINT32 *const numFamiliesRef);

typedef CEHResult (*iCEUIFontFamily_getFamilyAt) (ICEUIFontFamily* iUIFontFamily, struct ICEUIFontFamily* *const familiesOut, UINT32 index);

typedef CEHResult (*iCEUIFontFamily_dbgToString) (ICEUIFontFamily* iUIFontFamily, const UTF16CHAR* *const pStr16Out);

/*!
 * V-table of ICEUIFontFamily
 */
struct ICEUIFontFamily_vtbl
{
	iCEUIFontFamily_queryInterface	_queryInterface;	//!< Query interface.
	iCEUIFontFamily_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUIFontFamily_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUIFontFamily_getName	_getName;
	iCEUIFontFamily_getFamilies	_getFamilies;
	iCEUIFontFamily_getFamilyAt	_getFamilyAt;
	iCEUIFontFamily_dbgToString	_dbgToString;
};

/*! @}
 * end of ICEUIFontFamily
 */

/*! \defgroup ICEUIFontGlyphBuffer ICEUIFontGlyphBuffer
 * @{
 */

/*!
 * ID of ICEUIFontGlyphBuffer
 */
#define CEComIID_ICEUIFontGlyphBuffer 0x99a7cce4


/*!
* ICEUIFontGlyphBuffer
*/
typedef struct ICEUIFontGlyphBuffer
{
	const struct ICEUIFontGlyphBuffer_vtbl* _vtbl;
} ICEUIFontGlyphBuffer;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUIFontGlyphBuffer	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUIFontGlyphBuffer_queryInterface) (ICEUIFontGlyphBuffer* iUIFontGlyphBuffer, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUIFontGlyphBuffer_addRef) (ICEUIFontGlyphBuffer* iUIFontGlyphBuffer);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUIFontGlyphBuffer_release) (ICEUIFontGlyphBuffer* iUIFontGlyphBuffer);

typedef CEHResult (*iCEUIFontGlyphBuffer_initG16F) (ICEUIFontGlyphBuffer* iUIFontGlyphBuffer, struct ICEUIPlatformFont* iPf, const UINT16* gIndexes, const CEDimF* advances, INT_PTR nGlyphs);

typedef CEHResult (*iCEUIFontGlyphBuffer_getFirstPaltformFont) (ICEUIFontGlyphBuffer* iUIFontGlyphBuffer, struct ICEUIPlatformFont* *const iPfOut);

typedef CEHResult (*iCEUIFontGlyphBuffer_elementAt) (ICEUIFontGlyphBuffer* iUIFontGlyphBuffer, INT_PTR index, struct ICEUIPlatformFont* *const iPfOut, UINT32 *const gIndexOut, CEDimF *const advanceOut);

typedef CEHResult (*iCEUIFontGlyphBuffer_getGlyphExtents) (ICEUIFontGlyphBuffer* iUIFontGlyphBuffer, CEUITextExtents* extentsOut);

/*!
 * V-table of ICEUIFontGlyphBuffer
 */
struct ICEUIFontGlyphBuffer_vtbl
{
	iCEUIFontGlyphBuffer_queryInterface	_queryInterface;	//!< Query interface.
	iCEUIFontGlyphBuffer_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUIFontGlyphBuffer_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUIFontGlyphBuffer_initG16F	_initG16F;
	iCEUIFontGlyphBuffer_getFirstPaltformFont	_getFirstPaltformFont;
	iCEUIFontGlyphBuffer_elementAt	_elementAt;
	iCEUIFontGlyphBuffer_getGlyphExtents	_getGlyphExtents;
};

/*! @}
 * end of ICEUIFontGlyphBuffer
 */

/*! \defgroup ICEUITextMeasurer ICEUITextMeasurer
 * @{
 */

/*!
 * ID of ICEUITextMeasurer
 */
#define CEComIID_ICEUITextMeasurer 0x1ccd45a6


/*!
* ICEUITextMeasure
*/
typedef struct ICEUITextMeasurer
{
	const struct ICEUITextMeasurer_vtbl* _vtbl;
} ICEUITextMeasurer;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUITextMeasurer	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUITextMeasurer_queryInterface) (ICEUITextMeasurer* iUITextMeasurer, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUITextMeasurer_addRef) (ICEUITextMeasurer* iUITextMeasurer);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUITextMeasurer_release) (ICEUITextMeasurer* iUITextMeasurer);

typedef CEHResult (*iCEUITextMeasurer_resetEnumerator) (ICEUITextMeasurer* iUITextMeasurer);

typedef CEHResult (*iCEUITextMeasurer_next) (ICEUITextMeasurer* iUITextMeasurer);

	//! returns current measured glyph storage info.
	//!  
	//! pOptGlyphAdvanceXOut: width of current glyph. 
	//! pOptStartToCurGlyphRightEdgeWidthOut: width of current run. is equals to the right edge of current glyph.
	//! pOptGlyphArrayIndexOut: current index of glyphArray.
	//! pOptCharIndexOut: current index of character array.

typedef CEHResult (*iCEUITextMeasurer_current) (ICEUITextMeasurer* iUITextMeasurer, CEUTextMeasurement* pOptGlyphAdvanceXOut, CEUTextMeasurement* pOptStartToCurGlyphRightEdgeWidthOut, UINT32* pOptCharIndexOut, bool *const pOptIsRtLOut);

typedef CEHResult (*iCEUITextMeasurer_current2) (ICEUITextMeasurer* iUITextMeasurer, struct ICEUIPlatformFont* *const optPhyFontOut, eCEI18nScriptCode *const optScriptCodeOut, eCEUIGlyphLE *const optGlyhLeOut);

typedef CEHResult (*iCEUITextMeasurer_charIndexToGlyphInfo) (ICEUITextMeasurer* iUITextMeasurer, UINT32 charIndex, CEUTextMeasurement* pOptGlyphAdvanceXOut, CEUTextMeasurement* pOptStartToCurGlyphRightEdgeWidthOut, bool *const pOptIsRtLOut);

	//! returns specified text width.

typedef CEHResult (*iCEUITextMeasurer_measureWidth) (ICEUITextMeasurer* iUITextMeasurer, CEUTextMeasurement *const pWidthOut);

	//! returns left and right character boundary offset and distance to the specified x point in visual order.
	//! pOptLeftmostBoundaryOffset16Out, pOptRightmostBoundaryOffset16Out: first(leftmost) and last(rightmost) character boundary offset of source text in visual order.

typedef CEHResult (*iCEUITextMeasurer_positionToBoundaryOffset) (ICEUITextMeasurer* iUITextMeasurer, CEUTextMeasurement pointX, UINT32 *const pOptLeftBoundaryOffset16Out, UINT32 *const pOptRightBoundaryOffset16Out, CEUTextMeasurement *const pOptLeftDistanceOut, CEUTextMeasurement *const pOptRightDistanceOut, UINT32 *const pOptLeftmostBoundary16Out, UINT32 *const pOptRightmostBoundary16Out);

	//! returns leading and trailing word width.
	//! specify ICEI18nTextBreakTester to specify how to break text as the words.
	//! if specified width is wide enough to place the text then the text is broken at the most end of breakable point.
	//! if the specified width is narrow to place a character then returns 0 as numOfLeadingWordC16Out.

typedef CEHResult (*iCEUITextMeasurer_breakText) (ICEUITextMeasurer* iUITextMeasurer, CEUTextMeasurement width, UINT32 *const pOptNumOfLeadWordC16Out, UINT32 *const pOptNumOfTrailWordC16Out, CEUTextMeasurement *const pOptLeadWidthOut, CEUTextMeasurement *const pOptTrailWidthOut);

	//! returns glyph boundary region of the part of source string that is specified by pStrRange. returned regionOut may contain multiple rectangles.

typedef CEHResult (*iCEUITextMeasurer_getBoundaryRegion) (ICEUITextMeasurer* iUITextMeasurer, const CEUIFontStrRangeOption* pStrRanges, UINT32 numOfStrRanges, CEUTextMeasurement height, struct ICEDXRegion* *const regionOut);

	//! returns glyph buffer of this object.

typedef CEHResult (*iCEUITextMeasurer_getGlyphBuffer) (ICEUITextMeasurer* iUITextMeasurer, const CEUIFontStrRangeOption* pOptStrRange, struct ICEUIFontGlyphBuffer* *const iGBufferOut);

/*!
 * V-table of ICEUITextMeasurer
 */
struct ICEUITextMeasurer_vtbl
{
	iCEUITextMeasurer_queryInterface	_queryInterface;	//!< Query interface.
	iCEUITextMeasurer_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUITextMeasurer_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUITextMeasurer_resetEnumerator	_resetEnumerator;
	iCEUITextMeasurer_next	_next;
	iCEUITextMeasurer_current	_current;
	iCEUITextMeasurer_current2	_current2;
	iCEUITextMeasurer_charIndexToGlyphInfo	_charIndexToGlyphInfo;
	iCEUITextMeasurer_measureWidth	_measureWidth;
	iCEUITextMeasurer_positionToBoundaryOffset	_positionToBoundaryOffset;
	iCEUITextMeasurer_breakText	_breakText;
	iCEUITextMeasurer_getBoundaryRegion	_getBoundaryRegion;
	iCEUITextMeasurer_getGlyphBuffer	_getGlyphBuffer;
};

/*! @}
 * end of ICEUITextMeasurer
 */

/*! \defgroup ICEUIFont ICEUIFont
 * @{
 */

/*!
 * ID of ICEUIFont
 */
#define CEComIID_ICEUIFont 0x99ca81cb


/*!
* ICEUIFont
*/
typedef struct ICEUIFont
{
	const struct ICEUIFont_vtbl* _vtbl;
} ICEUIFont;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUIFont	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUIFont_queryInterface) (ICEUIFont* iUIFont, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUIFont_addRef) (ICEUIFont* iUIFont);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUIFont_release) (ICEUIFont* iUIFont);

typedef CEHResult (*iCEUIFont_getValue) (ICEUIFont* iUIFont, CEUIFontValueType type, INT32 *const valueOut);

typedef CEHResult (*iCEUIFont_getStyle) (ICEUIFont* iUIFont, CEPFFontStyle *const styleOut);

typedef CEHResult (*iCEUIFont_getFamilyName) (ICEUIFont* iUIFont, struct ICEUIFontFamily* *const familyNameOut);

	//! returns new ICEUIFontGlyphBuffer interface pointer.

typedef CEHResult (*iCEUIFont_createGlyphBuffer) (ICEUIFont* iUIFont, struct ICEUIFontGlyphBuffer* *const iGBufferOut);

typedef CEHResult (*iCEUIFont_toString) (ICEUIFont* iUIFont, struct ICEUString* *const iStrOut);

typedef CEHResult (*iCEUIFont_dbgToString) (ICEUIFont* iUIFont, const UTF16CHAR* *const pStr16Out);

/*!
 * V-table of ICEUIFont
 */
struct ICEUIFont_vtbl
{
	iCEUIFont_queryInterface	_queryInterface;	//!< Query interface.
	iCEUIFont_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUIFont_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUIFont_getValue	_getValue;
	iCEUIFont_getStyle	_getStyle;
	iCEUIFont_getFamilyName	_getFamilyName;
	iCEUIFont_createGlyphBuffer	_createGlyphBuffer;
	iCEUIFont_toString	_toString;
	iCEUIFont_dbgToString	_dbgToString;
};

/*! @}
 * end of ICEUIFont
 */

/*! \defgroup ICEUIPlatformFont ICEUIPlatformFont
 * @{
 */

/*!
 * ID of ICEUIPlatformFont
 */
#define CEComIID_ICEUIPlatformFont 0x4d363eb8

/*!
 * ICEUIPlatformFont
 */
typedef struct ICEUIPlatformFont
{
	const struct ICEUIPlatformFont_vtbl* _vtbl;
} ICEUIPlatformFont;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUIPlatformFont	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUIPlatformFont_queryInterface) (ICEUIPlatformFont* iUIPlatformFont, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUIPlatformFont_addRef) (ICEUIPlatformFont* iUIPlatformFont);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUIPlatformFont_release) (ICEUIPlatformFont* iUIPlatformFont);

typedef CEHResult (*iCEUIPlatformFont_getValue) (ICEUIPlatformFont* iUIPlatformFont, CEUIFontValueType type, INT32 *const valueOut);

typedef CEHResult (*iCEUIPlatformFont_getStyle) (ICEUIPlatformFont* iUIPlatformFont, CEPFFontStyle *const styleOut);

typedef CEHResult (*iCEUIPlatformFont_getFamilyName) (ICEUIPlatformFont* iUIPlatformFont, struct ICEUIFontFamily* *const familyNameOut);

	//! returns new ICEUIFontGlyphBuffer interface pointer.

typedef CEHResult (*iCEUIPlatformFont_createGlyphBuffer) (ICEUIPlatformFont* iUIPlatformFont, struct ICEUIFontGlyphBuffer* *const iGBufferOut);

typedef CEHResult (*iCEUIPlatformFont_toString) (ICEUIPlatformFont* iUIPlatformFont, struct ICEUString* *const iStrOut);

typedef CEHResult (*iCEUIPlatformFont_dbgToString) (ICEUIPlatformFont* iUIPlatformFont, const UTF16CHAR* *const pStr16Out);

	//! deprecated

typedef CEHResult (*iCEUIPlatformFont_drawGlyph) (ICEUIPlatformFont* iUIPlatformFont, struct ICEDXContext2D* csd, const CEPointBase* ptToDrawAt, UINT32 glyphIndex);

	//! deprecated

typedef CEHResult (*iCEUIPlatformFont_strokeGlyph) (ICEUIPlatformFont* iUIPlatformFont, struct ICEDXContext2D* csd, const CEPointBase* ptToDrawAt, UINT32 glyphIndex);

typedef CEHResult (*iCEUIPlatformFont_drawGlyphI) (ICEUIPlatformFont* iUIPlatformFont, struct ICEVGContext* ivgc, const CEPointBase* ptToDrawAt, UINT32 glyphIndex, CEPFGlyphAdvance *const optAdvanceOut);

typedef CEHResult (*iCEUIPlatformFont_strokeGlyphI) (ICEUIPlatformFont* iUIPlatformFont, struct ICEVGContext* ivgc, const CEPointBase* ptToDrawAt, UINT32 glyphIndex, CEPFGlyphAdvance *const optAdvanceOut);

typedef CEHResult (*iCEUIPlatformFont_addGlyphPathI) (ICEUIPlatformFont* iUIPlatformFont, struct ICEVGContext* ivgc, const CEPointBase* ptToDrawAt, UINT32 glyphIndex, CEPFGlyphAdvance *const optAdvanceOut);

typedef CEHResult (*iCEUIPlatformFont_getFaceInfo) (ICEUIPlatformFont* iUIPlatformFont, CEPFFaceInfo *const faceInfoOut);

typedef CEHResult (*iCEUIPlatformFont_getGlyphWidth) (ICEUIPlatformFont* iUIPlatformFont, UINT32 glyphIndex, INT32 *const widthOut);

typedef CEHResult (*iCEUIPlatformFont_getGlyphMetrics) (ICEUIPlatformFont* iUIPlatformFont, UINT32 glyphIndex, CEPFGlyphBBox* bboxOut, CEPFGlyphAdvance* advanceOut);

	//! nGlyphOutRef: [in/out] in: specify size of optGlyphsOut. out:max index of optGlyphsOut.
	//! return: JGSS_SUCCESS: more than one glyph index found. JGSS_ERR_NOTFOUND: no glyph index found.  

typedef CEHResult (*iCEUIPlatformFont_getGlyphs16) (ICEUIPlatformFont* iUIPlatformFont, const UTF16CHAR* pC16s, UINT32 nC16s, bool bNoSurrogatePairs, UINT32 *const optGlyphsOut, UINT32 *const optNGlyphsRef);

typedef CEHResult (*iCEUIPlatformFont_getICEPFRasterizerInfo) (ICEUIPlatformFont* iUIPlatformFont, struct ICEPFFace* *const faceOut, struct ICEPFGlyphLoader* *const glyphLoaderOut);

	//! returns smallcaps font of this platform font.

typedef CEHResult (*iCEUIPlatformFont_getSmallCapsFont) (ICEUIPlatformFont* iUIPlatformFont, struct ICEUIPlatformFont* *const smallCapsPfOut);

typedef CEHResult (*iCEUIPlatformFont_getScaledFont) (ICEUIPlatformFont* iUIPlatformFont, struct ICEUIPlatformFont* *const scaledPfOut, float scale);

/*!
 * V-table of ICEUIPlatformFont
 */
struct ICEUIPlatformFont_vtbl
{
	iCEUIPlatformFont_queryInterface	_queryInterface;	//!< Query interface.
	iCEUIPlatformFont_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUIPlatformFont_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUIPlatformFont_getValue	_getValue;
	iCEUIPlatformFont_getStyle	_getStyle;
	iCEUIPlatformFont_getFamilyName	_getFamilyName;
	iCEUIPlatformFont_createGlyphBuffer	_createGlyphBuffer;
	iCEUIPlatformFont_toString	_toString;
	iCEUIPlatformFont_dbgToString	_dbgToString;
	iCEUIPlatformFont_drawGlyph	_drawGlyph;
	iCEUIPlatformFont_strokeGlyph	_strokeGlyph;
	iCEUIPlatformFont_drawGlyphI	_drawGlyphI;
	iCEUIPlatformFont_strokeGlyphI	_strokeGlyphI;
	iCEUIPlatformFont_addGlyphPathI	_addGlyphPathI;
	iCEUIPlatformFont_getFaceInfo	_getFaceInfo;
	iCEUIPlatformFont_getGlyphWidth	_getGlyphWidth;
	iCEUIPlatformFont_getGlyphMetrics	_getGlyphMetrics;
	iCEUIPlatformFont_getGlyphs16	_getGlyphs16;
	iCEUIPlatformFont_getICEPFRasterizerInfo	_getICEPFRasterizerInfo;
	iCEUIPlatformFont_getSmallCapsFont	_getSmallCapsFont;
	iCEUIPlatformFont_getScaledFont	_getScaledFont;
};

/*! @}
 * end of ICEUIPlatformFont
 */

/*! \defgroup ICEUICompositeFont ICEUICompositeFont
 * @{
 */

/*!
 * ID of ICEUICompositeFont
 */
#define CEComIID_ICEUICompositeFont 0x5bb899e0

/*!
 * ICEUICompositeFont
 */
typedef struct ICEUICompositeFont
{
	const struct ICEUICompositeFont_vtbl* _vtbl;
} ICEUICompositeFont;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUICompositeFont	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUICompositeFont_queryInterface) (ICEUICompositeFont* iUICompositeFont, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUICompositeFont_addRef) (ICEUICompositeFont* iUICompositeFont);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUICompositeFont_release) (ICEUICompositeFont* iUICompositeFont);

typedef CEHResult (*iCEUICompositeFont_getValue) (ICEUICompositeFont* iUICompositeFont, CEUIFontValueType type, INT32 *const valueOut);

typedef CEHResult (*iCEUICompositeFont_getStyle) (ICEUICompositeFont* iUICompositeFont, CEPFFontStyle *const styleOut);

typedef CEHResult (*iCEUICompositeFont_getFamilyName) (ICEUICompositeFont* iUICompositeFont, struct ICEUIFontFamily* *const familyNameOut);

	//! returns new ICEUIFontGlyphBuffer interface pointer.

typedef CEHResult (*iCEUICompositeFont_createGlyphBuffer) (ICEUICompositeFont* iUICompositeFont, struct ICEUIFontGlyphBuffer* *const iGBufferOut);

typedef CEHResult (*iCEUICompositeFont_toString) (ICEUICompositeFont* iUICompositeFont, struct ICEUString* *const iStrOut);

typedef CEHResult (*iCEUICompositeFont_dbgToString) (ICEUICompositeFont* iUICompositeFont, const UTF16CHAR* *const pStr16Out);

typedef CEHResult (*iCEUICompositeFont_getLang) (ICEUICompositeFont* iUICompositeFont, struct ICEI18nLocale* *const langOut);

typedef CEHResult (*iCEUICompositeFont_getFirstPlatformFont) (ICEUICompositeFont* iUICompositeFont, struct ICEUIPlatformFont* *const pfOut);

typedef CEHResult (*iCEUICompositeFont_getPlatformFont) (ICEUICompositeFont* iUICompositeFont, UTF32CHAR c32, bool bUseCharMapper, struct ICEUIPlatformFont* *const pfOut, UINT32 *const optGlyphIndexOut);

typedef CEHResult (*iCEUICompositeFont_mapChar32) (ICEUICompositeFont* iUICompositeFont, UTF32CHAR c32, UTF32CHAR *const c32Out);

typedef CEHResult (*iCEUICompositeFont_measureUTF16CHARArray) (ICEUICompositeFont* iUICompositeFont, const UTF16CHAR* pC16s, const UINT32 numOfChar16, CEUIFontOption* pOptionRef, INT32 *const measuredLengthOut);

	//! deprecated

typedef CEHResult (*iCEUICompositeFont_drawUTF16CHARArray) (ICEUICompositeFont* iUICompositeFont, struct ICEDXContext2D* c2d, const CEPointBase* ptToDrawAt, const UTF16CHAR* pC16s, const UINT32 numOfChar16, CEUIFontOption* pOptionRef);

typedef CEHResult (*iCEUICompositeFont_drawUTF16CHARArrayI) (ICEUICompositeFont* iUICompositeFont, struct ICEVGContext* ivgc, const CERGBAColor* glyphColor, const CEPointBase* ptToDrawAt, const UTF16CHAR* pC16s, const UINT32 numOfChar16, CEUIFontOption* pOptionRef);

	//! returns new ICEUITextMeasure interface pointer for specified character array.

typedef CEHResult (*iCEUICompositeFont_getTextMeasurer) (ICEUICompositeFont* iUICompositeFont, const UTF16CHAR* pC16s, UINT32 numOfC16s, CEUIFontOption* pOptFontOptionRef, struct ICEUITextMeasurer* *const iTextMeasurerOut);

/*!
 * V-table of ICEUICompositeFont
 */
struct ICEUICompositeFont_vtbl
{
	iCEUICompositeFont_queryInterface	_queryInterface;	//!< Query interface.
	iCEUICompositeFont_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUICompositeFont_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUICompositeFont_getValue	_getValue;
	iCEUICompositeFont_getStyle	_getStyle;
	iCEUICompositeFont_getFamilyName	_getFamilyName;
	iCEUICompositeFont_createGlyphBuffer	_createGlyphBuffer;
	iCEUICompositeFont_toString	_toString;
	iCEUICompositeFont_dbgToString	_dbgToString;
	iCEUICompositeFont_getLang	_getLang;
	iCEUICompositeFont_getFirstPlatformFont	_getFirstPlatformFont;
	iCEUICompositeFont_getPlatformFont	_getPlatformFont;
	iCEUICompositeFont_mapChar32	_mapChar32;
	iCEUICompositeFont_measureUTF16CHARArray	_measureUTF16CHARArray;
	iCEUICompositeFont_drawUTF16CHARArray	_drawUTF16CHARArray;
	iCEUICompositeFont_drawUTF16CHARArrayI	_drawUTF16CHARArrayI;
	iCEUICompositeFont_getTextMeasurer	_getTextMeasurer;
};

/*! @}
 * end of ICEUICompositeFont
 */

/*! \defgroup ICEUIFontFamilyNameUtility ICEUIFontFamilyNameUtility
 * @{
 */

/*!
 * ID of ICEUIFontFamilyNameUtility
 */
#define CEComIID_ICEUIFontFamilyNameUtility 0xbac5a303


/*!
* ICEUIFontFamilyNameUtility
*/
typedef struct ICEUIFontFamilyNameUtility
{
	const struct ICEUIFontFamilyNameUtility_vtbl* _vtbl;
} ICEUIFontFamilyNameUtility;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUIFontFamilyNameUtility	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUIFontFamilyNameUtility_queryInterface) (ICEUIFontFamilyNameUtility* iUIFontFamilyNameUtility, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUIFontFamilyNameUtility_addRef) (ICEUIFontFamilyNameUtility* iUIFontFamilyNameUtility);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUIFontFamilyNameUtility_release) (ICEUIFontFamilyNameUtility* iUIFontFamilyNameUtility);

	//! parse specified string as a single font family.

typedef CEHResult (*iCEUIFontFamilyNameUtility_createSingleFontFamilyFromCStringLiteral) (ICEUIFontFamilyNameUtility* iUIFontFamilyNameUtility, const UCHAR8* str, UINT32 numOfChar8, struct ICEUIFontFamily* *const fontFamilyOut);

typedef CEHResult (*iCEUIFontFamilyNameUtility_createSingleFontFamilyFromUTF16Array) (ICEUIFontFamilyNameUtility* iUIFontFamilyNameUtility, const UTF16CHAR* str16, UINT32 numOfChar16, struct ICEUIFontFamily* *const fontFamilyOut);

	//! parse specified string as a ',' separated multiple font family list.

typedef CEHResult (*iCEUIFontFamilyNameUtility_createFontFamilyFromCStringLiteral) (ICEUIFontFamilyNameUtility* iUIFontFamilyNameUtility, const UCHAR8* str, UINT32 numOfChar8, struct ICEUIFontFamily* *const fontFamilyOut);

typedef CEHResult (*iCEUIFontFamilyNameUtility_createFontFamilyFromUTF16Array) (ICEUIFontFamilyNameUtility* iUIFontFamilyNameUtility, const UTF16CHAR* str16, UINT32 numOfChar16, struct ICEUIFontFamily* *const fontFamilyOut);

typedef CEHResult (*iCEUIFontFamilyNameUtility_createFontFamilyFromFamilyArray) (ICEUIFontFamilyNameUtility* iUIFontFamilyNameUtility, struct ICEUIFontFamily*const *const families, const UINT32 numOfFamilies, struct ICEUIFontFamily* *const fontFamilyOut);

/*!
 * V-table of ICEUIFontFamilyNameUtility
 */
struct ICEUIFontFamilyNameUtility_vtbl
{
	iCEUIFontFamilyNameUtility_queryInterface	_queryInterface;	//!< Query interface.
	iCEUIFontFamilyNameUtility_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUIFontFamilyNameUtility_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUIFontFamilyNameUtility_createSingleFontFamilyFromCStringLiteral	_createSingleFontFamilyFromCStringLiteral;
	iCEUIFontFamilyNameUtility_createSingleFontFamilyFromUTF16Array	_createSingleFontFamilyFromUTF16Array;
	iCEUIFontFamilyNameUtility_createFontFamilyFromCStringLiteral	_createFontFamilyFromCStringLiteral;
	iCEUIFontFamilyNameUtility_createFontFamilyFromUTF16Array	_createFontFamilyFromUTF16Array;
	iCEUIFontFamilyNameUtility_createFontFamilyFromFamilyArray	_createFontFamilyFromFamilyArray;
};

/*! @}
 * end of ICEUIFontFamilyNameUtility
 */

/*! \defgroup ICEUICompositeFontFactory ICEUICompositeFontFactory
 * @{
 */

/*!
 * ID of ICEUICompositeFontFactory
 */
#define CEComIID_ICEUICompositeFontFactory 0x489f2339


/*!
* ICEUICompositeFontFactory
*/
typedef struct ICEUICompositeFontFactory
{
	const struct ICEUICompositeFontFactory_vtbl* _vtbl;
} ICEUICompositeFontFactory;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUICompositeFontFactory	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUICompositeFontFactory_queryInterface) (ICEUICompositeFontFactory* iUICompositeFontFactory, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUICompositeFontFactory_addRef) (ICEUICompositeFontFactory* iUICompositeFontFactory);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUICompositeFontFactory_release) (ICEUICompositeFontFactory* iUICompositeFontFactory);

typedef CEHResult (*iCEUICompositeFontFactory_createCompositeFontBySizeInPixels) (ICEUICompositeFontFactory* iUICompositeFontFactory, const struct ICEI18nLocale* i18nLocale, struct ICEUIFontFamily* fontNameList, const CEPFFontStyle* style, INT32 sizeInPixels, bool bUseLastResortFallbackFont, struct ICEUICompositeFont* *const cfOut);

typedef CEHResult (*iCEUICompositeFontFactory_createCompositeFontBySizeInPoints) (ICEUICompositeFontFactory* iUICompositeFontFactory, const struct ICEI18nLocale* i18nLocale, struct ICEUIFontFamily* fontNameList, const CEPFFontStyle* style, CEFixed16d16 sizeInPoints, bool bUseLastResortFallbackFont, struct ICEUICompositeFont* *const cfOut);

typedef CEHResult (*iCEUICompositeFontFactory_getTraitsInFamily) (ICEUICompositeFontFactory* iUICompositeFontFactory, struct ICEUIFontFamily* fontFamily, CEPFFontStyle *const traits, UINT32 *const nTraitsRef);

typedef CEHResult (*iCEUICompositeFontFactory_handleResetInterface) (ICEUICompositeFontFactory* iUICompositeFontFactory);

typedef CEHResult (*iCEUICompositeFontFactory_registerCustomPlatformFontFace) (ICEUICompositeFontFactory* iUICompositeFontFactory, const UINT8* bytes, UINT32 nBytes, struct ICEUIFontFamily* fontFamily, struct ICEPFFace* *const faceOut);

typedef CEHResult (*iCEUICompositeFontFactory_unregisterCustomPlatformFontFace) (ICEUICompositeFontFactory* iUICompositeFontFactory, struct ICEUIFontFamily* fontFamily, struct ICEPFFace* pfFont);

typedef CEHResult (*iCEUICompositeFontFactory_createPlatformFontBySizeInPixels) (ICEUICompositeFontFactory* iUICompositeFontFactory, struct ICEUIFontFamily* fontFamily, const CEPFFontStyle* style, INT32 sizeInPixels, struct ICEUIPlatformFont* *const pfFontOut);

typedef CEHResult (*iCEUICompositeFontFactory_createPlatformFontBySizeInPoints) (ICEUICompositeFontFactory* iUICompositeFontFactory, struct ICEUIFontFamily* fontFamily, const CEPFFontStyle* style, CEFixed16d16 sizeInPoints, struct ICEUIPlatformFont* *const pfFontOut);

/*!
 * V-table of ICEUICompositeFontFactory
 */
struct ICEUICompositeFontFactory_vtbl
{
	iCEUICompositeFontFactory_queryInterface	_queryInterface;	//!< Query interface.
	iCEUICompositeFontFactory_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUICompositeFontFactory_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUICompositeFontFactory_createCompositeFontBySizeInPixels	_createCompositeFontBySizeInPixels;
	iCEUICompositeFontFactory_createCompositeFontBySizeInPoints	_createCompositeFontBySizeInPoints;
	iCEUICompositeFontFactory_getTraitsInFamily	_getTraitsInFamily;
	iCEUICompositeFontFactory_handleResetInterface	_handleResetInterface;
	iCEUICompositeFontFactory_registerCustomPlatformFontFace	_registerCustomPlatformFontFace;
	iCEUICompositeFontFactory_unregisterCustomPlatformFontFace	_unregisterCustomPlatformFontFace;
	iCEUICompositeFontFactory_createPlatformFontBySizeInPixels	_createPlatformFontBySizeInPixels;
	iCEUICompositeFontFactory_createPlatformFontBySizeInPoints	_createPlatformFontBySizeInPoints;
};

/*! @}
 * end of ICEUICompositeFontFactory
 */

/*! \defgroup ICEUIFontService ICEUIFontService
 * @{
 */

/*!
 * ID of ICEUIFontService
 */
#define CEComIID_ICEUIFontService 0x4d586f1c

/*!
 * ICEUIFontService
 */
typedef struct ICEUIFontService
{
	const struct ICEUIFontService_vtbl* _vtbl;
} ICEUIFontService;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iUIFontService	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEUIFontService_queryInterface) (ICEUIFontService* iUIFontService, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUIFontService_addRef) (ICEUIFontService* iUIFontService);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEUIFontService_release) (ICEUIFontService* iUIFontService);


/*!
* <b>Summary:</b>
* creates com instance and returns interface pointer of specified iid.
* 
* \param[in]	iUIFontService	Specifies interface pointer
* \param[in]	pUnkOuter	TBD.
* \param[in]	iid	 iid to create.
* \param[out]	ppv	created interface pointer.
*
* \return error code.
*/
typedef CEHResult (*iCEUIFontService_createInstance) (ICEUIFontService* iUIFontService, struct ICEUnknown* pUnkOuter, UINT32 iid, void* *const ppv);

typedef CEHResult (*iCEUIFontService_initService) (ICEUIFontService* iUIFontService, CEUIFontServiceInitArg* arg);

typedef CEHResult (*iCEUIFontService_getFontFamilyNameUtility) (ICEUIFontService* iUIFontService, struct ICEUIFontFamilyNameUtility* *const ffNameUtilOut);

typedef CEHResult (*iCEUIFontService_getCompositeFontFactory) (ICEUIFontService* iUIFontService, struct ICEUICompositeFontFactory* *const ccfOut);

typedef CEHResult (*iCEUIFontService_setOptionsBit) (ICEUIFontService* iUIFontService, eCEUIFontServiceOptionsBit optionBit, bool v);

typedef CEHResult (*iCEUIFontService_getOptionsBit) (ICEUIFontService* iUIFontService, eCEUIFontServiceOptionsBit optionBit, bool *const vOut);

typedef CEHResult (*iCEUIFontService_setRasterizerParams) (ICEUIFontService* iUIFontService, const UINT32* params, UINT32 numOfParams);

typedef CEHResult (*iCEUIFontService_getRasterizerParams) (ICEUIFontService* iUIFontService, const UINT32**params, UINT32*const numOfParams);

typedef CEHResult (*iCEUIFontService_shutdownService) (ICEUIFontService* iUIFontService);

/*!
 * V-table of ICEUIFontService
 */
struct ICEUIFontService_vtbl
{
	iCEUIFontService_queryInterface	_queryInterface;	//!< Query interface.
	iCEUIFontService_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEUIFontService_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEUIFontService_createInstance	_createInstance;
	iCEUIFontService_initService	_initService;
	iCEUIFontService_getFontFamilyNameUtility	_getFontFamilyNameUtility;
	iCEUIFontService_getCompositeFontFactory	_getCompositeFontFactory;
	iCEUIFontService_setOptionsBit	_setOptionsBit;
	iCEUIFontService_getOptionsBit	_getOptionsBit;
	iCEUIFontService_setRasterizerParams	_setRasterizerParams;
	iCEUIFontService_getRasterizerParams	_getRasterizerParams;
	iCEUIFontService_shutdownService	_shutdownService;
};

/*! @}
 * end of ICEUIFontService
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEUIFontFamilyRef CEComICEUIFontFamilyRef
 * @{
 */

class CEComICEUIFontFamilyRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUIFontFamilyRef() : _iUIFontFamily(0)
	{
#if defined(_DEBUG) || !defined(NDEBUG)
		dbgToString();
#endif //_DEBUG || !NDEBUG
	}
	CEComICEUIFontFamilyRef(ICEUIFontFamily* iOther) : _iUIFontFamily(0)
	{
		if (iOther)
		{
			_iUIFontFamily = iOther;
			_iUIFontFamily->_vtbl->_addRef(_iUIFontFamily);
		}
	}

	CEComICEUIFontFamilyRef(const CEComICEUIFontFamilyRef& other) : _iUIFontFamily(0)
	{
		if (other._iUIFontFamily)
		{
			_iUIFontFamily = other._iUIFontFamily;
			_iUIFontFamily->_vtbl->_addRef(_iUIFontFamily);
		}
	}

	~CEComICEUIFontFamilyRef()
	{
		if (_iUIFontFamily)
		{
			ICEUIFontFamily* tmp = _iUIFontFamily;
			_iUIFontFamily = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUIFontFamily* iUIFontFamily;
				void* _ptr;
			} uIntf;
			uIntf.iUIFontFamily = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUIFontFamily, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUIFontFamily)
				{
					ICEUIFontFamily* tmp = _iUIFontFamily;
					_iUIFontFamily = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUIFontFamily = uIntf.iUIFontFamily;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUIFontFamily* object() const	{ return _iUIFontFamily; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUIFontFamily* iOther)
	{
		if (_iUIFontFamily)
		{
			ICEUIFontFamily* tmp = _iUIFontFamily;
			_iUIFontFamily = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUIFontFamily = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUIFontFamily* detach()
	{
		ICEUIFontFamily* iIntf = _iUIFontFamily;
		_iUIFontFamily = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUIFontFamily* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUIFontFamily;
		if (_iUIFontFamily)
		{
			_iUIFontFamily->_vtbl->_addRef(_iUIFontFamily);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUIFontFamily*() const	{ return _iUIFontFamily; }
	FORCEINLINE_WITHOUT_DEBUG ICEUIFontFamily& operator*() const	{ return *_iUIFontFamily; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUIFontFamily == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUIFontFamily* iOther) const	{ return (_iUIFontFamily != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUIFontFamily* iOther) const	{ return (_iUIFontFamily == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUIFontFamilyRef& operator=(const CEComICEUIFontFamilyRef& other)	{ return operator=(other._iUIFontFamily); }

	CEComICEUIFontFamilyRef& operator=(const ICEUIFontFamily* iOther)
	{
		if (_iUIFontFamily != iOther)
		{
			if (_iUIFontFamily)
			{
				ICEUIFontFamily* tmp = _iUIFontFamily;
				_iUIFontFamily = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUIFontFamily = const_cast<ICEUIFontFamily*>(iOther);
			if (_iUIFontFamily)
			{
				_iUIFontFamily->_vtbl->_addRef(_iUIFontFamily);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUIFontFamily** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUIFontFamilyRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUIFontFamily && "has a com object reference. clear first.");
		return & _iUIFontFamily; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUIFontFamilyRef& other)
	{
		bool result = false;
		if (_iUIFontFamily)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUIFontFamily);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUIFontFamily ? _iUIFontFamily->_vtbl->_queryInterface(_iUIFontFamily, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUIFontFamily) { _iUIFontFamily->_vtbl->_addRef(_iUIFontFamily); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUIFontFamily) { _iUIFontFamily->_vtbl->_release(_iUIFontFamily); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getName(struct ICEUString* *const listNameOut)	{ return _iUIFontFamily ? _iUIFontFamily->_vtbl->_getName(_iUIFontFamily, listNameOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFamilies(struct ICEUIFontFamily* *const familiesOut, UINT32 *const numFamiliesRef)	{ return _iUIFontFamily ? _iUIFontFamily->_vtbl->_getFamilies(_iUIFontFamily, familiesOut, numFamiliesRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFamilyAt(struct ICEUIFontFamily* *const familiesOut, UINT32 index)	{ return _iUIFontFamily ? _iUIFontFamily->_vtbl->_getFamilyAt(_iUIFontFamily, familiesOut, index) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult dbgToString(const UTF16CHAR* *const pStr16Out)	{ return _iUIFontFamily ? _iUIFontFamily->_vtbl->_dbgToString(_iUIFontFamily, pStr16Out) : CE_SILK_ERR_UNINITIALIZED; }

#if defined(_DEBUG) || !defined(NDEBUG)
	const UTF16CHAR* dbgToString() { const UTF16CHAR* ret=0; if (_iUIFontFamily) { dbgToString(&ret); } return ret; }
#endif // _DEBUG || !NDEBUG


private:
	ICEUIFontFamily* _iUIFontFamily;
};

/*! @}
 * end of CEComICEUIFontFamilyRef
 */

/*! \defgroup CEComICEUIFontGlyphBufferRef CEComICEUIFontGlyphBufferRef
 * @{
 */

class CEComICEUIFontGlyphBufferRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUIFontGlyphBufferRef() : _iUIFontGlyphBuffer(0) {}
	CEComICEUIFontGlyphBufferRef(ICEUIFontGlyphBuffer* iOther) : _iUIFontGlyphBuffer(0)
	{
		if (iOther)
		{
			_iUIFontGlyphBuffer = iOther;
			_iUIFontGlyphBuffer->_vtbl->_addRef(_iUIFontGlyphBuffer);
		}
	}

	CEComICEUIFontGlyphBufferRef(const CEComICEUIFontGlyphBufferRef& other) : _iUIFontGlyphBuffer(0)
	{
		if (other._iUIFontGlyphBuffer)
		{
			_iUIFontGlyphBuffer = other._iUIFontGlyphBuffer;
			_iUIFontGlyphBuffer->_vtbl->_addRef(_iUIFontGlyphBuffer);
		}
	}

	~CEComICEUIFontGlyphBufferRef()
	{
		if (_iUIFontGlyphBuffer)
		{
			ICEUIFontGlyphBuffer* tmp = _iUIFontGlyphBuffer;
			_iUIFontGlyphBuffer = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUIFontGlyphBuffer* iUIFontGlyphBuffer;
				void* _ptr;
			} uIntf;
			uIntf.iUIFontGlyphBuffer = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUIFontGlyphBuffer, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUIFontGlyphBuffer)
				{
					ICEUIFontGlyphBuffer* tmp = _iUIFontGlyphBuffer;
					_iUIFontGlyphBuffer = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUIFontGlyphBuffer = uIntf.iUIFontGlyphBuffer;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUIFontGlyphBuffer* object() const	{ return _iUIFontGlyphBuffer; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUIFontGlyphBuffer* iOther)
	{
		if (_iUIFontGlyphBuffer)
		{
			ICEUIFontGlyphBuffer* tmp = _iUIFontGlyphBuffer;
			_iUIFontGlyphBuffer = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUIFontGlyphBuffer = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUIFontGlyphBuffer* detach()
	{
		ICEUIFontGlyphBuffer* iIntf = _iUIFontGlyphBuffer;
		_iUIFontGlyphBuffer = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUIFontGlyphBuffer* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUIFontGlyphBuffer;
		if (_iUIFontGlyphBuffer)
		{
			_iUIFontGlyphBuffer->_vtbl->_addRef(_iUIFontGlyphBuffer);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUIFontGlyphBuffer*() const	{ return _iUIFontGlyphBuffer; }
	FORCEINLINE_WITHOUT_DEBUG ICEUIFontGlyphBuffer& operator*() const	{ return *_iUIFontGlyphBuffer; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUIFontGlyphBuffer == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUIFontGlyphBuffer* iOther) const	{ return (_iUIFontGlyphBuffer != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUIFontGlyphBuffer* iOther) const	{ return (_iUIFontGlyphBuffer == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUIFontGlyphBufferRef& operator=(const CEComICEUIFontGlyphBufferRef& other)	{ return operator=(other._iUIFontGlyphBuffer); }

	CEComICEUIFontGlyphBufferRef& operator=(const ICEUIFontGlyphBuffer* iOther)
	{
		if (_iUIFontGlyphBuffer != iOther)
		{
			if (_iUIFontGlyphBuffer)
			{
				ICEUIFontGlyphBuffer* tmp = _iUIFontGlyphBuffer;
				_iUIFontGlyphBuffer = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUIFontGlyphBuffer = const_cast<ICEUIFontGlyphBuffer*>(iOther);
			if (_iUIFontGlyphBuffer)
			{
				_iUIFontGlyphBuffer->_vtbl->_addRef(_iUIFontGlyphBuffer);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUIFontGlyphBuffer** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUIFontGlyphBufferRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUIFontGlyphBuffer && "has a com object reference. clear first.");
		return & _iUIFontGlyphBuffer; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUIFontGlyphBufferRef& other)
	{
		bool result = false;
		if (_iUIFontGlyphBuffer)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUIFontGlyphBuffer);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUIFontGlyphBuffer ? _iUIFontGlyphBuffer->_vtbl->_queryInterface(_iUIFontGlyphBuffer, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUIFontGlyphBuffer) { _iUIFontGlyphBuffer->_vtbl->_addRef(_iUIFontGlyphBuffer); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUIFontGlyphBuffer) { _iUIFontGlyphBuffer->_vtbl->_release(_iUIFontGlyphBuffer); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initG16F(struct ICEUIPlatformFont* iPf, const UINT16* gIndexes, const CEDimF* advances, INT_PTR nGlyphs)	{ return _iUIFontGlyphBuffer ? _iUIFontGlyphBuffer->_vtbl->_initG16F(_iUIFontGlyphBuffer, iPf, gIndexes, advances, nGlyphs) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFirstPaltformFont(struct ICEUIPlatformFont* *const iPfOut)	{ return _iUIFontGlyphBuffer ? _iUIFontGlyphBuffer->_vtbl->_getFirstPaltformFont(_iUIFontGlyphBuffer, iPfOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult elementAt(INT_PTR index, struct ICEUIPlatformFont* *const iPfOut, UINT32 *const gIndexOut, CEDimF *const advanceOut)	{ return _iUIFontGlyphBuffer ? _iUIFontGlyphBuffer->_vtbl->_elementAt(_iUIFontGlyphBuffer, index, iPfOut, gIndexOut, advanceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphExtents(CEUITextExtents* extentsOut)	{ return _iUIFontGlyphBuffer ? _iUIFontGlyphBuffer->_vtbl->_getGlyphExtents(_iUIFontGlyphBuffer, extentsOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEUIFontGlyphBuffer* _iUIFontGlyphBuffer;
};

/*! @}
 * end of CEComICEUIFontGlyphBufferRef
 */

/*! \defgroup CEComICEUITextMeasurerRef CEComICEUITextMeasurerRef
 * @{
 */

class CEComICEUITextMeasurerRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUITextMeasurerRef() : _iUITextMeasurer(0) {}
	CEComICEUITextMeasurerRef(ICEUITextMeasurer* iOther) : _iUITextMeasurer(0)
	{
		if (iOther)
		{
			_iUITextMeasurer = iOther;
			_iUITextMeasurer->_vtbl->_addRef(_iUITextMeasurer);
		}
	}

	CEComICEUITextMeasurerRef(const CEComICEUITextMeasurerRef& other) : _iUITextMeasurer(0)
	{
		if (other._iUITextMeasurer)
		{
			_iUITextMeasurer = other._iUITextMeasurer;
			_iUITextMeasurer->_vtbl->_addRef(_iUITextMeasurer);
		}
	}

	~CEComICEUITextMeasurerRef()
	{
		if (_iUITextMeasurer)
		{
			ICEUITextMeasurer* tmp = _iUITextMeasurer;
			_iUITextMeasurer = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUITextMeasurer* iUITextMeasurer;
				void* _ptr;
			} uIntf;
			uIntf.iUITextMeasurer = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUITextMeasurer, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUITextMeasurer)
				{
					ICEUITextMeasurer* tmp = _iUITextMeasurer;
					_iUITextMeasurer = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUITextMeasurer = uIntf.iUITextMeasurer;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUITextMeasurer* object() const	{ return _iUITextMeasurer; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUITextMeasurer* iOther)
	{
		if (_iUITextMeasurer)
		{
			ICEUITextMeasurer* tmp = _iUITextMeasurer;
			_iUITextMeasurer = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUITextMeasurer = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUITextMeasurer* detach()
	{
		ICEUITextMeasurer* iIntf = _iUITextMeasurer;
		_iUITextMeasurer = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUITextMeasurer* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUITextMeasurer;
		if (_iUITextMeasurer)
		{
			_iUITextMeasurer->_vtbl->_addRef(_iUITextMeasurer);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUITextMeasurer*() const	{ return _iUITextMeasurer; }
	FORCEINLINE_WITHOUT_DEBUG ICEUITextMeasurer& operator*() const	{ return *_iUITextMeasurer; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUITextMeasurer == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUITextMeasurer* iOther) const	{ return (_iUITextMeasurer != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUITextMeasurer* iOther) const	{ return (_iUITextMeasurer == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUITextMeasurerRef& operator=(const CEComICEUITextMeasurerRef& other)	{ return operator=(other._iUITextMeasurer); }

	CEComICEUITextMeasurerRef& operator=(const ICEUITextMeasurer* iOther)
	{
		if (_iUITextMeasurer != iOther)
		{
			if (_iUITextMeasurer)
			{
				ICEUITextMeasurer* tmp = _iUITextMeasurer;
				_iUITextMeasurer = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUITextMeasurer = const_cast<ICEUITextMeasurer*>(iOther);
			if (_iUITextMeasurer)
			{
				_iUITextMeasurer->_vtbl->_addRef(_iUITextMeasurer);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUITextMeasurer** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUITextMeasurerRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUITextMeasurer && "has a com object reference. clear first.");
		return & _iUITextMeasurer; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUITextMeasurerRef& other)
	{
		bool result = false;
		if (_iUITextMeasurer)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUITextMeasurer);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUITextMeasurer ? _iUITextMeasurer->_vtbl->_queryInterface(_iUITextMeasurer, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUITextMeasurer) { _iUITextMeasurer->_vtbl->_addRef(_iUITextMeasurer); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUITextMeasurer) { _iUITextMeasurer->_vtbl->_release(_iUITextMeasurer); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult resetEnumerator()	{ return _iUITextMeasurer ? _iUITextMeasurer->_vtbl->_resetEnumerator(_iUITextMeasurer) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult next()	{ return _iUITextMeasurer ? _iUITextMeasurer->_vtbl->_next(_iUITextMeasurer) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult current(CEUTextMeasurement* pOptGlyphAdvanceXOut, CEUTextMeasurement* pOptStartToCurGlyphRightEdgeWidthOut, UINT32* pOptCharIndexOut, bool *const pOptIsRtLOut)	{ return _iUITextMeasurer ? _iUITextMeasurer->_vtbl->_current(_iUITextMeasurer, pOptGlyphAdvanceXOut, pOptStartToCurGlyphRightEdgeWidthOut, pOptCharIndexOut, pOptIsRtLOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult current2(struct ICEUIPlatformFont* *const optPhyFontOut, eCEI18nScriptCode *const optScriptCodeOut, eCEUIGlyphLE *const optGlyhLeOut)	{ return _iUITextMeasurer ? _iUITextMeasurer->_vtbl->_current2(_iUITextMeasurer, optPhyFontOut, optScriptCodeOut, optGlyhLeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult charIndexToGlyphInfo(UINT32 charIndex, CEUTextMeasurement* pOptGlyphAdvanceXOut, CEUTextMeasurement* pOptStartToCurGlyphRightEdgeWidthOut, bool *const pOptIsRtLOut)	{ return _iUITextMeasurer ? _iUITextMeasurer->_vtbl->_charIndexToGlyphInfo(_iUITextMeasurer, charIndex, pOptGlyphAdvanceXOut, pOptStartToCurGlyphRightEdgeWidthOut, pOptIsRtLOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult measureWidth(CEUTextMeasurement *const pWidthOut)	{ return _iUITextMeasurer ? _iUITextMeasurer->_vtbl->_measureWidth(_iUITextMeasurer, pWidthOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult positionToBoundaryOffset(CEUTextMeasurement pointX, UINT32 *const pOptLeftBoundaryOffset16Out, UINT32 *const pOptRightBoundaryOffset16Out, CEUTextMeasurement *const pOptLeftDistanceOut, CEUTextMeasurement *const pOptRightDistanceOut, UINT32 *const pOptLeftmostBoundary16Out, UINT32 *const pOptRightmostBoundary16Out)	{ return _iUITextMeasurer ? _iUITextMeasurer->_vtbl->_positionToBoundaryOffset(_iUITextMeasurer, pointX, pOptLeftBoundaryOffset16Out, pOptRightBoundaryOffset16Out, pOptLeftDistanceOut, pOptRightDistanceOut, pOptLeftmostBoundary16Out, pOptRightmostBoundary16Out) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult breakText(CEUTextMeasurement width, UINT32 *const pOptNumOfLeadWordC16Out, UINT32 *const pOptNumOfTrailWordC16Out, CEUTextMeasurement *const pOptLeadWidthOut, CEUTextMeasurement *const pOptTrailWidthOut)	{ return _iUITextMeasurer ? _iUITextMeasurer->_vtbl->_breakText(_iUITextMeasurer, width, pOptNumOfLeadWordC16Out, pOptNumOfTrailWordC16Out, pOptLeadWidthOut, pOptTrailWidthOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getBoundaryRegion(const CEUIFontStrRangeOption* pStrRanges, UINT32 numOfStrRanges, CEUTextMeasurement height, struct ICEDXRegion* *const regionOut)	{ return _iUITextMeasurer ? _iUITextMeasurer->_vtbl->_getBoundaryRegion(_iUITextMeasurer, pStrRanges, numOfStrRanges, height, regionOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphBuffer(const CEUIFontStrRangeOption* pOptStrRange, struct ICEUIFontGlyphBuffer* *const iGBufferOut)	{ return _iUITextMeasurer ? _iUITextMeasurer->_vtbl->_getGlyphBuffer(_iUITextMeasurer, pOptStrRange, iGBufferOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEUITextMeasurer* _iUITextMeasurer;
};

/*! @}
 * end of CEComICEUITextMeasurerRef
 */

/*! \defgroup CEComICEUIFontRef CEComICEUIFontRef
 * @{
 */

class CEComICEUIFontRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUIFontRef() : _iUIFont(0)
	{
#if defined(_DEBUG) || !defined(NDEBUG)
		dbgToString();
#endif //_DEBUG || !NDEBUG
	}
	CEComICEUIFontRef(ICEUIFont* iOther) : _iUIFont(0)
	{
		if (iOther)
		{
			_iUIFont = iOther;
			_iUIFont->_vtbl->_addRef(_iUIFont);
		}
	}

	CEComICEUIFontRef(const CEComICEUIFontRef& other) : _iUIFont(0)
	{
		if (other._iUIFont)
		{
			_iUIFont = other._iUIFont;
			_iUIFont->_vtbl->_addRef(_iUIFont);
		}
	}

	~CEComICEUIFontRef()
	{
		if (_iUIFont)
		{
			ICEUIFont* tmp = _iUIFont;
			_iUIFont = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUIFont* iUIFont;
				void* _ptr;
			} uIntf;
			uIntf.iUIFont = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUIFont, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUIFont)
				{
					ICEUIFont* tmp = _iUIFont;
					_iUIFont = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUIFont = uIntf.iUIFont;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUIFont* object() const	{ return _iUIFont; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUIFont* iOther)
	{
		if (_iUIFont)
		{
			ICEUIFont* tmp = _iUIFont;
			_iUIFont = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUIFont = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUIFont* detach()
	{
		ICEUIFont* iIntf = _iUIFont;
		_iUIFont = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUIFont* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUIFont;
		if (_iUIFont)
		{
			_iUIFont->_vtbl->_addRef(_iUIFont);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUIFont*() const	{ return _iUIFont; }
	FORCEINLINE_WITHOUT_DEBUG ICEUIFont& operator*() const	{ return *_iUIFont; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUIFont == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUIFont* iOther) const	{ return (_iUIFont != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUIFont* iOther) const	{ return (_iUIFont == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUIFontRef& operator=(const CEComICEUIFontRef& other)	{ return operator=(other._iUIFont); }

	CEComICEUIFontRef& operator=(const ICEUIFont* iOther)
	{
		if (_iUIFont != iOther)
		{
			if (_iUIFont)
			{
				ICEUIFont* tmp = _iUIFont;
				_iUIFont = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUIFont = const_cast<ICEUIFont*>(iOther);
			if (_iUIFont)
			{
				_iUIFont->_vtbl->_addRef(_iUIFont);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUIFont** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUIFontRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUIFont && "has a com object reference. clear first.");
		return & _iUIFont; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUIFontRef& other)
	{
		bool result = false;
		if (_iUIFont)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUIFont);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUIFont ? _iUIFont->_vtbl->_queryInterface(_iUIFont, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUIFont) { _iUIFont->_vtbl->_addRef(_iUIFont); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUIFont) { _iUIFont->_vtbl->_release(_iUIFont); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getValue(CEUIFontValueType type, INT32 *const valueOut)	{ return _iUIFont ? _iUIFont->_vtbl->_getValue(_iUIFont, type, valueOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStyle(CEPFFontStyle *const styleOut)	{ return _iUIFont ? _iUIFont->_vtbl->_getStyle(_iUIFont, styleOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFamilyName(struct ICEUIFontFamily* *const familyNameOut)	{ return _iUIFont ? _iUIFont->_vtbl->_getFamilyName(_iUIFont, familyNameOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createGlyphBuffer(struct ICEUIFontGlyphBuffer* *const iGBufferOut)	{ return _iUIFont ? _iUIFont->_vtbl->_createGlyphBuffer(_iUIFont, iGBufferOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult toString(struct ICEUString* *const iStrOut)	{ return _iUIFont ? _iUIFont->_vtbl->_toString(_iUIFont, iStrOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult dbgToString(const UTF16CHAR* *const pStr16Out)	{ return _iUIFont ? _iUIFont->_vtbl->_dbgToString(_iUIFont, pStr16Out) : CE_SILK_ERR_UNINITIALIZED; }

#if defined(_DEBUG) || !defined(NDEBUG)
	const UTF16CHAR* dbgToString() { const UTF16CHAR* ret=0; if (_iUIFont) { dbgToString(&ret); } return ret; }
#endif // _DEBUG || !NDEBUG


private:
	ICEUIFont* _iUIFont;
};

/*! @}
 * end of CEComICEUIFontRef
 */

/*! \defgroup CEComICEUIPlatformFontRef CEComICEUIPlatformFontRef
 * @{
 */

class CEComICEUIPlatformFontRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUIPlatformFontRef() : _iUIPlatformFont(0) {}
	CEComICEUIPlatformFontRef(ICEUIPlatformFont* iOther) : _iUIPlatformFont(0)
	{
		if (iOther)
		{
			_iUIPlatformFont = iOther;
			_iUIPlatformFont->_vtbl->_addRef(_iUIPlatformFont);
		}
	}

	CEComICEUIPlatformFontRef(const CEComICEUIPlatformFontRef& other) : _iUIPlatformFont(0)
	{
		if (other._iUIPlatformFont)
		{
			_iUIPlatformFont = other._iUIPlatformFont;
			_iUIPlatformFont->_vtbl->_addRef(_iUIPlatformFont);
		}
	}

	~CEComICEUIPlatformFontRef()
	{
		if (_iUIPlatformFont)
		{
			ICEUIPlatformFont* tmp = _iUIPlatformFont;
			_iUIPlatformFont = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUIPlatformFont* iUIPlatformFont;
				void* _ptr;
			} uIntf;
			uIntf.iUIPlatformFont = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUIPlatformFont, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUIPlatformFont)
				{
					ICEUIPlatformFont* tmp = _iUIPlatformFont;
					_iUIPlatformFont = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUIPlatformFont = uIntf.iUIPlatformFont;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUIPlatformFont* object() const	{ return _iUIPlatformFont; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUIPlatformFont* iOther)
	{
		if (_iUIPlatformFont)
		{
			ICEUIPlatformFont* tmp = _iUIPlatformFont;
			_iUIPlatformFont = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUIPlatformFont = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUIPlatformFont* detach()
	{
		ICEUIPlatformFont* iIntf = _iUIPlatformFont;
		_iUIPlatformFont = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUIPlatformFont* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUIPlatformFont;
		if (_iUIPlatformFont)
		{
			_iUIPlatformFont->_vtbl->_addRef(_iUIPlatformFont);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUIPlatformFont*() const	{ return _iUIPlatformFont; }
	FORCEINLINE_WITHOUT_DEBUG ICEUIPlatformFont& operator*() const	{ return *_iUIPlatformFont; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUIPlatformFont == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUIPlatformFont* iOther) const	{ return (_iUIPlatformFont != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUIPlatformFont* iOther) const	{ return (_iUIPlatformFont == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUIPlatformFontRef& operator=(const CEComICEUIPlatformFontRef& other)	{ return operator=(other._iUIPlatformFont); }

	CEComICEUIPlatformFontRef& operator=(const ICEUIPlatformFont* iOther)
	{
		if (_iUIPlatformFont != iOther)
		{
			if (_iUIPlatformFont)
			{
				ICEUIPlatformFont* tmp = _iUIPlatformFont;
				_iUIPlatformFont = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUIPlatformFont = const_cast<ICEUIPlatformFont*>(iOther);
			if (_iUIPlatformFont)
			{
				_iUIPlatformFont->_vtbl->_addRef(_iUIPlatformFont);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUIPlatformFont** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUIPlatformFontRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUIPlatformFont && "has a com object reference. clear first.");
		return & _iUIPlatformFont; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUIPlatformFontRef& other)
	{
		bool result = false;
		if (_iUIPlatformFont)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUIPlatformFont);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_queryInterface(_iUIPlatformFont, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUIPlatformFont) { _iUIPlatformFont->_vtbl->_addRef(_iUIPlatformFont); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUIPlatformFont) { _iUIPlatformFont->_vtbl->_release(_iUIPlatformFont); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getValue(CEUIFontValueType type, INT32 *const valueOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_getValue(_iUIPlatformFont, type, valueOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStyle(CEPFFontStyle *const styleOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_getStyle(_iUIPlatformFont, styleOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFamilyName(struct ICEUIFontFamily* *const familyNameOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_getFamilyName(_iUIPlatformFont, familyNameOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createGlyphBuffer(struct ICEUIFontGlyphBuffer* *const iGBufferOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_createGlyphBuffer(_iUIPlatformFont, iGBufferOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult toString(struct ICEUString* *const iStrOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_toString(_iUIPlatformFont, iStrOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult dbgToString(const UTF16CHAR* *const pStr16Out)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_dbgToString(_iUIPlatformFont, pStr16Out) : CE_SILK_ERR_UNINITIALIZED; }

#if defined(_DEBUG) || !defined(NDEBUG)
	const UTF16CHAR* dbgToString() { const UTF16CHAR* ret=0; if (_iUIPlatformFont) { dbgToString(&ret); } return ret; }
#endif // _DEBUG || !NDEBUG

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawGlyph(struct ICEDXContext2D* csd, const CEPointBase* ptToDrawAt, UINT32 glyphIndex)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_drawGlyph(_iUIPlatformFont, csd, ptToDrawAt, glyphIndex) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokeGlyph(struct ICEDXContext2D* csd, const CEPointBase* ptToDrawAt, UINT32 glyphIndex)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_strokeGlyph(_iUIPlatformFont, csd, ptToDrawAt, glyphIndex) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawGlyphI(struct ICEVGContext* ivgc, const CEPointBase* ptToDrawAt, UINT32 glyphIndex, CEPFGlyphAdvance *const optAdvanceOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_drawGlyphI(_iUIPlatformFont, ivgc, ptToDrawAt, glyphIndex, optAdvanceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokeGlyphI(struct ICEVGContext* ivgc, const CEPointBase* ptToDrawAt, UINT32 glyphIndex, CEPFGlyphAdvance *const optAdvanceOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_strokeGlyphI(_iUIPlatformFont, ivgc, ptToDrawAt, glyphIndex, optAdvanceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult addGlyphPathI(struct ICEVGContext* ivgc, const CEPointBase* ptToDrawAt, UINT32 glyphIndex, CEPFGlyphAdvance *const optAdvanceOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_addGlyphPathI(_iUIPlatformFont, ivgc, ptToDrawAt, glyphIndex, optAdvanceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFaceInfo(CEPFFaceInfo *const faceInfoOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_getFaceInfo(_iUIPlatformFont, faceInfoOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphWidth(UINT32 glyphIndex, INT32 *const widthOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_getGlyphWidth(_iUIPlatformFont, glyphIndex, widthOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphMetrics(UINT32 glyphIndex, CEPFGlyphBBox* bboxOut, CEPFGlyphAdvance* advanceOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_getGlyphMetrics(_iUIPlatformFont, glyphIndex, bboxOut, advanceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphs16(const UTF16CHAR* pC16s, UINT32 nC16s, bool bNoSurrogatePairs, UINT32 *const optGlyphsOut, UINT32 *const optNGlyphsRef)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_getGlyphs16(_iUIPlatformFont, pC16s, nC16s, bNoSurrogatePairs, optGlyphsOut, optNGlyphsRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getICEPFRasterizerInfo(struct ICEPFFace* *const faceOut, struct ICEPFGlyphLoader* *const glyphLoaderOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_getICEPFRasterizerInfo(_iUIPlatformFont, faceOut, glyphLoaderOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getSmallCapsFont(struct ICEUIPlatformFont* *const smallCapsPfOut)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_getSmallCapsFont(_iUIPlatformFont, smallCapsPfOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getScaledFont(struct ICEUIPlatformFont* *const scaledPfOut, float scale)	{ return _iUIPlatformFont ? _iUIPlatformFont->_vtbl->_getScaledFont(_iUIPlatformFont, scaledPfOut, scale) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEUIPlatformFont* _iUIPlatformFont;
};

/*! @}
 * end of CEComICEUIPlatformFontRef
 */

/*! \defgroup CEComICEUICompositeFontRef CEComICEUICompositeFontRef
 * @{
 */

class CEComICEUICompositeFontRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUICompositeFontRef() : _iUICompositeFont(0) {}
	CEComICEUICompositeFontRef(ICEUICompositeFont* iOther) : _iUICompositeFont(0)
	{
		if (iOther)
		{
			_iUICompositeFont = iOther;
			_iUICompositeFont->_vtbl->_addRef(_iUICompositeFont);
		}
	}

	CEComICEUICompositeFontRef(const CEComICEUICompositeFontRef& other) : _iUICompositeFont(0)
	{
		if (other._iUICompositeFont)
		{
			_iUICompositeFont = other._iUICompositeFont;
			_iUICompositeFont->_vtbl->_addRef(_iUICompositeFont);
		}
	}

	~CEComICEUICompositeFontRef()
	{
		if (_iUICompositeFont)
		{
			ICEUICompositeFont* tmp = _iUICompositeFont;
			_iUICompositeFont = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUICompositeFont* iUICompositeFont;
				void* _ptr;
			} uIntf;
			uIntf.iUICompositeFont = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUICompositeFont, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUICompositeFont)
				{
					ICEUICompositeFont* tmp = _iUICompositeFont;
					_iUICompositeFont = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUICompositeFont = uIntf.iUICompositeFont;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUICompositeFont* object() const	{ return _iUICompositeFont; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUICompositeFont* iOther)
	{
		if (_iUICompositeFont)
		{
			ICEUICompositeFont* tmp = _iUICompositeFont;
			_iUICompositeFont = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUICompositeFont = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUICompositeFont* detach()
	{
		ICEUICompositeFont* iIntf = _iUICompositeFont;
		_iUICompositeFont = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUICompositeFont* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUICompositeFont;
		if (_iUICompositeFont)
		{
			_iUICompositeFont->_vtbl->_addRef(_iUICompositeFont);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUICompositeFont*() const	{ return _iUICompositeFont; }
	FORCEINLINE_WITHOUT_DEBUG ICEUICompositeFont& operator*() const	{ return *_iUICompositeFont; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUICompositeFont == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUICompositeFont* iOther) const	{ return (_iUICompositeFont != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUICompositeFont* iOther) const	{ return (_iUICompositeFont == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUICompositeFontRef& operator=(const CEComICEUICompositeFontRef& other)	{ return operator=(other._iUICompositeFont); }

	CEComICEUICompositeFontRef& operator=(const ICEUICompositeFont* iOther)
	{
		if (_iUICompositeFont != iOther)
		{
			if (_iUICompositeFont)
			{
				ICEUICompositeFont* tmp = _iUICompositeFont;
				_iUICompositeFont = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUICompositeFont = const_cast<ICEUICompositeFont*>(iOther);
			if (_iUICompositeFont)
			{
				_iUICompositeFont->_vtbl->_addRef(_iUICompositeFont);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUICompositeFont** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUICompositeFontRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUICompositeFont && "has a com object reference. clear first.");
		return & _iUICompositeFont; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUICompositeFontRef& other)
	{
		bool result = false;
		if (_iUICompositeFont)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUICompositeFont);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_queryInterface(_iUICompositeFont, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUICompositeFont) { _iUICompositeFont->_vtbl->_addRef(_iUICompositeFont); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUICompositeFont) { _iUICompositeFont->_vtbl->_release(_iUICompositeFont); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getValue(CEUIFontValueType type, INT32 *const valueOut)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_getValue(_iUICompositeFont, type, valueOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStyle(CEPFFontStyle *const styleOut)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_getStyle(_iUICompositeFont, styleOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFamilyName(struct ICEUIFontFamily* *const familyNameOut)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_getFamilyName(_iUICompositeFont, familyNameOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createGlyphBuffer(struct ICEUIFontGlyphBuffer* *const iGBufferOut)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_createGlyphBuffer(_iUICompositeFont, iGBufferOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult toString(struct ICEUString* *const iStrOut)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_toString(_iUICompositeFont, iStrOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult dbgToString(const UTF16CHAR* *const pStr16Out)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_dbgToString(_iUICompositeFont, pStr16Out) : CE_SILK_ERR_UNINITIALIZED; }

#if defined(_DEBUG) || !defined(NDEBUG)
	const UTF16CHAR* dbgToString() { const UTF16CHAR* ret=0; if (_iUICompositeFont) { dbgToString(&ret); } return ret; }
#endif // _DEBUG || !NDEBUG

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLang(struct ICEI18nLocale* *const langOut)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_getLang(_iUICompositeFont, langOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFirstPlatformFont(struct ICEUIPlatformFont* *const pfOut)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_getFirstPlatformFont(_iUICompositeFont, pfOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getPlatformFont(UTF32CHAR c32, bool bUseCharMapper, struct ICEUIPlatformFont* *const pfOut, UINT32 *const optGlyphIndexOut)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_getPlatformFont(_iUICompositeFont, c32, bUseCharMapper, pfOut, optGlyphIndexOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult mapChar32(UTF32CHAR c32, UTF32CHAR *const c32Out)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_mapChar32(_iUICompositeFont, c32, c32Out) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult measureUTF16CHARArray(const UTF16CHAR* pC16s, const UINT32 numOfChar16, CEUIFontOption* pOptionRef, INT32 *const measuredLengthOut)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_measureUTF16CHARArray(_iUICompositeFont, pC16s, numOfChar16, pOptionRef, measuredLengthOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawUTF16CHARArray(struct ICEDXContext2D* c2d, const CEPointBase* ptToDrawAt, const UTF16CHAR* pC16s, const UINT32 numOfChar16, CEUIFontOption* pOptionRef)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_drawUTF16CHARArray(_iUICompositeFont, c2d, ptToDrawAt, pC16s, numOfChar16, pOptionRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawUTF16CHARArrayI(struct ICEVGContext* ivgc, const CERGBAColor* glyphColor, const CEPointBase* ptToDrawAt, const UTF16CHAR* pC16s, const UINT32 numOfChar16, CEUIFontOption* pOptionRef)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_drawUTF16CHARArrayI(_iUICompositeFont, ivgc, glyphColor, ptToDrawAt, pC16s, numOfChar16, pOptionRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getTextMeasurer(const UTF16CHAR* pC16s, UINT32 numOfC16s, CEUIFontOption* pOptFontOptionRef, struct ICEUITextMeasurer* *const iTextMeasurerOut)	{ return _iUICompositeFont ? _iUICompositeFont->_vtbl->_getTextMeasurer(_iUICompositeFont, pC16s, numOfC16s, pOptFontOptionRef, iTextMeasurerOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEUICompositeFont* _iUICompositeFont;
};

/*! @}
 * end of CEComICEUICompositeFontRef
 */

/*! \defgroup CEComICEUIFontFamilyNameUtilityRef CEComICEUIFontFamilyNameUtilityRef
 * @{
 */

class CEComICEUIFontFamilyNameUtilityRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUIFontFamilyNameUtilityRef() : _iUIFontFamilyNameUtility(0) {}
	CEComICEUIFontFamilyNameUtilityRef(ICEUIFontFamilyNameUtility* iOther) : _iUIFontFamilyNameUtility(0)
	{
		if (iOther)
		{
			_iUIFontFamilyNameUtility = iOther;
			_iUIFontFamilyNameUtility->_vtbl->_addRef(_iUIFontFamilyNameUtility);
		}
	}

	CEComICEUIFontFamilyNameUtilityRef(const CEComICEUIFontFamilyNameUtilityRef& other) : _iUIFontFamilyNameUtility(0)
	{
		if (other._iUIFontFamilyNameUtility)
		{
			_iUIFontFamilyNameUtility = other._iUIFontFamilyNameUtility;
			_iUIFontFamilyNameUtility->_vtbl->_addRef(_iUIFontFamilyNameUtility);
		}
	}

	~CEComICEUIFontFamilyNameUtilityRef()
	{
		if (_iUIFontFamilyNameUtility)
		{
			ICEUIFontFamilyNameUtility* tmp = _iUIFontFamilyNameUtility;
			_iUIFontFamilyNameUtility = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUIFontFamilyNameUtility* iUIFontFamilyNameUtility;
				void* _ptr;
			} uIntf;
			uIntf.iUIFontFamilyNameUtility = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUIFontFamilyNameUtility, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUIFontFamilyNameUtility)
				{
					ICEUIFontFamilyNameUtility* tmp = _iUIFontFamilyNameUtility;
					_iUIFontFamilyNameUtility = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUIFontFamilyNameUtility = uIntf.iUIFontFamilyNameUtility;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUIFontFamilyNameUtility* object() const	{ return _iUIFontFamilyNameUtility; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUIFontFamilyNameUtility* iOther)
	{
		if (_iUIFontFamilyNameUtility)
		{
			ICEUIFontFamilyNameUtility* tmp = _iUIFontFamilyNameUtility;
			_iUIFontFamilyNameUtility = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUIFontFamilyNameUtility = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUIFontFamilyNameUtility* detach()
	{
		ICEUIFontFamilyNameUtility* iIntf = _iUIFontFamilyNameUtility;
		_iUIFontFamilyNameUtility = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUIFontFamilyNameUtility* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUIFontFamilyNameUtility;
		if (_iUIFontFamilyNameUtility)
		{
			_iUIFontFamilyNameUtility->_vtbl->_addRef(_iUIFontFamilyNameUtility);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUIFontFamilyNameUtility*() const	{ return _iUIFontFamilyNameUtility; }
	FORCEINLINE_WITHOUT_DEBUG ICEUIFontFamilyNameUtility& operator*() const	{ return *_iUIFontFamilyNameUtility; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUIFontFamilyNameUtility == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUIFontFamilyNameUtility* iOther) const	{ return (_iUIFontFamilyNameUtility != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUIFontFamilyNameUtility* iOther) const	{ return (_iUIFontFamilyNameUtility == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUIFontFamilyNameUtilityRef& operator=(const CEComICEUIFontFamilyNameUtilityRef& other)	{ return operator=(other._iUIFontFamilyNameUtility); }

	CEComICEUIFontFamilyNameUtilityRef& operator=(const ICEUIFontFamilyNameUtility* iOther)
	{
		if (_iUIFontFamilyNameUtility != iOther)
		{
			if (_iUIFontFamilyNameUtility)
			{
				ICEUIFontFamilyNameUtility* tmp = _iUIFontFamilyNameUtility;
				_iUIFontFamilyNameUtility = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUIFontFamilyNameUtility = const_cast<ICEUIFontFamilyNameUtility*>(iOther);
			if (_iUIFontFamilyNameUtility)
			{
				_iUIFontFamilyNameUtility->_vtbl->_addRef(_iUIFontFamilyNameUtility);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUIFontFamilyNameUtility** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUIFontFamilyNameUtilityRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUIFontFamilyNameUtility && "has a com object reference. clear first.");
		return & _iUIFontFamilyNameUtility; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUIFontFamilyNameUtilityRef& other)
	{
		bool result = false;
		if (_iUIFontFamilyNameUtility)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUIFontFamilyNameUtility);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUIFontFamilyNameUtility ? _iUIFontFamilyNameUtility->_vtbl->_queryInterface(_iUIFontFamilyNameUtility, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUIFontFamilyNameUtility) { _iUIFontFamilyNameUtility->_vtbl->_addRef(_iUIFontFamilyNameUtility); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUIFontFamilyNameUtility) { _iUIFontFamilyNameUtility->_vtbl->_release(_iUIFontFamilyNameUtility); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createSingleFontFamilyFromCStringLiteral(const UCHAR8* str, UINT32 numOfChar8, struct ICEUIFontFamily* *const fontFamilyOut)	{ return _iUIFontFamilyNameUtility ? _iUIFontFamilyNameUtility->_vtbl->_createSingleFontFamilyFromCStringLiteral(_iUIFontFamilyNameUtility, str, numOfChar8, fontFamilyOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createSingleFontFamilyFromUTF16Array(const UTF16CHAR* str16, UINT32 numOfChar16, struct ICEUIFontFamily* *const fontFamilyOut)	{ return _iUIFontFamilyNameUtility ? _iUIFontFamilyNameUtility->_vtbl->_createSingleFontFamilyFromUTF16Array(_iUIFontFamilyNameUtility, str16, numOfChar16, fontFamilyOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createFontFamilyFromCStringLiteral(const UCHAR8* str, UINT32 numOfChar8, struct ICEUIFontFamily* *const fontFamilyOut)	{ return _iUIFontFamilyNameUtility ? _iUIFontFamilyNameUtility->_vtbl->_createFontFamilyFromCStringLiteral(_iUIFontFamilyNameUtility, str, numOfChar8, fontFamilyOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createFontFamilyFromUTF16Array(const UTF16CHAR* str16, UINT32 numOfChar16, struct ICEUIFontFamily* *const fontFamilyOut)	{ return _iUIFontFamilyNameUtility ? _iUIFontFamilyNameUtility->_vtbl->_createFontFamilyFromUTF16Array(_iUIFontFamilyNameUtility, str16, numOfChar16, fontFamilyOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createFontFamilyFromFamilyArray(struct ICEUIFontFamily*const *const families, const UINT32 numOfFamilies, struct ICEUIFontFamily* *const fontFamilyOut)	{ return _iUIFontFamilyNameUtility ? _iUIFontFamilyNameUtility->_vtbl->_createFontFamilyFromFamilyArray(_iUIFontFamilyNameUtility, families, numOfFamilies, fontFamilyOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEUIFontFamilyNameUtility* _iUIFontFamilyNameUtility;
};

/*! @}
 * end of CEComICEUIFontFamilyNameUtilityRef
 */

/*! \defgroup CEComICEUICompositeFontFactoryRef CEComICEUICompositeFontFactoryRef
 * @{
 */

class CEComICEUICompositeFontFactoryRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUICompositeFontFactoryRef() : _iUICompositeFontFactory(0) {}
	CEComICEUICompositeFontFactoryRef(ICEUICompositeFontFactory* iOther) : _iUICompositeFontFactory(0)
	{
		if (iOther)
		{
			_iUICompositeFontFactory = iOther;
			_iUICompositeFontFactory->_vtbl->_addRef(_iUICompositeFontFactory);
		}
	}

	CEComICEUICompositeFontFactoryRef(const CEComICEUICompositeFontFactoryRef& other) : _iUICompositeFontFactory(0)
	{
		if (other._iUICompositeFontFactory)
		{
			_iUICompositeFontFactory = other._iUICompositeFontFactory;
			_iUICompositeFontFactory->_vtbl->_addRef(_iUICompositeFontFactory);
		}
	}

	~CEComICEUICompositeFontFactoryRef()
	{
		if (_iUICompositeFontFactory)
		{
			ICEUICompositeFontFactory* tmp = _iUICompositeFontFactory;
			_iUICompositeFontFactory = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUICompositeFontFactory* iUICompositeFontFactory;
				void* _ptr;
			} uIntf;
			uIntf.iUICompositeFontFactory = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUICompositeFontFactory, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUICompositeFontFactory)
				{
					ICEUICompositeFontFactory* tmp = _iUICompositeFontFactory;
					_iUICompositeFontFactory = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUICompositeFontFactory = uIntf.iUICompositeFontFactory;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUICompositeFontFactory* object() const	{ return _iUICompositeFontFactory; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUICompositeFontFactory* iOther)
	{
		if (_iUICompositeFontFactory)
		{
			ICEUICompositeFontFactory* tmp = _iUICompositeFontFactory;
			_iUICompositeFontFactory = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUICompositeFontFactory = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUICompositeFontFactory* detach()
	{
		ICEUICompositeFontFactory* iIntf = _iUICompositeFontFactory;
		_iUICompositeFontFactory = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUICompositeFontFactory* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUICompositeFontFactory;
		if (_iUICompositeFontFactory)
		{
			_iUICompositeFontFactory->_vtbl->_addRef(_iUICompositeFontFactory);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUICompositeFontFactory*() const	{ return _iUICompositeFontFactory; }
	FORCEINLINE_WITHOUT_DEBUG ICEUICompositeFontFactory& operator*() const	{ return *_iUICompositeFontFactory; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUICompositeFontFactory == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUICompositeFontFactory* iOther) const	{ return (_iUICompositeFontFactory != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUICompositeFontFactory* iOther) const	{ return (_iUICompositeFontFactory == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUICompositeFontFactoryRef& operator=(const CEComICEUICompositeFontFactoryRef& other)	{ return operator=(other._iUICompositeFontFactory); }

	CEComICEUICompositeFontFactoryRef& operator=(const ICEUICompositeFontFactory* iOther)
	{
		if (_iUICompositeFontFactory != iOther)
		{
			if (_iUICompositeFontFactory)
			{
				ICEUICompositeFontFactory* tmp = _iUICompositeFontFactory;
				_iUICompositeFontFactory = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUICompositeFontFactory = const_cast<ICEUICompositeFontFactory*>(iOther);
			if (_iUICompositeFontFactory)
			{
				_iUICompositeFontFactory->_vtbl->_addRef(_iUICompositeFontFactory);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUICompositeFontFactory** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUICompositeFontFactoryRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUICompositeFontFactory && "has a com object reference. clear first.");
		return & _iUICompositeFontFactory; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUICompositeFontFactoryRef& other)
	{
		bool result = false;
		if (_iUICompositeFontFactory)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUICompositeFontFactory);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUICompositeFontFactory ? _iUICompositeFontFactory->_vtbl->_queryInterface(_iUICompositeFontFactory, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUICompositeFontFactory) { _iUICompositeFontFactory->_vtbl->_addRef(_iUICompositeFontFactory); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUICompositeFontFactory) { _iUICompositeFontFactory->_vtbl->_release(_iUICompositeFontFactory); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createCompositeFontBySizeInPixels(const struct ICEI18nLocale* i18nLocale, struct ICEUIFontFamily* fontNameList, const CEPFFontStyle* style, INT32 sizeInPixels, bool bUseLastResortFallbackFont, struct ICEUICompositeFont* *const cfOut)	{ return _iUICompositeFontFactory ? _iUICompositeFontFactory->_vtbl->_createCompositeFontBySizeInPixels(_iUICompositeFontFactory, i18nLocale, fontNameList, style, sizeInPixels, bUseLastResortFallbackFont, cfOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createCompositeFontBySizeInPoints(const struct ICEI18nLocale* i18nLocale, struct ICEUIFontFamily* fontNameList, const CEPFFontStyle* style, CEFixed16d16 sizeInPoints, bool bUseLastResortFallbackFont, struct ICEUICompositeFont* *const cfOut)	{ return _iUICompositeFontFactory ? _iUICompositeFontFactory->_vtbl->_createCompositeFontBySizeInPoints(_iUICompositeFontFactory, i18nLocale, fontNameList, style, sizeInPoints, bUseLastResortFallbackFont, cfOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getTraitsInFamily(struct ICEUIFontFamily* fontFamily, CEPFFontStyle *const traits, UINT32 *const nTraitsRef)	{ return _iUICompositeFontFactory ? _iUICompositeFontFactory->_vtbl->_getTraitsInFamily(_iUICompositeFontFactory, fontFamily, traits, nTraitsRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult handleResetInterface()	{ return _iUICompositeFontFactory ? _iUICompositeFontFactory->_vtbl->_handleResetInterface(_iUICompositeFontFactory) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult registerCustomPlatformFontFace(const UINT8* bytes, UINT32 nBytes, struct ICEUIFontFamily* fontFamily, struct ICEPFFace* *const faceOut)	{ return _iUICompositeFontFactory ? _iUICompositeFontFactory->_vtbl->_registerCustomPlatformFontFace(_iUICompositeFontFactory, bytes, nBytes, fontFamily, faceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unregisterCustomPlatformFontFace(struct ICEUIFontFamily* fontFamily, struct ICEPFFace* pfFont)	{ return _iUICompositeFontFactory ? _iUICompositeFontFactory->_vtbl->_unregisterCustomPlatformFontFace(_iUICompositeFontFactory, fontFamily, pfFont) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createPlatformFontBySizeInPixels(struct ICEUIFontFamily* fontFamily, const CEPFFontStyle* style, INT32 sizeInPixels, struct ICEUIPlatformFont* *const pfFontOut)	{ return _iUICompositeFontFactory ? _iUICompositeFontFactory->_vtbl->_createPlatformFontBySizeInPixels(_iUICompositeFontFactory, fontFamily, style, sizeInPixels, pfFontOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createPlatformFontBySizeInPoints(struct ICEUIFontFamily* fontFamily, const CEPFFontStyle* style, CEFixed16d16 sizeInPoints, struct ICEUIPlatformFont* *const pfFontOut)	{ return _iUICompositeFontFactory ? _iUICompositeFontFactory->_vtbl->_createPlatformFontBySizeInPoints(_iUICompositeFontFactory, fontFamily, style, sizeInPoints, pfFontOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEUICompositeFontFactory* _iUICompositeFontFactory;
};

/*! @}
 * end of CEComICEUICompositeFontFactoryRef
 */

/*! \defgroup CEComICEUIFontServiceRef CEComICEUIFontServiceRef
 * @{
 */

class CEComICEUIFontServiceRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEUIFontServiceRef() : _iUIFontService(0) {}
	CEComICEUIFontServiceRef(ICEUIFontService* iOther) : _iUIFontService(0)
	{
		if (iOther)
		{
			_iUIFontService = iOther;
			_iUIFontService->_vtbl->_addRef(_iUIFontService);
		}
	}

	CEComICEUIFontServiceRef(const CEComICEUIFontServiceRef& other) : _iUIFontService(0)
	{
		if (other._iUIFontService)
		{
			_iUIFontService = other._iUIFontService;
			_iUIFontService->_vtbl->_addRef(_iUIFontService);
		}
	}

	~CEComICEUIFontServiceRef()
	{
		if (_iUIFontService)
		{
			ICEUIFontService* tmp = _iUIFontService;
			_iUIFontService = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEUIFontService* iUIFontService;
				void* _ptr;
			} uIntf;
			uIntf.iUIFontService = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEUIFontService, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iUIFontService)
				{
					ICEUIFontService* tmp = _iUIFontService;
					_iUIFontService = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iUIFontService = uIntf.iUIFontService;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEUIFontService* object() const	{ return _iUIFontService; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEUIFontService* iOther)
	{
		if (_iUIFontService)
		{
			ICEUIFontService* tmp = _iUIFontService;
			_iUIFontService = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iUIFontService = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEUIFontService* detach()
	{
		ICEUIFontService* iIntf = _iUIFontService;
		_iUIFontService = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEUIFontService* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iUIFontService;
		if (_iUIFontService)
		{
			_iUIFontService->_vtbl->_addRef(_iUIFontService);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEUIFontService*() const	{ return _iUIFontService; }
	FORCEINLINE_WITHOUT_DEBUG ICEUIFontService& operator*() const	{ return *_iUIFontService; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iUIFontService == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEUIFontService* iOther) const	{ return (_iUIFontService != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEUIFontService* iOther) const	{ return (_iUIFontService == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEUIFontServiceRef& operator=(const CEComICEUIFontServiceRef& other)	{ return operator=(other._iUIFontService); }

	CEComICEUIFontServiceRef& operator=(const ICEUIFontService* iOther)
	{
		if (_iUIFontService != iOther)
		{
			if (_iUIFontService)
			{
				ICEUIFontService* tmp = _iUIFontService;
				_iUIFontService = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iUIFontService = const_cast<ICEUIFontService*>(iOther);
			if (_iUIFontService)
			{
				_iUIFontService->_vtbl->_addRef(_iUIFontService);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEUIFontService** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEUIFontServiceRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iUIFontService && "has a com object reference. clear first.");
		return & _iUIFontService; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEUIFontServiceRef& other)
	{
		bool result = false;
		if (_iUIFontService)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iUIFontService);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iUIFontService ? _iUIFontService->_vtbl->_queryInterface(_iUIFontService, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iUIFontService) { _iUIFontService->_vtbl->_addRef(_iUIFontService); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iUIFontService) { _iUIFontService->_vtbl->_release(_iUIFontService); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createInstance(struct ICEUnknown* pUnkOuter, UINT32 iid, void* *const ppv)	{ return _iUIFontService ? _iUIFontService->_vtbl->_createInstance(_iUIFontService, pUnkOuter, iid, ppv) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initService(CEUIFontServiceInitArg* arg)	{ return _iUIFontService ? _iUIFontService->_vtbl->_initService(_iUIFontService, arg) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFontFamilyNameUtility(struct ICEUIFontFamilyNameUtility* *const ffNameUtilOut)	{ return _iUIFontService ? _iUIFontService->_vtbl->_getFontFamilyNameUtility(_iUIFontService, ffNameUtilOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCompositeFontFactory(struct ICEUICompositeFontFactory* *const ccfOut)	{ return _iUIFontService ? _iUIFontService->_vtbl->_getCompositeFontFactory(_iUIFontService, ccfOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setOptionsBit(eCEUIFontServiceOptionsBit optionBit, bool v)	{ return _iUIFontService ? _iUIFontService->_vtbl->_setOptionsBit(_iUIFontService, optionBit, v) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getOptionsBit(eCEUIFontServiceOptionsBit optionBit, bool *const vOut)	{ return _iUIFontService ? _iUIFontService->_vtbl->_getOptionsBit(_iUIFontService, optionBit, vOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setRasterizerParams(const UINT32* params, UINT32 numOfParams)	{ return _iUIFontService ? _iUIFontService->_vtbl->_setRasterizerParams(_iUIFontService, params, numOfParams) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getRasterizerParams(const UINT32**params, UINT32*const numOfParams)	{ return _iUIFontService ? _iUIFontService->_vtbl->_getRasterizerParams(_iUIFontService, params, numOfParams) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult shutdownService()	{ return _iUIFontService ? _iUIFontService->_vtbl->_shutdownService(_iUIFontService) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEUIFontService* _iUIFontService;
};

/*! @}
 * end of CEComICEUIFontServiceRef
 */

#endif // __cplusplus

#endif
