///////////////////////////////////////////////////////////////////////////////
// Copyright 2004,2005,2006,2007,2009 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef CEVariant_h
#define CEVariant_h

///////////////////////////////////////////////////////////////////////////////
// Includes
///////////////////////////////////////////////////////////////////////////////

#include "CESysDefs.h"
#include "CEApiUnknown.h"
#include "CETypes.h"
#include "CESysMemory.h"

struct ICEScriptable;
struct ICEUString;

#ifdef __cplusplus
extern "C"
{
#endif

	////////////////////////////////////
	//  Variant Types for Scripting
	////////////////////////////////////

	enum eCEVariantType
	{
		eCEVariantType_Undefined = 0,
		eCEVariantType_Boolean,
		eCEVariantType_INT32,
		eCEVariantType_UINT64,
		eCEVariantType_Float,
		eCEVariantType_Number,		// double
		eCEVariantType_Color,
		eCEVariantType_PointF,
		eCEVariantType_ICEUnknown,	// all interface
		eCEVariantType_VoidPtr,		// NOT interface

		// the following types will be obsolete soon. don't use them!
		eCEVariantType_Null,
		//eCEVariantType_Object,
		eCEVariantType_String,
		eCEVariantType_Scriptable,
		eCEVariantType_ScmPtr,
	};

	//! Check whether IUnknown has refcount or not.
	//! Some IUnknown has no interface such as ICESymbol
	//! When set/get some IUnknown to CEVariant, 'addRef' and 'release' are unnecessary. 
	#define IUNK_HAS_INTERFACE(p)  (! ((reinterpret_cast<INT_PTR>(p)) & 1))

	//! We strongly recommend you to use CESmartVariant (C++ class) in your code.
	//! CEVariant is supposed to be used only in interface argments, where C++ classes cannot be used.
	struct CEVariant
	{
		eCEVariantType _type;
		union
		{
			bool	_boolValue;
			INT32	_int32Value;
			UINT64	_uint64Value;
			float	_floatValue;
			double	_doubleValue;
			CERGBAColor _colorValue;
			CEPointF	_pointfValue;
			struct
			{
				void*	_iUnk;
				UINT32	_iid;
			} _iUnknownValue;
			void* _voidPtrValue;

			// the following types will be obsolete soon. don't use them!
			ICEUString* _stringValue;
			ICEScriptable* _scriptableValue;
			void* _scmPtrValue;
		};
	};

	// accsessor
	// copy
	#define  CEVARIANT_COPY(dst, src)		\
		CESysCopyMemory(dst, sizeof (struct CEVariant), src, sizeof (struct CEVariant))

	// get type
	#define CEVARIANT_GET_TYPE(variant)   \
		(variant)._type

	// set type
	#define CEVARIANT_SET_TYPE(v, t)		\
		(v)._type = t

	// set undefined
	#define CEVARIANT_SET_UNDEFINED(v)			\
		CEVARIANT_SET_TYPE(v, eCEVariantType_Undefined)

	#define CEVARIANT_SET_NULL(v)				\
		CEVARIANT_SET_TYPE(v, eCEVariantType_Null)

	// set value
	#define CEVARIANT_SET_BOOLEAN(v, b) 			\
		{					\
			CEVARIANT_SET_TYPE(v, eCEVariantType_Boolean);	\
			(v)._boolValue = (b);			\
		}

	#define CEVARIANT_SET_NUMBER(v, n) 			\
		{					\
			CEVARIANT_SET_TYPE(v, eCEVariantType_Number);	\
			(v)._doubleValue = (n);			\
		}

	#define CEVARIANT_SET_STRING(v, s) 			\
		{					\
			CEVARIANT_SET_TYPE(v, eCEVariantType_String);	\
			(v)._stringValue = (s);			\
		}

	#define CEVARIANT_SET_INT32(v, i)   \
		{					\
			CEVARIANT_SET_TYPE(v, eCEVariantType_INT32);	\
			(v)._int32Value = (i);              \
		}

	#define CEVARIANT_SET_UINT64(v, i)   \
		{					\
			CEVARIANT_SET_TYPE(v, eCEVariantType_UINT64);	\
			(v)._uint64Value = (i);              \
		}

	#define CEVARIANT_SET_SCRIPTABLE(v, scr)   \
		{					\
			CEVARIANT_SET_TYPE(v, eCEVariantType_Scriptable);	\
			(v)._scriptableValue = (scr);              \
		}

	#define CEVARIANT_SET_ICEUNKNOWN(v, iUnk, iId)   \
		{					\
			CEVARIANT_SET_TYPE(v, eCEVariantType_ICEUnknown);	\
			(v)._iUnknownValue._iUnk = (iUnk);              \
			(v)._iUnknownValue._iid = (iId);              \
		}

	#define CEVARIANT_SET_SCMPTR(v, scmptr)   \
		{					\
			CEVARIANT_SET_TYPE(v, eCEVariantType_ScmPtr);	\
			(v)._scmPtrValue = (scmptr);              \
		}

	#define CEVARIANT_SET_VOIDPTR(v, voidptr)   \
		{					\
			CEVARIANT_SET_TYPE(v, eCEVariantType_VoidPtr);	\
			(v)._voidPtrValue = (voidptr);              \
		}

	// is type
	#define CEVARIANT_IS_TYPE(v, type)   \
		((v)._type == (type))


	////////////////////////////////////
	//  Variant Args for Scripting
	////////////////////////////////////
	struct CEVariantArgs
	{
		CEVariant*          _inArgs;
		UINT32				_numOfInArgs;
		CEVariant*			_outArgs;
		UINT32				_numOfOutArgs;
	};


#ifdef __cplusplus
}
#endif // __cplusplus


//----------------------------------------------------------------
// Smart variant
//----------------------------------------------------------------

#ifdef __cplusplus

//! Smart variant
//!
//! Why we need?
//! ------------
//! since CEVariant is C structure, you always have to take care of reference count
//! of the interface pointer stored in the variant.
//! CESmartVariant works as a smart pointer when it contains an interface pointer.
//!
//! Caution!!!
//! ----------
//! Even if CEVariant can contain eCEVariantType_String, eCEVariantType_Scriptable,
//! and eCEVariantType_ScmPtr, CESmartVariant does not use those types.
//! CESmartVariant contains ALL COM interface as eCEVariantType_ICEUnknown.
//! You can check the actual interface type by _iUnknownValue._iid.
//!
//! Usage
//! -----
//! 
//! * initialize
//!
//! CESmartVariant var0;								// typed as undefined
//! CESmartVariant var1(true);
//! CESmartVariant var2(123);
//! CESmartVariant var3(456.789);
//! CESmartVariant var4(CEComIID_ICEMyObj, iMyObj);		// addRef'ed
//! CESmartVariant var5(var4);							// addRef'ed
//!
//! * setter
//!
//! CESmartVariant var10;
//! var10 = false;
//! var10 = 123;
//! var10 = -456.78;
//! var10 = var1;
//! var10.set(CEComIID_ICEMyObj, iMyObj);				// addRef'ed
//!
//! * getter
//! note that getter may fail when the value's actual type is different
//! from the destination variable.
//!
//! CEHResult err;
//! CESmartVariant var20(true);
//! bool boolVar;
//! double doubleVar;
//! err = var20.get(boolVar);								// success
//! err = var20.get(doubleVar);								// fail
//! 
//! CESmartVariant var21(CEComIID_ICEMyObj, iMyObj);
//! CEComICEMyObjRef iMyObjRef;
//! CEComICEFooRef iFooRef;
//! err = var21.get(CEComIID_ICEMyObj, (void**)&iMyObjRef);			// success. addRef'ed
//! err = var21.get(CEComIID_ICEFoo, (void**)&iFooRef);				// fail
//!
//! // success if the object implements ICEFoo. otherwise fail.
//! err = var21.getByQueryInterface(CEComIID_ICEFoo, &iFooRef);
class CESmartVariant : public CEVariant
{
public:

	//////////////////////////////
	// constructor / destructor
	//////////////////////////////

	CESmartVariant() 
	{
		_type = eCEVariantType_Undefined;
		_int32Value = 0;
		_uint64Value = 0;
	}

	CESmartVariant(bool val)
	{
		_type = eCEVariantType_Boolean;
		_boolValue = val;
	}

	CESmartVariant(INT32 val)
	{
		_type = eCEVariantType_INT32;
		_int32Value = val;
	}

	CESmartVariant(UINT64 val)
	{
		_type = eCEVariantType_UINT64;
		_uint64Value = val;
	}

	CESmartVariant(float val)
	{
		_type = eCEVariantType_Float;
		_floatValue = val;
	}

	CESmartVariant(double val)
	{
		_type = eCEVariantType_Number;
		_doubleValue = val;
	}

	CESmartVariant(CERGBAColor& val)
	{
		_type = eCEVariantType_Color;
		_colorValue = val;
	}

	CESmartVariant(CEPointF& val)
	{
		_type = eCEVariantType_PointF;
		_pointfValue = val;
	}

	CESmartVariant(UINT32 iid, void* iUnk)
	{
		_type = eCEVariantType_ICEUnknown;
		_iUnknownValue._iid = iid;
		_iUnknownValue._iUnk = iUnk;

		if(iUnk && IUNK_HAS_INTERFACE(iUnk))
		{
			ICEUnknown* iUnknown = static_cast<ICEUnknown*>(iUnk);
			iUnknown->_vtbl->_addRef(iUnknown);
		}
	}

	CESmartVariant(const CEVariant& other)
	{
		CESysCopyMemory(this, sizeof(CESmartVariant), &other, sizeof(CESmartVariant));
		if((_type == eCEVariantType_ICEUnknown) && (_iUnknownValue._iUnk) && IUNK_HAS_INTERFACE(_iUnknownValue._iUnk))
		{
			ICEUnknown* iUnknown = static_cast<ICEUnknown*>(_iUnknownValue._iUnk);
			iUnknown->_vtbl->_addRef(iUnknown);
		}
	}

	CESmartVariant(const CESmartVariant& other)
	{
		CESysCopyMemory(this, sizeof(CESmartVariant), &other, sizeof(CESmartVariant));
		if((_type == eCEVariantType_ICEUnknown) && (_iUnknownValue._iUnk) && IUNK_HAS_INTERFACE(_iUnknownValue._iUnk))
		{
			ICEUnknown* iUnknown = static_cast<ICEUnknown*>(_iUnknownValue._iUnk);
			iUnknown->_vtbl->_addRef(iUnknown);
		}
	}

	~CESmartVariant()
	{
		_release();
	}

	//////////////////////////////
	// setter
	//////////////////////////////

	CESmartVariant& operator=(const bool val)
	{
		_release();

		_type = eCEVariantType_Boolean;
		_boolValue = val;

		return(*this);
	}

	CESmartVariant& operator=(const INT32 val)
	{
		_release();

		_type = eCEVariantType_INT32;
		_int32Value = val;

		return(*this);
	}

	CESmartVariant& operator=(const UINT64 val)
	{
		_release();

		_type = eCEVariantType_UINT64;
		_uint64Value = val;

		return(*this);
	}

	CESmartVariant& operator=(const float val)
	{
		_release();

		_type = eCEVariantType_Float;
		_floatValue = val;

		return(*this);
	}

	CESmartVariant& operator=(const double val)
	{
		_release();

		_type = eCEVariantType_Number;
		_doubleValue = val;

		return(*this);
	}

	CESmartVariant& operator=(const CERGBAColor& val)
	{
		_release();

		_type = eCEVariantType_Color;
		_colorValue = val;

		return(*this);
	}

	CESmartVariant& operator=(const CEPointF& val)
	{
		_release();

		_type = eCEVariantType_PointF;
		_pointfValue = val;

		return(*this);
	}

	CESmartVariant& operator=(const CEVariant& other)
	{
		_release();

		CESysCopyMemory(this, sizeof(CESmartVariant), &other, sizeof(CESmartVariant));
		if((_type == eCEVariantType_ICEUnknown) && (_iUnknownValue._iUnk) && IUNK_HAS_INTERFACE(_iUnknownValue._iUnk))
		{
			ICEUnknown* iUnknown = static_cast<ICEUnknown*>(_iUnknownValue._iUnk);
			iUnknown->_vtbl->_addRef(iUnknown);
		}

		return(*this);
	}

	CESmartVariant& operator=(const CESmartVariant& other)
	{
		_release();

		CESysCopyMemory(this, sizeof(CESmartVariant), &other, sizeof(CESmartVariant));
		if((_type == eCEVariantType_ICEUnknown) && (_iUnknownValue._iUnk) && IUNK_HAS_INTERFACE(_iUnknownValue._iUnk))
		{
			ICEUnknown* iUnknown = static_cast<ICEUnknown*>(_iUnknownValue._iUnk);
			iUnknown->_vtbl->_addRef(iUnknown);
		}

		return(*this);
	}

	//! you cannot set an interface pointer by "=" operator. use "set()".
	void set(UINT32 iid, void* iUnk)
	{
		_release();

		_type = eCEVariantType_ICEUnknown;
		_iUnknownValue._iid = iid;
		_iUnknownValue._iUnk = iUnk;

		if(iUnk && IUNK_HAS_INTERFACE(iUnk))
		{
			ICEUnknown* iUnknown = static_cast<ICEUnknown*>(iUnk);
			iUnknown->_vtbl->_addRef(iUnknown);
		}
	}

	///////////////////////////////////////////////////////////////
	// getter
	// note that getter may fail when the value's actual type is different
	// from the destination variable.
	///////////////////////////////////////////////////////////////

	CEHResult get(bool& valOut)
	{
		CEHResult err = CE_SILK_ERR_BADARGS;

		if(_type == eCEVariantType_Boolean)
		{
			valOut = _boolValue;
			err = CE_S_OK;
		}

		return(err);
	}

	CEHResult get(INT32& valOut)
	{
		CEHResult err = CE_SILK_ERR_BADARGS;

		if(_type == eCEVariantType_INT32)
		{
			valOut = _int32Value;
			err = CE_S_OK;
		}

		return(err);
	}

	CEHResult get(UINT64& valOut)
	{
		CEHResult err = CE_SILK_ERR_BADARGS;

		if(_type == eCEVariantType_UINT64)
		{
			valOut = _uint64Value;
			err = CE_S_OK;
		}

		return(err);
	}

	CEHResult get(float& valOut)
	{
		CEHResult err = CE_SILK_ERR_BADARGS;

		if(_type == eCEVariantType_Float)
		{
			valOut = _floatValue;
			err = CE_S_OK;
		}

		return(err);
	}

	CEHResult get(double& valOut)
	{
		CEHResult err = CE_SILK_ERR_BADARGS;

		if(_type == eCEVariantType_Number)
		{
			valOut = _doubleValue;
			err = CE_S_OK;
		}

		return(err);
	}

	CEHResult get(CERGBAColor& valOut)
	{
		CEHResult err = CE_SILK_ERR_BADARGS;

		if(_type == eCEVariantType_Color)
		{
			valOut = _colorValue;
			err = CE_S_OK;
		}

		return(err);
	}

	CEHResult get(CEPointF& valOut)
	{
		CEHResult err = CE_SILK_ERR_BADARGS;

		if(_type == eCEVariantType_PointF)
		{
			valOut = _pointfValue;
			err = CE_S_OK;
		}

		return(err);
	}

	//! when you strongly assume this variant to hold a specific typed interface,
	//! this method is recommended. but when you are not sure the interface type,
	//! we recommend you to use getByQueryInterface().
	//!
	//! the returned interface is addRef'ed.
	CEHResult get(UINT32 iid, void** const piUnkOut)
	{
		CEHResult err = CE_SILK_ERR_BADARGS;

		if(piUnkOut && (_type == eCEVariantType_ICEUnknown) && (_iUnknownValue._iid == iid))
		{
			*piUnkOut = _iUnknownValue._iUnk;
			if(*piUnkOut && IUNK_HAS_INTERFACE(*piUnkOut))
			{
				ICEUnknown* iUnknown = static_cast<ICEUnknown*>(*piUnkOut);
				iUnknown->_vtbl->_addRef(iUnknown);
			}

			err = CE_S_OK;
		}

		return(err);
	}

	//! the returned interface is addRef'ed.
	CEHResult getByQueryInterface(UINT32 iid, void** const piUnkOut)
	{
		CEHResult err = CE_SILK_ERR_BADARGS;

		if(piUnkOut && (_type == eCEVariantType_ICEUnknown))
		{
			if(_iUnknownValue._iUnk && IUNK_HAS_INTERFACE(_iUnknownValue._iUnk))
			{
				ICEUnknown* iUnknown = static_cast<ICEUnknown*>(_iUnknownValue._iUnk);
				err = iUnknown->_vtbl->_queryInterface(iUnknown, iid, piUnkOut);
			}
			else
			{
				*piUnkOut = NULL;
				err = CE_S_OK;
			}
		}

		return(err);
	}

private:
	//////////////////////////////
	// misc
	//////////////////////////////

	void _release()
	{
		if((_type == eCEVariantType_ICEUnknown) && (_iUnknownValue._iUnk) && IUNK_HAS_INTERFACE(_iUnknownValue._iUnk))
		{
			ICEUnknown* iUnknown = static_cast<ICEUnknown*>(_iUnknownValue._iUnk);
			iUnknown->_vtbl->_release(iUnknown);
		} 
	}
};

#endif // __cplusplus

#endif  //CEVariant_h
