///////////////////////////////////////////////////////////////////////////////
// Copyright 2008,2009 Sony Corporation
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
// 
///////////////////////////////////////////////////////////////////////////////
#include "WebKitSilkPrefix.h"

#include "CEWebKitImpl.h"
#include "CEWebKitFrameWindow.h"

#include "CEWebKit.h"
#include "SystemTime.h"

#include "InitializeThreading.h"
#include "MainThread.h"
#include "PageGroup.h"
#include "Cache.h"
#include "JSDOMWindow.h"
#include "ResourceHandleManagerCore.h"
#include <wtf/CurrentTime.h>
#include "PluginDatabase.h"
#include "ThreadGlobalData.h"
#include "GCController.h"
#if ENABLE(DATABASE)
#include "DatabaseTracker.h"
#endif
#include "PageCache.h"
#include "FontCache.h"
#include "TextBreakIterator.h"
#include <unicode/ubrk.h>
#include "ICEVG.h"

#if USE(ACCELERATED_COMPOSITING)
#include "GraphicsLayerSilk.h"
#endif //#if USE(ACCELERATED_COMPOSITING)

const UINT32 _defaultCacheMinDeadBytes = 0;
const UINT32 _defaultCacheMaxDeadBytes = 8192 * 1024;
const UINT32 _defaultCacheTotalBytes = 8192 * 1024;

const UINT32 _emptyCacheMinDeadBytes = 1;
const UINT32 _emptyCacheMaxDeadBytes = 1;
const UINT32 _emptyCacheTotalBytes = 2;

static inline void _shutdownWebCoreStaticData()
{
	// cleanup TextBreakIterators.
	{
		const UChar str[] = {'a', 0};
		WebCore::TextBreakIterator* itrs[] = {
			WebCore::characterBreakIterator(str, 1),
			WebCore::cursorMovementIterator(str, 1),
			WebCore::wordBreakIterator(str, 1),
			WebCore::lineBreakIterator(str, 1),
			WebCore::sentenceBreakIterator(str, 1)
		};
		int sz = sizeof(itrs)/sizeof(WebCore::TextBreakIterator*);
		int i = 0;
		for (i = 0; i < sz; i++)
		{
			if (itrs[i])
				ubrk_close(itrs[i]);
		}
	}
}

CEWebKitImpl* CEWebKitImpl::_self = NULL;
CEWebKitImpl::CEWebKitImpl() :
	_sharedTimerFiredFunction(NULL),
	_thread(NULL),
	_shortTermTimeoutCount(0),
	_lockingOutInLowMemory(false),
	_timeoutReservedAfterLowMemory(false),
	_initFlagOfPlugin(false)
{
	CEASSERT(NULL == _self);
	_self = this;
}

CEHResult CEWebKitImpl::init(ICEHtmlWebKitEnvironment* iEnvironment, ICENetwork* iCENetwork)
{
	JSC::initializeThreading();
	WebCore::PageGroup::setShouldTrackVisitedLinks(true);
	
	_iEnvironmentRef = iEnvironment;

	_cacheMinDeadBytes = _defaultCacheMinDeadBytes;
	UINT32 maxResourceCacheSize = 0;
	CEHResult hr = _iEnvironmentRef.getMaxResourceCacheSize(&maxResourceCacheSize);
	if (CESucceeded(hr))
	{
		_cacheMaxDeadBytes = maxResourceCacheSize;
		_cacheTotalBytes = maxResourceCacheSize;
	}
	else
	{
		_cacheMaxDeadBytes = _defaultCacheMaxDeadBytes;
		_cacheTotalBytes = _defaultCacheTotalBytes;
	}
	WebCore::cache()->setDisabled(false);
	WebCore::cache()->setCapacities(_cacheMinDeadBytes, _cacheMaxDeadBytes, _cacheTotalBytes);
	WebCore::cache()->setDeadDecodedDataDeletionInterval(1);
	
	hr = _iEnvironmentRef.getCEUThread(&_thread);
	if (CESucceeded(hr)) 
	{
        WTF::setMainCEUThreadRef(_thread);
	}
	if (CESucceeded(hr))
	{
		hr = CEComGetThreadContext(CEComIID_ICEHtmlOneshotTimerManager, reinterpret_cast<void**>(&_iTimerManagerRef));
	}
	if (CESucceeded(hr))
	{
		CEComICEVGFactoryRef vgFactory;
		hr = CEComGetThreadContext(CEComIID_ICEVGFactory, reinterpret_cast<void**>(&vgFactory));
		if (CESucceeded(hr))
		{
			extern CEHResult VGSupport_init(ICEVGFactory* );
			hr =  VGSupport_init(vgFactory);
		}
	}

#if USE(ACCELERATED_COMPOSITING)
	if (CESucceeded(hr))
	{
		CEComICEMRLFactoryRef mrlFactory;
		hr = CEComGetThreadContext(CEComIID_ICEMRLFactory, reinterpret_cast<void**>(&mrlFactory));
		if (CESucceeded(hr))
			hr = WebCore::GraphicsLayerSilk::Env::init(mrlFactory);
	}
#endif //#if USE(ACCELERATED_COMPOSITING)

	if (CESucceeded(hr))
	{
		extern CEHResult CENetworkSupport_init(ICENetwork*);
		hr = CENetworkSupport_init(iCENetwork);
	}

#if ENABLE(DATABASE)
    if (CESucceeded(hr)) 
    {
        CEComICEUStringRef path;
        hr = _iEnvironmentRef.getDatabasePath(&path);
        if (CESucceeded(hr)) 
        {
            WebCore::String str;
            hr = str.fromICEUString(path);
            if (CESucceeded(hr)) 
                WebCore::DatabaseTracker::tracker().setDatabaseDirectoryPath(str);
        }
    }
#endif

	if (CESucceeded(hr))
	{
		if (!WebCore::JSDOMWindow::commonJSGlobalData())
		{
			hr = CE_SILK_ERR_MEMERR;
		}
	}

	return hr;
}

CEHResult CEWebKitImpl::shutdown()
{
	CEHResult hr = CE_SILK_ERR_BUSY;

	WebCore::pageCache()->releaseAutoreleasedPagesNow();
	WebCore::gcController().garbageCollectNow();

	if (!_shortTermTimeoutCount)
	{
		{
			extern void CENetworkSupport_shutdown();
			CENetworkSupport_shutdown();
		}

		stopSharedTimer();

		WebCore::cache()->setCapacities(_emptyCacheMinDeadBytes, _emptyCacheMaxDeadBytes, _emptyCacheTotalBytes);

		{
			extern void CEUIFontSupport_shutdown();
			CEUIFontSupport_shutdown();
		}

#if USE(ACCELERATED_COMPOSITING)
		{
			WebCore::GraphicsLayerSilk::Env::shutdown();
		}
#endif //#if USE(ACCELERATED_COMPOSITING)
		{
			extern void VGSupport_shutdown();
			VGSupport_shutdown();
		}

#if USE(ACCELERATED_COMPOSITING)
		{
			CEMRL_FPS_18278_SHUTDOWN;
		}
#endif //#if USE(ACCELERATED_COMPOSITING)

		_iEnvironmentRef = NULL;
		_iTimerManagerRef = NULL;
		_jsPrivilegeExtObjectListener = NULL;
		_initFlagOfPlugin = false;

		_shutdownWebCoreStaticData();
		{
			// see ThreadGlobalData& threadGlobalData()
#if ENABLE(WORKERS)
			CEASSERT("FIXME: but how to fix this.");
#else //#if ENABLE(WORKERS)
			delete &WebCore::threadGlobalData();
#endif //#if ENABLE(WORKERS)
		}
		hr = CE_S_OK;
	}
	return hr;
}

CEHResult CEWebKitImpl::createWebKitFrameWindow(ICEWebKitFrameWindow** iFrameWindowOut)
{
	CEHResult hr = CE_SILK_ERR_BADARGS;
	if (iFrameWindowOut)
	{
		CEWebKitFrameWindow* frameWindow = new CEWebKitFrameWindow();
		if (frameWindow)
		{
			CEComICEWebKitFrameWindowRef iWebKitFrameWindowRef = CEWebKitFrameWindow::toICEWebKitFrameWindow(frameWindow);
			*iFrameWindowOut = iWebKitFrameWindowRef.detach();
			hr = CE_S_OK;
		}
		else
		{
			hr = CE_SILK_ERR_MEMERR;
		}
	}
	return hr;
}

CEHResult CEWebKitImpl::pruneCache()
{
    WebCore::Cache* cache = WebCore::cache();
    if (cache)
    {
        cache->setCapacities(_emptyCacheMinDeadBytes, _emptyCacheMaxDeadBytes, _emptyCacheTotalBytes);
        cache->setCapacities(_cacheMinDeadBytes, _cacheMaxDeadBytes, _cacheTotalBytes);
    }

    WebCore::FontCache* fontCache = WebCore::fontCache();
    if (fontCache)
    {
        size_t numFontCache = fontCache->fontDataCount();
        size_t numInactiveFontCache = fontCache->inactiveFontDataCount();
        fontCache->purgeInactiveFontData(numInactiveFontCache);
    }

	return CE_S_OK;
}

CEHResult CEWebKitImpl::setProxy(bool isProxyEnable, ICEUString* host, UINT16 port)
{
	CEASSERT(0 && "CEWebKitImpl::setProxy is notimplemented!! \n");
	return CE_SILK_ERR_NOTIMPLEMENTED;
}

CEHResult CEWebKitImpl::setCookieEnable(bool isCookieEnable)
{
	CEHResult hr = CE_S_OK;
	hr = WebCore::ResourceHandleManager::setCookieEnable(isCookieEnable);
	return hr;
}

CEHResult CEWebKitImpl::deleteAllCookies()
{
	CEHResult hr = CE_S_OK;
#if USE(CENETWORK) || USE(XAI)
	hr = WebCore::ResourceHandleManager::deleteAllCookies();
#endif
	return hr;
}

CEHResult CEWebKitImpl::setAcceptLanguage(ICEUString* acceptLanguage)
{
	return CE_S_OK;
}

CEHResult CEWebKitImpl::setCustomHeaderFileds(UINT16 numFields, ICEUString* const* fieldNames, ICEUString* const* fieldValues)
{
	CEHResult hr = CE_S_OK;
	return hr;
}

void CEWebKitImpl::_messageHandlerProc(void* msg)
{
	if (_sharedTimerFiredFunction && msg == _sharedTimerFiredFunction)
	{
		--_shortTermTimeoutCount;
		if (checkLockingOutInLowMemory())
		{
			reserveTimeoutAfterLowMemory();
		}
		else
		{
			(*_sharedTimerFiredFunction)();
		}
	}
}

CEHResult CEWebKitImpl::handleTimerEvent()
{
	CEComGlobalUnlock();
	CEASSERT(_sharedTimerFiredFunction);
	if (_sharedTimerFiredFunction)
	{
		if (checkLockingOutInLowMemory())
		{
			reserveTimeoutAfterLowMemory();
		}
		else
		{
			(*_sharedTimerFiredFunction)();
		}
	}
	CEComGlobalLock();
	return CE_S_OK;
}

void CEWebKitImpl::setSharedTimerFireTime(double fireTime)
{
	INT64 timeInUSec = (fireTime - currentTime()) * 1000.0 * 1000.0;
	if (timeInUSec < 0)
	{
		timeInUSec = 0;
	}

//	if (timeInUSec > 0)
	if (1) // TODO: kawasaki: Bug 17904 Comment 5
	{
		_iTimerManagerRef.setOneshotTimer(CEWebKit::toICEHtmlOneshotTimerEventListener(this), timeInUSec);
	}
	else
	{
		if (0 == _shortTermTimeoutCount)
		{
			CEHResult hr = CEUThread_postMessage(_thread, (void*)_sharedTimerFiredFunction, &messageHandlerProc, NULL, 0);
			if (CESucceeded(hr))
			{
				++_shortTermTimeoutCount;
			}
		}
	}
}

void CEWebKitImpl::stopSharedTimer()
{
	// Do nothing.
}

CEHResult CEWebKitImpl::preparePluginInfo()
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if(!_initFlagOfPlugin)
	{
		CEComICEUStringRef iPluginPathRef;
		hr = _iEnvironmentRef.getPluginPath(&iPluginPathRef);
		if (CESucceeded(hr))
		{
			WebCore::String pluginPath;
			hr = pluginPath.fromICEUString(iPluginPathRef);
			if (CESucceeded(hr))
			{
				// Reading plugin information read form
				WebCore::PluginDatabase::installedPlugins()->addExtraPluginDirectory(pluginPath);
				_initFlagOfPlugin = true;
				hr = CE_S_OK;
			}
		}
	}
	else
	{
		hr = CE_S_OK;
	}
	return hr;
}
 

namespace WebCore {

void setSharedTimerFiredFunction(void (*f)())
{
	CEWebKitImpl::getInstance()->setSharedTimerFiredFunction(f);
}

void setSharedTimerFireTime(double fireTime)
{
	CEWebKitImpl::getInstance()->setSharedTimerFireTime(fireTime);
}

void stopSharedTimer()
{
	CEWebKitImpl::getInstance()->stopSharedTimer();
}

};
