/*
 * Copyright (C) 2007 Kevin Ollivier  All rights reserved.
 * Copyright     2010 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
#include "WebKitSilkPrefix.h"

#include "FrameView.h"
#include "GraphicsContext.h"
#include "CEWebKitFrameViewImpl.h"
#include "CEWebKitFrameWindowImpl.h"
#include "CEWebKitWindowImpl.h"

CEWebKitFrameViewImpl::CEWebKitFrameViewImpl(CEWebKitWindowPrivate* window)
: _pFrameView(NULL), _windowPrivate(window)
{
}
CEWebKitFrameViewImpl::~CEWebKitFrameViewImpl()
{
	_pFrameView = NULL;
	_windowPrivate = NULL;
}
CEHResult CEWebKitFrameViewImpl::draw(WebCore::GraphicsContext* ctx, const WebCore::IntRect& rect)
{
	if(_pFrameView)
	{
		WebCore::IntRect drawRect = rect;
		ctx->save();

		// tlanslate subframe location.
		ctx->translate(_pFrameView->x(), _pFrameView->y());
		drawRect.move(-(_pFrameView->x()), -(_pFrameView->y()));

		// cliping subframe rect.
		if (_pFrameView->parent())
			ctx->clip(_pFrameView->visibleContentRect());
		else
			ctx->clip(WebCore::IntRect(WebCore::IntPoint(0,0), _pFrameView->contentsSize()));

		_pFrameView->paintContents(ctx, drawRect);
		ctx->restore();
	}
	return CE_S_OK;
}
CEHResult CEWebKitFrameViewImpl::contentsResized()
{
	CEHResult err = CE_SILK_ERR_BADSTATE;
	if (_pFrameView)
	{
		_pFrameView->setNeedsLayout();
		err = CE_S_OK;
	}
	return err;
}
CEHResult CEWebKitFrameViewImpl::setScrollPosition(const WebCore::IntPoint& position)
{
	CEHResult err = CE_SILK_ERR_BADSTATE;

	if (_windowPrivate)
	{
		CEWebKitFrameWindowImpl* fw = kit(_windowPrivate->getFrame()->page());
		if (fw)
		{
			err = fw->setScrollPosition(position.x(), position.y());
			if (!err && _pFrameView)
			{
				_pFrameView->setWasScrolledByUser(true);
			}
		}
	}
	return err;
}
CEHResult CEWebKitFrameViewImpl::setFrameView(WebCore::FrameView* pFrameView)
{
	CEHResult err = CE_SILK_ERR_BADARGS;
	if (pFrameView)
	{
		_pFrameView = pFrameView;
		err = CE_S_OK;
	}
	return err;
}
CEHResult CEWebKitFrameViewImpl::notifyChangeContentSize(const WebCore::IntSize size, const WebCore::IntSize fixedLayoutSize, const float zoomFactor)
{
	CEHResult err = CE_SILK_ERR_BADSTATE;

	if (_windowPrivate)
	{
		if (_windowPrivate->getFrame() == _windowPrivate->getFrame()->page()->mainFrame())
		{
			CEWebKitFrameWindowImpl* fw = kit(_windowPrivate->getFrame()->page());
			if (fw)
			{
				err = fw->notifyChangeContentSize(size.width(), size.height(), fixedLayoutSize.width(), fixedLayoutSize.height(), zoomFactor);
			}
		}
	}
	return err;
}
CEHResult CEWebKitFrameViewImpl::getDeviceScreenRect(WebCore::IntRect *const deviceScreenRectOut)
{
	CEHResult err = CE_SILK_ERR_BADSTATE;
	if (_windowPrivate)
	{
		if (_windowPrivate->getFrame())
		{
			CEWebKitFrameWindowImpl* fw = kit(_windowPrivate->getFrame()->page());
			if (fw)
			{
			  CERect deviceScreenCERect;
				err = fw->getDeviceScreenRect(&deviceScreenCERect);
				if (!err)
				  {
				    deviceScreenRectOut->setX(deviceScreenCERect._x);
				    deviceScreenRectOut->setY(deviceScreenCERect._y);
				    deviceScreenRectOut->setWidth(deviceScreenCERect._width);
				    deviceScreenRectOut->setHeight(deviceScreenCERect._height);
				  }
			}
		}
	}
	return err;
}
CEHResult CEWebKitFrameViewImpl::initializeIME(bool password, bool singleLine, CEWebKitWidgetInputType inputType, const unsigned char* defaultStringStrUTF8, unsigned int defaultStringLen, INT32 maxLength, INT32 imeRect_x, INT32 imeRect_y, INT32 imeRect_width, INT32 imeRect_height, bool isPlugin, WebCore::Widget* pTarget)
{
	CEHResult err = CE_SILK_ERR_BADSTATE;
	if (_windowPrivate)
	{
		if (_windowPrivate->getFrame())
		{
			CEWebKitFrameWindowImpl* fw = kit(_windowPrivate->getFrame()->page());
			if (fw)
			{
				err = fw->initializeIME(password, singleLine, inputType, defaultStringStrUTF8, defaultStringLen, maxLength, imeRect_x, imeRect_y, imeRect_width, imeRect_height, isPlugin, pTarget);
			}
		}
	}
	return err;
}
CEHResult CEWebKitFrameViewImpl::shutdownIME(bool nextIsTextInput)
{
	CEHResult err = CE_SILK_ERR_BADSTATE;
	if (_windowPrivate)
	{
		if (_windowPrivate->getFrame())
		{
			CEWebKitFrameWindowImpl* fw = kit(_windowPrivate->getFrame()->page());
			if (fw)
			{
				err = fw->shutdownIME(nextIsTextInput);
			}
		}
	}
	return err;
}

CEHResult CEWebKitFrameViewImpl::getNetworkEventListener(ICENetworkEventListener** listener)
{
	CEHResult err = CE_SILK_ERR_BADSTATE;
	CEComICENetworkEventListenerRef networkListener = NULL;
	CEComICEHtmlWebKitFrameWindowChromeClientRef chromeclient = NULL;
	if (_windowPrivate)
	{
		if (_windowPrivate->getFrame())
		{
			CEWebKitFrameWindowImpl* fw = kit(_windowPrivate->getFrame()->page());
			if (fw)
			{
				chromeclient = fw->getChromeClient();
				if (chromeclient)
				{
					err = networkListener.initByQueryInterface(chromeclient);
					if (CESucceeded(err))
					{
						err = networkListener.copyTo(listener);
					}
				}
			}
		}
	}
	return err;
}

CEHResult CEWebKitFrameViewImpl::getUpdateDocRect(WebCore::FloatRect* const updateRect)
{
	CEHResult err = CE_SILK_ERR_BADSTATE;
	if (_windowPrivate)
	{
		if (_windowPrivate->getFrame())
		{
			CEWebKitFrameWindowImpl* fw = kit(_windowPrivate->getFrame()->page());
			if (fw)
			{
				err = fw->getUpdateDocRect(updateRect);
			}
		}
	}
	return err;
}

float CEWebKitFrameViewImpl::getZoomFactor()
{
	float zoomFactor = 1.0f;
	if (_windowPrivate)
	{
		if (_windowPrivate->getFrame())
		{
			CEWebKitFrameWindowImpl* fw = kit(_windowPrivate->getFrame()->page());
			if (fw)
			{
				zoomFactor = fw->getZoomFactor();
			}
		}
	}
	return zoomFactor;
}

CEHResult CEWebKitFrameViewImpl::getScrollMode(CEWebKitScrollMode *const scrollMode)
{	
	CEHResult err = CE_SILK_ERR_BADSTATE;
	if (_windowPrivate)
	{
		if (_windowPrivate->getFrame())
		{
			CEWebKitFrameWindowImpl* fw = kit(_windowPrivate->getFrame()->page());
			if (fw)
			{
				err = fw->getScrollMode(scrollMode);
			}
		}
	}
	return err;
}

