/*
 * Copyright (C) 2006, 2007 Apple Inc.  All rights reserved.
 * Copyright (C) 2008 Collabora Ltd. All rights reserved.
 * Copyright     2009 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
#include "config.h"
#include "HostWindow.h"
#include "PluginView.h"
#include <runtime/JSLock.h>

#include "EventNames.h"
#include "Frame.h"
#include "FrameView.h"
#include "GraphicsContext.h"
#include "HTMLNames.h"
#include "HTMLPlugInElement.h"
#include "NotImplemented.h"
#include "npapi.h"
#include "npruntime_impl.h"
#include "PluginMainThreadScheduler.h"
#include "PluginPackage.h"
#include "ScriptController.h"
#include "Timer.h"
#include "MouseEvent.h"

#include "CEUITypes.h"
#include "ICEHtmlWebKit.h"
#include "npapi_silk.h"

using std::max;
using std::min;

//#define ENABLE_LOG_OF_PLUGINVIEWSILK
#define NEW_PAINT_PLUGINVIEW_SILK
//#define NEW_PAINT_PLUGINVIEW_SILK2

namespace WebCore {
using namespace HTMLNames;

class PluginTimer : public TimerBase
{
public:
	PluginTimer(NPP instance, PluginPackage* plugin) : TimerBase(), m_instance(instance), m_plugin(plugin), m_markForDelete(false) {}
	virtual ~PluginTimer();

    virtual void fired();
	NPP& getInstance() { return m_instance; }
	void markForDelete() { m_markForDelete = true; }

private:
	NPP m_instance;
	RefPtr<PluginPackage> m_plugin;
	bool m_markForDelete;
};

static WTF::Vector<PluginTimer*> pluginTimerVector;

PluginTimer::~PluginTimer() {
	size_t position = pluginTimerVector.find(this);
	pluginTimerVector.remove(position);
}

void PluginTimer::fired() {
	if (!m_markForDelete) {

		// To do:
		// I have to decide timer event paramters.
		CETimerEventParam timerParam;
		::memset(&timerParam, 0x00, sizeof(CETimerEventParam));

		NPEvent npEvent;
		npEvent.event = CENP_WM_TIMER;
		npEvent.lParam = reinterpret_cast<uint32>(&timerParam);

		if (m_plugin->pluginFuncs()->event) {
			JSC::JSLock::DropAllLocks dropAllLocks(false);
			m_plugin->pluginFuncs()->event(m_instance, &npEvent);
		}
		//
	}

	delete this; // don't reuse me!
}

void PluginView::init()
{
    if (m_haveInitialized)
        return;
	// Silk's plugin only supports windowless plugin.
	m_isWindowed = false;
    m_haveInitialized = true;

    if (!m_plugin) {
        ASSERT(m_status == PluginStatusCanNotFindPlugin);
        return;
    }

    if (!m_plugin->load()) {
        m_plugin = 0;
        m_status = PluginStatusCanNotLoadPlugin;
        return;
    }

    if (!start()) {
        m_status = PluginStatusCanNotLoadPlugin;
        return;
    }

	m_matDocToPluginCoordinateScale.makeIdentity();
	m_matPlugToDocCoordinate.makeIdentity();

    m_npWindow.type = NPWindowTypeDrawable; // The Silk plugin always has fake window less plugin. 
	m_npWindow.window = 0;
    m_npWindow.x = 0;
    m_npWindow.y = 0;
    m_npWindow.width = 0;
    m_npWindow.height = 0;
    m_npWindow.clipRect.left = 0;
    m_npWindow.clipRect.top = 0;
    m_npWindow.clipRect.right = 0;
    m_npWindow.clipRect.bottom = 0;

	if (!m_plugin->quirks().contains(PluginQuirkDeferFirstSetWindowCall))
		setNPWindowRect(frameRect());

	show();
    m_status = PluginStatusLoadedSuccessfully;
}

void PluginView::updatePluginWidget()
{

	/******************************************/
	// The definition of coordinate
	/******************************************/
	// m_windowRect --> The TopFrame coordinate
	// m_clipRect --> The plugin coordinate

	if (!parent())
        return;

    ASSERT(parent()->isFrameView());
    FrameView* frameView = static_cast<FrameView*>(parent());

    IntRect oldWindowRect = m_windowRect;
    IntRect oldClipRect = m_clipRect;
	
	m_windowRect = IntRect(frameView->contentsToWindow(frameRect().location()), frameRect().size());

	//m_clipRect is indicated by PluginCoordinate
    m_clipRect = windowClipRect();
    m_clipRect.move(-m_windowRect.x(), -m_windowRect.y()); // convert for plugin coordinate

    if (platformPluginWidget() && (m_windowRect != oldWindowRect || m_clipRect != oldClipRect)) {

        setCallingPlugin(true);

        // To prevent flashes while scrolling, we disable drawing during the window
        // update process by clipping the window to the zero rect.

        bool clipToZeroRect = !m_plugin->quirks().contains(PluginQuirkDontClipToZeroRectWhenScrolling);
        setCallingPlugin(false);
    }

}

void PluginView::setFocus()
{
	Widget::setFocus();
}

void PluginView::show()
{
	Widget::show();
    setSelfVisible(true);
}

void PluginView::hide()
{
	Widget::hide();
    setSelfVisible(false);
}

#if !defined(NEW_PAINT_PLUGINVIEW_SILK)
void PluginView::paint(GraphicsContext* context, const IntRect& rect)
{
	if (!m_isStarted)
	{
		paintMissingPluginIcon(context, rect);
        return; 
	}

    if (context->paintingDisabled())
        return;
    ASSERT(parent()->isFrameView());
	FrameView* frameView = static_cast<FrameView*>(parent());
	if(frameView)
	{
		m_npWindow.type = NPWindowTypeDrawable;
		m_npWindow.window = 0;
		if (context && context->platformContext())
		{
			context->save();
			CEComICEUnknownRef iUnkGCRef = NULL;
			CEHResult hr = CE_S_OK;

			CEComICEVGContextRef vgc(context->platformContext());
			hr = iUnkGCRef.initByQueryInterface(vgc);

			if (CESucceeded(hr)) 
			{		
				CEComICEVGSurfaceRef iSurface;
				CEHResult err = vgc.getTargetSurface(&iSurface);
				if(!err)
				{
					CEDim dim;
					err = iSurface.getDimension(&dim);
					if(!err && dim._width>0 && dim._height>0 )
					{
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
						CEComDebugPrintf("this:%08x plugin view rect: x:%d, y:%d, width:%d, height:%d \n", this, rect.x(), rect.y(), rect.width(), rect.height() );
						CEComDebugPrintf("this:%08x plugin view frameRect x:%d, y:%d, w:%d, h:%d\n", this, frameRect().location().x(), frameRect().location().y(), frameRect().width(), frameRect().height());
						CEComDebugPrintf("this:%08x plugin view dim: width:%d, height:%d \n", this, dim._width, dim._height);
						CEComDebugPrintf("this:%08x plugin view x(), y(): x:%d y:%d\n", this, x(), y());	
#endif
						unsigned char* pAddr =NULL;
						INT32 stride=0;
						err = iSurface.getImageDataRef( &stride, (UCHAR8* *const)&pAddr );
						if(!err)
						{
							CEDXMatrix2DAffineD mat;
							vgc.getMatrix2DAffine(&mat);

#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
							CEComDebugPrintf("this:%08x plugin view mat:\n xx:%f xy:%f x0:%f\n, yx:%f yy:%f y0:%f \n\n", this, mat._xx, mat._xy, mat._x0, mat._yx, mat._yy, mat._y0);
#endif
						
							///////////////////////////////////////
							// Document Area
							//
							//
							// ----------------------------------
							// | Plugin Area                    |
							// |                                |
							// |                   ???????????????????????????
							// |                   ?------------|---         ?
							// |                   ?| clipRect  |  |<=rect   ?
							// |                   ?|           |  |         ?
							// |                   ?------------|---         ?
							// --------------------?-------------            ?
							//                     ? Buffer                  ?
							//                     ???????????????????????????
							TransformationMatrix matDocToPluginCoordinate; // The convert matrixt from document coordinate to plugin coordinate
							matDocToPluginCoordinate.setMatrix(mat._xx, 0.0, 0.0,mat._yy, 0.0, 0.0);
							TransformationMatrix matMove(1.0, 0.0, 0.0, 1.0f, -x(), -y());
							matDocToPluginCoordinate.multLeft(matMove);
							m_matPlugToDocCoordinate = matDocToPluginCoordinate.inverse();

#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
							CEComDebugPrintf("this:%08x plugin view matDocToPluginCoordinate:\n m11:%f m12:%f m13:%f m14:%f\n m21:%f m22:%f m23:%f m24:%f\n m31:%f m32:%f m33:%f m34:%f\n m41:%f m42:%f m43:%f m44:%f\n\n", this, 
								matDocToPluginCoordinate.m11(), matDocToPluginCoordinate.m12(), matDocToPluginCoordinate.m13(), matDocToPluginCoordinate.m14(),
								matDocToPluginCoordinate.m21(), matDocToPluginCoordinate.m22(), matDocToPluginCoordinate.m23(), matDocToPluginCoordinate.m24(),
								matDocToPluginCoordinate.m31(), matDocToPluginCoordinate.m32(), matDocToPluginCoordinate.m33(), matDocToPluginCoordinate.m34(),
								matDocToPluginCoordinate.m41(), matDocToPluginCoordinate.m42(), matDocToPluginCoordinate.m43(), matDocToPluginCoordinate.m44()
								);

							CEComDebugPrintf("this;%08x plugin view scale X:%f Y:%f \n", this, mat._xx, mat._yy);
#endif

							/////////////////////////////////////////////////////////////////////
							// Calc plgFrakeRect
						    // The plgFrameRect is the Plugin Area on document coordinate.
							IntRect docFrameRect = frameRect(); //Calling frameRect() can get PluginArea on document coordinate 

							double tempResultL; // left
							double tempResultT; // top
							matDocToPluginCoordinate.map(docFrameRect.x(), docFrameRect.y(), tempResultL, tempResultT);

							double tempResultR; // right
							double tempResultB; // botom
							matDocToPluginCoordinate.map(docFrameRect.x()+docFrameRect.width(), docFrameRect.y()+docFrameRect.height(), tempResultR, tempResultB);

							FloatRect fplgFrameRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);
							INT32 l = static_cast<INT32>(ceilf(fplgFrameRect.x()));
							INT32 t = static_cast<INT32>(ceilf(fplgFrameRect.y()));
							INT32 r = static_cast<INT32>(floorf(fplgFrameRect.right()));
							INT32 b = static_cast<INT32>(floorf(fplgFrameRect.bottom()));

							IntRect plgFrameRect = IntRect(l, t, r - l, b - t);
							
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
							CEComDebugPrintf("this:%08x plugin view plgFrameRect x:%d, y:%d, right:%d bottom:%d width:%d, height:%d \n", this, plgFrameRect.x(), plgFrameRect.y(), plgFrameRect.right(), plgFrameRect.bottom(), plgFrameRect.width(), plgFrameRect.height());
#endif
							/////////////////////////////////////////////////////////////////////
							IntRect docRect = rect;
							IntRect rectOnThisCoord = docRect;
							tempResultL = 0.0;
							tempResultT = 0.0;
							tempResultR = 0.0;
							tempResultB = 0.0;

							matDocToPluginCoordinate.map(rectOnThisCoord.x(), rectOnThisCoord.y(), tempResultL, tempResultT);
							matDocToPluginCoordinate.map(rectOnThisCoord.x()+rectOnThisCoord.width(), rectOnThisCoord.y()+rectOnThisCoord.height(), tempResultR, tempResultB);

							FloatRect fplgDrawRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);
						
							l = static_cast<INT32>(ceilf(fplgDrawRect.x()));
							t = static_cast<INT32>(ceilf(fplgDrawRect.y()));
							r = static_cast<INT32>(floorf(fplgDrawRect.right()));
							b = static_cast<INT32>(floorf(fplgDrawRect.bottom()));

							IntRect plgDrawRect = IntRect(l, t, r - l, b - t);//enclosingIntRect(fplgRectRect);


							INT32 moveX = static_cast<INT32>(floorf(rect.x()*mat._xx+mat._x0));
							INT32 moveY = static_cast<INT32>(floorf(rect.y()*mat._yy+mat._y0));

							// workaround code
							if(frameView->parent())
							{
								IntRect dimRect = IntRect(plgDrawRect.x()-moveX, plgDrawRect.y()-moveY, dim._width, dim._height);
								plgDrawRect.intersect(dimRect);
							}
							
							/////////////////////////////////////////////////////////////////////
							// Calc plgClipRect
							// The plgClipRect is clipRect on Plugin coordinate. ClipRect indicates drawing rect on plugin's src data (Plugin's drawing src data)
							IntRect plgClipRect = plgFrameRect;
							plgClipRect.intersect(plgDrawRect);
							m_clipRect = plgClipRect;
							////////////////////////////////////////////////////////////////////////

							// Send the data of docPluginRect and m_clipRect for Plugin.
							setNPWindowRect(plgFrameRect);



#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
							CEComDebugPrintf("this:%08x plugin view plgDrawRect: x:%d y:%d, width:%d, height:%d \n", this, plgDrawRect.x(), plgDrawRect.y(), plgDrawRect.width(), plgDrawRect.height());
							CEComDebugPrintf("this:%08x plugin view plgClipRect: x:%d y:%d, width:%d, height:%d \n", this, plgClipRect.x(), plgClipRect.y(), plgClipRect.width(), plgClipRect.height());
#endif

							NPEvent npEvent;
							npEvent.event = CENP_WM_PAINT;
							CENPBuffer npBuffer;
							npBuffer.pitch = stride;
							npBuffer.bytePerPixel = static_cast<int32>(stride/dim._width);
							
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
							CEComDebugPrintf("this:%08x plugin view rectmove x:%d, y:%d \n",this,moveX, moveY);
#endif
					
							if(moveY>0)
							{
								pAddr+=stride*moveY; 
							}
							if(moveX>0)
							{
								pAddr+=npBuffer.bytePerPixel*moveX; 
							}
					
							npBuffer.pbuffer = pAddr;
							npEvent.wParam = reinterpret_cast<uint32>(&npBuffer);

							CENPRect npClipRect;

							npClipRect._left = plgDrawRect.x();
							npClipRect._right = plgDrawRect.x()+plgDrawRect.width();
							npClipRect._top = plgDrawRect.y();
							npClipRect._bottom =plgDrawRect.y()+plgDrawRect.height();

							npEvent.lParam = reinterpret_cast<uint32>(&npClipRect);
					
							dispatchNPEvent(npEvent);
						}
					}
				}
			}
			context->restore();
		}
	}
}
#endif

void PluginView::handleKeyboardEvent(KeyboardEvent* event)
{
}

void PluginView::handleMouseEvent(MouseEvent* event)
{
	NPEvent npEvent;
	IntPoint p = static_cast<FrameView*>(parent())->contentsToWindow(IntPoint(event->pageX(), event->pageY()));

	p.setX(p.x()-m_windowRect.x());
	p.setY(p.y()-m_windowRect.y());

	double plgX;
	double plgY;
	m_matDocToPluginCoordinateScale.map(p.x(), p.y(), plgX, plgY);
	
	CENPPoint point;
	point._x = ceil(plgX);
	point._y = ceil(plgY);

	npEvent.lParam = reinterpret_cast<uint32>(&point);
	npEvent.wParam = 0;

	if (event->type() == eventNames().mousemoveEvent) 
	{
		npEvent.event = CENP_WM_MOUSEMOVE;
	}
	else if (event->type() == eventNames().mousedownEvent)
	{
		focusPluginElement();
		switch (event->button()) 
		{
		case 0:
			npEvent.event = CENP_WM_LBUTTONDOWN;
			break;
		default:
			break;
		}
	} 
	else if (event->type() == eventNames().mouseupEvent) 
	{
		switch (event->button()) 
		{
		case 0:
			npEvent.event = CENP_WM_LBUTTONUP;
			break;
		default:
			break;
		}
	}
	else
		return;

	JSC::JSLock::DropAllLocks dropAllLocks(false);
	if (!dispatchNPEvent(npEvent))
		event->setDefaultHandled();
}

void PluginView::setParent(ScrollView* parent)
{
    Widget::setParent(parent);

    if (parent)
	{
        init();
    }
}

void PluginView::setNPWindowRect(const IntRect& rect)
{	
    if (!m_isStarted || !parent() || !m_plugin->pluginFuncs()->setwindow)
        return;

    m_npWindow.window = NULL;
	m_npWindow.x = 0;
    m_npWindow.y = 0;

    m_npWindow.width = rect.width();
    m_npWindow.height = rect.height();

    m_npWindow.clipRect.left = m_clipRect.x();
    m_npWindow.clipRect.top = m_clipRect.y();
    m_npWindow.clipRect.right = m_clipRect.width()+m_clipRect.x();
    m_npWindow.clipRect.bottom = m_clipRect.height()+m_clipRect.y();

	PluginView::setCurrentPluginView(this);
    JSC::JSLock::DropAllLocks dropAllLocks(false);
    setCallingPlugin(true);
    m_plugin->pluginFuncs()->setwindow(m_instance, &m_npWindow);
    setCallingPlugin(false);
    PluginView::setCurrentPluginView(0);
}

void PluginView::setParentVisible(bool visible)
{
}

void PluginView::stop()
{
    if (!m_isStarted)
        return;

    HashSet<RefPtr<PluginStream> > streams = m_streams;
    HashSet<RefPtr<PluginStream> >::iterator end = streams.end();
    for (HashSet<RefPtr<PluginStream> >::iterator it = streams.begin(); it != end; ++it) {
        (*it)->stop();
        disconnectStream((*it).get());
    }

    ASSERT(m_streams.isEmpty());

    m_isStarted = false;

    JSC::JSLock::DropAllLocks dropAllLocks(false);

    m_npWindow.window = 0;
    if (m_plugin->pluginFuncs()->setwindow && !m_plugin->quirks().contains(PluginQuirkDontSetNullWindowHandleOnDestroy)) {
        PluginView::setCurrentPluginView(this);
        setCallingPlugin(true);
        m_plugin->pluginFuncs()->setwindow(m_instance, &m_npWindow);
        setCallingPlugin(false);
        PluginView::setCurrentPluginView(0);
    }

    PluginMainThreadScheduler::scheduler().unregisterPlugin(m_instance);

    // Destroy the plugin
    PluginView::setCurrentPluginView(this);
    setCallingPlugin(true);
    m_plugin->pluginFuncs()->destroy(m_instance, 0);
    setCallingPlugin(false);
    PluginView::setCurrentPluginView(0);

	for (Vector<PluginTimer*>::iterator it = pluginTimerVector.begin(); it != pluginTimerVector.end(); ++it) {
		PluginTimer* timer = *it;
		if (timer && timer->getInstance()->pdata == m_instance->pdata) {
			timer->markForDelete();
		}
	}

    m_instance->pdata = 0;
}

static const char* MozillaUserAgent = "Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.8.1) Gecko/20061010 Firefox/2.0";
const char* PluginView::userAgent()
{
	if (m_plugin->quirks().contains(PluginQuirkWantsMozillaUserAgent))
        return MozillaUserAgent;

    if (m_userAgent.isNull())
        m_userAgent = m_parentFrame->loader()->userAgent(m_url).utf8();
    return m_userAgent.data();
}

const char* PluginView::userAgentStatic()
{
	return MozillaUserAgent;
}

NPError PluginView::handlePostReadFile(Vector<char>& buffer, uint32 len, const char* buf)
{
	return NPERR_NO_ERROR;
}

NPError PluginView::getValueStatic(NPNVariable variable, void* value)
{
	if (value)
	{
		switch (variable) {
		case NPNVToolkit:
			*((uint32 *)value) = 0;
			return NPERR_NO_ERROR;
		case NPNVSupportsXEmbedBool:
			*((uint32 *)value) = false;
			return NPERR_NO_ERROR;
		case NPNVjavascriptEnabledBool:
			*((uint32 *)value) = true;
			return NPERR_NO_ERROR;
		default:
			return NPERR_GENERIC_ERROR;
		}
	}
	return NPERR_GENERIC_ERROR;
}

NPError PluginView::getValue(NPNVariable variable, void* value)
{
	if (value)
	{
		switch (variable) {
			case NPNVWindowNPObject: {
				if (m_isJavaScriptPaused)
					return NPERR_GENERIC_ERROR;

				NPObject* windowScriptObject = m_parentFrame->script()->windowScriptNPObject();

				// Return value is expected to be retained, as described here: <http://www.mozilla.org/projects/plugin/npruntime.html>
				if (windowScriptObject)
					_NPN_RetainObject(windowScriptObject);

				void** v = (void**)value;
				*v = windowScriptObject;
            
				return NPERR_NO_ERROR;
			}

			case NPNVPluginElementNPObject: {
				if (m_isJavaScriptPaused)
					return NPERR_GENERIC_ERROR;

				NPObject* pluginScriptObject = 0;

				if (m_element->hasTagName(appletTag) || m_element->hasTagName(embedTag) || m_element->hasTagName(objectTag))
					pluginScriptObject = static_cast<HTMLPlugInElement*>(m_element)->getNPObject();

				// Return value is expected to be retained, as described here: <http://www.mozilla.org/projects/plugin/npruntime.html>
				if (pluginScriptObject)
					_NPN_RetainObject(pluginScriptObject);

				void** v = (void**)value;
				*v = pluginScriptObject;

				return NPERR_NO_ERROR;
			}
			default:
        	    return getValueStatic(variable, value);
		}
	}
	return NPERR_GENERIC_ERROR;
}

NPError PluginView::setValue(NPPVariable variable, void* value)
{
	NPError npErr = NPERR_NO_ERROR;
    switch (variable) {
    case NPPVpluginWindowBool:
		// ignore
        // m_isWindowed = value;
        return npErr;
    case NPPVpluginTransparentBool:
        m_isTransparent = value;
        return npErr;
    case NPPVpluginTimerInterval: {
		PluginTimer* timer = new PluginTimer(instance(), plugin());
		if (timer && value)
		{
			pluginTimerVector.append(timer);

			int usec = *static_cast<int*>(value);
			double sec = static_cast<double>(usec) / 1000.0 / 1000.0;
			timer->startOneShot(sec);
		}
        return npErr;
	}
	case NPPVpluginInitializeIME: {
		if (root() && root()->hostWindow())
		{
			PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
			if(platformWindow)
			{
				CENPInitializeIME* paramIME = static_cast<CENPInitializeIME*>(value);
				if(paramIME)
				{
					CEWebKitWidgetInputType inputType = CEWebKitWidgetInputType_Undefined;
					if((paramIME->imeMode & CENPImeModeFlag_Type_Text) == CENPImeModeFlag_Type_Text)
					{
						inputType = CEWebKitWidgetInputType_TEXT;
					}
					else if((paramIME->imeMode & CENPImeModeFlag_Type_Search) == CENPImeModeFlag_Type_Search)
					{
						inputType = CEWebKitWidgetInputType_SEARCH;
					}
					else if((paramIME->imeMode & CENPImeModeFlag_Type_Tel) == CENPImeModeFlag_Type_Tel)
					{
						inputType = CEWebKitWidgetInputType_TELEPHONE;
					}
					else if((paramIME->imeMode & CENPImeModeFlag_Type_Url) == CENPImeModeFlag_Type_Url)
					{
						inputType = CEWebKitWidgetInputType_URL;
					}
					else if((paramIME->imeMode & CENPImeModeFlag_Type_Email) == CENPImeModeFlag_Type_Email)
					{
						inputType = CEWebKitWidgetInputType_EMAIL;
					}		
					else if((paramIME->imeMode & CENPImeModeFlag_Type_Password) == CENPImeModeFlag_Type_Password)
					{
						inputType = CEWebKitWidgetInputType_PASSWORD;
					}
					else if((paramIME->imeMode & CENPImeModeFlag_Type_Number) == CENPImeModeFlag_Type_Number)
					{
						inputType = CEWebKitWidgetInputType_NUMBER;
					}
					else
					{
						npErr = NPERR_INVALID_PARAM;
					}

					bool isPassword = false;
					if((paramIME->imeMode & CENPImeModeFlag_Password) == CENPImeModeFlag_Password)
					{
						isPassword = true;
					}

					bool isSingleLine = false;
					if((paramIME->imeMode & CENPImeModeFlag_SingleLine) == CENPImeModeFlag_SingleLine)
					{
						isSingleLine = true;
					}

					if(npErr == NPERR_NO_ERROR)
					{

#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
						printf("initializeIME len:%d \n",  paramIME->iDefaultStringLen);
						for(int i=0; i< paramIME->iDefaultStringLen; i++)
						{
							printf("%x ", paramIME->iDefaultStringStr[i]);
						}
						printf("\n");
#endif
						CEHResult hr = platformWindow->initializeIME(isPassword, isSingleLine, inputType, paramIME->iDefaultStringStr,  paramIME->iDefaultStringLen, paramIME->maxLength, 0, 0, 0, 0, true, this);
						if(hr)
						{
							npErr = NPERR_GENERIC_ERROR;
						}
					}
				}
			}
		}
	}
		return npErr;
	case NPPVpluginShutdownIME:{
		if (root() && root()->hostWindow())
		{
			PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
			if(platformWindow)
			{
				CEHResult hr = platformWindow->shutdownIME(false); 
				if(hr)
				{
					npErr = NPERR_GENERIC_ERROR;
				}
			}
		}
	}
		return npErr;
    default:
        notImplemented();
        return NPERR_GENERIC_ERROR;
    }
}

void PluginView::invalidateRect(const IntRect& rect)
{
	if (parent()) {
		IntRect dirtyRect;
		if (rect.width()>0 && rect.height()>0)
		{
			double tempResultL; // left
			double tempResultT; // top
			double tempResultR; // right
			double tempResultB; // botom
			m_matPlugToDocCoordinate.map(rect.x(), rect.y(), tempResultL, tempResultT);
			m_matPlugToDocCoordinate.map(rect.right(), rect.bottom(), tempResultR, tempResultB);

			FloatRect fdocInvalidateRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);
			INT32 l = static_cast<INT32>(floorf(fdocInvalidateRect.x()));
			INT32 t = static_cast<INT32>(floorf(fdocInvalidateRect.y()));
			INT32 r = static_cast<INT32>(ceilf(fdocInvalidateRect.right()));
			INT32 b = static_cast<INT32>(ceilf(fdocInvalidateRect.bottom()));

			dirtyRect = IntRect(l, t, r - l, b - t);
			dirtyRect.intersect(frameRect());
		}
		else
		{
			dirtyRect = frameRect();
		}

		PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
		if(platformWindow)
		{
			CEWebKitScrollMode scrollMode;
			platformWindow->getScrollMode(&scrollMode);
			if (scrollMode == CEWebKitScrollMode_Window)
			{
				ASSERT(parent()->isFrameView());
				FrameView* frameView = static_cast<FrameView*>(parent());
				FrameView* nowFrameView = frameView;
				bool isParent = true;

				while(isParent)
				{
					dirtyRect.move(-nowFrameView->scrollX(), -nowFrameView->scrollY());
					dirtyRect.move(nowFrameView->x(), nowFrameView->y());
					ScrollView* parentScroll = nowFrameView->parent();
					if (parentScroll)
					{
						nowFrameView = static_cast<FrameView*>(parentScroll);
					}
					else
					{
						isParent = false;
					}
				}
				
				nowFrameView->invalidateRect(dirtyRect);
			}
			else
			{
				ASSERT(parent()->isFrameView());
				FrameView* frameView = static_cast<FrameView*>(parent());
				if (frameView)
				{
					ScrollView* parentScroll = frameView->parent();
					while(parentScroll)
					{
						frameView = static_cast<FrameView*>(parentScroll);
						parentScroll = parentScroll->parent();
					}
				}
		#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
				//	CEComDebugPrintf("PluginView::invalidateRect plg coord x:%d, y:%d, right:%d, bottom:%d, width:%d, height:%d\n", rect.x(), rect.y(), rect.right(), rect.bottom(), rect.width(), rect.height());
				//	CEComDebugPrintf("PluginView::invalidateRect doc coord x:%d, y:%d, right:%d, bottom:%d, width:%d, height:%d\n", dirtyRect.x(), dirtyRect.y(), dirtyRect.right(), dirtyRect.bottom(), dirtyRect.width(), dirtyRect.height());
		#endif

				parent()->invalidateRect(dirtyRect);
			}
		}
	}
}

void PluginView::invalidateRect(NPRect* rect)
{
	IntRect diryRect;
	diryRect.setX(0);
	diryRect.setY(0);
	diryRect.setWidth(0);
	diryRect.setHeight(0);
	if (rect) {
		diryRect.setX(rect->left);
		diryRect.setY(rect->top);
		diryRect.setWidth(rect->right - rect->left);
		diryRect.setHeight(rect->bottom - rect->top);
	}
	invalidateRect(diryRect);
}

void PluginView::forceRedraw()
{
}

bool PluginView::dispatchNPEvent(NPEvent& npEvent)
{
    if (!m_plugin->pluginFuncs()->event)
        return true;

    JSC::JSLock::DropAllLocks dropAllLocks(false);
    setCallingPlugin(true);
    bool result = m_plugin->pluginFuncs()->event(m_instance, &npEvent);
    setCallingPlugin(false);

    return result;
}

PluginView::~PluginView()
{
    stop();
	m_matDocToPluginCoordinateScale.makeIdentity();
	m_matPlugToDocCoordinate.makeIdentity();

    deleteAllValues(m_requests);

    freeStringArray(m_paramNames, m_paramCount);
    freeStringArray(m_paramValues, m_paramCount);

    m_parentFrame->script()->cleanupScriptObjectsForPlugin(this);

    if (m_plugin && !(m_plugin->quirks().contains(PluginQuirkDontUnloadPlugin)))
        m_plugin->unload();
}

bool PluginView::setCompositionText(unsigned char* iTextStr, unsigned int iTextLen)
{
	CENPSetCompositionText compositionParam;
	::memset(&compositionParam, 0x00, sizeof(CENPSetCompositionText));
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
	printf("insertText len:%d \n", iTextLen);
	for(int i=0; i<iTextLen; i++)
	{
		printf("%x ", iTextStr[i]);
	}
	printf("\n");
#endif
	NPEvent npEvent;
	npEvent.event = CENP_WM_IME_COMPOSITIONTEXT;
	compositionParam.iTextStr = iTextStr;
	compositionParam.iTextLen = iTextLen;
	npEvent.lParam = reinterpret_cast<uint32>(&compositionParam);
	npEvent.wParam = 0;

	if (m_plugin->pluginFuncs()->event) {
		JSC::JSLock::DropAllLocks dropAllLocks(false);
		m_plugin->pluginFuncs()->event(m_instance, &npEvent);
	}

	return true;
}
bool PluginView::endComposition()
{
	NPEvent npEvent;
	npEvent.event = CENP_WM_IME_ENDCOMPOSITION;
	npEvent.lParam = 0;
	npEvent.wParam = 0;
	if (m_plugin->pluginFuncs()->event) {
		JSC::JSLock::DropAllLocks dropAllLocks(false);
		m_plugin->pluginFuncs()->event(m_instance, &npEvent);
	}
	return true;
}
bool PluginView::clearComponentText()
{
	NPEvent npEvent;
	npEvent.event = CENP_WM_IME_CLEARCOMPONENTTEXT;
	npEvent.lParam = 0;
	npEvent.wParam = 0;
	if (m_plugin->pluginFuncs()->event) {
		JSC::JSLock::DropAllLocks dropAllLocks(false);
		m_plugin->pluginFuncs()->event(m_instance, &npEvent);
	}
	return true;
}
bool PluginView::exitComponent()
{
	NPEvent npEvent;
	npEvent.event = CENP_WM_IME_EXITCOMPONENT;
	npEvent.lParam = 0;
	npEvent.wParam = 0;
	if (m_plugin->pluginFuncs()->event) {
		JSC::JSLock::DropAllLocks dropAllLocks(false);
		m_plugin->pluginFuncs()->event(m_instance, &npEvent);
	}
	return true;
}


#if defined(NEW_PAINT_PLUGINVIEW_SILK)
void PluginView::paint(GraphicsContext* context, const IntRect& rect)
{
	if (!m_isStarted)
	{
		paintMissingPluginIcon(context, rect);
        return; 
	}

#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
	CEComDebugPrintf("this:%08x plugin view rect: x:%d, y:%d, width:%d, height:%d \n", this, rect.x(), rect.y(), rect.width(), rect.height() );
	CEComDebugPrintf("this:%08x plugin view frameRect x:%d, y:%d, w:%d, h:%d\n", this, frameRect().location().x(), frameRect().location().y(), frameRect().width(), frameRect().height());
	CEComDebugPrintf("this:%08x plugin view x(), y(): x:%d y:%d\n", this, x(), y());
#endif
    if (context->paintingDisabled())
        return;
    ASSERT(parent()->isFrameView());
	FrameView* frameView = static_cast<FrameView*>(parent());
	if (frameView)
	{
		IntRect trect = rect;
		IntRect tframeRect = frameRect();
		IntRect tframeRectNoClip = tframeRect;
	
		// For IFrame
		// The document coordinate is used by PluginViewSilk::paint, when point is calculating.
		// Converting from FrameView coordinate to DocumentCoordinate. This is inverse calculation of ScrollView::paint()
		PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
		if(platformWindow)
		{
			CEWebKitScrollMode scrollMode;
			platformWindow->getScrollMode(&scrollMode);
			if (scrollMode == CEWebKitScrollMode_Window)
			{
				FrameView* nowFrameView = frameView;
				FrameView* firstParentView = frameView;
				bool isParent = true;
				while(isParent)
				{
					trect.move(-nowFrameView->scrollX(), -nowFrameView->scrollY());
					trect.move(nowFrameView->x(), nowFrameView->y());
					tframeRect.move(-nowFrameView->scrollX(), -nowFrameView->scrollY());
					tframeRect.move(nowFrameView->frameRect().location().x(), nowFrameView->frameRect().location().y());
					tframeRect.intersect(nowFrameView->frameRect());
					ScrollView* parentScroll = nowFrameView->parent();
					if (parentScroll)
					{
						nowFrameView = static_cast<FrameView*>(parentScroll);
					}
					else
					{
						isParent = false;
					}
				}
			}
			else
			{
				ScrollView* parentScroll = frameView->parent();
				while(parentScroll)
				{
					trect.move(-frameView->scrollX(), -frameView->scrollY());
					trect.move(frameView->x(), frameView->y());
					tframeRect.move(-frameView->scrollX(), -frameView->scrollY());
					tframeRect.move(frameView->frameRect().location().x(), frameView->frameRect().location().y());
					tframeRect.intersect(frameView->frameRect());

					frameView = static_cast<FrameView*>(parentScroll);
					parentScroll = parentScroll->parent();
				}
			}
		}
		//
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
		CEComDebugPrintf("this:%08x plugin view trect: x:%d, y:%d, width:%d, height:%d \n", this, trect.x(), trect.y(), trect.width(), trect.height() );
		CEComDebugPrintf("this:%08x plugin view tframeRect x:%d, y:%d, w:%d, h:%d\n", this, tframeRect.location().x(), tframeRect.location().y(), tframeRect.width(), tframeRect.height());
		CEComDebugPrintf("this:%08x plugin view tframeRectNoClip x:%d, y:%d, w:%d, h:%d\n", this, tframeRectNoClip.location().x(), tframeRectNoClip.location().y(), tframeRectNoClip.width(), tframeRectNoClip.height());
#endif

		m_npWindow.type = NPWindowTypeDrawable;
		m_npWindow.window = 0;
		if (context && context->platformContext())
		{
			context->save();
			CEComICEUnknownRef iUnkGCRef = NULL;
			CEHResult hr = CE_S_OK;

			CEComICEVGContextRef vgc(context->platformContext());
			hr = iUnkGCRef.initByQueryInterface(vgc);

			if (CESucceeded(hr)) 
			{		
				CEComICEVGSurfaceRef iSurface;
				CEHResult err = vgc.getTargetSurface(&iSurface);
				if (!err)
				{
					CEDim dim;
					err = iSurface.getDimension(&dim);
					if (!err && dim._width>0 && dim._height>0 )
					{
						PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
						if (platformWindow)
						{
							WebCore::FloatRect updateDoc; 
							err = platformWindow->getUpdateDocRect(&updateDoc);
							if (!err)
							{
								if (updateDoc.width() > 0.0f && updateDoc.height() > 0.0f)
								{
									TransformationMatrix matDocToPluginCoordinate; // The convert matrixt from document coordinate to Plugin coordinate
									matDocToPluginCoordinate.setMatrix(static_cast<double>(dim._width)/static_cast<double>(updateDoc.width()), 0.0, 0.0, static_cast<double>(dim._height)/static_cast<double>(updateDoc.height()), 0.0, 0.0);
									TransformationMatrix matMove(1.0, 0.0, 0.0, 1.0f, -tframeRect.x(), -tframeRect.y());
									matDocToPluginCoordinate.multLeft(matMove);

									//m_matPlugToDocCoordinate = matDocToPluginCoordinate.inverse();

									CEDXMatrix2DAffineD mat;
									vgc.getMatrix2DAffine(&mat);
									TransformationMatrix matDocToPluginCoordinateVirtual; // The convert matrixt from document coordinate to Plugin coordinate
									matDocToPluginCoordinateVirtual.setMatrix(mat._xx, 0.0, 0.0,mat._yy, 0.0, 0.0);
									matDocToPluginCoordinateVirtual.multLeft(matMove);

									m_matPlugToDocCoordinate = matDocToPluginCoordinateVirtual.inverse();
									m_matDocToPluginCoordinateScale.setMatrix(mat._xx, 0.0, 0.0,mat._yy, 0.0, 0.0);
								
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
							CEComDebugPrintf("this:%08x plugin view matDocToPluginCoordinate:\n m11:%f m12:%f m13:%f m14:%f\n m21:%f m22:%f m23:%f m24:%f\n m31:%f m32:%f m33:%f m34:%f\n m41:%f m42:%f m43:%f m44:%f\n\n", this, 
								matDocToPluginCoordinate.m11(), matDocToPluginCoordinate.m12(), matDocToPluginCoordinate.m13(), matDocToPluginCoordinate.m14(),
								matDocToPluginCoordinate.m21(), matDocToPluginCoordinate.m22(), matDocToPluginCoordinate.m23(), matDocToPluginCoordinate.m24(),
								matDocToPluginCoordinate.m31(), matDocToPluginCoordinate.m32(), matDocToPluginCoordinate.m33(), matDocToPluginCoordinate.m34(),
								matDocToPluginCoordinate.m41(), matDocToPluginCoordinate.m42(), matDocToPluginCoordinate.m43(), matDocToPluginCoordinate.m44()
								);
#endif
									IntRect docClipRect = tframeRect; // clipRect on DocumentCoordinate.
									docClipRect.intersect(trect);

									double tempResultL; // left
									double tempResultT; // top
									double tempResultR; // right
									double tempResultB; // botom

									matDocToPluginCoordinate.map(docClipRect.x(), docClipRect.y(), tempResultL, tempResultT);
									matDocToPluginCoordinate.map(docClipRect.x()+docClipRect.width(), docClipRect.y()+docClipRect.height(), tempResultR, tempResultB);
									FloatRect plgVirtualClipRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);

									INT32 l = static_cast<INT32>(floorf(plgVirtualClipRect.x()));
									INT32 t = static_cast<INT32>(floorf(plgVirtualClipRect.y()));
									INT32 r = static_cast<INT32>(floorf(plgVirtualClipRect.right()));
									INT32 b = static_cast<INT32>(floorf(plgVirtualClipRect.bottom()));

									IntRect plgClipRect = IntRect(l, t, r - l, b - t);

									m_clipRect = plgClipRect;

#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
									CEComDebugPrintf("this:%08x plugin view x(), y(): x:%d y:%d\n", this, x(), y());
									CEComDebugPrintf("this:%08x plugin view plgVirtualClipRect: x:%f y:%f, width:%f, height:%f \n", this, plgVirtualClipRect.x(), plgVirtualClipRect.y(), plgVirtualClipRect.width(), plgVirtualClipRect.height());
									CEComDebugPrintf("this:%08x plugin view m_clipRect: x:%d y:%d, width:%d, height:%d \n", this, m_clipRect.x(), m_clipRect.y(), m_clipRect.width(), m_clipRect.height());
									CEComDebugPrintf("this:%08x plugin view dim: width:%d, height:%d \n", this, dim._width, dim._height);
#endif
									// <<FrameRect On PluginCoord>>
									// Calc PluginRect size.
									matDocToPluginCoordinateVirtual.map(tframeRect.x(), tframeRect.y(), tempResultL, tempResultT);
									matDocToPluginCoordinateVirtual.map(tframeRect.x()+tframeRectNoClip.width(), tframeRect.y()+tframeRectNoClip.height(), tempResultR, tempResultB);
									FloatRect plgVirtualFrameRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);

									l = static_cast<INT32>(floorf(plgVirtualFrameRect.x()));
									t = static_cast<INT32>(floorf(plgVirtualFrameRect.y()));
									r = static_cast<INT32>(floorf(plgVirtualFrameRect.right()));
									b = static_cast<INT32>(floorf(plgVirtualFrameRect.bottom()));

									IntRect plgFrameRect = IntRect(l, t, r - l, b - t);

#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
									CEComDebugPrintf("this:%08x plugin view plgFrameRect: x:%d y:%d, width:%d, height:%d \n", this, plgFrameRect.x(), plgFrameRect.y(), plgFrameRect.width(), plgFrameRect.height());
#endif
									setNPWindowRect(plgFrameRect);
									//

									//<<NPRect>>
									IntRect docrect = IntRect(static_cast<INT32>(updateDoc.x()), static_cast<INT32>(updateDoc.y()), static_cast<INT32>(updateDoc.width()), static_cast<INT32>(updateDoc.height()));
									matDocToPluginCoordinate.map(docrect.x(), docrect.y(), tempResultL, tempResultT);
									matDocToPluginCoordinate.map(docrect.x()+docrect.width(), docrect.y()+docrect.height(), tempResultR, tempResultB);

									FloatRect plgVirtualDocRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);
									l = static_cast<INT32>(floorf(plgVirtualDocRect.x()));
									t = static_cast<INT32>(floorf(plgVirtualDocRect.y()));
									r = static_cast<INT32>(floorf(plgVirtualDocRect.right()));
									b = static_cast<INT32>(floorf(plgVirtualDocRect.bottom()));

									IntRect plgRect = IntRect(l, t, r - l, b - t);
									
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
									CEComDebugPrintf("this:%08x plugin view plgRect: x:%d y:%d, width:%d, height:%d \n", this, plgRect.x(), plgRect.y(), plgRect.width(), plgRect.height());
#endif
									unsigned char* pAddr =NULL;
									INT32 stride=0;
									err = iSurface.getImageDataRef( &stride, (UCHAR8* *const)&pAddr );
									if (!err)
									{
										NPEvent npEvent;
										npEvent.event = CENP_WM_PAINT;
										CENPBuffer npBuffer;
										npBuffer.pitch = stride;
										npBuffer.bytePerPixel = static_cast<int32>(stride/dim._width);

										npBuffer.pbuffer = pAddr;
										npEvent.wParam = reinterpret_cast<uint32>(&npBuffer);

										CENPRect npClipRect;
										npClipRect._left = plgRect.x();
										npClipRect._right = plgRect.x()+plgRect.width();
										npClipRect._top = plgRect.y();
										npClipRect._bottom =plgRect.y()+plgRect.height();

										npEvent.lParam = reinterpret_cast<uint32>(&npClipRect);
						
										dispatchNPEvent(npEvent);
									}
								}
							}
						}
					}
				}
			}
		}
		context->restore();
	}
}
#endif

#if defined(NEW_PAINT_PLUGINVIEW_SILK2)

void PluginView::paint(GraphicsContext* context, const IntRect& rect)
{
	if (!m_isStarted)
	{
		paintMissingPluginIcon(context, rect);
        return; 
	}

    if (context->paintingDisabled())
        return;
    ASSERT(parent()->isFrameView());
	FrameView* frameView = static_cast<FrameView*>(parent());
	if(frameView)
	{
		m_npWindow.type = NPWindowTypeDrawable;
		m_npWindow.window = 0;
		if (context && context->platformContext())
		{
			context->save();
			CEComICEUnknownRef iUnkGCRef = NULL;
			CEHResult hr = CE_S_OK;

			CEComICEVGContextRef vgc(context->platformContext());
			hr = iUnkGCRef.initByQueryInterface(vgc);

			if (CESucceeded(hr)) 
			{		
				CEComICEVGSurfaceRef iSurface;
				CEHResult err = vgc.getTargetSurface(&iSurface);
				if(!err)
				{
					CEDim dim;
					err = iSurface.getDimension(&dim);
					if(!err && dim._width>0 && dim._height>0 )
					{
						PlatformWidget platformWindow = root()->hostWindow()->platformWindow();
						if(platformWindow)
						{
							WebCore::FloatRect updateDoc; 
							err = platformWindow->getUpdateDocRect(&updateDoc);
							if(!err)
							{
								if(updateDoc.width() > 0.0f && updateDoc.height() > 0.0f)
								{
									
									CEDXMatrix2DAffineD mat;
									vgc.getMatrix2DAffine(&mat);

									TransformationMatrix matDocToVirtualDoc; // Convert the coordinate from document to virtual document.
									matDocToVirtualDoc.setMatrix(
										mat._xx, 0.0, 
										0.0, mat._yy,
										0.0, 0.0);

									double tempResultL; // left
									double tempResultT; // top
									double tempResultR; // right
									double tempResultB; // botom

									matDocToVirtualDoc.map(updateDoc.x(), updateDoc.y(), tempResultL, tempResultT);
									matDocToVirtualDoc.map(updateDoc.x()+updateDoc.width(), updateDoc.y()+updateDoc.height(), tempResultR, tempResultB);
									FloatRect virtualUpdateRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);

									TransformationMatrix matVirtualDocToSurface; // Convert the coordinate from virtual document to surface.
									matVirtualDocToSurface.setMatrix(
										dim._width/virtualUpdateRect.width(), 0.0,
										0.0, dim._height/virtualUpdateRect.height(),
										0.0, 0.0);

									IntRect docframeRect = frameRect(); //frameRect on document.
									
									matDocToVirtualDoc.map(docframeRect.x(), docframeRect.y(), tempResultL, tempResultT);
									matDocToVirtualDoc.map(docframeRect.x()+docframeRect.width(), docframeRect.y()+docframeRect.height(), tempResultR, tempResultB);
									FloatRect virtualFrameRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT); //frameRect on virtual document

									TransformationMatrix matDocToSurface = matDocToVirtualDoc; // Convert the coordinate from document to surface.
									matDocToSurface.multLeft(matVirtualDocToSurface);

									matDocToSurface.map(docframeRect.x(), docframeRect.y(), tempResultL, tempResultT);
									matDocToSurface.map(docframeRect.x()+docframeRect.width(), docframeRect.y()+docframeRect.height(), tempResultR, tempResultB);
									FloatRect surFrameRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT); //frameRect on surface.

									TransformationMatrix matSurfaceToPlugin; // Convert the coordinate from surface to plugin.

									INT32 surX = static_cast<INT32>(floorf(surFrameRect.x()));
									INT32 surY = static_cast<INT32>(floorf(surFrameRect.y()));

#if 1
									matSurfaceToPlugin.setMatrix(
										1.0, 0.0,
										0.0, 1.0,
										-surX, -surY);
#else
									matSurfaceToPlugin.setMatrix(
										1.0, 0.0,
										0.0, 1.0,
										-surFrameRect.x(), -surFrameRect.y());
#endif
									
									TransformationMatrix matDocToPlugin = matDocToSurface;
									matDocToPlugin.multiply(matSurfaceToPlugin);

//#if 0
									m_matPlugToDocCoordinate = matDocToPlugin.inverse();
//#else
									m_matPlugToDocCoordinate.setMatrix(
										1.0/mat._xx, 0.0, 
										0.0, 1.0/mat._yy,
										docframeRect.x(), docframeRect.y());
//#endif

									TransformationMatrix matMove(1.0, 0.0, 0.0, 1.0f, -frameRect().x(), -frameRect().y());
									TransformationMatrix matDocToPluginCoordinate; // The convert matrixt from document coordinate to Plugin coordinate
									matDocToPluginCoordinate.setMatrix(static_cast<double>(dim._width)/static_cast<double>(updateDoc.width()), 0.0, 0.0, static_cast<double>(dim._height)/static_cast<double>(updateDoc.height()), 0.0, 0.0);
									
									matDocToPluginCoordinate.multLeft(matMove);

								//	m_matPlugToDocCoordinate = matDocToPluginCoordinate.inverse();
								
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
						/*	CEComDebugPrintf("this:%08x plugin view matDocToPluginCoordinate:\n m11:%f m12:%f m13:%f m14:%f\n m21:%f m22:%f m23:%f m24:%f\n m31:%f m32:%f m33:%f m34:%f\n m41:%f m42:%f m43:%f m44:%f\n\n", this, 
								matDocToPluginCoordinate.m11(), matDocToPluginCoordinate.m12(), matDocToPluginCoordinate.m13(), matDocToPluginCoordinate.m14(),
								matDocToPluginCoordinate.m21(), matDocToPluginCoordinate.m22(), matDocToPluginCoordinate.m23(), matDocToPluginCoordinate.m24(),
								matDocToPluginCoordinate.m31(), matDocToPluginCoordinate.m32(), matDocToPluginCoordinate.m33(), matDocToPluginCoordinate.m34(),
								matDocToPluginCoordinate.m41(), matDocToPluginCoordinate.m42(), matDocToPluginCoordinate.m43(), matDocToPluginCoordinate.m44()
								);
							CEComDebugPrintf("this:%08x plugin view matDocToPlugin:\n m11:%f m12:%f m13:%f m14:%f\n m21:%f m22:%f m23:%f m24:%f\n m31:%f m32:%f m33:%f m34:%f\n m41:%f m42:%f m43:%f m44:%f\n\n", this, 
								matDocToPlugin.m11(), matDocToPlugin.m12(), matDocToPlugin.m13(), matDocToPlugin.m14(),
								matDocToPlugin.m21(), matDocToPlugin.m22(), matDocToPlugin.m23(), matDocToPlugin.m24(),
								matDocToPlugin.m31(), matDocToPlugin.m32(), matDocToPlugin.m33(), matDocToPlugin.m34(),
								matDocToPlugin.m41(), matDocToPlugin.m42(), matDocToPlugin.m43(), matDocToPlugin.m44()
								);*/
#endif
									IntRect docClipRect = frameRect(); // clipRect on DocumentCoordinate.
									docClipRect.intersect(rect);

		
									matDocToPlugin.map(docClipRect.x(), docClipRect.y(), tempResultL, tempResultT);
									matDocToPlugin.map(docClipRect.x()+docClipRect.width(), docClipRect.y()+docClipRect.height(), tempResultR, tempResultB);
									FloatRect plgVirtualClipRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);

									INT32 l = static_cast<INT32>(floorf(plgVirtualClipRect.x()));
									INT32 t = static_cast<INT32>(floorf(plgVirtualClipRect.y()));
									INT32 r = static_cast<INT32>(floorf(plgVirtualClipRect.right()));
									INT32 b = static_cast<INT32>(floorf(plgVirtualClipRect.bottom()));

									IntRect plgClipRect = IntRect(l, t, r - l, b - t);

									m_clipRect = plgClipRect;

#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
									CEComDebugPrintf("this:%08x plugin view rect: x:%d, y:%d, right:%d, bottom:%d, width:%d, height:%d \n", this, rect.x(), rect.y(), rect.right(), rect.bottom(), rect.width(), rect.height() );
									//CEComDebugPrintf("this:%08x plugin view x(), y(): x:%d y:%d\n", this, x(), y());
									//CEComDebugPrintf("this:%08x plugin view plgVirtualClipRect: x:%f y:%f, width:%f, height:%f \n", this, plgVirtualClipRect.x(), plgVirtualClipRect.y(), plgVirtualClipRect.width(), plgVirtualClipRect.height());
									CEComDebugPrintf("this:%08x plugin view m_clipRect: x:%d y:%d, width:%d, height:%d \n", this, m_clipRect.x(), m_clipRect.y(), m_clipRect.width(), m_clipRect.height());
									//CEComDebugPrintf("this:%08x plugin view dim: width:%d, height:%d \n", this, dim._width, dim._height);
#endif
									// <<FrameRect On PluginCoord>>
									// Calc PluginRect size.
									matDocToPlugin.map(frameRect().x(), frameRect().y(), tempResultL, tempResultT);
									matDocToPlugin.map(frameRect().x()+frameRect().width(), frameRect().y()+frameRect().height(), tempResultR, tempResultB);

									FloatRect plgVirtualFrameRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);

									l = static_cast<INT32>(floorf(plgVirtualFrameRect.x()));
									t = static_cast<INT32>(floorf(plgVirtualFrameRect.y()));
									r = static_cast<INT32>(floorf(plgVirtualFrameRect.right()));
									b = static_cast<INT32>(floorf(plgVirtualFrameRect.bottom()));

									IntRect plgFrameRect = IntRect(l, t, r - l, b - t);

#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
									CEComDebugPrintf("this:%08x plugin view plgFrameRect: x:%d y:%d, width:%d, height:%d \n", this, plgFrameRect.x(), plgFrameRect.y(), plgFrameRect.width(), plgFrameRect.height());
#endif
									setNPWindowRect(plgFrameRect);
									//

									//<<NPRect>>
									IntRect docrect = IntRect(static_cast<INT32>(updateDoc.x()), static_cast<INT32>(updateDoc.y()), static_cast<INT32>(updateDoc.width()), static_cast<INT32>(updateDoc.height()));
									matDocToPlugin.map(docrect.x(), docrect.y(), tempResultL, tempResultT);
									matDocToPlugin.map(docrect.x()+docrect.width(), docrect.y()+docrect.height(), tempResultR, tempResultB);

									FloatRect plgVirtualDocRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);
									l = static_cast<INT32>(floorf(plgVirtualDocRect.x()));
									t = static_cast<INT32>(floorf(plgVirtualDocRect.y()));
									r = static_cast<INT32>(floorf(plgVirtualDocRect.right()));
									b = static_cast<INT32>(floorf(plgVirtualDocRect.bottom()));

									IntRect plgRect = IntRect(l, t, r - l, b - t);
									
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
								//	CEComDebugPrintf("this:%08x plugin view plgRect: x:%d y:%d, width:%d, height:%d \n", this, plgRect.x(), plgRect.y(), plgRect.width(), plgRect.height());
#endif
									unsigned char* pAddr =NULL;
									INT32 stride=0;
									err = iSurface.getImageDataRef( &stride, (UCHAR8* *const)&pAddr );
									printf("PluginView::getImageDataRef pAddr :%08x \n", pAddr);
									if(!err)
									{
										NPEvent npEvent;
										npEvent.event = CENP_WM_PAINT;
										CENPBuffer npBuffer;
										npBuffer.pitch = stride;
										npBuffer.bytePerPixel = static_cast<int32>(stride/dim._width);

										npBuffer.pbuffer = pAddr;
										npEvent.wParam = reinterpret_cast<uint32>(&npBuffer);

										CENPRect npClipRect;
										npClipRect._left = plgRect.x();
										npClipRect._right = plgRect.x()+plgRect.width();
										npClipRect._top = plgRect.y();
										npClipRect._bottom =plgRect.y()+plgRect.height();

										npEvent.lParam = reinterpret_cast<uint32>(&npClipRect);
						
										dispatchNPEvent(npEvent);
									}
								}
							}
						}
					}
				}
			}
		}
		context->restore();
	}
}



void PluginView::invalidateRect(const IntRect& rect)
{
	if (parent()) {

		double tempResultL; // left
		double tempResultT; // top
		double tempResultR; // right
		double tempResultB; // botom
		m_matPlugToDocCoordinate.map(rect.x(), rect.y(), tempResultL, tempResultT);
		m_matPlugToDocCoordinate.map(rect.right(), rect.bottom(), tempResultR, tempResultB);

		FloatRect fdocInvalidateRect = FloatRect(tempResultL, tempResultT, tempResultR-tempResultL, tempResultB-tempResultT);
		INT32 l = static_cast<INT32>(ceilf(fdocInvalidateRect.x()));
		INT32 t = static_cast<INT32>(ceilf(fdocInvalidateRect.y()));
		INT32 r = static_cast<INT32>(floorf(fdocInvalidateRect.right()));
		INT32 b = static_cast<INT32>(floorf(fdocInvalidateRect.bottom()));

		IntRect intRect = IntRect(l, t, r - l, b - t);
#if defined(ENABLE_LOG_OF_PLUGINVIEWSILK)
		CEComDebugPrintf("PluginView::invalidateRect plg coord x:%d, y:%d, right:%d, bottom:%d, width:%d, height:%d\n", rect.x(), rect.y(), rect.right(), rect.bottom(), rect.width(), rect.height());
		CEComDebugPrintf("PluginView::invalidateRect doc float coord x:%f, y:%f, right:%f, bottom:%f, width:%f, height:%f \n", fdocInvalidateRect.x(), fdocInvalidateRect.y(), fdocInvalidateRect.right(), fdocInvalidateRect.bottom(), fdocInvalidateRect.width(), fdocInvalidateRect.height());

		CEComDebugPrintf("PluginView::invalidateRect doc coord x:%d, y:%d, right:%d, bottom:%d, width:%d, height:%d\n", intRect.x(), intRect.y(), intRect.right(), intRect.bottom(), intRect.width(), intRect.height());
#endif
		parent()->invalidateRect(intRect);
	}
}

#endif // #defined(NEW_PAINT_PLUGINVIEW_SILK2)

} // namespace WebCore
 