/*
 * Copyright (C) 2006, 2007 Apple Inc.
 * Copyright     2009 Sony Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */
#include "config.h"

#include "GraphicsContext.h"
#include "GraphicsContextPlatformPrivateSilk.h"
#include "IntRect.h"
#include "RenderThemeSilk.h"

namespace WebCore {

static RenderTheme* pTheme;

RenderTheme* theme()
{
	if (!pTheme)
	{
		pTheme = new RenderThemeSilk;
	}
    return pTheme;
}

PassRefPtr<RenderTheme> RenderTheme::themeForPage(Page* page)
{
    static RenderTheme* rt = new RenderThemeSilk;
    return rt;
}

RenderThemeSilk::RenderThemeSilk() : 
	_checkboxWidth(0),
	_checkboxHeight(0),
	_radioWidth(0),
	_radioHeight(0),
	_dropDownButtonWidth(0),
	_dropDownButtonHeight(0)
{
	CEComGetThreadContext(CEComIID_ICEHtmlWebKitTheme, reinterpret_cast<void**>(&_iWebKitThemeRef));
	_iWebKitThemeRef.release();
}
	
RenderThemeSilk::~RenderThemeSilk()
{
}

bool RenderThemeSilk::paintControl(CEHtmlDrawWidget type, CEHtmlDrawWidgetState state, RenderObject* ro, const RenderObject::PaintInfo& pi, const IntRect& r)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (ro && pi.context)
	{
		CERect paintRect = {r.x(), r.y(), r.width(), r.height()};
		CERect dirtyRect = {pi.rect.x(), pi.rect.y(), pi.rect.width(), pi.rect.height()};

		hr = _iWebKitThemeRef.drawWidget(
			pi.context->platformContext(),
			type,
			state,
			&paintRect,
			&dirtyRect
			);
	}
	return CEFailed(hr); // return false if succeeded my painting.
}

CEHtmlDrawWidgetState RenderThemeSilk::_convertWidgetState(RenderObject* o)
{
	return static_cast<CEHtmlDrawWidgetState>(
			(isEnabled(o) ? CEHtmlDrawWidgetState_Enable : 0) |
			(isChecked(o) ? CEHtmlDrawWidgetState_Checked : 0) |
			(isPressed(o) ? CEHtmlDrawWidgetState_Pushed  : 0)
		);
}

void RenderThemeSilk::_setDropDownButtonSize() const
{
	if (!_dropDownButtonWidth && !_dropDownButtonHeight)
	{
		CEComICEHtmlWebKitThemeRef iWebKitThemeRef = _iWebKitThemeRef;
		iWebKitThemeRef.getWidgetSize(CEHtmlDrawWidget_MenulistButton, CEHtmlDrawWidgetState_Enable, const_cast<UINT32*>(&_dropDownButtonWidth), const_cast<UINT32*>(&_dropDownButtonHeight));
	}
}

bool RenderThemeSilk::paintButton(RenderObject* o, const RenderObject::PaintInfo& i, const IntRect& r)
{
	return paintControl(
		CEHtmlDrawWidget_Button, _convertWidgetState(o),
		o, i, r);
}

bool RenderThemeSilk::paintCheckbox(RenderObject* o, const RenderObject::PaintInfo& i, const IntRect& r)
{
	return paintControl(
		CEHtmlDrawWidget_Checkbox, _convertWidgetState(o),
		o, i, r);
}

void RenderThemeSilk::setCheckboxSize(RenderStyle* style) const
{
	if (!_checkboxWidth && !_checkboxHeight)
	{
		CEComICEHtmlWebKitThemeRef iWebKitThemeRef = _iWebKitThemeRef;
		iWebKitThemeRef.getWidgetSize(CEHtmlDrawWidget_Checkbox, CEHtmlDrawWidgetState_Enable, const_cast<UINT32*>(&_checkboxWidth), const_cast<UINT32*>(&_checkboxHeight));
	}

	if (style)
	{
		style->setWidth(Length(static_cast<int>(_checkboxWidth*style->effectiveZoom()), Fixed));
		style->setHeight(Length(static_cast<int>(_checkboxWidth*style->effectiveZoom()), Fixed));
	}
}

bool RenderThemeSilk::paintRadio(RenderObject* o, const RenderObject::PaintInfo& i, const IntRect& r)
{
	return paintControl(
		CEHtmlDrawWidget_Radio, _convertWidgetState(o),
		o, i, r);
}

void RenderThemeSilk::setRadioSize(RenderStyle* style) const
{
	if (!_radioWidth && !_radioHeight)
	{
		CEComICEHtmlWebKitThemeRef iWebKitThemeRef = _iWebKitThemeRef;
		iWebKitThemeRef.getWidgetSize(CEHtmlDrawWidget_Radio, CEHtmlDrawWidgetState_Enable, const_cast<UINT32*>(&_radioWidth), const_cast<UINT32*>(&_radioHeight));
	}

	if (style)
	{
		style->setWidth(Length(static_cast<int>(_radioWidth*style->effectiveZoom()), Fixed));
		style->setHeight(Length(static_cast<int>(_radioWidth*style->effectiveZoom()), Fixed));
	}
}

bool RenderThemeSilk::paintTextField(RenderObject* o, const RenderObject::PaintInfo& i, const IntRect& r)
{
	return paintControl(
		CEHtmlDrawWidget_TextField, _convertWidgetState(o),
		o, i, r);
}

bool RenderThemeSilk::paintTextArea(RenderObject* o, const RenderObject::PaintInfo& i, const IntRect& r)
{
	return paintControl(
		CEHtmlDrawWidget_TextArea, _convertWidgetState(o),
		o, i, r);
}

void RenderThemeSilk::adjustMenuListButtonStyle(CSSStyleSelector* css, RenderStyle* style, Element* e) const
{
    _setDropDownButtonSize();

    // These are the paddings needed to place the text correctly in the <select> box
    const int dropDownBoxPaddingTop    = 2;
    const int dropDownBoxPaddingRight  = style->direction() == LTR ? 4 + static_cast<int>(_dropDownButtonWidth*style->effectiveZoom()) : 4;
    const int dropDownBoxPaddingBottom = 2;
    const int dropDownBoxPaddingLeft   = style->direction() == LTR ? 4 : 4 + static_cast<int>(_dropDownButtonWidth*style->effectiveZoom());
    // The <select> box must be at least 12px high for the button to render nicely on Windows
    const int dropDownBoxMinHeight = 12;
    
    // Position the text correctly within the select box and make the box wide enough to fit the dropdown button
    style->setPaddingTop(Length(dropDownBoxPaddingTop, Fixed));
    style->setPaddingRight(Length(dropDownBoxPaddingRight, Fixed));
    style->setPaddingBottom(Length(dropDownBoxPaddingBottom, Fixed));
    style->setPaddingLeft(Length(dropDownBoxPaddingLeft, Fixed));

    // Height is locked to auto
    style->setHeight(Length(Auto));

    // Calculate our min-height
    int minHeight = style->font().height();
    minHeight = max(minHeight, dropDownBoxMinHeight);

    style->setMinHeight(Length(minHeight, Fixed));
    
    // White-space is locked to pre
    style->setWhiteSpace(PRE);
}

bool RenderThemeSilk::paintMenuListButton(RenderObject* o, const RenderObject::PaintInfo& i, const IntRect& r)
{
    _setDropDownButtonSize();

    IntRect buttonRect(r);
    if (o->style()->direction() == LTR)
        buttonRect.setX(buttonRect.right() - static_cast<int>(_dropDownButtonWidth*o->style()->effectiveZoom()));
    buttonRect.setWidth(static_cast<int>(_dropDownButtonWidth*o->style()->effectiveZoom()));
	
	return paintControl(
		CEHtmlDrawWidget_MenulistButton, _convertWidgetState(o),
		o, i, buttonRect);
}

bool RenderThemeSilk::paintMenuList(RenderObject* o, const RenderObject::PaintInfo& i, const IntRect& r)
{
    // The outer box of a menu list is just a text field.  Paint it first.
	paintControl(
		CEHtmlDrawWidget_TextArea, _convertWidgetState(o),
		o, i, r);
    
    return paintMenuListButton(o, i, r);
}

void RenderThemeSilk::adjustMenuListStyle(CSSStyleSelector* selector, RenderStyle* style, Element* e) const
{
    style->resetBorder();
    adjustMenuListButtonStyle(selector, style, e);
}

bool RenderThemeSilk::paintSearchField(RenderObject* o, const RenderObject::PaintInfo& i, const IntRect& r)
{
	return paintTextField(o, i, r);
}

void RenderThemeSilk::adjustSearchFieldCancelButtonStyle(CSSStyleSelector*, RenderStyle*, Element*) const
{
}

bool RenderThemeSilk::paintSearchFieldCancelButton(RenderObject*, const RenderObject::PaintInfo&, const IntRect&)
{
	return false;
}

		double RenderThemeSilk::caretBlinkInterval() const
{
	UINT32 intervalTime = 500; // [ms]
	ICEHtmlWebKitTheme* iWebKitTheme = _iWebKitThemeRef.object();
	if (iWebKitTheme)
	{
		iWebKitTheme->_vtbl->_getCaretBlinkInterval(iWebKitTheme, &intervalTime);
	}
	return static_cast<double>(intervalTime) / 1000.0;
}

bool RenderThemeSilk::supportsFocusRing(const RenderStyle* style) const
{
	return true;
}

}; // namespace WebCore
