///////////////////////////////////////////////////////////////////////////////
// Copyright 2004,2005,2006,2007 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef __CETLTypes_h__
#define __CETLTypes_h__

#include "CESysDefs.h"
#include "CEPFRasterizer.h"
#include "ICEI18nSupport.h"

#ifdef __cplusplus
extern "C" {
#endif 

#define CETL_CHAR_REPLACEMENT_CODE CEI18n_C_OBJECT_REPLACEMENT

//! real value type
//! dependeing on the platform, this could be INT32 or float.
typedef union CETLReal
{
	INT32 _intVal;
	float _floatVal;
} CETLReal;


#if defined(__cplusplus)

#if CE_OS(WIN32) || CE_OS(LINUX) || CE_OS(XAI) || CE_OS(HYDRA) || CE_OS(ANDROID)
//! C++ class wrapper for CETLReal (float version)
class CETLRealValue
{
public:
	// default value is 0.
	CETLRealValue()             {_val._floatVal = 0.0f;}

	// convert from...
	CETLRealValue(CETLReal val) {_val = val;}
	CETLRealValue(float val)    {_val._floatVal = val;}
	CETLRealValue(INT16 val)    {_val._floatVal = (float)val;}
	CETLRealValue(UINT16 val)   {_val._floatVal = (float)val;}
	CETLRealValue(INT32 val)    {_val._floatVal = (float)val;}
	CETLRealValue(UINT32 val)   {_val._floatVal = (float)val;}

	// since CEFixed26d6 is the alias of INT32, you cannot define the constructor for it.
	void setFixed26d6(CEFixed26d6 val)  {_val._floatVal = ((float)val) / 64.0f;}

	// convert to...
	operator CETLReal()    {return _val;}
	operator float()       {return _val._floatVal;}
	operator double()      {return (double)(_val._floatVal);}
	operator long()        {return (INT32)(_val._floatVal);}
	operator INT32()       {return (INT32)(_val._floatVal);}

	// act as a CETLReal pointer.
	CETLReal* operator &()
	{
		return(&_val);
	}

	CETLRealValue operator + (CETLRealValue otherVal) const
	{
		return(CETLRealValue(_val._floatVal + otherVal._val._floatVal));
	}

	CETLRealValue operator - (CETLRealValue otherVal) const
	{
		return(CETLRealValue(_val._floatVal - otherVal._val._floatVal));
	}

	CETLRealValue operator * (CETLRealValue otherVal) const
	{
		return(CETLRealValue(_val._floatVal * otherVal._val._floatVal));
	}

	CETLRealValue operator / (CETLRealValue otherVal) const
	{
		return(CETLRealValue(_val._floatVal / otherVal._val._floatVal));
	}

	int operator == (CETLRealValue otherVal) const
	{
		return(_val._floatVal == otherVal._val._floatVal);
	}
	int operator != (CETLRealValue otherVal) const
	{
		return(_val._floatVal != otherVal._val._floatVal);
	}
	int operator > (CETLRealValue otherVal) const
	{
		return(_val._floatVal > otherVal._val._floatVal);
	}
	int operator < (CETLRealValue otherVal) const
	{
		return(_val._floatVal < otherVal._val._floatVal);
	}
private:
	CETLReal _val;
};
#else
#error Please define your own CETLRealValue
#endif


#endif // defined(__cplusplus)


typedef enum 
{
	eCETLIMEStyleAttr_None		= 0,		//! < no style (e.g. Result String)
	eCETLIMEStyleAttr_TargetConverted,		//! < selected		, converted.
	eCETLIMEStyleAttr_Converted,			//! < not selected	, converted.
	eCETLIMEStyleAttr_Input,			//! < not selected	, not converted.
	eCETLIMEStyleAttr_TargetNotConverted,		//! < selected		, not converted
	eCETLIMEStyleAttr_FixedConvert,			//! < selected		, is to fixed convert
}eCETLIMEStyleAttr;

//! ICETLStyle's attributes
typedef enum eCETLStyleAttribute
{
	eCETLStyleAttribute_Color			= (1 << 0),	//!< Desc :		font color.
													//!< Type :		CERGBAColor*
													//!< Default :	opaque black

	eCETLStyleAttribute_FontSize		= (1 << 1),	//!< Desc :		font size, in point.
													//!< Type :		CETLReal
													//!< Default :	system dependent.

	eCETLStyleAttribute_Bold			= (1 << 2),	//!< Desc :		boldface
													//!< Type :		bool
													//!< Default :	false

	eCETLStyleAttribute_CharReplacement	= (1 << 3),	//!< Desc :		character replacement
													//!< Type :		ICETLCharReplacement*
													//!< Default :	NULL

	eCETLStyleAttribute_IME				= (1 << 4),	//!> Desc :		IME style
													//!< Type :		UINT32
													//!< Default :	eCETLIMEStyleAttr_None(0)

	eCETLStyleAttribute_FontPenAdvance	= (1 << 5),	//!< Desc :		explicitly specified pen advance, which is used 
													//!<			instead of the normal pen advance. (in point)
													//!< Type :		CETLReal
													//!< Default :	0. when 0 is specified, normal pen advance is used.
													//!< Note :		This is a special workaround to support BDRE's Zeta.
													//!<			YOU SHOULD NOT USE THIS ATTRIBUTE in other situation.

	eCETLStyleAttribute_GlyphColorIndices = (1 << 6)	//!< Desc :		convert table glyph color to palette index.
														//!< Type :		UINT8* (address of UINT8[32])
														//!< Default :	NULL

} eCETLStyleAttribute;

//! ICETLLayout's attributes for layouting.
typedef enum eCETLLayoutAttribute
{
	eCETLLayoutAttribute_LineWidth,				//!< Desc :		line width for layouting, in point.
												//!< Type :		CETLReal
												//!< Default :	0 (means infinity)
	eCETLLayoutAttribute_Alignment,				//!< Desc :		Alignment, in frac[0.0, 1.0].
												//!< Type :		CETLReal
												//!< Default :	0 (means left aligned)
	eCETLLayoutAttribute_Justification,			//!< Desc :		Justification, in frac[0.0, 1.0].
												//!< Type :		CETLReal
												//!< Default :	0 (means no justification)
	eCETLLayoutAttribute_EnableEllipsis,		//!< Desc :		use Ellipsis or not
												//!< Type :		bool
												//!< Default :	0 (means disable Ellipsis)
	eCETLLayoutAttribute_LineHeight,			//!< Desc :		baseline-to-baseline distance, in pixel.
												//!<			valid only in ICEDXContext2D::setStateTLR(), since
												//!<			ICEDXContext2D handles multiple line layout, while
												//!<			ICETLLayout does not.
												//!< Type :		CETLReal
												//!< Default :	0.  when 0 is specified, the baseline-to-baseline
												//!<			distance should be computed as :
												//!<			ascent - descent + linegap.
	eCETLLayoutAttribute_MultiLine,				//!< Desc :		true when multi-line layout is expected.
												//!<			valid only when used with context2D.
												//!< Type :		bool
												//!< Default :	false
	eCETLLayoutAttribute_LineAlignment,			//!< Desc :		vertical alignment of the lines, in frac[0.0, 1.0].
												//!<			valid only when used with context2D.
												//!< Type :		CETLReal
												//!< Default :	0 (means top aligned)
												//!<
												//!<                         multiline=false         multiline=true
												//!<                         
												//!<                         +------+                +------+
												//!<                         |abcdef|                |abcdef|
												//!< lineAlignment=0.0       |      |                |ghij  |
												//!<                         |      |                |      |
												//!<                         |      |                |      |
												//!<                         +------+                +------+
												//!< 
												//!<                         +------+                +------+
												//!<                         |      |                |      |
												//!< lineAlignment=0.5       |abcdef|                |abcdef|
												//!<                         |      |                |ghij  |
												//!<                         |      |                |      |
												//!<                         +------+                +------+
												//!< 
												//!<                         +------+                +------+
												//!<                         |      |                |      |
												//!< lineAlignment=1.0       |      |                |      |
												//!<                         |      |                |abcdef|
												//!<                         |abcdef|                |ghij  |
												//!<                         +------+                +------+
	eCETLLayoutAttribute_AllowNewLine			//!< Desc :		New line, true is allow. false is forbid.
												//!< Type :		bool
												//!< Default :	true
} eCETLLayoutAttribute;

//! One specific line's attributes generated by ICETLLayout.
typedef enum eCETLLineAttribute
{
	eCETLLineAttribute_LineAscent,				//!< Desc :		max ascent of all glyphs on the line, in point.
												//!< Type :		CETLReal
												//!< Default :	N/A

	eCETLLineAttribute_LineDescent,				//!< Desc :		min descent of all glyphs on the line, in point. note the value is negative!
												//!< Type :		CETLReal
												//!< Default :	N/A

	eCETLLineAttribute_LineGap					//!< Desc :		max line gap of all glyphs on the line, in point.
												//!<			The distance that must be placed between two lines of text.
												//!<			The baseline-to-baseline distance should be computed as:
												//!<			ascent - descent + linegap, if you use the typographic values.
												//!< Type :		CETLReal
												//!< Default :	N/A
} eCETLLineAttribute;

#ifdef __cplusplus
}
#endif

#endif //__CETLTypes_h__
