///////////////////////////////////////////////////////////////////////////////
// Copyright 2009 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef __CESysDefs_h__
#define __CESysDefs_h__

/////////////////////////////////////
// target dispatching macro
/////////////////////////////////////

#define CE_HAVE(feature) (defined( CE_HAVE_##feature ) && CE_HAVE_##feature)
#define CE_OS(name) (defined( CE_OS_##name ) && CE_OS_##name)
#define CE_COMPILER(name) (defined( CE_COMPILER_##name ) && CE_COMPILER_##name)
#define CE_USE(feature) (defined( CE_USE_##feature ) && CE_USE_##feature)
#define CE_ENABLE(feature) (defined( CE_ENABLE_##feature ) && CE_ENABLE_##feature)


/////////////////////////////////////
// cpp hacks
/////////////////////////////////////

#define _CE_STATIC_CONCAT( First, Second ) First ## Second
#define CE_STATIC_CONCAT( First, Second ) _CE_STATIC_CONCAT(First, Second)


/////////////////////////////////////
// static assert
/////////////////////////////////////

#define CE_STATIC_ASSERT_LINE() __LINE__
#define CE_STATIC_ASSERT( Expression ) typedef char CE_STATIC_CONCAT( static_assert_fail_, CE_STATIC_CONCAT(Line_, CE_STATIC_ASSERT_LINE())) [ (Expression) ? 1 : -1 ];


/////////////////////////////////////
// target definitions
/////////////////////////////////////

// TODO: following definition should be supplied from Makefile or Project files.
#if defined(_MSC_VER)
    #define CESYSDEFS_TARGET_H "CESysTargetWin32.h"
#elif defined(PLATFORM_XAI)
    #define CESYSDEFS_TARGET_H "CESysTarget_ppu.h"
#elif CE_OS(LINUX)
    #define CESYSDEFS_TARGET_H "CESysTarget_linux.h"
#elif CE_OS(ANDROID)
    #define CESYSDEFS_TARGET_H "CESysTargetAndroid.h"
#elif defined(__SNC__)
    #define CESYSDEFS_TARGET_H "CESysTargetHydra.h"
#else
    #error unknown platform
#endif


#if !defined(CESYSDEFS_TARGET_H)
#error CESYSDEFS_TARGET_H should be supplied from Makefile or Project files to specify the build target
#else
#include CESYSDEFS_TARGET_H
#endif

/////////////////////////////////////
// Compiler-dependent definitions
/////////////////////////////////////

#include CESYS_COMPILER_H


/////////////////////////////////////
// Primitive Types
/////////////////////////////////////

// stdint.h:  for int*_t types
#if CE_HAVE(STDINT_H)
#include <stdint.h>
#endif

#if CE_HAVE(BASETSD_H)
#include <BaseTsd.h>
#endif

// stddef.h:  for size_t, NULL, offsetof  
#if CE_HAVE(STDDEF_H)
#include <stddef.h>
#endif

// stdarg.h:  for variable arguments
#if CE_HAVE(STDARG_H)
#include <stdarg.h>
#endif

#ifdef __cplusplus
extern "C" 
{
#endif // __cplusplus

	// integer types
	#if CE_HAVE(STDINT_H)
		typedef uint8_t		UINT8;
		typedef int8_t		INT8;
		typedef uint16_t	UINT16;
		typedef int16_t		INT16;
		typedef uint32_t	UINT32;
		typedef int32_t		INT32;
		typedef uint64_t	UINT64;
		typedef int64_t		INT64;
	    typedef intptr_t    INT_PTR;
		typedef uintptr_t   UINT_PTR;
	#endif // CE_HAVE(STDINT_H)

    #define CE_INT64_C(c)  c ## LL
    #define CE_UINT64_C(c) c ## ULL

	typedef volatile CE_ALIGNED(4) INT32  CESysAtomicINT32;
	typedef volatile CE_ALIGNED(4) UINT32 CESysAtomicUINT32;
	typedef volatile CE_ALIGNED(8) INT64  CESysAtomicINT64;
	typedef volatile CE_ALIGNED(8) UINT64 CESysAtomicUINT64;

	// char types
	typedef char CEAsciiChar;
	typedef UINT8 CEUtf8Char;
	typedef UINT16 CEUtf16Char;
	typedef UINT32 CEUtf32Char;

	// float / double
	typedef float CEFloat;
	typedef double CEDouble;

#ifdef __cplusplus
};
#endif // __cplusplus

CE_STATIC_ASSERT(sizeof(INT8) == 1);
CE_STATIC_ASSERT(sizeof(INT16) == 2);
CE_STATIC_ASSERT(sizeof(INT32) == 4);
CE_STATIC_ASSERT(sizeof(INT64) == 8);
CE_STATIC_ASSERT(sizeof(UINT8) == 1);
CE_STATIC_ASSERT(sizeof(UINT16) == 2);
CE_STATIC_ASSERT(sizeof(UINT32) == 4);
CE_STATIC_ASSERT(sizeof(UINT64) == 8);


// when used in C (not C++), "bool" is not a built-in type.
#if !defined(__cplusplus)
typedef char bool;
#endif

// INT_PTR/UINT_PTR constant
#if CE_OS(WIN32) && defined(WIN64)
// LLP64 (Win64)
#define INT_PTR_CONST(c)	(c ## LL)
#define UINT_PTR_CONST(c)	(c ## ULL)
#else
// ILP32 (32bit OS) or LP64(Linux, MacOSX, Solaris)
#define INT_PTR_CONST(c)	(c ## L)
#define UINT_PTR_CONST(c)	(c ## UL)
#endif

// CEOFFSETOF(type, member)
// type specifies the name of the struct/class
// member specifies the name of the member variable whose offset is desired
#if defined(offsetof)
	#define CEOFFSETOF(TYPE,MEMBER)	offsetof(TYPE,MEMBER)
#elif CE_COMPILER(GCC)
    #if (CE_GCC_VER < CE_GCC_VER_NUM(3,0,0))
        #define CEOFFSETOF(TYPE,MEMBER)   ((UINT32)(&TYPE::MEMBER) & 0xfffffffe)
    #elif (CE_GCC_VER < CE_GCC_VER_NUM(3,3,0))
        // workaround to suppress the stupid warning
        #define CEOFFSETOF(TYPE,MEMBER)   (((USIZE_T)&(((TYPE *)0x1000)->MEMBER))-0x1000)
    #else
        #define CEOFFSETOF(TYPE,MEMBER) (__builtin_offsetof (TYPE, MEMBER))
    #endif
#else
	#error Is 'offsetof' undefined in stddef.h?
#endif


/////////////////////////////////////
// Primitive Type Limits
/////////////////////////////////////

// limits.h:  for int types MIN/MAX
#if CE_HAVE(LIMITS_H)
#include <limits.h>
#endif

#define CE_INT8_MAX		SCHAR_MAX
#define CE_INT8_MIN		SCHAR_MIN
#define CE_UINT8_MAX	UCHAR_MAX

#define CE_INT16_MAX	SHRT_MAX
#define CE_INT16_MIN	SHRT_MIN
#define CE_UINT16_MAX	USHRT_MAX

#define CE_INT32_MAX	INT_MAX
#define CE_INT32_MIN	INT_MIN
#define CE_UINT32_MAX	UINT_MAX

#define CE_INT64_MAX	LLONG_MAX
#define CE_INT64_MIN	LLONG_MIN
#define CE_UINT64_MAX	ULLONG_MAX


/////////////////////////////////////
// OS-independent runtime types
/////////////////////////////////////

typedef INT64 ce_off_t;


/////////////////////////////////////
// OS-dependent definitions
/////////////////////////////////////

#include  CESYS_OSDEFS_H  // CESYS_OSDEFS_H should be supplied from Makefile or Project files to specify the build target




/////////////////////////////////////
// Target system standard headers
/////////////////////////////////////

#if CE_HAVE(CTYPE_H)
#include <ctype.h>
#endif

#if CE_HAVE(STRING_H)
#include <string.h>
#endif

#if CE_HAVE(STDIO_H)
#include <stdio.h>
#endif

#if CE_HAVE(STDLIB_H)
#include <stdlib.h>
#endif

#if CE_HAVE(FLOAT_H)
#include <float.h>
#endif

#if CE_HAVE(MATH_H)
#include <math.h>
#endif

#ifdef __cplusplus

#if CE_HAVE(NEW)
#include <new>
#endif

#endif // __cplusplus


/////////////////////////////////////
// Error code
/////////////////////////////////////

//  CEHResult is a type to represent the error status.
//  Following is a bitwise structure of CEHResult:
//
//   3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1
//   1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
//  +-+-+-+-+-+---------------------+-------------------------------+
//  |S|0|1|0|s|    Facility         |               Code            |
//  +-+-+-+-+-+---------------------+-------------------------------+
//
//   S(Severity) -- 0:Success, 
//                  1:Fail
//   R           -- always 0  (reserved bit for severity bit of NTSTATUS)
//   C(Customer) -- 0: MS COM defined HRESULT
//                  1: Silk or its application
//   s           -- 0: Silk defined errors
//                  1: Silk application errors

//  SRCN
//  1000  0x80000000-0x8fffffff:  HRESULT range
//  1001  0x90000000-0x9fffffff:  NTSTATUS range
//  1010  0xa0000000-0xafffffff:  Silk error range
//  1011  0xb0000000-0xbfffffff:  (Silk reserved)
//  1100  0xc0000000-0xcfffffff:  (HRESULT reserved)
//  1101  0xd0000000-0xdfffffff:  NTSTATUS range
//  1110  0xe0000000-0xefffffff:  (Silk reserved)
//  1111  0xf0000000-0xffffffff:  (Silk reserved)

#define _CEMAKEHRESULT(sev, fac, code) \
	(((UINT32)(sev)<<31) | ((UINT32)(fac)<<16) | ((UINT32)code) | 0x20000000UL)

#ifdef __cplusplus
extern "C"
{
#endif

	// sevirity code in CEHResult
	typedef enum 
	{
		eCESeverity_Success = 0,
		eCESeverity_Fail = 1,
	} eCESeverity;

	// facility code in CEHResult
	typedef enum 
	{
		/////////////////////////////////////////////////////////
		// HRESULT reserved range  (0x0   -- 0x0ff)
		/////////////////////////////////////////////////////////
		eCEFacility_Null = 0,	// FACILITY_NULL


		/////////////////////////////////////////////////////////
		// NTSTATUS reserved range (0x100 -- 0x1ff)
		/////////////////////////////////////////////////////////

		/////////////////////////////////////////////////////////
		// SilkCore reserved range (0x200 -- 0x27f)
		/////////////////////////////////////////////////////////

		eCEFacility_System	= 0x201,
		eCEFacility_SYS = eCEFacility_System,  // abbrev

		eCEFacility_FileSystem	= 0x202,
		eCEFacility_FS = eCEFacility_FileSystem, // abbrev

		eCEFacility_Socket		= 0x203,
		eCEFacility_SOCKET =  eCEFacility_Socket,

		eCEFacility_DNS			= 0x204,

		eCEFacility_SSL			= 0x205,

		eCEFacility_Silk = 0x240,
		eCEFacility_SILK= eCEFacility_Silk,		

		eCEFacility_SYSWRAP = 0x241,

		/////////////////////////////////////////////////////////
		// Silk Application range  (0x280 -- 0x2ff)
		/////////////////////////////////////////////////////////
	} eCEFacility;


	#define CEHRESULT_FACILITY_START(fac) \
		CE_##fac##_ERR_BASE = _CEMAKEHRESULT(eCESeverity_Fail, eCEFacility_##fac, 0),

	#define CEHRESULT_FACILITY_END(fac)

	#define CEHRESULT_BLOCK_START(fac, blkname, blkcode) \
		CE_##fac##_ERR_##blkname##_BASE = _CEMAKEHRESULT(eCESeverity_Fail, eCEFacility_##fac, blkcode * 256),

	#define CEHRESULT_BLOCK_END(fac, blkname)

	#define CEHRESULT_ERR(fac, err) \
		CE_##fac##_ERR_##err,


	typedef enum
	{
		CE_S_OK = 0,

		#include "CESysError.h"

	} CEHResult; // enum CEHResult


	#undef CEHRESULT_FACILITY_START
	#undef CEHRESULT_FACILITY_END

	#undef CEHRESULT_BLOCK_START
	#undef CEHRESULT_BLOCK_END
	#undef CEHRESULT_ERR


    #define CEMAKEHRESULT(sev, fac, code)   ((CEHResult)_CEMAKEHRESULT(sev, fac, code))
    #define CEHRESULT_FACILITY(hr)          ((eCEFacility)(((hr) >> 16) & 0x2ff))
    #define CESUCCEEDED(hr)                 ((INT32)(hr) >= 0)
    #define CEFAILED(hr)                    ((INT32)(hr) < 0)

    #if CE_OS(WIN32)
        typedef DWORD CENativeError;
    #else
        typedef int CENativeError;
    #endif

	CE_DLL_EXPORT CEHResult CESysConvertNativeErrorToHResult(CENativeError err, eCEFacility fac);
	CE_DLL_EXPORT CENativeError CESysConvertHResultToNativeError(CEHResult hr);
	CE_DLL_EXPORT void CESysSetLastError(CEHResult hr);
	CE_DLL_EXPORT CEHResult CESysGetLastError();

#ifdef __cplusplus
};
#endif

#ifdef __cplusplus
CE_FORCEINLINE CEHResult   CEMakeHResult(eCESeverity sev, eCEFacility fac, UINT16 code) { return CEMAKEHRESULT(sev, fac, code); }
CE_FORCEINLINE eCEFacility CEHResult_Facility(CEHResult hr) { return CEHRESULT_FACILITY(hr); }
CE_FORCEINLINE bool CESucceeded(CEHResult hr) { return (bool)CESUCCEEDED(hr); }
CE_FORCEINLINE bool CEFailed(CEHResult hr)    { return (bool)CEFAILED(hr); }
#endif // __cplusplus

#undef _CEMAKEHRESULT


/////////////////////////////////////
// DLL support
/////////////////////////////////////

#define CEDLL_PROCESS_ATTACH   1    // DLL_PROCESS_ATTACH
#define CEDLL_PROCESS_DETACH   0    // DLL_PROCESS_DETACH

/////////////////////////////////////
// Inline functions
/////////////////////////////////////

#ifdef __cplusplus

template <class T>
CE_FORCEINLINE T CEMax(T a, T b)  { return (a > b ? a : b); }

template <class T>
CE_FORCEINLINE T CEMin(T a, T b)  { return (a < b ? a : b); }

#define CE_MAX(a,b)	CEMax((a), (b))
#define CE_MIN(a,b)	CEMin((a), (b))

#else // __cplusplus

#define CE_MAX(a,b)	((a) > (b) ? (a) : (b))
#define CE_MIN(a,b)	((a) < (b) ? (a) : (b))

#endif // __cplusplus

#define CE_ARRAY_SIZE(x) (sizeof(x)/sizeof((x)[0]))




/////////////////////////////////////////////////////////////////////
// Backward Compatibility
// TODO: following code should be moved to sstypes.h
/////////////////////////////////////////////////////////////////////

#if CE_OS(ANDROID)
#if defined(__release)
#undef __release
#endif
#endif

#define JGSYS_LITTLE_ENDIAN CE_LITTLE_ENDIAN

// <---- ssconfig workaround ----

/*!
* Use sub-pixel accurate text layout.
* define USE_SUBPXL_ACCURATE_TEXTLAYOUT when you want to use CETL module for sub-pixel accurate
* text layout.
*/
#define USE_SUBPXL_ACCURATE_TEXTLAYOUT

// enable assert (debug/release build).
//#define FORCE_ENABLE_ASSERT

#define USE_WEBKIT

/*!
* use i18n icu4c binding.
*/
#if !defined(USE_CEI18N_ICU)
#error USE_CEI18N_ICU should be defined in CESYSDEFS_TARGET_H
#endif

/*!
 * Configure Acropolis Silk Core feature for linux & win build
 */
#if defined(PLATFORM_SHOKA)
#define DISABLE_CONTROL_SCROLLBAR
#define ENABLE_SIMPLE_URL_FILTER
#define ENABLE_SET_PROXY_IMMEDIATELY
#define ENABLE_SHOKA_GRAYSCALE
#define _REPLACE_ERROR_CODE_ENABLED
#define TOGGLE_LISTBOX
#endif
// ---- ssconfig workaround ---->

// <---- sstypes.h to CESysDefs.h migration helper macro ----
// basic character types
typedef char	CHAR8;
typedef UINT8	UCHAR8;

typedef UINT16	UTF16CHAR;
typedef UINT32	UTF32CHAR;

#define UWCHAR unsigned short // 16-bit UNICODE character
#ifdef _UNICODE
#define	UTCHAR UWCHAR // 16-bit UNICODE character
#else
#define	UTCHAR char   
#endif

#define UINT32_UNKNOWN_SIZE 0xffffffff
#define INVALID_UCS4_CHAR	0xffffffff
#define INVALID_UCS2_CHAR	0xffff

// unsigned type to represent the maximum number of bytes to which a pointer can refer.
typedef size_t         USIZE_T;
// ---- sstypes.h to CESysDefs.h migration helper macro ---->

// <----   sstypes.h to CESysDefs.h migration helper macro ----
#if !defined(DLL_EXPORT)
#define DLL_EXPORT CE_DLL_EXPORT
#endif

#if !defined(FORCEINLINE)
#define FORCEINLINE CE_FORCEINLINE
#endif

// TODO: 'FORCEINLINE_WITHOUT_DEBUG' will be renamed later
#if defined(DEBUG)
#define FORCEINLINE_WITHOUT_DEBUG
#else
#define FORCEINLINE_WITHOUT_DEBUG FORCEINLINE
#endif
// ----   sstypes.h to CESysDefs.h migration helper macro ---->


//--------------------------------------------------------------------
// DllMain
//--------------------------------------------------------------------

#if defined(STATIC_LINK)
#define CECOM_DLL_FUNC(modulename, funcname)  modulename##_##funcname
#else // defined(STATIC_LINK)
#define CECOM_DLL_FUNC(modulename, funcname)  funcname
#endif // defined(STATIC_LINK)

//--------------------------------------------------------------------
// Color conversion
//--------------------------------------------------------------------

//for GrayScale.
//#define RGB2L(r, g, b) (0.299 * r + 0.587 * g + 0.114 * b)
#define RGB2L(r, g, b) ((77 * r + 150 * g + 29 * b) >> 8) //integer-ver. for speed-up



/////////////////////////////////////
// target depenedent header filenames
/////////////////////////////////////

#if !defined(CESYSATOMICOPS_TARGET_H)
#error CESYSATOMICOPS_TARGET_H should be defined in CESYSDEFS_TARGET_H
#endif

#if !defined(CESYSCONFIG_TARGET_H)
#error CESYSCONFIG_TARGET_H should be defined in CESYSDEFS_TARGET_H
#endif

#if !defined(CESYSCRITICALSECTION_TARGET_H)
#error CESYSCRITICALSECTION_TARGET_H should be defined in CESYSDEFS_TARGET_H
#endif

#if !defined(CESYSSOCKET_TARGET_H)
#error CESYSSOCKET_TARGET_H should be defined in CESYSDEFS_TARGET_H
#endif



#endif // __CESysDefs_h__
