///////////////////////////////////////////////////////////////////////////////
// Copyright 2004,2008,2009 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef __CESysAtomicOps_h__
#define __CESysAtomicOps_h__

#include "CESysDefs.h"
#include "CESysConfig.h" // macros defined in CESysConfig*.h will be moved to CESysOSDefs*.h

////////////////////////////////////////
// platform dependent CESysAtomicOps
////////////////////////////////////////

#include CESYSATOMICOPS_TARGET_H


////////////////////////////////////////
// C-based atomic primitives
////////////////////////////////////////

#ifdef __cplusplus
extern "C"
{
#endif

	////////////////////////////////////////
	// Compare-And-Swap
	////////////////////////////////////////

	//! Execute Compare-And-Swap(CAS) operation on the specified variable as an atomic operation
	//! \param[in/out] dest          specifies the variable where execute CAS operation
	//! \param[in]     newValue      specifies the value to set to 'dest', if the comparison between 'dest' and 'expectedValue' success.
	//! \param[in]     expectedValue specifies the value to be compared with dest
	//! \return the initial value of dest parameter
#if !CE_HAVE(NATIVE_CAS32)
	CE_DLL_EXPORT INT32 CESysInterlockedCompareExchange(CESysAtomicINT32* dest, INT32 newValue, INT32 expectedValue);
#endif

#if !CE_HAVE(NATIVE_CAS64)
	CE_DLL_EXPORT INT64 CESysInterlockedCompareExchange64(CESysAtomicINT64* dest, INT64 newValue, INT64 expectedValue);
#endif

    // obsolete?
	static CE_FORCEINLINE void CESysInterlockedInit(CESysAtomicINT32* var)
	{
		*var = 0;
	}

	//! Add the value to the specified variable as an atomic operation
	//! \param[in/out] var     specifies the variable to be added
	//! \param[in]     value   the value to add
	//! \return the initial value of dest parameter
#if !CE_HAVE(NATIVE_CESysInterlockedExchangeAdd)
	static CE_FORCEINLINE INT32 CESysInterlockedExchangeAdd(CESysAtomicINT32* var, INT32 value)
	{
		INT32 expected, newValue;
		do
		{ 
			expected = *var; 
			newValue = expected + value; 
		} while (CESysInterlockedCompareExchange(reinterpret_cast<CESysAtomicINT32*>(var), newValue, expected) != expected);
		return expected;
	}
#endif //!CE_HAVE(NATIVE_CESysInterlockedExchangeAdd)

	static CE_FORCEINLINE INT64 CESysInterlockedExchangeAdd64(CESysAtomicINT64* var, INT64 value)
	{
		INT64 expected, newValue;
		do
		{ 
			expected = *var; 
			newValue = expected + value; 
		} while (CESysInterlockedCompareExchange64(reinterpret_cast<CESysAtomicINT64*>(var), newValue, expected) != expected);
		return expected;
	}

	/*! Increment the specified variable atomically				   */	
	/*! \param[in/out] var     specifies the variable to increment */	
	/*! \return the incremented value							   */	
#if !CE_HAVE(NATIVE_CESysInterlockedIncrement)
	static CE_FORCEINLINE INT32 CESysInterlockedIncrement(CESysAtomicINT32* var)
	{ 
		return CESysInterlockedExchangeAdd(var, 1)+1;
	} 
#endif //!CE_HAVE(NATIVE_CESysInterlockedIncrement)

	/*! Decrement the specified variable atomically				   */	
	/*! \param[in/out] var     specifies the variable to decrement */	
	/*! \return the decremented value							   */	
#if !CE_HAVE(NATIVE_CESysInterlockedDecrement)
	static CE_FORCEINLINE INT32 CESysInterlockedDecrement(CESysAtomicINT32* var)
	{
		return CESysInterlockedExchangeAdd(var, -1)-1;
	}
#endif //!CE_HAVE(NATIVE_CESysInterlockedDecrement)

	static CE_FORCEINLINE INT64 CESysInterlockedIncrement64(CESysAtomicINT64* var)
	{ 
		return CESysInterlockedExchangeAdd64(var, 1)+1;
	} 

	static CE_FORCEINLINE INT64 CESysInterlockedDecrement64(CESysAtomicINT64* var)
	{
		return CESysInterlockedExchangeAdd64(var, -1)-1;
	}



	/*! reset the specified bit in the varibale atomically		        */	\
	/*! \param[inout] var  pointer to the variable to be bit-reset      */	\
	/*! \param[in] bitpos  bit position to be reset. (LSB=0)	        */	\
	/*! \return the previous bit value of the specified bit-position	*/  \
	static CE_FORCEINLINE UINT8 CESysInterlockedBitTestAndReset(CESysAtomicUINT32* var, UINT32 bitpos)
	{
		UINT32 mask = 1U << bitpos;
		UINT32 expected, newValue;
		do
		{ 
			expected = *var; 
			newValue = expected & ~mask; 
		} while (CESysInterlockedCompareExchange(reinterpret_cast<CESysAtomicINT32*>(var), (INT32)newValue, (INT32)expected) != (INT32)expected);
		return (UINT8)((expected & mask) >> bitpos);
	} 

	/*! set the specified bit in the varibale atomically		        */ \
	/*! \param[inout] var  pointer to the variable to be bit-set        */ \
	/*! \param[in] bitpos  bit position to be set. (LSB=0)			    */ \
	/*! \return the previous bit value of the specified bit-position	*/ \
	static CE_FORCEINLINE UINT8 CESysInterlockedBitTestAndSet(volatile UINT32* var, UINT32 bitpos)
	{
		UINT32 mask = 1U << bitpos;
		UINT32 expected, newValue;
		do
		{ 
			expected = *var; 
			newValue = expected | mask; 
		} while (CESysInterlockedCompareExchange(reinterpret_cast<CESysAtomicINT32*>(var), (INT32)newValue, (INT32)expected) != (INT32)expected);
		return (UINT8)((expected & mask) >> bitpos);
	}

	static CE_FORCEINLINE UINT32 CESysPopCount(UINT32 bits)
	{
		bits = (bits & 0x55555555) + (bits >> 1 & 0x55555555);
		bits = (bits & 0x33333333) + (bits >> 2 & 0x33333333);
		bits = (bits & 0x0f0f0f0f) + (bits >> 4 & 0x0f0f0f0f);
		bits = (bits & 0x00ff00ff) + (bits >> 8 & 0x00ff00ff);
		return (bits & 0x0000ffff) + (bits >>16 & 0x0000ffff);	
	}

	static CE_FORCEINLINE UINT32 CESysPopCount64(UINT64 bits)
	{
		UINT32 hi = (UINT32)(bits >> 32);
		UINT32 lo = (UINT32)(bits & 0xffffffff);
		return CESysPopCount(lo) + CESysPopCount(hi);
	}

	//! count the number of the leading zeros
	//! \return return 32 if val is zero. otherwise return the number of leading bits.
	static CE_FORCEINLINE UINT32 CESysCountLeadingZero(UINT32 x)
	{
		UINT32 y;
		UINT32 n = 32;
		y = x >> 16; if (y != 0) { n = n - 16; x = y; }
		y = x >>  8; if (y != 0) { n = n -  8; x = y; }
		y = x >>  4; if (y != 0) { n = n -  4; x = y; }
		y = x >>  2; if (y != 0) { n = n -  2; x = y; }
		y = x >>  1; if (y != 0) { return n - 2; }
		return n - x;
	}

	static CE_FORCEINLINE UINT32 CESysCountLeadingZero64(UINT64 bits)
	{
		UINT32 hi = (UINT32)(bits >> 32);
		UINT32 lo = (UINT32)(bits & 0xffffffff);
		UINT32 n = CESysCountLeadingZero(hi);
		if (n == 32)
		{
			n += CESysCountLeadingZero(lo);
		}
		return n;
	}

	//! count the number of the trailing zeros
	//! \return return 32 if val is zero. otherwise return the number of trailing bits.
	static CE_FORCEINLINE UINT32 CESysCountTrailingZero(UINT32 val)
	{
		INT32 x = (INT32)val;
		x = (x & (-x))-1; 
		return CESysPopCount(x);
	}

	static CE_FORCEINLINE UINT64 CESysCountTrailingZero64(UINT64 val)
	{
		INT64 x = (INT64)val;
		x = (x & (-x))-1; 
		return CESysPopCount64(x);
	}

#ifdef __cplusplus
}; // extern "C"
#endif

#endif // __CESysAtomicOps_h__
