///////////////////////////////////////////////////////////////////////////////
// Copyright 2004,2005,2006,2009 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef CEPFRasterizer_h
#define CEPFRasterizer_h

#include "CESysDefs.h"
#include "ceutypes.h"
#include "CECom.h"
#include "CEUITypes.h"

#ifdef __cplusplus
extern "C" {
#endif  //__cplusplus

	#define CEPF_BAD_GLYPHINDEX(_gi) (_gi == 0)
	#define CEPF_DUMMY_GLYPHINDEX(_gi) (_gi == 0xffff || _gi == 0xfffe)

	typedef struct ICEPFRasterizer ICEPFRasterizer;

	// 26.6 format fixed point value
	typedef INT32 CEFixed26d6;

	//! rendering flags (e.g. 1bit or 8bit, anti-aliased, ...)
	typedef enum {
		// rendering options bits.
		eCEPFRasterizerOptions_NONE				= 0,
		eCEPFRasterizerOptions_ALWAYS_AA_OUTLINE	= (1<<0),
		eCEPFRasterizerOptions_MONOCHROME		= (1<<1),
		eCEPFRasterizerOptions_LAST
	} eCEPFRasterizerOptions;

	struct CEPFGlyphBitmapInfo;
	struct CEPFGlyphMetrics;
	struct CEPFGlyphLoaderParam;
	struct ICEPFGlyphLoaderCB;

	//! ICEPRRasterizer initialize parameter
	typedef struct CEPFRasterizerInitParam
	{
		UINT32 hRes;					          //! horizontal resolution in dpi
		UINT32 vRes;							  //! vertical resolution in dpi
		UINT32 eCEPFRasterizerOptionsMask;		  //! rendering flags (e.g. 1bit or 8bit, anti-aliased, ...)
		ICEPFGlyphLoaderCB* iGlyphLoaderCB;		  //! address to the glyph loader callback.
	} CEPFRasterizerInitParam;


	// ICEPRRasterizer activate parameter
	typedef struct CEPFRasterizerActivateParam
	{
		void* pGlyphCacheMemory;				//! pre-allocated memory buffer for glyph cache.
		UINT32 glyphCacheMemorySizeInBytes;		//! size of pGlyphCacheMemory in bytes.
	} CEPFRasterizerActivateParam;

	typedef enum eCEPFFontCharacterMap
	{
		eCEPFFontCharacterMap_unicode = 0 ,
		eCEPFFontCharacterMap_unicode_BML_mono, //! BML specific unicode like encoding. ascii(0x20 - 0x7e) characters will be converted to the extended charactes area.
		eCEPFFontCharacterMap_LAST = eCEPFFontCharacterMap_unicode_BML_mono
	} eCEPFFontCharacterMap;

	typedef enum eCEPFFontStyle
	{
		eCEPFFontStyle_Italic		= 1 << 0,
		eCEPFFontStyle_Oblique		= 1 << 1
	} eCEPFFontStyle;

	typedef enum eCEPFFontStyleWidthClass
	{
		eCEPFFontStyleWidthClass_UltraCondensed	= 0,
		eCEPFFontStyleWidthClass_ExtraCondensed	= 1,
		eCEPFFontStyleWidthClass_Condensed		= 2,
		eCEPFFontStyleWidthClass_SemiCondensed	= 3,
		eCEPFFontStyleWidthClass_Medium			= 4,	//! normal width class.
		eCEPFFontStyleWidthClass_SemiExpanded	= 5,
		eCEPFFontStyleWidthClass_Expanded		= 6,
		eCEPFFontStyleWidthClass_ExtraExpanded	= 7,
		eCEPFFontStyleWidthClass_UltraExpanded	= 8
	} eCEPFFontStyleWidthClass;

	typedef enum eCEPFFontStyleWeightClass
	{
		eCEPFFontStyleWeightClass_100 = 0,
		eCEPFFontStyleWeightClass_200 = 1,
		eCEPFFontStyleWeightClass_300 = 2,
		eCEPFFontStyleWeightClass_400 = 3,
		eCEPFFontStyleWeightClass_500 = 4,
		eCEPFFontStyleWeightClass_600 = 5,
		eCEPFFontStyleWeightClass_700 = 6,
		eCEPFFontStyleWeightClass_800 = 7,
		eCEPFFontStyleWeightClass_900 = 8,

		eCEPFFontStyleWeightClass_Thin		= 0,
		eCEPFFontStyleWeightClass_ExtraLight= 1,
		eCEPFFontStyleWeightClass_Light		= 2,
		eCEPFFontStyleWeightClass_Normal	= 3,
		eCEPFFontStyleWeightClass_Medium	= 4,
		eCEPFFontStyleWeightClass_SemiBold	= 5,
		eCEPFFontStyleWeightClass_Bold		= 6,
		eCEPFFontStyleWeightClass_ExtraBold = 7,
		eCEPFFontStyleWeightClass_Black		= 8
	} eCEPFFontStyleWeightClass;

	//! font style.
	typedef struct CEPFFontStyle
	{
		eCEPFFontStyleWeightClass weightClass;
		eCEPFFontStyleWidthClass widthClass;
		UINT16 eCEPFFontStyleBits;
	} CEPFFontStyle;

	//! font face informations.
	typedef struct CEPFFaceInfo 
	{
		const UTF16CHAR* familyName;			//! font-family name. this pointer is owned by ICEPFFace, DON'T free this pointer.
		UINT32		 familyNameSizeInBytes;		//! font-family name size in bytes.
		INT32		 heightInPixels;			//! This member variable indicates the size of none scalable font.
		bool		 isScalable;				//! true: scalable font.
		CEPFFontStyle style;					//! face style.
		INT32		 faceIndex;					//! font face index.
		UINT32		 unitsPerEm;				//! units per em.
		INT32		 ascent; 					//! ascent in em unit 
		INT32		 descent; 					//! descent in em unit
		INT32		 height; 					//! height in em unit
	} CEPFFaceInfo;

	//! scaled font face informations.
	typedef struct CEPFScaledFaceInfo
	{
		INT32		 height;					//! height in pixels of character cell (ascent + internal leading + descent)
		INT32		 ascent; 					//! ascent in pixels - distance from baseline of font to top of character cell 
		INT32		 descent; 					//! descent in pixels - distance from baseline of font to bottom of character cell 
		INT32		 linegap; 					//! linegap in pixels of character cell (height - ascent + descent)
		INT32		 maxCharWidth;				//! max character width in pixels 
		INT32		 underlineThickness;		//! thickness of underline in pixels. 0: not supported. 
		INT32		 strikeThroughThickness;	//! thickness of strikethrough line in pixels. 0: not supported.
	} CEPFScaledFaceInfo;

	//! CEPFGlyphLoder setting parameters.
	typedef struct CEPFGlyphLoaderParam
	{
		CEFixed26d6	 sizeInPoints;				//! size in points. 
		CEPFFontStyle glyphStyleToRender;		//! combination of eCEPFFontStyle bits.
	} CEPFGlyphLoaderParam;

	typedef struct CEPFGlyphBBox
	{
		INT32	xMin;		//! left side of bbox in pixels.
		INT32	xMax;		//! right side of bbox in pixels.
		INT32	yMin;		//! bottom side of bbox in pixels.
		INT32	yMax;		//! upper side of bbox in pixels.
	} CEPFGlyphBBox;

	typedef struct CEPFGlyphAdvance
	{
		INT32	xAdvance;	//! pen advance in the horizontal mode in pixels.
		INT32	yAdvance;	//! pen advance in the vertical mode in pixels. (just reserved. not used)
	} CEPFGlyphAdvance;

	typedef struct CEPFGlyphMetricsPt
	{
#if defined(USE_SUBPXL_ACCURATE_TEXTLAYOUT)
		// point-based metrics
		CEFixed26d6	xMinPt;		//! left side of bbox in points.
		CEFixed26d6	xMaxPt;		//! right side of bbox in points.
		CEFixed26d6	yMinPt;		//! bottom side of bbox in points.
		CEFixed26d6	yMaxPt;		//! upper side of bbox in points.
		CEFixed26d6	xAdvancePt;	//! pen advance in the horizontal mode in points.
#endif // defined(USE_SUBPXL_ACCURATE_TEXTLAYOUT)
	} CEPFGlyphMetricsPt;

	//! rasterized glyph bitmap pixel format.
	typedef enum eCEPFPixelFormat {
		eCEPFPixelFormat_FIRST,
		eCEPFPixelFormat_A1,    //! 1bit/pixel
		eCEPFPixelFormat_A8,    //! 8bit/pixel
		eCEPFPixelFormat_X1A7,  
		eCEPFPixelFormat_LAST =  eCEPFPixelFormat_X1A7,
		eCEPFPixelFormat_Unknown
	} eCEPFPixelFormat;

	//! rasterized glyph bitmap informations.
	typedef struct CEPFGlyphBitmapInfo
	{
		CEPFGlyphBBox bbox;
		CEPFGlyphAdvance advance;
		void* baseAddr;		//! base address of the glyph bitmap
		INT32 pitch;		//! the stride between rasters in bytes
		UINT16 width;		//! glyph bitmap width in pixels
		UINT16 height;		//! glyph bitmap height in pixels
		eCEPFPixelFormat pixelFormat;  //! bitmap pixel format. one of the eCEPFPixelFormat.
		CEPFFontStyle glyphStyle;  //! actual style of rendered glyph bitmap. combination of eCEPFFontStyle bits.
	} CEPFGlyphBitmapInfo;

	typedef enum eCEPFCurveTag
	{
		eCEPFCurveTag_On = 1,
		eCEPFCurveTag_Conic = 0,
		eCEPFCurveTag_Cubic = 2
	} eCEPFCurveTag;

	//! glyph outline informations.
	typedef struct CEPFGlyphOutlineInfo
	{
		CEPFGlyphBBox bbox;
		CEPFGlyphAdvance advance;
		UINT16 nContours;
		UINT16 nPoints;
		UINT16* contours;
		CEPointF* points;
		UINT8* eCEPFCurveTags;
	} CEPFGlyphOutlineInfo;

	#define CEPFFONTSTYLE_REGULAR {eCEPFFontStyleWeightClass_Normal, eCEPFFontStyleWidthClass_Medium, 0}

	inline void CEPFFontStyle_InitRegularStyle(CEPFFontStyle *const styleOut)
	{
		if (styleOut)
		{
			styleOut->weightClass = eCEPFFontStyleWeightClass_Normal;
			styleOut->widthClass = eCEPFFontStyleWidthClass_Medium;
			styleOut->eCEPFFontStyleBits = 0;
		}
	}

	inline bool CEPFFontStyleEquals(const CEPFFontStyle *const style1, const CEPFFontStyle *const style2)
	{
		if (style1 && style2)
		{
			return (
				style1->eCEPFFontStyleBits == style2->eCEPFFontStyleBits &&
				style1->weightClass == style2->weightClass &&
				style1->widthClass == style2->widthClass
				);
		}
		return false;
	}
#ifdef __cplusplus
}
#endif // __cplusplus

#endif // CEPFRasterizer_h


